/***************************************************************************
					threadengine.cpp  -  description
						 -------------------
		begin                : Tue Jul 4 2000
		copyright            : (C) 2000 by Jozef Kosoru
		email                : jozef.kosoru@pobox.sk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "control.h"
#include <cstdio>
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <ctime>
#include <new>
#include <iostream>
#include <qstring.h>
#include <qsocketnotifier.h>
#include <klocale.h>
#include <kmessagebox.h>
#include "threadengine.h"
#include "tuneroid.h"
#include "alsasound.h"
#include "pitchdetector.h"
#include "tuneroptions.h"
#include "tonedetermine.h"

ThreadEngine::ThreadEngine(QObject* parent) throw(std::runtime_error)
 : QObject(parent)
{
	if (::pipe(m_fds) == -1)
	{
		throw std::runtime_error("Unable to create pipes!");
	}

	m_notifier = new QSocketNotifier(m_fds[0], QSocketNotifier::Read);
	connect(m_notifier, SIGNAL(activated(int)), SLOT(messageHandler()));
	pthread_mutex_init(&m_messageLock, NULL);
	m_exitThread = true;
	m_toneDetermine = (static_cast<Tuneroid*>(parent))->getToneDetermine();

}

ThreadEngine::~ThreadEngine()
{
	if (!m_exitThread)
	{
		stopThread();
	}

	delete m_notifier;

	::close(m_fds[0]);
	::close(m_fds[1]);
	pthread_mutex_destroy(&m_messageLock);

#ifdef _DEBUG
	std::cerr << "ThreadEngine deleted..." << std::endl;
#endif
}

void ThreadEngine::startThread() throw(std::runtime_error)
{
	m_exitThread = false;
	m_isError = false;

	if (pthread_create(&m_thread, NULL, Thread::th_analyzer, reinterpret_cast<void*>(parent())))
	{
		throw std::runtime_error("Can't create another thread!");
	}
}

void ThreadEngine::stopThread()
{
	m_exitThread = true;

	if (pthread_join(m_thread, NULL))
	{
		throw std::runtime_error("Can't join to a thread!");
	}
}

// call from thread
void ThreadEngine::postMessage()
{
	static const char c_buf = 0;
	if (::write(m_fds[1], &c_buf, sizeof(c_buf)) == -1)
	{
		::perror("Writing to pipe");
		pthread_exit(0);
	}
}

// call from thread
void ThreadEngine::setError(const char* const errorText)
{
	pthread_mutex_lock(&m_messageLock);
	m_isError = true;
	m_sErrorText = errorText;
	pthread_mutex_unlock(&m_messageLock);
	postMessage();
}

// call from thread
void ThreadEngine::setFreq(const float freq)
{
	pthread_mutex_lock(&m_messageLock);
	m_freq = freq;
	pthread_mutex_unlock(&m_messageLock);
	postMessage();
}

void ThreadEngine::messageHandler()
{
	static char c_buf;
	if (::read(m_fds[0], &c_buf, sizeof(c_buf)) == -1)
	{
		throw std::runtime_error("Unable to read from pipe!");
	}

	pthread_mutex_lock(&m_messageLock);
    const bool bIsError = m_isError;
    if (m_isError)
    {
		m_isError = false;
	}	

    const float fFreq = m_freq;
	pthread_mutex_unlock(&m_messageLock);
	
	if (!bIsError)
	{
		emit signalIndicatorChanged(m_toneDetermine->getDivergency(fFreq));
		emit signalToneChanged(m_toneDetermine->getTone(fFreq));
		emit signalFreqChanged(fFreq);
	}
	else
	{
		emit signalThreadError(i18n(m_sErrorText.c_str()));
	}
}

////// thread
void* Thread::th_analyzer(void* ptr)
{

	Tuneroid* tuneroid = reinterpret_cast<Tuneroid*>(ptr);
	ThreadEngine* engine = tuneroid->m_engine;
	TunerOptions* options = tuneroid->m_options;

	PitchDetector_16bit* detector_16 = 0;
	PitchDetector_8bit* detector_8 = 0;

	try {

		AlsaSound soundDevice(options);

		assert(options->m_iResolution == 16 || options->m_iResolution == 8);
		const bool res16_bit = options->m_iResolution == 16;

		if (res16_bit)
		{
			detector_16 = new PitchDetector_16bit(options);
		}
		else
		{
			detector_8 = new PitchDetector_8bit(options);
		}

		// loop until exit
		while (!engine->m_exitThread)
		{
			soundDevice.capture();   // fill buffer with recorded samples
			if (res16_bit)
			{
				engine->setFreq(detector_16->analyzeBuffer());
			}
			else
			{
				engine->setFreq(detector_8->analyzeBuffer());
			}
		}

		delete detector_16;
		detector_16 = 0;
		delete detector_8;
		detector_8 = 0;

	}
	catch (const std::exception& err)
	{
		delete detector_16;
		delete detector_8;

		engine->setError(err.what());
	}

	return NULL;
}
