/***************************************************************************
					tonedetermine.cpp  -  description
						 -------------------
		begin                : Sat Jul 8 2000
		copyright            : (C) 2000 by Jozef Kosoru
		email                : jozef.kosoru@pobox.sk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "control.h"
#include <cassert>
#include <iostream>
#include "tonedetermine.h"
#include "tuneroptions.h"

ToneDetermine::ToneDetermine(const TunerOptions* options)
{
	setNewFreqOf_A(options);
}

void ToneDetermine::setNewFreqOf_A(const TunerOptions* options)
{
	m_lastFreq = -999;
	m_logFreqOf_A = std::log(static_cast<double>(options->m_freqOf_A));
	m_logToneFreq[0] = m_logFreqOf_A;
	for (int count = 1; count < 12; ++count)
	{
		m_logToneFreq[count] = m_logToneFreq[count - 1] + m_toneIncrease;
	}
}

ToneDetermine::~ToneDetermine()
{

}

int ToneDetermine::getTone(const float freq)
{
	if (m_lastFreq != freq)
	{
		analyzeFreq(freq);
	}

	return m_tone;
}

int ToneDetermine::getDivergency(const float freq)
{
	if (m_lastFreq != freq)
	{
		analyzeFreq(freq);
	}

	return m_divergence;
}

void ToneDetermine::analyzeFreq(const float freq)
{
	m_lastFreq = freq;
	if (freq == -1)
	{
		m_divergence = 999; //OFF
		m_tone = 999;       //OFF
		return;
	}

	double logFreq = std::log(static_cast<double>(freq));
	const double bottomLimit = m_logFreqOf_A - (m_toneIncrease / 2);
	const double topLimit = m_logFreqOf_A + (m_log2 + (m_toneIncrease / 2));

	while (logFreq < bottomLimit)
	{
		logFreq += m_log2;
	}

	while (logFreq >= topLimit)
	{
		logFreq -= m_log2;
	}

	double totalDiverg = m_toneIncrease / 2;
	int resultTone = 0;
	for (int count = 0; count < 12; ++count)
	{
		double curDiverg = std::fabs(m_logToneFreq[count] - logFreq);
		if (totalDiverg > curDiverg)
		{
			totalDiverg = curDiverg;
			resultTone = count;
		}
	}

	// Compute values for TunerDiplay Indicator
	// <-32,0,32> , 999=OFF
	const double oneIndic = (m_toneIncrease / 2) / 32;
	if (logFreq < m_logToneFreq[resultTone])
	{	// -32 -- 0
		m_divergence = -roundToInt(totalDiverg / oneIndic);
		assert(m_divergence > -33 && m_divergence < 1);
	}
	else
	{                                  // 0 -- 32
		m_divergence = roundToInt(totalDiverg / oneIndic);
		assert(m_divergence > -1 && m_divergence < 33);
	}

	// Convert tones to a TunerDisplay specification
	// here: 0=A 11=#G
	// TunerDisplay: 0=C 11=B
	resultTone += 9;
	if (resultTone > 11)
	{
		resultTone -= 12;
	}

	assert(resultTone > -1 && resultTone < 12);
	m_tone = resultTone;
#ifdef _DEBUG_TOD
	std::cerr << "Tone: " << m_tone << std::endl;
#endif
}

const double ToneDetermine::m_toneIncrease = M_LN2 / 12.0;	//Log(2)/12
const double ToneDetermine::m_log2 = M_LN2;					//Log(2)
