/***************************************************************************
					tunerdisplay.cpp  -  description
						 -------------------
		begin                : Fri Feb 4 2000
		copyright            : (C) 2000 by Jozef Kosoru
		email                : jozef.kosoru@pobox.sk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "control.h"
#include <cassert>
#include <cstdio>
#include <qpixmap.h>
#include <qpainter.h>
#include <qtimer.h>
#include <kstandarddirs.h>
#include <kglobal.h>
#include "tunerdisplay.h"

#define DISPLAY_H_SIZE  493
#define DISPLAY_V_SIZE  247
#define DISPLAY_OFF     "pics/Display-OFF.png"
#define DISPLAY_ON      "pics/Display-ON.png"
#define DISPLAY_TONE    "pics/Display-TONE.png"
#define DISPLAY_DIGITS  "pics/Display-DIGITS.png"
#define DIGIT_SIZE_W    20
#define DIGIT_SIZE_H    28
#define POINT_H_POS     244
#define POINT_V_POS     127
#define POINT_SIZE      4
#define X_TONE_CROP_OFFSET  (-71)
#define Y_TONE_CROP_OFFSET  (-141)
#define SLOW_BLINK      400   // msec
#define FAST_BLINK      150

const short int TunerDisplay::s_LIndicator[32][4]=
{
	{237,47,241,79},  //-1
	{231,47,235,79},  //-2
	{226,47,229,79},  //-3
	{220,47,224,79},  //-4
	{214,47,218,79},  //-5
	{209,47,212,80},  //-6
	{203,47,207,81},  //-7
	{197,47,201,81},  //-8
	{191,47,195,82},  //-9
	{185,47,189,83},  //-10
	{180,48,183,84},  //-11
	{174,48,178,85},  //-12
	{168,49,172,86},  //-13
	{162,50,166,88},  //-14
	{156,51,160,90},  //-15
	{150,52,154,92},  //-16
	{144,53,148,94},  //-17
	{138,54,142,97},  //-18
	{132,55,136,99},  //-19
	{126,56,130,102}, //-20
	{120,58,124,104}, //-21
	{114,59,118,107}, //-22
	{108,61,112,110}, //-23
	{102,63,106,113}, //-24
	{95,65,100,117},  //-25
	{88,67,93,121},   //-26
	{80,71,86,126},   //-27
	{70,75,78,133},   //-28
	{59,81,68,141},   //-29
	{47,88,56,150},   //-30
	{34,97,44,161},   //-31
	{19,107,31,174}   //-32
};

const short int TunerDisplay::s_RIndicator[32][4]=
{
	{251,47,255,79},  //1
	{257,47,260,79},  //2
	{262,47,266,79},  //3
	{268,47,271,79},  //4
	{273,47,277,79},  //5
	{279,47,283,79},  //6
	{285,47,289,80},  //7
	{291,47,295,80},  //8
	{297,47,300,81},  //9
	{302,47,306,83},  //10
	{308,48,312,84},  //11
	{314,48,318,85},  //12
	{320,49,324,87},  //13
	{326,50,330,88},  //14
	{332,51,336,90},  //15
	{338,51,342,92},  //16
	{344,52,348,94},  //17
	{350,54,354,97},  //18
	{356,55,360,99},  //19
	{362,56,366,102}, //20
	{368,58,372,104}, //21
	{374,59,378,107}, //22
	{380,61,384,110}, //23
	{386,63,390,113}, //24
	{392,65,396,117}, //25
	{398,67,403,121}, //26
	{406,71,412,126}, //27
	{414,75,421,133}, //28
	{424,81,433,141}, //29
	{435,88,445,150}, //30
	{447,97,458,161}, //31
	{461,107,473,174} //32
};

const short int TunerDisplay::s_TUNEDmarks[8][4]=
{
	{226,88,265,98},  //0 TUNED (Display-OFF)
	{203,18,224,42},  //1 Left Green Arrow
	{233,18,258,42},  //2 Red Circle
	{267,18,288,42},  //3 Right Geen Arrow

	{207,41,246,49},  //4 TUNED (Display-ON)
	{184,64,205,88},  //5 Left Green Arrow
	{214,64,239,88},  //6 Red Circle
	{248,64,269,88}   //7 Right Geen Arrow
};

const short int TunerDisplay::s_TonePosition[12][4]=
{
	{71,188,111,227},   //0 C
	{97,141,137,180},   //1 C#
	{123,188,162,227},  //2 D
	{149,141,188,180},  //3 D#
	{175,188,214,227},  //4 E
	{226,188,265,227},  //5 F
	{252,141,291,180},  //6 F#
	{278,188,317,227},  //7 G
	{303,141,343,180},  //8 G#
	{329,188,368,227},  //9 A
	{355,141,394,180},  //10  A#
	{381,188,420,227}   //11  B
};

const short int TunerDisplay::s_DigitPosition[7][2] =
{
	{164,103},
	{184,103},
	{204,103},
	{224,103},
	{253,103},
	{273,103},
	{293,103}
};

void TunerDisplay::resetDisplay()
{
	slotChangeIndicator(999);
	slotChangeTone(999);
	slotChangeFrequency(-1);
}

TunerDisplay::TunerDisplay(QWidget *parent, const char *name )
 : QFrame(parent, name)
{
	// set frame style
	setFrameStyle(Panel | Raised);
	setLineWidth(2);
	setMidLineWidth(0);
	setBackgroundMode(NoBackground);
	m_frameWidth = frameWidth();

	const KStandardDirs* const dirs = KGlobal::dirs();
	// load pixmaps
	m_display_OFF = new QPixmap(dirs->findResource("appdata", DISPLAY_OFF), "PNG",
								ColorOnly | DiffuseDither);
	m_display_OFF->setOptimization(QPixmap::NoOptim);

	m_display_ON = new QPixmap(dirs->findResource("appdata", DISPLAY_ON), "PNG",
							   ColorOnly | DiffuseDither);
	m_display_ON->setOptimization(QPixmap::NoOptim);

	m_display_TONE = new QPixmap(dirs->findResource("appdata", DISPLAY_TONE), "PNG",
								 ColorOnly | DiffuseDither);
	m_display_TONE->setOptimization(QPixmap::NoOptim);

	m_display_DIGITS = new QPixmap(dirs->findResource("appdata", DISPLAY_DIGITS), "PNG",
								   ColorOnly | DiffuseDither);

	m_tempDigit = new QPixmap(DIGIT_SIZE_W, DIGIT_SIZE_H);

	// default values
	m_LArrow = m_RArrow = true;
	m_Indicator = 999;
	m_LArrow_c = m_RArrow_c = false;
	m_Indicator_c = m_Tone_c = 999;
	m_freq_c = m_freq = -1;

	// make timer
	m_blinkArrowTimer = new QTimer(this);
	connect(m_blinkArrowTimer, SIGNAL(timeout()), this, SLOT(slotDrawBlinkArrow()));
	m_blinkArrowTimer->start(SLOW_BLINK);   //start default blinking of arrows

	// set size
	setFixedSize(DISPLAY_H_SIZE + (m_frameWidth * 2), DISPLAY_V_SIZE + (m_frameWidth * 2));
}

TunerDisplay::~TunerDisplay()
{
	delete m_display_OFF;
	delete m_display_ON;
	delete m_display_TONE;
	delete m_display_DIGITS;
	delete m_tempDigit;
#ifdef _DEBUG_TD
		debug("TunerDisplay delete...");
#endif
}

void TunerDisplay::paintEvent(QPaintEvent* e)
{
	// draw background
	QRect tpmRect(e->rect());
	tpmRect.moveBy(-m_frameWidth, -m_frameWidth);
	bitBlt(this, e->rect().topLeft(), m_display_OFF, tpmRect, CopyROP, true);

	// draw Indicators
	redrawON();

	// draw Frame
	QPainter p(this);
	drawFrame(&p);
}

void TunerDisplay::updateON()
{
	int firstGrayBar = 0, lastGrayBar = 0;
	int firstLightBar = 0, lastLightBar = 0;

	if (m_Indicator == 0 || m_Indicator == 999)
	{
		// TUNED
		if (!m_Indicator && m_Indicator_c)
		{
			drawTUNED(true);
		}
		else if (m_Indicator == 999 && m_Indicator_c != 999)
		{
			drawTUNED(false);
		}

		// Indicators
		firstLightBar = 999;
		if (m_Indicator_c == 999)
		{
			 firstGrayBar = 999;
		}
		else if (m_Indicator_c < 0)
		{
			firstGrayBar = m_Indicator_c;
			lastGrayBar = -1;
		}
		else if (m_Indicator_c > 0)
		{
			firstGrayBar = 1;
			lastGrayBar = m_Indicator_c;
		}
	}
	else if (m_Indicator_c == 0 || m_Indicator_c == 999)
	{
		// TUNED
		if (!m_Indicator_c && m_Indicator)
		{
			drawTUNED(false);
		}

		// Indicators
		firstGrayBar = 999;

		if (m_Indicator == 999)
		{
			firstLightBar = 999;
		}
		else if (m_Indicator < 0)
		{
			firstLightBar = m_Indicator;
			lastLightBar = -1;
		}
		else if (m_Indicator > 0)
		{
			firstLightBar = 1;
			lastLightBar = m_Indicator;
		}
	}
	else if (m_Indicator_c < 0 && m_Indicator < 0)
	{
		if (m_Indicator_c < m_Indicator)
		{
			firstGrayBar = m_Indicator_c;
			lastGrayBar = m_Indicator - 1;
			firstLightBar = 999;	// we don't need to redraw light bars
		}
		else
		{
			firstGrayBar = 999;		// we don't need to redraw gray bars
			firstLightBar = m_Indicator;
			lastLightBar = m_Indicator_c - 1;
		}
	}
	else if (m_Indicator_c > 0 && m_Indicator > 0)
	{
		if (m_Indicator_c > m_Indicator)
		{
			firstGrayBar = m_Indicator + 1;
			lastGrayBar = m_Indicator_c;
			firstLightBar = 999;
		}
		else
		{
			firstGrayBar = 999;
			firstLightBar = m_Indicator_c + 1;
			lastLightBar = m_Indicator;
		}
	}
	else if (m_Indicator_c > 0 && m_Indicator < 0)
	{
		firstGrayBar = 1;
		lastGrayBar = m_Indicator_c;
		firstLightBar = m_Indicator;
		lastLightBar = -1;
	}
	else
	{	// if((m_Indicator_c<0) && (m_Indicator>0)){
		firstGrayBar = m_Indicator_c;
		lastGrayBar = -1;
		firstLightBar = 1;
		lastLightBar = m_Indicator;
	}

	////////////// draw bars /////////////////////////////
	if (firstGrayBar != 999)
	{
		for (int counter = firstGrayBar; counter <= lastGrayBar; ++counter)
		{
			drawGrayBar(counter);
		}
	}

	if (firstLightBar != 999)
	{
		for (int counter = firstLightBar; counter<=lastLightBar; ++counter)
		{
			drawLightBar(counter);
		}
	}
}

void TunerDisplay::drawGrayBar(int numOfBar)
{
	bitBlt(this, getGrayPoint(numOfBar), m_display_OFF, getGrayRect(numOfBar), CopyROP, true);
}

void TunerDisplay::drawLightBar(int numOfBar)
{
	bitBlt(this, getGrayPoint(numOfBar), m_display_ON, getLightRect(numOfBar), CopyROP, true);
}


void TunerDisplay::redrawON()
{
	// TUNED
	if (m_LArrow_c)
	{
		drawLEFTarrow(true);
	}

	if (m_RArrow_c)
	{
		drawRIGHTarrow(true);
	}

	// Tone
	if (m_Tone_c != 999)
	{
		drawTone(m_Tone_c, true);
	}

	// Frequency
	drawFreq(true);

	// Indicator
	if (m_Indicator == 999)
	{
		return;
	}

	if (!m_Indicator)
	{
		drawTUNED(true);
	}

	int firstLightBar, lastLightBar;

	if (m_Indicator < 0)
	{
		firstLightBar = m_Indicator;
		lastLightBar = -1;
	}
	else
	{
		firstLightBar = 1;
		lastLightBar = m_Indicator;
	}

	for (int counter = firstLightBar; counter <= lastLightBar; ++counter)
	{
		drawLightBar(counter);
	}
}

void TunerDisplay::slotChangeIndicator(int newValue)
{
#ifdef _DEBUG_TD
		debug("Indicator=%d", newValue);
		assert((newValue > -33 && newValue < 33) || newValue == 999);
#endif

	if (m_Indicator_c == newValue)
	{
		return;
	}

	m_Indicator = newValue;

	// draw Indicator
	updateON();

	// draw blinking arrows
	if (!m_Indicator)
	{
		drawLEFTarrow(false);
		drawRIGHTarrow(false);
		m_LArrow_c = m_RArrow_c = false;
		m_blinkArrowTimer->stop();
	}
	else if(m_Indicator == 999)
	{
		m_LArrow = m_RArrow = true;
		m_LArrow_c = m_RArrow_c = false;
		m_blinkArrowTimer->changeInterval(SLOW_BLINK);
	}
	else if (m_Indicator < 0 && (m_Indicator_c >= 0 || m_Indicator_c == 999))
	{
		m_LArrow = m_RArrow_c = false;
		m_RArrow = m_LArrow_c = true;
		m_blinkArrowTimer->changeInterval(FAST_BLINK);
	}
	else if (m_Indicator > 0 && (m_Indicator_c <= 0 || m_Indicator_c == 999))
	{
		m_LArrow = m_RArrow_c = true;
		m_RArrow = m_LArrow_c = false;
		m_blinkArrowTimer->changeInterval(FAST_BLINK);
	}

	m_Indicator_c = m_Indicator;
}

void TunerDisplay::slotChangeTone(int tone)
{
#ifdef _DEBUG_TD
	debug("Tone=%d", tone);
	assert((tone > -1 && tone < 12) || tone == 999);
#endif

	if(tone == m_Tone_c)
	{
		return;
	}
	else if( tone == 999 && m_Tone_c != 999)
	{
		drawTone(m_Tone_c, false);
		m_Tone_c = 999;
	}
	else
	{
		drawTone(tone, true);

		if (m_Tone_c != 999)
		{
#ifdef _DEBUG_TD
			debug("OFF: %d", m_Tone_c);
#endif
			drawTone(m_Tone_c, false);
		}

		m_Tone_c = tone;
	}
}

void TunerDisplay::slotChangeFrequency(float freq)
{
	if (freq == m_freq_c)
		return;

	m_freq = freq;

	if (freq != -1)
	{
		sprintf(m_freqStr, "%#8.3f", freq);
#ifdef _DEBUG_TD
		debug("FREQ=%s", m_freqStr);
#endif
	}

	///// draw frequency
	drawFreq(false);
}

void TunerDisplay::drawTUNED(bool light)
{
	const QPoint topLeftTUNED(s_TUNEDmarks[0][0] + m_frameWidth, s_TUNEDmarks[0][1] + m_frameWidth);
	const QPoint topLeftRED(s_TUNEDmarks[2][0] + m_frameWidth, s_TUNEDmarks[2][1] + m_frameWidth);

	if (light)
	{	// draw ON

		{	// TUNED
			const QPoint tmpPoint(s_TUNEDmarks[4][0], s_TUNEDmarks[4][1]);
			const QRect rrect(tmpPoint, QPoint(s_TUNEDmarks[4][2], s_TUNEDmarks[4][3]));
			bitBlt(this, topLeftTUNED, m_display_ON, rrect, CopyROP, true);
		}

		{	// Red Circle
			const QPoint tmpPoint(s_TUNEDmarks[6][0], s_TUNEDmarks[6][1]);
			const QRect rrect(tmpPoint, QPoint(s_TUNEDmarks[6][2], s_TUNEDmarks[6][3]));
			bitBlt(this, topLeftRED, m_display_ON, rrect, CopyROP, true);
		}
	}
	else
	{	// draw OFF

		{	// TUNED
			const QPoint tmpPoint(s_TUNEDmarks[0][0], s_TUNEDmarks[0][1]);
			const QRect rrect(tmpPoint, QPoint(s_TUNEDmarks[0][2], s_TUNEDmarks[0][3]));
			bitBlt(this, topLeftTUNED, m_display_OFF, rrect, CopyROP, true);
		}

		{	// Red Circle
			const QPoint tmpPoint(s_TUNEDmarks[2][0], s_TUNEDmarks[2][1]);
			const QRect rrect(tmpPoint, QPoint(s_TUNEDmarks[2][2], s_TUNEDmarks[2][3]));
			bitBlt(this, topLeftRED, m_display_OFF, rrect, CopyROP, true);
		}
	}
}

void TunerDisplay::drawLEFTarrow(bool light)
{
	const QPoint topLeftTUNED(s_TUNEDmarks[1][0] + m_frameWidth, s_TUNEDmarks[1][1] + m_frameWidth);

	if (light)
	{	// draw ON
		const QPoint tmpPoint(s_TUNEDmarks[5][0], s_TUNEDmarks[5][1]);
		const QRect rrect(tmpPoint, QPoint(s_TUNEDmarks[5][2], s_TUNEDmarks[5][3]));
		bitBlt(this, topLeftTUNED, m_display_ON, rrect, CopyROP, true);
	}
	else
	{	// draw OFF
		const QPoint tmpPoint(s_TUNEDmarks[1][0], s_TUNEDmarks[1][1]);
		const QRect rrect(tmpPoint, QPoint(s_TUNEDmarks[1][2], s_TUNEDmarks[1][3]));
		bitBlt(this, topLeftTUNED, m_display_OFF, rrect, CopyROP, true);
	}
}

void TunerDisplay::drawRIGHTarrow(bool light)
{
	const QPoint topLeftTUNED(s_TUNEDmarks[3][0] + m_frameWidth, s_TUNEDmarks[3][1] + m_frameWidth);

	if (light)
	{	// draw ON
		const QPoint tmpPoint(s_TUNEDmarks[7][0], s_TUNEDmarks[7][1]);
		const QRect rrect(tmpPoint, QPoint(s_TUNEDmarks[7][2], s_TUNEDmarks[7][3]));
		bitBlt(this, topLeftTUNED, m_display_ON, rrect, CopyROP, true);
	}
	else
	{	// draw OFF
		const QPoint tmpPoint(s_TUNEDmarks[3][0], s_TUNEDmarks[3][1]);
		const QRect rrect(tmpPoint, QPoint(s_TUNEDmarks[3][2], s_TUNEDmarks[3][3]));
		bitBlt(this, topLeftTUNED, m_display_OFF, rrect, CopyROP, true);
	}
}

void TunerDisplay::slotDrawBlinkArrow()
{
	// leftArrow
	if (m_LArrow && !m_LArrow_c)
	{
		drawLEFTarrow(true);
		m_LArrow_c = true;
	}
	else if ((m_LArrow && m_LArrow_c) || (!m_LArrow && m_LArrow_c))
	{
		drawLEFTarrow(false);
		m_LArrow_c = false;
	}

	// rightArrow
	if (m_RArrow && !m_RArrow_c)
	{
		drawRIGHTarrow(true);
		m_RArrow_c = true;
	}
	else if ((m_RArrow && m_RArrow_c) || (!m_RArrow && m_RArrow_c))
	{
		drawRIGHTarrow(false);
		m_RArrow_c = false;
	}
}

void TunerDisplay::drawTone(int tone, bool light)
{
	const QPoint topLeftPos(s_TonePosition[tone][0] + m_frameWidth,
							s_TonePosition[tone][1] + m_frameWidth);
	if (light)
	{	// draw ON
		const QPoint topLeftPoint(s_TonePosition[tone][0] + X_TONE_CROP_OFFSET,
								  s_TonePosition[tone][1] + Y_TONE_CROP_OFFSET);
		const QPoint bottomRightPoint(s_TonePosition[tone][2] + X_TONE_CROP_OFFSET,
									  s_TonePosition[tone][3] + Y_TONE_CROP_OFFSET);
		const QRect rrect(topLeftPoint, bottomRightPoint);

		bitBlt(this, topLeftPos, m_display_TONE, rrect, CopyROP, true);
	}
	else
	{	// draw OFF
		const QPoint topLeftPoint(s_TonePosition[tone][0],
								  s_TonePosition[tone][1]);
		const QPoint bottomRightPoint(s_TonePosition[tone][2],
									  s_TonePosition[tone][3]);
		const QRect rrect(topLeftPoint, bottomRightPoint);

		bitBlt(this, topLeftPos, m_display_OFF, rrect, CopyROP, true);
	}
}

void TunerDisplay::drawFreq(bool redraw)
{
	if (m_freq == -1)
	{	// -1 means: switch off whole number display

		if (!redraw)
		{
			// draw gray numbers
			for(int count = 0; count < 7; ++count)
			{
				const QPoint topLeftPos(s_DigitPosition[count][0] + m_frameWidth,
										s_DigitPosition[count][1] + m_frameWidth);
				const QRect rrect(s_DigitPosition[count][0], s_DigitPosition[count][1],
								  DIGIT_SIZE_W,DIGIT_SIZE_H);

				bitBlt(this, topLeftPos, m_display_OFF, rrect, CopyROP, true);
			}

			// draw gray point "."
			const QPoint topLeftPos(POINT_H_POS + m_frameWidth, POINT_V_POS + m_frameWidth);
			const QRect rrect(POINT_H_POS, POINT_V_POS, POINT_SIZE, POINT_SIZE);

			bitBlt(this, topLeftPos, m_display_OFF, rrect, CopyROP, true);
		}
	}
	else
	{	// draw appropriate number

		if (!redraw)
		{	// update (draw also background)

			const QPoint zeroPoint(0, 0);
			const QRect digitRect(0, 0, DIGIT_SIZE_W, DIGIT_SIZE_H);

			for (int count = 0; count < 7; ++count)
			{
				const QRect rrect(s_DigitPosition[count][0], s_DigitPosition[count][1],
								  DIGIT_SIZE_W,DIGIT_SIZE_H);

				bitBlt(m_tempDigit, zeroPoint, m_display_OFF, rrect, CopyROP, true);

				if (!(count < 3 && m_freqStr[count] == ' '))
				{
					const int digOffset = (count < 4) ? count : count + 1;

					assert(m_freqStr[digOffset] >= '0' && m_freqStr[digOffset] <= '9');

					const int digit = m_freqStr[digOffset] - '0';
					const QRect rrect2(digit * DIGIT_SIZE_W, 0, DIGIT_SIZE_W, DIGIT_SIZE_H);

					bitBlt(m_tempDigit, zeroPoint, m_display_DIGITS, rrect2, CopyROP, false);
				}

				const QPoint topLeftPos(s_DigitPosition[count][0] + m_frameWidth,
										s_DigitPosition[count][1] + m_frameWidth);

				bitBlt(this, topLeftPos, m_tempDigit, digitRect, CopyROP, true);
			}
		}
		else
		{	// redraw (don't draw background)

			for (int count = 0; count < 7; ++count)
			{
				if (count < 3 && m_freqStr[count] == ' ')
					continue;

				const QPoint topLeftPos(s_DigitPosition[count][0] + m_frameWidth,
										s_DigitPosition[count][1] + m_frameWidth);

				const int digOffset = (count < 4) ? count : count+1;

				assert(m_freqStr[digOffset] >= '0' && m_freqStr[digOffset] <= '9');

				const int digit = m_freqStr[digOffset] - '0';
				const QRect rrect(digit * DIGIT_SIZE_W, 0, DIGIT_SIZE_W, DIGIT_SIZE_H);

				bitBlt(this, topLeftPos, m_display_DIGITS, rrect, CopyROP, false);
			}
		}

		// draw blue point "."
		if (redraw || m_freq_c == -1)
		{
			const QPoint topLeftPos(POINT_H_POS + m_frameWidth, POINT_V_POS + m_frameWidth);
			const QRect rrect(10 * DIGIT_SIZE_W, DIGIT_SIZE_H - POINT_SIZE, POINT_SIZE, POINT_SIZE);

			bitBlt(this, topLeftPos, m_display_DIGITS, rrect, CopyROP, true);

			assert(m_freqStr[4] == '.');
		}
	}

	m_freq_c = m_freq;
}
