/*
   Copyright (C) 2001-2008 Thomas Capricelli <orzel@freehackers.org>
*/

/* Qt */
#include <qpainter.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <qmessagebox.h>
#include <qspinbox.h>
#include <qpushbutton.h>
#include <qdom.h>
#include <qfile.h>
#include <QList>
#include <QTextStream>
#include <QTextCodec>

/* KDE */
#ifdef ENABLE_KDE
#include <kdebug.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kdebug.h>
#endif

/* Opale */
#include "document.h"
#include "ui_fileoptions.h"
#include "listboxeditorimpl.h"
#include "ui_replacedialog.h"
#include "ui_recordDialog.h"
#include "ui_pattern.h"
#include "ui_repeat.h"



static OpaleRecord * item2record( QTreeWidgetItem *item )
{
//	Q_ASSERT(it->inherits("OpaleRecord"));
	return ( OpaleRecord* )item ;
}




/*
 * OpalDatae
 */
OpaleDocument::OpaleDocument( QWidget* parent)
	: QTreeWidget( parent)
{
	resetData();
    setHeaderLabels(QStringList()<<
		tr("Date") << 		// 0
		tr("Description") << 
		tr("Amount") << 	// 2
		tr("X") << 
		tr("Method") << 	// 4
		tr("Future?") << 
		tr("Balance") << 	// 6
		tr("Category")
	);
	setAlternatingRowColors(true);
	setRootIsDecorated(false);

	qApp->setStyleSheet("QTreeView { alternate-background-color: #bebeff; }");

	connect( this, SIGNAL(itemActivated(QTreeWidgetItem*,int)), this, SLOT(editRecord(QTreeWidgetItem*,int)));
	connect( this, SIGNAL(itemClicked( QTreeWidgetItem*, int)), this, SLOT(itemclicked( QTreeWidgetItem*, int)));
}


void OpaleDocument::resetData(void)
{
	// new empty..
	b_modified=false;
	s_currency =  tr("Euro");
	s_account_name =  tr("Bank's name");

	categorySList.clear();
	categorySList.append(tr("Misc"));
	emit categoriesUpdated();
	methodSList.clear();
	methodSList.append(tr("Check"));
	emit methodsUpdated();
	patterns.clear(); syncPatterns();
	clear();
	emit listUpdated();
}



bool OpaleDocument::loadXML(const QDomDocument &doc )
{
	QDomNode n;
	int i=0;

	qDebug("begin of loadXML");
	/* check document type */
	if ( doc.doctype().name() != "opale") {
		qDebug("doc.doctype().name() != opaleL");
		return false;
		}

	/* check mimetype */
	qDebug("check mimetype");
	QDomElement account = doc.documentElement();
	if ( account.attribute( "mime" ) != "application/x-opale" ) {
		qDebug("account.attribute( mime ) != application/x-opale");
		return false;
		}


	/* read account information */
	s_currency = account.attribute("currency");
	if ("" == s_currency ) s_currency="Euros";
	s_account_name = account.attribute("account_name");

	/* read the method section */
	qDebug("read the method section... ");
	QDomElement methodlist = account.namedItem("howlist").toElement();
	methodSList.clear();
	n = methodlist.firstChild();
	i=0;
	while (!n.isNull()) {
		QDomElement e = n.toElement();
		if (i!=e.attribute("num").toInt()) {
			qDebug("num=%d in method : %s element, %d was expected, file is corrupted", e.attribute("num").toInt(), qPrintable(e.attribute("name")),i);
			return false;
		}
		i++;
		methodSList.append(e.attribute("name"));
		n = n.nextSibling();
	}
	qDebug("%d elements", i);
	emit methodsUpdated();

	/* read the category section */
	qDebug("read the category section... ");
	QDomElement categorylist = account.namedItem("categorylist").toElement();
	categorySList.clear();
	n = categorylist.firstChild();
	i=0;
	while (!n.isNull()) {
		QDomElement e = n.toElement();
		if (i!=e.attribute("num").toInt()) {
			qDebug("num=%d in categorie: %s element, %d was expected, file is corrupted", e.attribute("num").toInt(), qPrintable(e.attribute("name")),i);
			return false;
		}
		i++;
		categorySList.append(e.attribute("name"));
		n = n.nextSibling();
	}
	qDebug("%d elements", i);
	emit categoriesUpdated();

	/* read the patterns section */
	qDebug("read the patterns section...");
	QDomElement patternslist = account.namedItem("patternlist").toElement();

	patterns.clear();
	n = patternslist.firstChild();
	i = 0;
	while (!n.isNull()) {
		patterns.append( OpalePattern(this, n.toElement()) );
		n = n.nextSibling();
		i++;
	}
	syncPatterns(); // one single emit at the end
	qDebug("%d elements", i);

	/* read the data section */
	qDebug("read the data section...");
	QDomElement records = account.namedItem("recordlist").toElement();

	clear();
	n = records.firstChild();
	i=0;
	while (!n.isNull()) {
		new OpaleRecord(this, n.toElement());
		n = n.nextSibling();
		i++;
	}
	emit listUpdated();
	qDebug("%d elements", i);

	qDebug("end of loadXML");

	// by priority, last one is the 
	resizeColumnToContents(0); // date
	resizeColumnToContents(1); // description
	resizeColumnToContents(2); // amount
	resizeColumnToContents(3); // checked
	resizeColumnToContents(4); // method
	resizeColumnToContents(5); // future?
	resizeColumnToContents(6); //balance
	resizeColumnToContents(7); // category
	setModified(false);
	emit listUpdated();
	gotoNow();
	return true;
}


/*

<?xml version="1.0" encoding="UTF-8"?><!DOCTYPE opale >
<account
	mime="application/x-opale"
	editor="Opale"
	currency="Euro" 
	account_name="bank_name" >

	<howlist>
		<how name="Check" num="0" />
		<how name="asdsf" num="1" />
		<how name="asdf" num="2" />
	</howlist>

	<categorylist>
		<category name="Check" num="0" />
		<category name="asdsf" num="1" />
		<category name="asdf" num="2" />
	</categorylist>

	<data>
		<record amount="12300" when="2008-06-16" what="Burps pour la shbrob" how="3" category="1" />
		<record amount="12300" when="2008-06-16" what="Burps pour la shbrob" how="3" category="1" />
		<record amount="12300" when="2008-06-16" what="Burps pour la shbrob" how="3" category="1" />
	</data>
</account>


 */

QDomDocument OpaleDocument::saveXML()
{
	QDomDocument doc("opale");
	doc.appendChild( doc.createProcessingInstruction( "xml", "version=\"1.0\" encoding=\"UTF-8\"" ) );
	QDomElement account = doc.createElement( "account" ); 
	account.setAttribute( "mime",	"application/x-opale" );
	account.setAttribute( "editor",	"Opale" );
	account.setAttribute( "account_name", s_account_name );
	account.setAttribute( "currency", s_currency);
	doc.appendChild( account );

	/* record the method names */
	QDomElement methodlist = doc.createElement( "howlist" ); 

	for ( int i=0; i< methodSList.count(); i++) {
		QDomElement record = doc.createElement( "how" );
		record.setAttribute("name", methodSList[i]);
		record.setAttribute("num", i);
		methodlist.appendChild(record);
        }
	account.appendChild(methodlist);

	/* record the category names */
	QDomElement categorylist = doc.createElement( "categorylist" ); 

	for ( int i=0; i< categorySList.count(); i++) {
		QDomElement record = doc.createElement( "category" );
		record.setAttribute("name", categorySList[i]);
		record.setAttribute("num", i);
		categorylist.appendChild(record);
        }
	account.appendChild(categorylist);

	/* record the patterns */
	QDomElement patternslist = doc.createElement( "patternlist" ); 
	foreach (OpalePattern p, patterns)
		patternslist.appendChild( p.domElement(doc));
	account.appendChild(patternslist);


	/* record the data */
	QDomElement qde = doc.createElement( "recordlist" ); 
	for (int i=0; i< topLevelItemCount() ;i ++)
		qde.appendChild( item2record(topLevelItem(i))->domElement(doc));
	account.appendChild(qde);

	setModified(false);
	return doc;
}

void OpaleDocument::setModified(bool f)
{
	b_modified=f;
	sortItems(0,Qt::DescendingOrder);
}

void OpaleDocument::fillDataStats(StatsData &statsData, QDate begin , QDate end)
{
	statsData.methodPositive.resize(methodSList.count());
	statsData.categoryPositive.resize( categorySList.count() );
	statsData.methodNegative.resize(methodSList.count());
	statsData.categoryNegative.resize( categorySList.count() );

	// clear them all
	int i;
	for (i=0; i< statsData.methodPositive.count(); i++)
		statsData.methodPositive[i] = 0;
	for (i=0; i< statsData.methodNegative.count(); i++)
		statsData.methodNegative[i] = 0;
	for (i=0; i< statsData.categoryPositive.count(); i++)
		statsData.categoryPositive[i] = 0;
	for (i=0; i< statsData.categoryNegative.count(); i++)
		statsData.categoryNegative[i] = 0;

	// We start from the end of the list, which corresponds to the
	// start in the time
	for (int i=topLevelItemCount()-1; i>=0 ;i--) {
		OpaleRecord *r = item2record(topLevelItem(i));
		
		// the end;
		if (r->date>end) break;

		// begin
		if (r->date>begin)   {
			//fill
			amount_t m = r->m_amount;
			if ( m>0 ) {
				statsData.methodPositive[r->method]+= m;
				statsData.categoryPositive[r->category]+= m;
			} else {
				m = -m;
				statsData.methodNegative[r->method]+= m;
				statsData.categoryNegative[r->category]+= m;
			}
		}
	} // for
}

void OpaleDocument::fillDataGraph(QList<GraphData> &chartData  , QDate begin, QDate end, amount_t &a_min, amount_t &a_max)
{
	GraphData gdata;
	bool onemore=false;

	a_min = a_max = 0; 
	chartData.clear();

	// We start from the end of the list, which corresponds to the
	// start in the time
	for (int i=topLevelItemCount()-1; i>=0 ;i--) {
		OpaleRecord *r = item2record(topLevelItem(i));

		// add the previous one, so that we are sure to have one
		// point _before_ the actual "begin".
		if (r->date>begin) chartData.append(gdata);

		// compute
		r->addToChart(gdata);

		// min/max
		if (begin < r->date &&  r->date <= end) {
#ifdef DISPLAY_BEST_CASE 
			if (gdata.best>a_max) a_max = gdata.best;
#else
			if (gdata.balance>a_max) a_max = gdata.balance;
#endif
			if (gdata.worst<a_min) a_min = gdata.worst;
		}

		// the end (after everything, so that we got one more point)
		if (r->date>end) {
			if (onemore)
				break;
			else onemore=true;
		}
	} // for
//    qDebug("OpaleDocument::fillDataGraph  end  min/max = " AMOUNT_ESCAPE "/" AMOUNT_ESCAPE, a_min, a_max);

	// safeguard
	if (a_max == a_min ) {
		a_min = -10 * 100;
		a_max = +10 * 100;
	}
//    qDebug("OpaleDocument::fillDataGraph  safeguard min/max = " AMOUNT_ESCAPE "/" AMOUNT_ESCAPE, a_min, a_max);
}

void OpaleDocument::updateTotal(Total &total)
{
	total.global = 0l;	total.utt = 0l;
	total.uttsafe = 0l;	total.checked = 0l;

	for (int i=topLevelItemCount()-1; i>=0 ;i--)
		item2record(topLevelItem(i))->updateTotal( total );
}


void OpaleDocument::fillCombos(QComboBox *method, QComboBox *category)
{
	QStringList::iterator it;

	for (int i=0; i<methodSList.count(); i++)
		method->insertItem( i, methodSList[i]);
	for (int i=0; i<categorySList.count(); i++)
		category->insertItem( i, categorySList[i]);
}


int OpaleDocument::checkMethodCombos(QComboBox *method)
{
// XXX orzel : re-enable auto-add
/*
	int i = methodSList.findIndex(method->currentText());
	if ( i==-1 ) {
		qDebug("checkMethodCombos : adding");
		i = methodSList.count();
		methodSList.append(method->currentText());
	}
	return i;
	*/
	int i = method->currentIndex();
	Q_ASSERT(i>=0); Q_ASSERT(i<(int)methodSList.count());
	return i;
}

int OpaleDocument::checkCategoryCombos(QComboBox *category)
{
/* XXX orzel : reenable
	int i = categorySList.findIndex(category->currentText());
	if ( i==-1 ) {
		qDebug("checkCategoryCombos : adding");
		i = categorySList.count();
		categorySList.append(category->currentText());
	}
	return i;
*/

	int i = category->currentIndex();
	Q_ASSERT(i>=0); Q_ASSERT(i<(int)categorySList.count());
	return i;
}


void OpaleDocument::touched(void)
{
	setModified(true); emit listUpdated();
}

void OpaleDocument::gotoNow( void )
{
	int idx=0;
	QDate now = QDate::currentDate();

	for (idx=0; idx< topLevelItemCount(); idx++) {
		if ( item2record(topLevelItem(idx))->date < now )
				break;
	}
	scrollToItem(topLevelItem(idx), QAbstractItemView::PositionAtCenter);
	scrollToItem(topLevelItem(idx), QAbstractItemView::PositionAtCenter);
}


void OpaleDocument::addPattern(OpalePattern newp)
{
	patterns.append(newp);
	syncPatterns();
}


void OpaleDocument::syncPatterns(void)
{
	QStringList	namelist;
	for (int i=0; i< patterns.count(); i++)
		namelist.append( patterns.at(i).name);
	emit newPatterns ( namelist );
}

int OpaleDocument::editFileOptions(void)
{
	
	int result;

	Ui_fileOptionsDialog ui;
	QDialog dialog(parentWidget());
	ui.setupUi(&dialog);

	ui.edit_currency->setText(currency());
	ui.edit_account_name->setText(account_name());
	result = dialog.exec();

	if (result) {
	//	setInfos(ui.edit_currency->text(), ui.edit_account_name->text());
		s_currency = ui.edit_currency->text();
		s_account_name = ui.edit_account_name->text();
		touched();
	}
	return result;
}

int OpaleDocument::editCategories(void)
{
	// create dialog
	ListBoxEditor dialog( this, editCategory, parentWidget(), categorySList);
	dialog.setWindowTitle(  tr(  "Edit Categories List" ) );

//useless	connect(dialog.preview, SIGNAL(selectionChanged(void)), this, SIGNAL(dataUpdated(void)));

	int result;

	result = dialog.exec();

	Q_ASSERT(result); // no cancel button

	// fill categorySList
	categorySList.clear();
	for ( int i=0; i<dialog.preview->count(); i++ ) {
		Q_ASSERT( dialog.preview->item(i));
		Q_ASSERT( dialog.preview->item(i)->text() != QString());
		categorySList.append(dialog.preview->item(i)->text());
	}

	touched();
	emit categoriesUpdated();
	return result;
}

int OpaleDocument::editMethods(void)
{
	// create dialog
	ListBoxEditor dialog( this, editMethod, parentWidget(), methodSList);
	dialog.setWindowTitle(tr("Edit List"));

	int result;

	result = dialog.exec();

	Q_ASSERT(result); // no cancel button

	// fill methodSList
	methodSList.clear();
	for ( int i=0; i<dialog.preview->count(); i++ ) {
		Q_ASSERT( dialog.preview->item(i));
		Q_ASSERT( dialog.preview->item(i)->text() != QString());
		methodSList.append(dialog.preview->item(i)->text());
	}

	touched();
	emit methodsUpdated();
	return result;
}


bool OpaleDocument::canDelete( int idx, editBoxWhat_t editBoxWhat )
{
	QStringList &list = (editCategory==editBoxWhat)?categorySList:methodSList;

	// check non empty
	qDebug("OpaleDocument::canDelete : count is %d", list.count());
	if ( list.count()<=1 ){
#ifdef ENABLE_KDE
		KMessageBox::error(0, tr("You can't delete the last item!"), tr("Not enough data"));
#else
		QMessageBox::critical(0, tr("Not enough data"), tr("You can't delete the last item!"));
#endif
		return false;
		}

	// check if does exist
	bool found=false;
	if ( editCategory==editBoxWhat ){

		for (int i=0; i< topLevelItemCount() ;i ++)  {
			if ( idx == item2record(topLevelItem(i))->category ) {
				found=true; break;
			}
		}
		for ( int i = 0; !found && i< patterns.count(); ++ i )
			if ( idx==patterns.at(i).category ) { found=true; break; }
	} else {
		for (int i=0; i< topLevelItemCount() ;i ++)  {
			if ( idx == item2record(topLevelItem(i))->method) {
				found=true; break;
			}
		}
		for ( int i = 0; !found && i< patterns.count(); ++ i )
			if ( idx==patterns.at(i).method ) { found=true; break; }
	}

	int newidx=0;
	if ( found ) {

		// check what to replace
		QDialog dialog(0);
		Ui_replaceDialog ui;
		ui.setupUi(&dialog);

		if ( editCategory==editBoxWhat ){
			dialog.setWindowTitle(  tr(  "Category replace dialog" ) );
			for ( int i=0; i< categorySList.count(); i++)
				if ( i!=idx ) ui.combo->insertItem(i, categorySList[i] );
		} else {
			dialog.setWindowTitle(  tr(  "Method replace dialog" ) );
			for ( int i=0; i< methodSList.count(); i++)
				if ( i!=idx ) ui.combo->insertItem(i, methodSList[i] );
		}

		if ( !dialog.exec() )
			return false;
		newidx = ui.combo->currentIndex();
// TODO : check		if (newidx>=idx)newidx++;
// as we now have an 'index' arg in insertItem, probably useless. Cleaner
// this way
	}



	// actually delete it
	if ( editCategory==editBoxWhat )
		removeCategory(idx, newidx);
	else
		removeMethod(idx, newidx);

	return true;
}


void OpaleDocument::removeMethod(int num, int newnum)
{
	Q_ASSERT(num!=newnum);
	Q_ASSERT(num<methodSList.count());

	if ( newnum>num ) newnum--; // newnum after removing num..

	// Update records...
	for (int i=0; i< topLevelItemCount() ; i++) {
		OpaleRecord *r = item2record(topLevelItem(i));
		if ( r->method==num ) { r->method=newnum; }
		else if ( r->method>num ) r->method--;
		Q_ASSERT(r->method<methodSList.count()-1);
	}
	// ... and patterns
	for ( int i = 0; i< patterns.count(); i++ ) {
		OpalePattern &p = patterns[i];
		if ( p.method==num ) { p.method=newnum;}
		else if ( p.method>num ) p.method--;
		Q_ASSERT(p.method<methodSList.count()-1);
		}
	// change the list
	// XXX : orzel : Qt4 provides QList::removeAt()
	for ( int i=num+1; i< methodSList.count(); i++)
		methodSList[i-1] =  methodSList[i];
	methodSList.pop_back();

	emit methodsUpdated();
	touched();
}

void OpaleDocument::removeCategory(int num, int newnum)
{
	Q_ASSERT(num!=newnum);
	Q_ASSERT(num<categorySList.count());

	if ( newnum>num ) newnum--; // newnum after removing num..

	// Update records...
	for (int i=0; i< topLevelItemCount() ;i ++) {
		OpaleRecord *r = item2record(topLevelItem(i));
		if ( r->category==num ) { r->category=newnum;}
		else if ( r->category>num ) r->category--;
		Q_ASSERT(r->category<categorySList.count()-1);
	}
	// ... and patterns
	for ( int i = 0; i< patterns.count(); ++ i ) {
		OpalePattern &p = patterns[i];
		if ( p.category==num ) { p.category=newnum;}
		else if ( p.category>num ) p.category--;
		Q_ASSERT(p.category<categorySList.count()-1);
	}
	// change the list
	// XXX : orzel : Qt4 provides QList::removeAt()
	for ( int i=num+1; i< categorySList.count(); i++)
		categorySList[i-1] =  categorySList[i];
	categorySList.pop_back();

	emit categoriesUpdated();
	touched();
}


bool OpaleDocument::loadXML( const QString filename )
{
	QFile file( filename );
	if ( !file.open( QIODevice::ReadOnly ) ) {
		QString msg;
		switch(file.error()) {
			case QFile::PermissionsError:
				msg = QString( tr("The permissions of this file prevent you from accessing it."));
				break;
			case QFile::OpenError:
				msg = QString( tr("The file could not be opened. It probably means that the file does not exist."));
				break;
			default:
				msg = QString( tr("Can't open the file, error : %1")).arg(file.error());
		}
		QMessageBox::critical(0, "Opale", msg);
		return false;
	}

	QDomDocument doc;
	QString errorMsg;
	int errorLine, errorColumn;
	if ( !doc.setContent(&file, &errorMsg, &errorLine, &errorColumn ) ) {
		QString msg= "Error parsing the input file\n";
		msg += "error text : \""+errorMsg+"\"\n";
		msg += QString("      line : %1\n" ).arg(errorLine);
		msg += QString("    column : %1\n" ).arg(errorColumn);
		QMessageBox::critical( 0, "Opale", msg );
		file.close();
		return false;
	}
	bool b=loadXML(doc);
	file.close();
	return b;
}

bool OpaleDocument::saveXML(QString filename)
{
	Q_ASSERT(filename!=QString::null); // shouldn't happen either

	QFile file( filename );
	if ( !file.open( QIODevice::Truncate | QIODevice::WriteOnly ) ) {
		QMessageBox::critical(0, "Opale", "Can't open the file for writing" );
		return false;
	}

	QTextStream stream( &file );
    stream.setCodec(QTextCodec::codecForName("UTF-8"));
	stream << saveXML().toString(1);

	file.close();

	return true;
}



#if 0
void OpaleDocument::fillList(void)
{
	// find item for 'today'
	for (int i=0; i< topLevelItemCount() ;i ++)
		if (topLevelItem(i)->text(5)==QString("")){ // '5' is "Future?"
			scrollToItem(topLevelItem(i), QAbstractItemView::PositionAtCenter);
			return;
		}
}
#endif

void OpaleDocument::editRecord(QTreeWidgetItem *item, int column)
{
	if (!item) // nothing selected
		return;

	// handle check/uncheck
	if ( 3==column ) {
		((OpaleRecord*)item)->switchCheck();
		touched();
		return;
	}

	OpaleRecord *record = item2record( item );
	int result;
	bool checked = record->checked();

	if (!record) {
		qDebug("OpaleDocument::editRecord(KListViewItem *item):unexpected item selected ?!");
		return;
	}

	QDialog dialog(this);
	Ui_inputRecordDialog ui;
	ui.setupUi(&dialog);
	ui.PushButton1->setFocus(); // for some reason the datepicker get the focus..
	fillCombos(ui.edit_method, ui.edit_category);
	record->to_dialog(&ui);

	result = dialog.exec();
	if (result) {
		record->from_dialog(&ui);
		record->setChecked(checked);
		sortItems(0,Qt::DescendingOrder);
		touched();
	}
}


void OpaleDocument::new_pattern(void)
{
	qDebug("OpaleDocument::new_pattern() called");

	OpalePattern pattern(this);
	int result;

	QDialog dialog(this);
	Ui_inputPatternDialog ui;
	ui.setupUi(&dialog);
	fillCombos(ui.edit_method, ui.edit_category);

	pattern.to_dialog(&ui);

	result = dialog.exec();
	if (result)
		addPattern(OpalePattern(this, &ui) );
}



void OpaleDocument::repeat_record(void)
{
	// presently cut is really a 'delete'
	qDebug("OpaleDocument::repeat_record() called");

	OpaleRecord *orig_record = item2record(currentItem());

	/* gui */
	int result;

	QDialog dialog(this);
	Ui_inputRepeatDialog ui;
	ui.setupUi(&dialog);

	result = dialog.exec();

	if (!result) return; // nothing to do


	int i;
	int n = ui.spinBoxNb->value();
	int periodNb = ui.spinBoxPeriod->value();
	int periodUnit = ui.editPeriod->currentIndex();

	for (i=0; i< n; i++) {
		OpaleRecord *r = new OpaleRecord(this, *orig_record);
		r->unCheck();
		r->addTime( (i+1)*periodNb, periodUnit);
	}
	touched();
}



void OpaleDocument::delete_record(void)
{
//		qDebug("OpaleDocument::cut() called");
		OpaleRecord *item= item2record( currentItem() );

		item->setSelected(true);
		scrollToItem(item, QAbstractItemView::PositionAtTop);

		QMessageBox::StandardButton button = QMessageBox::warning(  this, "Delete a record", tr( "-->\"%1\"<--\n\nDo you really want to delete this record ?" ).arg(item->name), QMessageBox::Ok | QMessageBox::Cancel);
		switch (button) {
			case QMessageBox::Cancel:
					break;
			case QMessageBox::Ok:
					delete item;
					touched();
					break;
			default: // ??
				qFatal("unhandled case in OpaleDocument::delete_record");
				break;
		};

}

void OpaleDocument::check_record(void)
{
	QList<QTreeWidgetItem *> selected = selectedItems();
	Q_ASSERT(selected.count()==1);
	if (selected.count()!=1)
		return;

	OpaleRecord *record = item2record( selected[0] );

	if (!record) {
		qDebug("OpaleDocument::spacePressedRecord(QListViewItem *item):unexpected item selected ?!");
		return;
	}
	record->switchCheck();
	touched();
}


void OpaleDocument::itemclicked( QTreeWidgetItem * item, int c )
{
	if (!item) // nothing selected
		return;
	// column third is 'checked'
	if ( 3==c ) {
		((OpaleRecord*)item)->switchCheck();
		touched();
	}
}

#include "document.moc"

