
/* Qt */
#include <QUrl>
#include <QMenuBar>
#include <QToolBar>
#include <QFileDialog>
#include <QDesktopServices>
#include <QCloseEvent>
#include <QSettings>
#include <QMessageBox>
#include <QPushButton>
#include <QLabel>

/* KDE */
#ifdef ENABLE_KDE

#include <kactioncollection.h>
#include <kapplication.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kstatusbar.h>
#include <klocale.h>
#include <kdebug.h>
#include <kurl.h>
#include <kconfig.h>

#else

#include <qapplication.h> // qApp
#include <qstatusbar.h>
#include <QMenu>
#include "kdeclasses/kselectaction.h"

#endif

/* Opale */
#include "mainwindow.h"
#include "document.h"
#include "graph.h"
#include "stats.h"
#include "ui_recordDialog.h"
#include "ui_aboutDialog.h"

MainWindow::MainWindow(QWidget *parent)
	:KMainWindow(parent)
	,mydata(0)
{
	setWindowTitle( tr("Opale - Main window") );

	resize(1200, 750);

	// construct mainwidget
	QFont _font("Utopia", 15);

	QStatusBar *sb=statusBar();
	Q_ASSERT(sb);
	sb->addWidget(l_utt= new QLabel( sb)); Q_ASSERT(l_utt);
	sb->addWidget(l_uttsafe= new QLabel( sb)); Q_ASSERT(l_uttsafe);
	sb->addWidget(l_checked= new QLabel( sb)); Q_ASSERT(l_checked);

	// data
	mydata = new OpaleDocument(this);
	Q_ASSERT(mydata);
	connect( mydata, SIGNAL(newPatterns(QStringList)), this, SLOT(newPatterns(QStringList)));
	connect( mydata, SIGNAL(listUpdated(void)), this, SLOT(updateTotal(void)));

	setCentralWidget(mydata);

	createActions();
#ifndef ENABLE_KDE
	createMenus();
	createToolBars();
#endif

	sb->show();

	emit setEnabled(false);
}


MainWindow::~MainWindow( )
{
}

/*
 * GUI stuff
 */

void MainWindow::createActions(void)
{
#ifdef ENABLE_KDE
		KAction *action;
		// setup File menu
		KStdAction::openNew( this, SLOT( newFile() ), actionCollection())->setShortcutText( "" );
		KStdAction::open( this, SLOT( open() ), actionCollection() );

		// recenter = KStdAction::openRecent( this, SLOT( openRecent( int ) ), actionCollection() );
		KStdAction::save( this, SLOT( save() ), actionCollection() );
		KStdAction::saveAs( this, SLOT( saveAs() ), actionCollection() );
		KStdAction::close( this, SLOT( file_close() ), actionCollection() );
		// XXX orzel : TODO KStdAction::print( this, SLOT( print() ), actionCollection() );
		// XXX orzel : TODO KStdAction::mail( this, SLOT( mail() ), actionCollection() );
		KStdAction::quit( this, SLOT( app_close() ), actionCollection() );

		// setup edit menu
		/*
		   XXX orzel : to clean up 
		   KStdAction::undo( this, SLOT( undo() ), actionCollection() );
		   KStdAction::redo( this, SLOT( redo() ), actionCollection() );
		   KStdAction::cut( this, SLOT( cut() ), actionCollection() );
		   KStdAction::copy( this, SLOT( copy() ), actionCollection() );
		   KStdAction::paste( this, SLOT( paste() ), actionCollection() );
		   KStdAction::selectAll( this, SLOT( select_all() ), actionCollection()
		   );
		   KStdAction::find( this, SLOT( search() ), actionCollection() );
		   KStdAction::findNext( this, SLOT( search_again() ), actionCollection()
		   );
		   KStdAction::replace( this, SLOT( replace() ), actionCollection() );
		 */

		action = new KAction( tr("New &record"), "add", QKeySequence("Ctrl+N" ), this, SLOT( new_record() ),
						actionCollection(), "new_record");
		connect(this, SIGNAL( setEnabled(bool)), action, SLOT(setEnabled(bool) ));

		action = new KAction( tr("&Repeat record"), "editpaste", 0, mydata, SLOT( repeat_record() ),
						actionCollection(), "repeat");
		connect(this, SIGNAL( setEnabled(bool)), action, SLOT(setEnabled(bool) ));

		action = new KAction( tr("&Delete record"), "editdelete", QKeySequence("Ctrl+D" ),
						mydata, SLOT( delete_record() ), actionCollection(), "delete_record" );
		connect(this, SIGNAL( setEnabled(bool)), action, SLOT(setEnabled(bool) ));

		action = new KAction( tr("&Check/Uncheck record"), 0, Qt::Key_Space,
						mydata, SLOT( check_record() ), actionCollection(), "check_record" );
		connect(this, SIGNAL( setEnabled(bool)), action, SLOT(setEnabled(bool) ));

		action = new KAction( tr("Edit file &options..."), "queue", 0, mydata, SLOT( editFileOptions() ),
						actionCollection(), "edit_file_options");
		connect(this, SIGNAL( setEnabled(bool)), action, SLOT(setEnabled(bool) ));

		action = new KAction( tr("Edit &Categories..."), 0, 0, mydata, SLOT( editCategories() ),
						actionCollection(), "edit_categories");
		connect(this, SIGNAL( setEnabled(bool)), action, SLOT(setEnabled(bool) ));

		action = new KAction( tr("Edit &Methods..."), 0, 0, mydata, SLOT( editMethods() ),
						actionCollection(), "edit_types");
		connect(this, SIGNAL( setEnabled(bool)), action, SLOT(setEnabled(bool) ));

		action = new KAction( tr("New &pattern"), "queue", 0, mydata, SLOT( new_pattern() ),
						actionCollection(), "new_pattern");
		connect(this, SIGNAL( setEnabled(bool)), action, SLOT(setEnabled(bool) ));

		action = new KAction( tr("&Graph..."), "opale_chart", 0, this, SLOT( get_graph() ),
						actionCollection(), "get_graph");
		connect(this, SIGNAL( setEnabled(bool)), action, SLOT(setEnabled(bool) ));

		action = new KAction( tr("&Statistics..."), "opale_stats", 0, this, SLOT( get_stats() ),
						actionCollection(), "get_stats");
		connect(this, SIGNAL( setEnabled(bool)), action, SLOT(setEnabled(bool) ));

		action = new KAction( tr("&Go to current day"), "redo", 0, mydata, SLOT( gotoNow() ),
						actionCollection(), "goto_now");
		connect(this, SIGNAL( setEnabled(bool)), action, SLOT(setEnabled(bool) ));

		// help menu
		action = new KAction( tr("&Manual..."), "bug", 0, this, SLOT( openManual() ),
						actionCollection(), "open_manual");
		action = new KAction( tr("&About Opale"), "opale", 0, this, SLOT( aboutOpale() ),
						actionCollection(), "about_opale");
		action = new KAction( tr("&Report bug..."), "bug", 0, this, SLOT( reportBug() ),
						actionCollection(), "report_bug");

		KConfig *config = KGlobal::config();
		m_recent->loadEntries( config );

		createGUI( "opale.rc" );
#else
        // Qt-only stuff:
		newAct = new QAction(QIcon(":/images/new.png"), tr("&New"), this);
		newAct->setShortcut(tr("Ctrl+N"));
		newAct->setStatusTip(tr("Create a new file"));
		connect(newAct, SIGNAL(triggered()), this, SLOT(newFile()));

		openAct = new QAction(QIcon(":/images/open.png"), tr("&Open..."), this);
		openAct->setShortcut(tr("Ctrl+O"));
		openAct->setStatusTip(tr("Open an existing file"));
		connect(openAct, SIGNAL(triggered()), this, SLOT(open()));

		saveAct = new QAction(QIcon(":/images/save.png"), tr("&Save"), this);
		saveAct->setShortcut(tr("Ctrl+S"));
		saveAct->setStatusTip(tr("Save the document to disk"));
		connect(saveAct, SIGNAL(triggered()), this, SLOT(save()));

		saveAsAct = new QAction(tr("Save &As..."), this);
		saveAsAct->setStatusTip(tr("Save the document under a new name"));
		connect(saveAsAct, SIGNAL(triggered()), this, SLOT(saveAs()));

		closeAct = new QAction(tr("&Close..."), this);
		closeAct->setShortcut(tr("Ctrl+W"));
		closeAct->setStatusTip(tr("Close the current document"));
		connect(closeAct, SIGNAL(triggered()), this, SLOT(file_close()));

		exitAct = new QAction(tr("&Quit"), this);
		exitAct->setShortcut(tr("Ctrl+Q"));
		exitAct->setStatusTip(tr("Quit the application"));
		connect(exitAct, SIGNAL(triggered()), this, SLOT(app_close()));

		newRecordAct = new QAction(QIcon(":/images/newrecord.png"), tr("New &record"), this);
		newRecordAct->setShortcut(tr("Ctrl+N"));
		newRecordAct->setStatusTip(tr("Add a new record"));
		connect(newRecordAct, SIGNAL(triggered()), this, SLOT(new_record()));

		repeatRecordAct = new QAction(QIcon(":/images/repeat.png"), tr("&Duplicate record"), this);
		repeatRecordAct->setStatusTip(tr("Copy this record, possibly several times"));
		connect(repeatRecordAct, SIGNAL(triggered()), mydata, SLOT(repeat_record()));

		deleteRecordAct = new QAction(QIcon(":/images/delete.png"), tr("&Delete record"), this);
		deleteRecordAct->setShortcut(tr("Ctrl+D"));
		deleteRecordAct->setStatusTip(tr("Delete selected record"));
		connect(deleteRecordAct, SIGNAL(triggered()), mydata, SLOT(delete_record()));


		checkRecordAct = new QAction(tr("&Check/Uncheck record"), this);
		checkRecordAct->setShortcut(Qt::Key_Space);
		checkRecordAct->setStatusTip(tr("Switch checked status for the current record"));
		connect(checkRecordAct, SIGNAL(triggered()), mydata, SLOT(check_record()));

		editfileOptionsAct = new QAction(tr("Edit file &options..."), this);
		editfileOptionsAct->setStatusTip(tr("Edit parameters for the current file."));
		connect(editfileOptionsAct, SIGNAL(triggered()), mydata, SLOT(editFileOptions()));

		editCategoriesAct = new QAction(tr("Edit &Categories..."), this);
		editCategoriesAct->setStatusTip(tr("Add, edit and delete categories."));
		connect(editCategoriesAct, SIGNAL(triggered()), mydata, SLOT(editCategories()));

		editMethodsAct = new QAction(tr("Edit &Methods..."), this);
		editMethodsAct->setStatusTip(tr("Add, edit and delete  methods."));
		connect(editMethodsAct, SIGNAL(triggered()), mydata, SLOT(editMethods()));

		newPatternAct = new QAction(tr("New &pattern"), this);
		newPatternAct->setStatusTip(tr("Create a new pattern."));
		connect(newPatternAct, SIGNAL(triggered()), mydata, SLOT(new_pattern()));

		getGraphAct = new QAction(QIcon(":/images/opale_chart.png"), tr("&Graph..."), this);
		getGraphAct->setShortcut(tr("F2"));
		getGraphAct->setStatusTip(tr("Open a window to browse the data using a graphical view."));
		connect(getGraphAct, SIGNAL(triggered()), this, SLOT(get_graph()));

		getStatsAct = new QAction(QIcon(":/images/opale_stats.png"), tr("&Statistics..."), this);
		getStatsAct->setShortcut(tr("F3"));
		getStatsAct->setStatusTip(tr("Open a new window to browse statistics."));
		connect(getStatsAct, SIGNAL(triggered()), this, SLOT(get_stats()));

		gotoNowAct = new QAction(QIcon(":/images/opale_gotonow.png"), tr("&Go to current day"), this);
		gotoNowAct->setShortcut(tr("F4"));
		gotoNowAct->setStatusTip(tr("Position the main view on top of the current day."));
		connect(gotoNowAct, SIGNAL(triggered()), mydata, SLOT(gotoNow()));

		openManualAct = new QAction(tr("&Manual..."), this);
		openManualAct->setStatusTip(tr("Open the manual, online web page."));
		connect(openManualAct, SIGNAL(triggered()), this, SLOT(openManual()));

		reportBugAct = new QAction(tr("&Report bug..."), this);
		reportBugAct->setStatusTip(tr("Open a browser on Opale web bugtracker."));
		connect(reportBugAct, SIGNAL(triggered()), this, SLOT(reportBug()));

		aboutAct = new QAction(tr("&About"), this);
		aboutAct->setStatusTip(tr("Show the application's About box"));
		connect(aboutAct, SIGNAL(triggered()), this, SLOT(aboutOpale()));

        /* enable/disable specific actions when there's no file loaded */
		connect(this, SIGNAL( setEnabled(bool)), saveAct, SLOT(setEnabled(bool) ));
        connect(this, SIGNAL( setEnabled(bool)), saveAsAct, SLOT(setEnabled(bool) ));
        connect(this, SIGNAL( setEnabled(bool)), closeAct, SLOT(setEnabled(bool) ));
        connect(this, SIGNAL( setEnabled(bool)), newRecordAct, SLOT(setEnabled(bool) ));
        connect(this, SIGNAL( setEnabled(bool)), repeatRecordAct, SLOT(setEnabled(bool) ));
        connect(this, SIGNAL( setEnabled(bool)), deleteRecordAct, SLOT(setEnabled(bool) ));
        connect(this, SIGNAL( setEnabled(bool)), checkRecordAct, SLOT(setEnabled(bool) ));
        connect(this, SIGNAL( setEnabled(bool)), editfileOptionsAct, SLOT(setEnabled(bool) ));
        connect(this, SIGNAL( setEnabled(bool)), editCategoriesAct, SLOT(setEnabled(bool) ));
        connect(this, SIGNAL( setEnabled(bool)), editMethodsAct, SLOT(setEnabled(bool) ));
        connect(this, SIGNAL( setEnabled(bool)), newPatternAct, SLOT(setEnabled(bool) ));
        connect(this, SIGNAL( setEnabled(bool)), getGraphAct, SLOT(setEnabled(bool) ));
        connect(this, SIGNAL( setEnabled(bool)), getStatsAct, SLOT(setEnabled(bool) ));
        connect(this, SIGNAL( setEnabled(bool)), gotoNowAct, SLOT(setEnabled(bool) ));
#endif
        // in both cases:

		m_patternaction = new KSelectAction(QIcon(":/images/editcopy.png"), tr("New record &from pattern"), this);
		m_patternaction->setStatusTip(tr("Create a new record, prefilled with the entries of the pattern and open the edit box."));
		connect(m_patternaction, SIGNAL(triggered(int)), this, SLOT(new_record_from_pattern(int)));
		connect(this, SIGNAL( setEnabled(bool)), m_patternaction, SLOT(setEnabled(bool) ));

        for (int i = 0; i < MaxRecentFiles; ++i) {
            recentFileActs[i] = new QAction(this);
            recentFileActs[i]->setVisible(false);
            connect(recentFileActs[i], SIGNAL(triggered()),
                    this, SLOT(openRecentFile()));
        }

}


#ifndef ENABLE_KDE

void MainWindow::setCurrentFile(const QString &_filename)
{
	qDebug("MainWindow::setCurrentFile : %s", qPrintable(_filename));
	filename = _filename;

    // title
    if (filename.isEmpty())
        setWindowTitle(tr("New file - Opale"));
    else
        setWindowTitle(tr("%1 - %2").arg(strippedName(filename)).arg(tr("Opale")));

    // recent files
    QSettings settings("Freehackers", "Opale");
    QStringList files = settings.value("recentFileList").toStringList();
    files.removeAll(filename);
    files.prepend(filename);
    while (files.size() > MaxRecentFiles)
        files.removeLast();

    settings.setValue("recentFileList", files);

    foreach (QWidget *widget, QApplication::topLevelWidgets()) {
        MainWindow *mainWin = qobject_cast<MainWindow *>(widget);
        if (mainWin)
            mainWin->updateRecentFileActions();
    }
}

void MainWindow::createMenus()
{
    fileMenu = menuBar()->addMenu(tr("&File"));
    fileMenu->addAction(newAct);
    fileMenu->addAction(openAct);
    fileMenu->addAction(saveAct);
    fileMenu->addAction(saveAsAct);
    fileMenu->addSeparator();
    fileMenu->addAction(closeAct);

    separatorAct = fileMenu->addSeparator();
    for (int i = 0; i < MaxRecentFiles; ++i)
        fileMenu->addAction(recentFileActs[i]);

    fileMenu->addSeparator();
    fileMenu->addAction(exitAct);

    updateRecentFileActions();

    editMenu = menuBar()->addMenu(tr("&Edit"));
	editMenu->addAction(newRecordAct);
	editMenu->addAction(m_patternaction);
	editMenu->addAction(deleteRecordAct);
	editMenu->addAction(repeatRecordAct);
    editMenu->addSeparator();
	editMenu->addAction(checkRecordAct);
    editMenu->addSeparator();
    editMenu->addAction(newPatternAct);


    viewMenu= menuBar()->addMenu(tr("&View"));
	viewMenu->addAction(getGraphAct);
	viewMenu->addAction(getStatsAct);
    viewMenu->addSeparator();
	viewMenu->addAction(gotoNowAct);

    settingsMenu = menuBar()->addMenu(tr("&Settings"));
    settingsMenu->addAction(editfileOptionsAct);
    settingsMenu->addAction(editCategoriesAct);
    settingsMenu->addAction(editMethodsAct);

    menuBar()->addSeparator();

    helpMenu = menuBar()->addMenu(tr("&Help"));
    helpMenu->addAction(openManualAct);
    helpMenu->addAction(reportBugAct);
    helpMenu->addSeparator();
    helpMenu->addAction(aboutAct);
}

void MainWindow::createToolBars()
{
    fileToolBar = addToolBar(tr("File"));
    fileToolBar->addAction(openAct);
    fileToolBar->addAction(saveAct);

    editToolBar = addToolBar(tr("Edit"));
	editToolBar->addAction(newRecordAct);
	editToolBar->addAction(deleteRecordAct);
	editToolBar->addAction(repeatRecordAct);
	editToolBar->addAction(getGraphAct);
	editToolBar->addAction(getStatsAct);
	editToolBar->addAction(gotoNowAct);
}
#endif

void MainWindow::newPatterns(QStringList l )
{
	m_patternaction->setItems(l);
	m_patternaction->setComboWidth(300);
}


/*
 * FILE STUFF
 */

void MainWindow::newFile()
{
	if ( filename!=QString::null && !file_close() ) return;
	// XX todo, makes OpaleDocument a member, not a pointer

	if ( !mydata->editFileOptions() ) // cancel
		return;

	mydata->resetData();
	filename = QString("");
	Q_ASSERT(mydata);

	mydata->show(); emit setEnabled(true);

	updateTotal();

}

void MainWindow::openfilename( QString _filename )
{
	Q_ASSERT(filename==QString::null); // no file opened yet, cause
			// only main.cc  and  ::open_file() calls this

	if ( _filename==QString::null ) return;

	mydata->resetData();
	if ( !mydata->loadXML( _filename) ) {
		qWarning("MainWindow::openfilename : error returned from mydata->loadXML()");
		return;
	}

    setCurrentFile(_filename);

	mydata->show(); emit setEnabled(true);
	updateTotal();
	qDebug("file loaded successfully");
}

void MainWindow::openRecentFile()
{
    QAction *action = qobject_cast<QAction *>(sender());
    if (action) {
        if ( filename!=QString::null && !file_close() ) return;
        openfilename(action->data().toString());
    }
}


void MainWindow::open()
{
	if ( filename!=QString::null && !file_close() ) return;

	QString s = QFileDialog::getOpenFileName( this, 
			tr( "Choose a file" ), QString::null, tr( "Opale file (*.opale)" ));
	if ( s==QString::null ) return; // user canceled

	openfilename( s );
}

void MainWindow::closeEvent(  QCloseEvent * e )
{
	if (app_close())
			e->accept();
	else
			e->ignore();
}

bool MainWindow::app_close()
{
		if ( filename==QString::null || file_close() ) {
#ifdef ENABLE_KDE
				m_recent->saveEntries( KGlobal::config() );
#endif
				return true;
		}
		return false;
}

bool MainWindow::file_close()
{
	if ( filename==QString::null ) {
		QMessageBox::critical(  this, "Opale", tr( "No file is currently opened" ) );
		return true;
	}

	if ( mydata->isModified()) {
		int ans=QMessageBox::question(  this, "Opale", tr( "Current file is modified, do you want to save it ?" ), tr( "Yes" ), tr( "No" ), tr( "Cancel" ));
		qDebug("open : question answer is %d", ans);
		switch (ans) {
			case 0: // yes
				if ( ! save() ) return false;
				break;
			case 1: // no
				// continue after switch, delete data
				break;
			case 2: // cancel
			case -1: // cancel
				return false;
				break;
			default: // ??
				qFatal("unhandled case in MainWindow::open");
				break;
		};
	}

	mydata->resetData();
	filename = QString::null;
	emit setEnabled(false);
//	qDebug("File closed");

	return true;
}

bool MainWindow::save()
{
	if ( filename==QString::null ) {
		QMessageBox::critical(  this, "Opale", tr( "No file is currently opened" ) );
		return true;
	}
	if ( !mydata->isModified()) return true;

	if ( filename.isEmpty() )
		return saveAs();

	/* actually save the file */
	mydata->saveXML( filename );

	return true;
}

bool MainWindow::saveAs()
{
	if ( filename==QString::null ) {
		QMessageBox::critical(  this, "Opale", tr( "No file is currently opened" ) );
		return true;
	}

	QString s = QFileDialog::getSaveFileName( this, tr( "Choose a file"), QString::null, tr( "Opale file (*.opale)" ));
	if ( s==QString::null ) return false; // user canceled

	filename = s;

	/* actually save the file */
	mydata->saveXML( filename );

	return true;
}


/*
 * SLOT STUFF
 */

void MainWindow::updateTotal()
{
//	qDebug("MainWindow::updateTotal()");
	mydata->updateTotal( total );
	displayTotal();
}

void MainWindow::displayTotal()
{
	l_utt->setText( tr( "Balance :" )+fillLabel(total.utt) );
	l_uttsafe->setText( tr( "Safe Balance :" ) + fillLabel(total.uttsafe) );
	l_checked->setText( tr( "Checked Balance :" ) + fillLabel(total.checked, true) );
}

QString	MainWindow::fillLabel(amount_t amount, bool exact)
{
	QString	currency = mydata->currency();

	if (!exact) {
		int av = (amount>=0)?amount:-amount;
		if (av > 800000000) {
			amount /= 1000000;
			currency.prepend("M");
		} else if ( av > 800000) {
			amount /= 1000;
			currency.prepend("k");
		}
	}
	return QString("%1 %3").arg( QString().sprintf("%12.2f", amount/100.)).arg(currency);

}

void MainWindow::new_record_from_pattern(int i)
{
	qDebug( "MainWindow::new_record_from_pattern() called" );

	int result;
	OpaleRecord *record = mydata->newRecordFromPattern(i);

	QDialog dialog(this);
	Ui_inputRecordDialog ui;
	ui.setupUi(&dialog);
	ui.PushButton1->setFocus(); // for some reason the datepicker get the focus..
	mydata->fillCombos(ui.edit_method, ui.edit_category);

	record->to_dialog(&ui);

	result = dialog.exec();

	delete record;

	if (result) {
		new OpaleRecord(mydata, &ui);
		mydata->touched();
	}
}


void MainWindow::new_record(void)
{
	qDebug( "MainWindow::new_record() called" );

	OpaleRecord record(mydata);
	int result;

	QDialog dialog(this);
	Ui_inputRecordDialog ui;
	ui.setupUi(&dialog);
	ui.PushButton1->setFocus(); // for some reason the datepicker get the focus..
	mydata->fillCombos(ui.edit_method, ui.edit_category);

	record.to_dialog(&ui);

	result = dialog.exec();
	if (result) {
	qDebug( "MainWindow::new_record() called           asdfasdsdj ksdlaf -> added" );
		new OpaleRecord(mydata, &ui);
		mydata->touched();
	}
}

void MainWindow::get_graph(void)
{
	if (mydata->recordsCount() < 3) {
#ifdef ENABLE_KDE
		KMessageBox::error(this, tr("Sorry, you don't have enough data yet to display a chart"), tr("Not enough data"));
#else
		QMessageBox::critical(this, tr("Not enough data"), tr("Sorry, you don't have enough data yet to display a chart"));
#endif
		return;
	}
	OpaleGraph *og = new OpaleGraph(mydata , this);
	og->show();
}



void MainWindow::get_stats(void)
{
	if (mydata->recordsCount() < 3) {
#ifdef ENABLE_KDE
		KMessageBox::error(this, tr("Sorry, you don't have enough data yet to display stats"), tr("Not enough data"));
#else
		QMessageBox::critical(this, tr("Not enough data"), tr("Sorry, you don't have enough data yet to display stats"));
#endif
		return;
	}
	OpaleStats *og = new OpaleStats(mydata , this);
	og->show();
}

void MainWindow::aboutOpale()
{
	QDialog d(this);
	Ui_aboutDialog ui;
	ui.setupUi(&d);

    ui.textBrowser->append( QString("\nVersion: %1").arg(version));
	d.exec();
}

void MainWindow::openManual()
{
		QDesktopServices::openUrl(QUrl("http://www.freehackers.org/Opale:Manual"));
}

void MainWindow::reportBug()
{
		QDesktopServices::openUrl(QUrl("http://labs.freehackers.org/projects/opale/issues"));
}

void MainWindow::updateRecentFileActions()
{
    QSettings settings("Freehackers", "Opale");
    QStringList files = settings.value("recentFileList").toStringList();

    int numRecentFiles = qMin(files.size(), (int)MaxRecentFiles);

    for (int i = 0; i < numRecentFiles; ++i) {
        QString text = tr("&%1 %2").arg(i + 1).arg(strippedName(files[i]));
        recentFileActs[i]->setText(text);
        recentFileActs[i]->setData(files[i]);
        recentFileActs[i]->setVisible(true);
    }
    for (int j = numRecentFiles; j < MaxRecentFiles; ++j)
        recentFileActs[j]->setVisible(false);

    separatorAct->setVisible(numRecentFiles > 0);
}


QString MainWindow::strippedName(const QString &fullFileName)
{
    return QFileInfo(fullFileName).fileName();
}

#include "mainwindow.moc"

