/*
   Copyright (C) 2001-2005 Thomas Capricelli <orzel@freehackers.org>
*/

/* libc */
#include <math.h>	// HUGE_VAL

/* qt */
#include <qdom.h>
#include <qpainter.h>
#include <qlineedit.h>
#include <qvalidator.h>
#include <qcombobox.h>

/* kde */
#ifdef ENABLE_KDE
#include <kdatepicker.h>
#include <kdebug.h>
#else
#include <q3datetimeedit.h>
#include <qpixmap.h>
#include <qobject.h>
#define KDatePicker Q3DateEdit
#endif

/* opale */
#include "record.h"
#include "document.h"
#include "ui_recordDialog.h"
#include "ui_pattern.h"


#include "checked.xpm"
#include "unchecked.xpm"


/*
 * OpalePattern
 */


OpalePattern::OpalePattern(OpaleDocument *d, QString _name, amount_t _amount)
{
	b_valid = false;

	p_data = d;
	name = _name;
	m_amount = _amount;
	method = 0;
	category = 0;
}


OpalePattern::OpalePattern(const OpalePattern &other)
{
	b_valid = false;

	p_data = other.p_data;
	name = other.name;
	m_amount = other.m_amount;
	method = other.method;
	category = other.category;
}


// TODO : check ok after each read
OpalePattern::OpalePattern(OpaleDocument *d, const QDomElement &e)
{
	bool ok;
	b_valid = true;

	p_data = d;

	if ( e.isNull() || (e.tagName() != "record" && e.tagName() != "pattern") ) {  
		b_valid=false;
		return;
	}

	m_amount = e.attribute("amount").toLongLong( &ok );
	if (!ok) {
		m_amount = 0;
		b_valid = false;
	}
	name = e.attribute("what");
	method = e.attribute("how").toInt( &ok ); // for compatibility
	method = e.attribute("method").toInt( &ok );
	category = e.attribute("category").toInt( &ok );
	if (!ok) {
		qDebug( "OpalePattern::OpalePattern(QDomElement): invalide " );
		b_valid = false;
	}
}

OpalePattern::OpalePattern(OpaleDocument *d, Ui_inputPatternDialog *dialog )
{
	p_data = d;
	from_dialog( dialog );
}



void OpalePattern::from_dialog(Ui_inputPatternDialog *dialog)
{
	b_valid = true;

	method = p_data->checkMethodCombos(dialog->edit_method);
	category = p_data->checkCategoryCombos(dialog->edit_category);
	name = dialog->edit_what->text();
//	method = dialog->edit_method->currentItem();

	double	f;
	bool	ok;

	f =  dialog->edit_amount->text().toDouble(&ok);
	Q_ASSERT(ok);
#if 0
	qDebug("XXX 0 XXX bool is %s\n", ok?"true":"false");
	qDebug("XXX 1 XXX %s\n", dialog->edit_amount->text().latin1());
	qDebug("XXX 1 XXX %f\n", f);

	for (unsigned int i = 0; i<sizeof(double); i++) qDebug("f[%d]=%d\n", i, (unsigned int)((char *)(&f))[i] );
//	f=-f;qDebug("\n"); 
//	for (int i = 0; i<sizeof(double); i++) qDebug("f[%d]=%d\n", i, (unsigned int)((char *)(&f))[i] );

#endif
	f=rint(f*100.);
#if 0
	qDebug("\n");
	for (unsigned int i = 0; i<sizeof(double); i++) qDebug("f[%d]=%d\n", i, (unsigned int)((char *)(&f))[i] );

	qDebug("XXX 2 XXX %f\n", f);
	qDebug("XXX 3 XXX %d\n", (int)f);
#endif
	m_amount = (amount_t)f;
//	qDebug("amount*100 : " AMOUNT_ESCAPE "\n", m_amount);
}


QDomElement OpalePattern::domElement(QDomDocument doc)
{
	QDomElement record = doc.createElement( "pattern" );

	record.setAttribute("amount", (int)m_amount);
	record.setAttribute("what", name);
	record.setAttribute("method", method);
	record.setAttribute("category", category);

	return record;
}


void OpalePattern::to_dialog(Ui_inputPatternDialog *dialog)
{
	dialog->edit_what->setText(name);
	dialog->edit_amount->setText(QString().sprintf("%.2f", m_amount/100.));
	dialog->edit_amount->setValidator(new QDoubleValidator(-HUGE_VAL,HUGE_VAL,2, dialog->edit_amount));
	dialog->edit_method->setCurrentIndex(method);
	dialog->edit_category->setCurrentIndex(category);
}



/*
 * OpaleRecord
 */

OpaleRecord::OpaleRecord(OpaleDocument *qlv, QString _name, amount_t _amount)
	: OpalePattern(qlv, _name, _amount)
	, QTreeWidgetItem ( qlv )
{
	date = QDate::currentDate();
	check = false;

	updateOutput();
}

OpaleRecord::OpaleRecord(OpaleDocument *qlv, const OpalePattern &p)
       	: OpalePattern(p) 
	, QTreeWidgetItem( qlv )
{
	date = QDate::currentDate();
	check = false;

	updateOutput();
}

OpaleRecord::OpaleRecord(OpaleDocument *qlv, const OpaleRecord &r)
       	: OpalePattern(r) 
	, QTreeWidgetItem( qlv )
{
	date = r.date;
	check = r.check;

	updateOutput();
}


OpaleRecord::OpaleRecord(OpaleDocument *qlv, const QDomElement &e)
	:OpalePattern(qlv, e)
	, QTreeWidgetItem( qlv )
{
	bool ok;

	if ( e.isNull() || e.tagName() != "record" ) {  
		b_valid=false;
		return;
	}

	// new date
	date = QDate::fromString( e.attribute("date"), Qt::ISODate );
	// old date, fallback
	if ( !  date.isValid() ) {
		qDebug( "Bad \"date\" attribute in OpaleRecord::OpaleRecord( KListView *qlv, const QDomElement &e ), fallback to \"when\"" );
		QDateTime qdt;
		qdt.setTime_t(  e.attribute("when").toInt( &ok ) );
		if (!ok) {
			qDebug( "Bad \"when\" attribute in OpaleRecord::OpaleRecord( KListView *qlv, const QDomElement &e )" );
			b_valid = false;
		}
		else date=qdt.date();
	}
	if (  e.attribute("checked") == QString("yes") )
		check = true;
	else if (  e.attribute("checked") == QString("no") )
		check = false;
	else {
		qDebug( "Bad \"checked\" attribute in OpaleRecord::OpaleRecord( KListView *qlv, const QDomElement &e )" );
		check = false;
	}

	updateOutput();
}

OpaleRecord::OpaleRecord(OpaleDocument *qlv, Ui_inputRecordDialog *dialog)
	: OpalePattern( qlv )
	, QTreeWidgetItem( qlv )
{
	from_dialog( dialog );
}


QDomElement OpaleRecord::domElement(QDomDocument doc)
{
	QDomElement record = doc.createElement( "record" );

	record.setAttribute("amount", m_amount);
	record.setAttribute("date",  date.toString( Qt::ISODate) );
	record.setAttribute("what", name);
	record.setAttribute("method", method);
	record.setAttribute("category", category);
	record.setAttribute("checked", check?"yes":"no");

	return record;
}

void OpaleRecord::from_dialog(Ui_inputRecordDialog *dialog)
{
	b_valid = true;

	method = p_data->checkMethodCombos(dialog->edit_method);
	category = p_data->checkCategoryCombos(dialog->edit_category);
	name = dialog->edit_what->text();
//	method = dialog->edit_method->currentItem();

	double	f;
	bool	ok;

	f =  dialog->edit_amount->text().toDouble(&ok);
	Q_ASSERT(ok);
#if 0
	qDebug("XXX 0 XXX bool is %s\n", ok?"true":"false");
	qDebug("XXX 1 XXX %s\n", dialog->edit_amount->text().latin1());
	qDebug("XXX 1 XXX %f\n", f);

//	for (unsigned int i = 0; i<sizeof(double); i++) qDebug("f[%d]=%d, ", i, (unsigned int)((char *)(&f))[i] );
#endif
	f=rint(f*100.);
	// work around a bug on i386 and optimisation : see
	// [2007-08-13 20:31] <pinskia> orzel: read http://www.validlab.com/goldberg/paper.pdf
	// [2007-08-13 20:31] <pinskia> orzel: and http://gcc.gnu.org/bugzilla/show_bug.cgi?id=323
#if 0
	qDebug("\n");
	for (unsigned int i = 0; i<sizeof(double); i++) qDebug("f[%d]=%d, ", i, (unsigned int)((char *)(&f))[i] );

	qDebug("XXX 2 XXX %e\n", f);
	qDebug("XXX 3 XXX %d\n", (int)f);

	f = 2040.;
	qDebug("\n");
	for (unsigned int i = 0; i<sizeof(double); i++) qDebug("f[%d]=%d, ", i, (unsigned int)((char *)(&f))[i] );
	qDebug("XXX 2bis XXX %e\n", f);
	qDebug("XXX 3bis XXX %d\n", (int)f);
#endif
	m_amount = (amount_t)f;
//	qDebug("amount*100 : " AMOUNT_ESCAPE "\n", m_amount);

	check = false;
	date = dialog->calendarWidget->selectedDate();


	updateOutput();
}

void OpaleRecord::to_dialog(Ui_inputRecordDialog *dialog)
{
	dialog->edit_what->setText(name);
	dialog->edit_amount->setText(QString().sprintf("%.2f", m_amount/100.));
	dialog->edit_amount->setValidator(new QDoubleValidator(-HUGE_VAL,HUGE_VAL,2, dialog->edit_amount));
	dialog->edit_method->setCurrentIndex(method);
	dialog->edit_category->setCurrentIndex(category);

	dialog->calendarWidget->setSelectedDate(date);
}


void OpaleRecord::updateOutput(void)
{
	static QPixmap pixOk((const char**)checked_pixmap), pixNok((const char**)unchecked_pixmap);
	setText( 0, QString().sprintf("%02d/%02d/%04d", date.day(), date.month(), date.year() ));
	setText( 1, name );
	setText( 2, QString("%1 %2").arg( QString().sprintf("%12.2f", m_amount/100.)).arg(p_data->currency()) );
	setIcon( 3, check? pixOk:pixNok);
	setText( 4, p_data->method(method) );
	setText( 5, isFutur()?QObject::tr("Future"):"" );
	setText( 7, p_data->category(category) );

	setTextAlignment(2, Qt::AlignRight); // amount
	setTextAlignment(3, Qt::AlignHCenter); // checked
	setTextAlignment(4, Qt::AlignRight); // method
	setTextAlignment(6, Qt::AlignRight);  // balance
	setTextAlignment(7, Qt::AlignRight); // category
}

void OpaleRecord::updateTotal(Total &t)
{
	t.global	+= m_amount;
	if (isPast()) {
		t.utt		+= m_amount;
		t.uttsafe	+= safeAmount();
	}
	if (check)
		t.checked	+= m_amount;

	// update display
	setText( 6, QString("%1 %2").arg( QString().sprintf("%12.2f", t.global/100.)).arg(p_data->currency()) );
	updateOutput();
}


/*
KListViewItem * OpaleRecord::fillListView(Total &t, KListView *qlv)
{
	KListViewItem	*_item = NULL;

	Q_ASSERT(qlv);
	t.global	+= m_amount;

	_item = new KListViewItem( qlv,
		QString().sprintf("%02d/%02d/%04d", date.day(), date.month(), date.year() ), 
		name,
		QString("%1 %2").arg( QString().sprintf("%12.2f", m_amount/100.)).arg(p_data->currency()),
		check()?"X":"",
		p_data->method(method),
		isFutur()?tr("Future"):"",
		QString("%1 %2").arg( QString().sprintf("%12.2f", t.global/100.)).arg(p_data->currency()),
		p_data->category(category) );

	if (isPast()) {
		t.utt		+= m_amount;
		t.uttsafe	+= safeAmount();
	}
	if (check)
		t.check	+= m_amount;

	return _item;
}
*/

void OpaleRecord::addToChart( GraphData &d )
{
	d.qdt.setDate(date );
	d.balance+=m_amount;
	if ( m_amount>=0 ) {
		d.best += m_amount;
		d.best2 += m_amount;
		if ( check ) d.worst += m_amount;
		if ( check || isPast()  ) d.worst2 += m_amount;
	} else {
		d.worst += m_amount;
		d.worst2 += m_amount;
		if ( check ) d.best += m_amount;
		if ( check || isPast()  ) d.best2 += m_amount;
	}

}


bool OpaleRecord::isFutur(void)
{
	return ( QDate::currentDate() < date );
}

/**
  *  add some time to the current record
  *  @param unit is 0 for day, 1 for week, 2 for month, and 3 for year
  *  @param nb is how many of those needs to be added
  */
void OpaleRecord::addTime(int nb, int unit)
{
	switch(unit) {
		default:
			qDebug( "unknown unit in OpaleRecord::addTime" );
			return;
		case 0:
			date=date.addDays(nb);
			break;
		case 1:
			date=date.addDays(7*nb);
			break;
		case 2:
			date=date.addMonths(nb);
			break;
		case 3:
			date=date.addYears(nb);
			break;
	}
}

bool OpaleRecord::operator< ( const QTreeWidgetItem & o) const
{
	// Yes, this is ugly.
	const OpaleRecord *other = ( const OpaleRecord*) &o;
	// first compare the date 
	if (date < other->date)
			return true;
	if (date > other->date)
			return false;
	// ==??
	// if not, compare the amount, as we need a deterministic/reproducable way to sort things
	return m_amount < other->m_amount;
}

