/*
   Copyright (C) 2001-2005 Thomas Capricelli <orzel@freehackers.org>
*/

#ifndef OPALE_RECORD
#define OPALE_RECORD

/* unix */
#include <time.h>

/* qt */
#include <qstring.h>
#include <qdatetime.h>
#include <QVector>
#include <QTreeWidget>

/* KDE */
#ifdef ENABLE_KDE
#include <klocale.h>
#else
//#include <q3listview.h>
#endif

/* opale */
#include "opale.h"

class QPainter;
class QDomElement;
class QDomDocument;
class Ui_inputRecordDialog;
class Ui_inputPatternDialog;
class OpaleDocument;


/**
  * @short Atom of data used to display the graph.
  *
  * It holds the data representing a point in time, with the different
  * values corresponding to the different curves displayed : best, worst,
  * balance, ...
  *
  * @author Thomas Capricelli <orzel@freehackers.org>
  */
struct GraphData {
	GraphData(void ){balance=best=worst=best2=worst2=0;}
	QDateTime	qdt;
	amount_t	balance;
	amount_t	best;
	amount_t	worst;
	amount_t	best2;
	amount_t	worst2;
};

/**
  * @short Data for the Pie graph.
  *
  * It holds the data that the Pie graph needs to know, such as the total
  * for every category or every method.
  *
  * @author Thomas Capricelli <orzel@freehackers.org>
  */
struct StatsData {
	QVector<int> methodPositive;
	QVector<int> categoryPositive;
	QVector<int> methodNegative;
	QVector<int> categoryNegative;
};


/**
  * @short Store the data representing a pattern
  *
  * A pattern is an record with some or all field already completed. (but
  * the date, of course)
  *
  * @author Thomas Capricelli <orzel@freehackers.org>
  */
class OpalePattern
{
	friend class OpaleDocument;

public:
	OpalePattern(OpaleDocument *d, QString _name="<empty>", amount_t _amount=0);
	OpalePattern(OpaleDocument *d, const QDomElement &);
	OpalePattern(OpaleDocument *d, Ui_inputPatternDialog *);
	OpalePattern(const OpalePattern &other);

	/** Returns whether this pattern is valid or not. This is used to
	  * check validity after using a constructor */
	bool		isValid(void) { return b_valid; }
	amount_t	amount(void) { return m_amount; }

	/** Returns a QDomElement for this pattern */
	QDomElement	domElement(QDomDocument);
	/** Fill the dialog with values from this record */
	void		to_dialog(Ui_inputPatternDialog *);
	/** Fill this record from values from the dialog */
	void		from_dialog(Ui_inputPatternDialog *);

protected:
	bool		b_valid;
	QString		name;
	/** amount in the OpaleDocument currency. The actual amount stored is
	  * 100*real_amount, so as to use fixed-point decimal.
	  * see http://en.wikipedia.org/wiki/Fixed-point_arithmetic
	  */
	amount_t	m_amount;
	/** The method index, refering to the method table in OpaleDocument */
	int		method;
	/** The category index, refering to the category table in OpaleDocument */
	int		category;
	
	OpaleDocument *p_data; // associated data
};


/**
  * @short Store the data representing a record
  *
  * A record gathers all the information for a given change on the account
  *
  * @author Thomas Capricelli <orzel@freehackers.org>
  */
class OpaleRecord : public OpalePattern, public QTreeWidgetItem
{
	friend class OpaleDocument;
public:
	OpaleRecord (OpaleDocument *d, QString _name="<empty>", amount_t _amount=0);
	OpaleRecord (OpaleDocument *d, const QDomElement &);
	OpaleRecord (OpaleDocument *d, Ui_inputRecordDialog * );
	OpaleRecord (OpaleDocument *, const OpalePattern &p);
	OpaleRecord (OpaleDocument *, const OpaleRecord &r);

	QDomElement	domElement(QDomDocument);
	/** Fill the dialog with values from this record */
	void		to_dialog(Ui_inputRecordDialog *);
	/** Fill this record from values from the dialog */
	void		from_dialog(Ui_inputRecordDialog *);

	void            updateTotal(Total &t);
//	QTreeWidgetItem*	fillListView(Total &t, KListView *);

	/** reimplements QTreeWidgetItem::operator<() */
	bool 		operator< ( const QTreeWidgetItem & other ) const ;

//	void	check(void) {check = true;}
	void	unCheck(void) { check = false;}
	/** getter for the 'checked' attributes */
	bool	checked(void) { return check; }
	/** setter for the 'checked' attributes */
	void	setChecked(bool f) { check = f; }
	/** switch this record between checked and uncheck */
	void    switchCheck(void) { check = !check; }


	
	amount_t	safeAmount(void) { return (check||m_amount<0)?m_amount:0l; }
	void            addToChart( GraphData & );
	/** returns true if the date for this record is in the future */
	bool		isFutur(void);
	/** returns true if the date for this record is in the past */
	bool		isPast(void) { return !isFutur(); }
	/**
	  *  add some time to the current record
	  *  @param unit is 0 for day, 1 for week, 2 for month, and 3 for year
	  *  @param nb is how many of those needs to be added
	  */
	void		addTime(int nb, int unit);
private:
	void            updateOutput(void);
	QDate	date;
	bool	check;
};

#endif /* OPALE_RECORD */
