/*
*  RAL -- Rubrica Addressbook Library
*  file: abook.h
*  
*  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 3 of the License
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifndef _R_ABOOK_H
#define _R_ABOOK_H 

#include <glib.h>
#include <glib-object.h>

#include "card.h"
#include "error.h"

#define R_ABOOK_TYPE             (r_abook_get_type())

#define R_ABOOK(obj)             (G_TYPE_CHECK_INSTANCE_CAST((obj),    \
                                  R_ABOOK_TYPE, RAbook))

#define R_ABOOK_CLASS(klass)     (G_TYPE_CHECK_CLASS_CAST((klass),     \
                                  R_ABOOK_TYPE, RAbookClass))

#define IS_R_ABOOK(obj)          (G_TYPE_CHECK_INSTANCE_TYPE((obj),    \
                                  R_ABOOK_TYPE))

#define IS_R_ABOOK_CLASS(klass)  (G_TYPE_CHECK_CLASS_TYPE((klass),     \
                                  R_ABOOK_TYPE))

#define R_ABOOK_GET_CLASS(klass) (G_TYPE_INSTANCE_GET_CLASS((klass),  \
                                  R_ABOOK_TYPE, RAbookClass))



typedef struct _RAbook        RAbook;  
typedef struct _RAbookClass   RAbookClass;
typedef struct _RAbookPrivate RAbookPrivate;


struct _RAbook
{
  GObject parent; 
  
  RAbookPrivate *priv;
};


struct _RAbookClass
{
  GObjectClass parent_class;
  
  /* methods */
  gboolean (*abook_open_file) (RAbook* abook, gchar* filename);
  gboolean (*abook_save_file) (RAbook* abook, gchar* filename, gint rate);
  gboolean (*abook_overwrite) (RAbook* abook, gint rate);

  /* signals */
  void (*open_fail)           (RAbook* abook, RError* err);
  void (*save_fail)           (RAbook* abook, RError* err);

  void (*card_read)           (RAbook* abook, gpointer card);
  void (*card_added)          (RAbook* abook, gpointer card);
  void (*card_deleted)        (RAbook* abook, gpointer card);
  void (*card_destroyed)      (RAbook* abook, gchar* name);
  void (*card_recovered)      (RAbook* abook, gpointer card);
  void (*card_replaced)       (RAbook* abook, gpointer card);

  void (*card_cutted)         (RAbook* abook, gpointer card); 
  void (*card_pasted)         (RAbook* abook, gpointer card); 

  void (*need_name)           (RAbook* abook);
  void (*addressbook_changed) (RAbook* abook);
  void (*addressbook_read)    (RAbook* abook);
  void (*addressbook_saved)   (RAbook* abook);
  void (*addressbook_closed)  (RAbook* abook);
};


GType    r_abook_get_type            (void);


RAbook*  r_abook_new                  (void);
void     r_abook_free                 (RAbook* abook);

gboolean r_abook_open_file            (RAbook* abook, gchar* fname);
gboolean r_abook_save_file            (RAbook* abook, gchar* fname, 
				       gint compression_rate);
gboolean r_abook_overwrite_file       (RAbook* abook, gboolean backup, 
				       gint compression_rate);

RAbook*  r_abook_copy                 (RAbook* abook);

gboolean r_abook_load_plugin          (RAbook* abook, const gchar* name);
gboolean r_abook_plugin_from_file     (RAbook* abook, gchar* filename);
gchar*   r_abook_get_plugin_extension (RAbook* abook);
gchar*   r_abook_get_plugin_name      (RAbook* abook);
gpointer r_abook_get_engine           (RAbook* abook);
gpointer r_abook_get_plugin           (RAbook* abook);

gint     r_abook_get_items            (RAbook* abook);
gint     r_abook_get_deleted          (RAbook* abook);
gboolean r_abook_is_empty             (RAbook* abook);

void     r_abook_add_card             (RAbook* abook, RCard* card);
void     r_abook_add_loaded_card      (RAbook* abook, RCard* card);
void     r_abook_paste_card           (RAbook* abook, RCard* card);
void     r_abook_empty_trash          (RAbook* abook);
void     r_abook_delete_card          (RAbook* abook, RCard* card);
void     r_abook_destroy_card         (RAbook* abook, RCard* card);
gboolean r_abook_recovery_card        (RAbook* abook, RCard* card);

void     r_abook_replace_card         (RAbook* abook, RCard* old, RCard* new);

gpointer r_abook_get_card             (RAbook* abook);
gpointer r_abook_get_next_card        (RAbook* abook);
gpointer r_abook_get_prev_card        (RAbook* abook);
void     r_abook_reset_book           (RAbook* abook);

void     r_abook_foreach_card         (RAbook* abook, RFunc func, 
				       gpointer data);

gpointer r_abook_get_card_by_id       (RAbook* abook, glong id);
gpointer r_abook_get_selected_card    (RAbook* abook);
void     r_abook_unselect_cards       (RAbook* abook);

gpointer r_abook_get_groups_box       (RAbook* abook);

GList*   r_abook_find_cards_by_group  (RAbook* abook, const gchar* group_name);
GList*   r_abook_find_cards_by_type   (RAbook* abook, const gchar* type);
GList*   r_abook_find_cards_by_genre  (RAbook* abook, const gchar* genre);
GList*   r_abook_find_cards_by_rate   (RAbook* abook, RRate rate);

GList*   r_abook_search               (RAbook* abook, const gchar* str);
GList*   r_abook_search_date          (RAbook* abook, gint date, 
				       SearchType type);
GList*   r_abook_search_between       (RAbook* abook, gint first, 
				       gint second, SearchType type);


#endif
