//
// anyRemote
// a bluetooth remote for your PC.
//
// Copyright (C) 2006,2007,2008,2009 Mikhail Fedotov <anyremote@mail.ru>
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
//

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <termios.h>
#include <sys/stat.h>

#include "parse.h"
#include "utils.h"

#define AUTOCONN_STR   	"AutoConnect"
#define AUTOREPEAT_STR	"AutoRepeat"
#define BAUD_STR    	"Baudrate"
#define CHARSET_STR    	"CharSet"
#define CMER_OFF_STR   	"CmerOff"
#define CMER_ON_STR   	"CmerOn"
#define DEVICE_STR   	"Device"
#define LOG_STR    	"Log"
#define RETRY_STR    	"RetrySecs"
#define SCREEN_STR      "Screen"
#define TOMAIN_STR   	"ToMainMenu"

#include <regex.h>

#define REGEX_KEYVAL   "^[[:space:]]*(\\[[[:alpha:]]+\\])([^=]*)$|^[[:space:]]*([^=%]*[^[:space:]]{1})[[:space:]]*=[[:space:]]*(.*[^[:space:]]{1})[[:space:]]*$"
#define REGEX_CMDBYCMD "[[:space:]]*(Exec|ExecAndSet|Make|Get|Set|Send|ExecAndSend|Timer|Macro|Load|Include|SendCKPD|Emulate)[[:space:]]*(\\()|[[:space:]]*(Exit)[[:space:]]*;{0,1}"

#define REGEX_SET    "[[:space:]]*(bg|editfield|filemanager|fg|font|fullscreen|icons|list|iconlist|menu|parameter|skin|status|text|title|volume|image|cover|var)[[:space:]]*,[[:space:]]*(.*)|[[:space:]]*(vibrate|repaint|disconnect)[[:space:]]*"
#define REGEX_GET    "[[:space:]]*(screen_size|cover_size|model|version|cursor|ping|password)[[:space:]]*|[[:space:]]*(is_exists)[[:space:]]*,[[:space:]]*(.*)[[:space:]]*,[[:space:]]*(.*)[[:space:]]*"
#define REGEX_TIMER  "[[:space:]]*([^[:space:]]+)[[:space:]]*,[[:space:]]*([[:digit:]]*)[[:space:]]*,[[:space:]]*([[:digit:]]*)[[:space:]]*$|[[:space:]]*([^[:space:]]+)[[:space:]]*,[[:space:]]*(cancel|pause|reset|restart|continue)"
#define REGEX_MAKE   "[[:space:]]*(remote|mode|var)[[:space:]]*,[[:space:]]*(.*)[[:space:]]*|[[:space:]]*(exit|flush|stop|disconnect|none)[[:space:]]*"

#define REGEX_SET_TL "[[:space:]]*(fg|bg|font|select)[[:space:]]*,(.*)|[[:space:]]*(add|replace)[[:space:]]*,[[:space:]]*([^[:space:]]{1}[^,]*[^[:space:]]{1})[[:space:]]*,[[:space:]]*(.*)|[[:space:]]*(close)[[:space:]]*,[[:space:]]*(clear)[[:space:]]*|[[:space:]]*(clear|close|show)[[:space:]]*"
#define REGEX_SET_MN "[[:space:]]*(add|replace)[[:space:]]*,(.*)|[[:space:]]*(clear)[[:space:]]*"
#define REGEX_SET_WM "[[:space:]]*(icon|window)[[:space:]]*,(.*)|[[:space:]]*(remove_all|clear_cache|show|close|cursor|nocursor|dynamic_cursor)[[:space:]]*|[[:space:]]*(set_cursor),[[:space:]]*(.*)[[:space:]]*"
#define REGEX_SET_FM "[[:space:]]*(add|replace|select)[[:space:]]*,[[:space:]]*(left|right)[[:space:]]*,[[:space:]]*(.*)|[[:space:]]*(close|show)[[:space:]]*"
#define REGEX_SET_VR "^[[:space:]]*([^,]*)[[:space:]]*,[[:space:]]*(.*)[[:space:]]*"

#define SEPARATOR 	'/'

extern char tmp[MAXMAXLEN];

static char load_buf[MAXCMDLEN]; 

regex_t *keyVal   = NULL;
regex_t *cmdByCmd = NULL;

mode       *modes        = NULL;
mode       *currentMode  = NULL;
mode       *internalMode = NULL;
mode 	   *defMode      = NULL;
type_alias *aliases      = NULL;
type_key   *alarms       = NULL;

CONF conf = { 0, 		// Log
              B19200, 		// Baudrate
              60, 		// Retrysecs
              0, 		// use screen
              0, 		// auto connect
              0,		// auto repeat
              "8859-1",         // charset
              DEFAULT_DEVICE,   // device
	      "",               // to main menu
    	      "",               // CMER ON
	      "",               // CMER OFF
	      MODEL_DEFAULT,	// Model
	      NULL,             // Cfg. dir
	      -1,		// Front-end port    
	      0,		// Work with anyremote2html
	      0,		// Ask password ?
	      "anyRemote",      // Service name
	      0,		// set uid
	      0};        	// set gid

type_key* findExact(mode *mode, char *key)
{
	//logger("DEBUG", "findExact()");
        
        if (mode == NULL || key == NULL) {
        	logger("DEBUG", "findExact() input is empty ?");
        	return NULL;
        }
	type_key* It = mode->keys;

	// Search exact command
	while (It != NULL && It->key != NULL && strcmp(It->key,key) != 0) {
		//sprintf(tmp,"findExact search >%s< compare to >%s<", It->key, key);
		//logger("DEBUG",tmp);   

		It = (type_key*) It->next;
        	//logger("DEBUG", "findExact() next loop");
	}
        //logger("DEBUG", "findExact() exiting");
	return It;
}

static mode* findMode(const char *name) 
{
	mode *mp = modes;
	while (mp && strcmp(name, mp->name) != 0) {
		mp = (mode*) mp->next;
	}
	if (mp) {
		DEBUG2("findMode() %s = %s",name,mp->name);
	}
	return mp;
}

void setMode (const char *modeName) 
{
	if (modeName ==  NULL) {
               logger("DEBUG", "setMode() NULL input");
               return;
	}
	DEBUG2("setMode() to %s", modeName);

	mode *mp = findMode(modeName);

	if (mp) {
		currentMode = mp;
		DEBUG2("setMode() new mode was set to %s", currentMode->name);
	} else {
		logger("DEBUG", "setMode() new mode did not found");
	}

	return;
}

static void setCfgDir(char *d) 
{
	//printf("CFG DIR set TO >%s<\n",(d?d:"NULL"));
	if (conf.cfgDir != NULL) {
        	free(conf.cfgDir);
		conf.cfgDir = NULL;
	}
	
        if (getenv("AR_CFGDIR")) {	// override if set
		conf.cfgDir = strdup(getenv("AR_CFGDIR"));
		//printf("CFG DIR >%s<\n",(conf.cfgDir?conf.cfgDir:"NULL"));
		return;
        } 
	
	
	if (d != NULL) {	// search ../Utils directory
	
		char* try2 = malloc(strlen(d) + 10); // + "/../Utils"
		strcpy(try2,d);
		strcat(try2,"/../Utils");
		
		struct stat buf;

        	if(stat(try2, &buf) == 0) {
			try2[strlen(try2)-6] = '\0';
			conf.cfgDir = strdup(try2);
			
			free(try2);
			//printf("CFG DIR >%s<\n",(conf.cfgDir?conf.cfgDir:"NULL"));
			return;
		}
		free(try2);
        } 
	
	// last resort: use {prefix}/anyremote/cfg-data directory
        #ifdef DATADIR
 	conf.cfgDir = malloc(strlen(DATADIR)+20);	// + "/anyremote/cfg-data"
        strcpy(conf.cfgDir,DATADIR);
        strcat(conf.cfgDir,"/anyremote/cfg-data");
	#else
 	conf.cfgDir = malloc(2);
        strcpy(conf.cfgDir,".");
        #endif

	//printf("CFG DIR >%s<\n",(conf.cfgDir?conf.cfgDir:"NULL"));
}

/* ----------------- Setup configuration functions ------------------------- */

static int setBaudrate(int rate) 
{
  switch (rate) {
	case 300:
		return B300;
	case 1200:
		return B1200;
	case 2400:
		return B2400;
	case 9600:
		return B9600;
	case 19200:
        	return B19200;
	case 38400:
        	return 38400;
	#ifdef B57600
	case 57600:
		return B57600;
	#endif
	#ifdef B115200
	case 115200:
		return B115200;
	#endif
	#ifdef B230400
	case 230400:
		return B230400;
	#endif
	default:
		printf("WARNING: bad baudrate %d, defaulting to 9600\n", rate);
  }

  return B9600;
}

/* Parses command line options and set flags. */
int parse_opts(int argc, char *argv[])
{
	int i;
	for(i=1; i<argc; i++) {
		if (strcmp(argv[i],"-log")==0 || strcmp(argv[i],"-l")==0) {
			conf.log=2;
		} else if (strcmp(argv[i],"-fe")==0 && i+1<argc) {
			conf.frontEnd=atoi(argv[++i]);
		} else if ((strcmp(argv[i],"--serial")==0 || strcmp(argv[i],"-s")==0) && i+1<argc) {
			strcpy(conf.device, argv[++i]);
		} else if ((strcmp(argv[i],"--baud")==0 || strcmp(argv[i],"-b")==0) && i+1<argc) {
			conf.baudrate = setBaudrate(atoi(argv[++i]));
		} else if (strcmp(argv[i],"--noscreen")==0 || strcmp(argv[i],"-n")==0) {
			conf.screen = 0;
		} else if (strcmp(argv[i],"-http")==0) {
			conf.http = 1;
		} else if (strcmp(argv[i],"--autoconnect")==0 || strcmp(argv[i],"-a")==0) {
			conf.autoConnect = 1;
		} else if (strcmp(argv[i],"--user")==0 || strcmp(argv[i],"-u")==0) {
			if(getUidGid(argv[++i], &conf.uid, &conf.gid)!=EXIT_OK)
				printf("WARNING: bad username %s\n", argv[i]);
		} else if (strcmp(argv[i],"-name")==0) {
                        strncpy(conf.serviceName, argv[++i],MTEXTLEN-1);
			conf.serviceName[MTEXTLEN-1] = '\0';
		} else if (strcmp(argv[i],"-password")==0) {
                        conf.pass = 1;
		} else if (strcmp(argv[i],"-f")==0) {
			++i;
			continue;   // already processed this parameter
		// -h and -v handled in main.c
		} else {
			printf("ERROR: Unknown input parameter %s\n", argv[i]);
		}
	}
	return 1;
}

static void regexpPrepare(regex_t *r, char *p) 
{
	int err_no=0;

	if((err_no=regcomp(r, p, REG_EXTENDED))!=0) {
		size_t length; 
		char *buffer;
		length = regerror (err_no, r, NULL, 0);
		buffer = malloc(length);
		regerror (err_no, r, buffer, length);
		printf("regexpPrepare(): regcomp() error %s", buffer);
                
		free(buffer);
		regfree(r);
                free(r);
                
		exit(1);	// exit since we can't parse cfg
	}

	return;
}

static void deleteSpaces(char **str, int afterCommas)
{
	char *n = (*str); 
	char *r = (*str);
        int  c  = 0;

	//printf("INFO : deleteSpaces in: >%s<\n", *str);

        while (*r != '\0') { 
		if (isspace(*r) && c >= afterCommas) { 
			r++; 
		} else { 
			if (*r == ',') { 
				c++; 
			} 
			*n = *r;
			r++; 
			n++; 
		} 
	} 
	*n = '\0';

	//printf("INFO : deleteSpaces out: >%s<\n", *str);
}

static void normalizeSequence(char **str)
{
	int isSpace = 0;
	
	char *n = (*str); 
	char *r = (*str);

	//printf("INFO : normalizeSequence in: >%s<\n", *str);

	while (*r != '\0') {
		if (isspace(*r)) {
			// Skip spaces
			if(!isSpace) {
				isSpace = 1;
			}
			r++;
		} else if (*r == '(' && r > *str && strstr(*str,"$$")) {  // This is parametrized command Command($$), we have to delete all spaces from brace
			r--;
			deleteSpaces(&r, 0);
			return;
		} else {
			if(isSpace) {
				*n = ' ';
				n++;
			}
			// Copy all symbols
			*n = *r;
			r++;
			n++;
			isSpace = 0;
		}
	}
	*n = '\0';
	    
	//printf("INFO : normalizeSequence out: >%s<\n", *str);
}

static regmatch_t* allocRegmatch(int no_sub)
{
	regmatch_t* result;
	if ((result = (regmatch_t *) malloc(sizeof(regmatch_t) * no_sub))==0) {
        	printf("allocRegmatch(): No more memory");
        	exit(1);
        }
        return result;
}

static int parseKeyValue(char *in, char ** tag, char ** value)
{
	//printf("INFO: parseKeyValue() %s (%d)\n", in, (int)keyVal->re_nsub+1);

	size_t no_sub = keyVal->re_nsub+1;    
	regmatch_t* result = allocRegmatch(no_sub);

	if (in[strlen(in) - 1] == '\n') {
		in[strlen(in) - 1] = '\0';
	}

	*value =  NULL;
	*tag   =  NULL;
	if (regexec(keyVal, in, no_sub, result, 0)==0) {
		int use1 = 1;
		int use2 = 2;
		if (result[1].rm_so == -1) {
			if (result[3].rm_so == -1) {
				printf("parseKeyValue(): Incorrectly formed command (1) %s\n", in);
				use1 = -1;
			}
			use1 = 3;
		} 
		if (result[2].rm_so == -1) {
			if (result[4].rm_so == -1) {
				printf("parseKeyValue(): Incorrectly formed command (2) %s\n", in);
				use2 = -1;
			}
			use2 = 4;
		} 

		if (use1 > 0) {
			*tag   = in + result[use1].rm_so;
			*(in + result[use1].rm_eo) = '\0';
		}
 
		if (use2 > 0) {
			if (result[use2].rm_so == -1 || result[use2].rm_eo == -1) {
				*value =  NULL;
			} else {  
				*value = in + result[use2].rm_so;
				*(in + result[use2].rm_eo) = '\0';
			}
		}
		
		/*if(*tag != NULL) {
			 printf("INFO : Got tag   : >%s<\n", *tag);
		} else {
			printf("INFO : Got tag : >NULL<\n");
		}
		if(*value != NULL) {
			 printf("INFO : Got value : >%s<\n", *value);
		}else {
			 printf("INFO : Got value : >NULL<\n");
		}*/
		
	//} else {
	//	printf("INFO : Not matched\n");
	}

	free(result);
	return EXIT_OK;
}

char* id2Cmd (int cmdId)
{
	switch(cmdId) {
		case ID_EXIT:		    return CMD_EXIT;
		case ID_EXEC:		    return CMD_EXEC;
		case ID_SENDCKPD:	    return CMD_SENDCKPD;
		case ID_SET:		    return CMD_SET;
		case ID_EXECSET:	    return CMD_EXECSET;
		case ID_TIMER:  	    return CMD_TIMER;
		case ID_SEND:		    return CMD_SEND;
		case ID_EXECSEND:	    return CMD_EXECSEND;
		case ID_MACRO:  	    return CMD_MACRO;
		case ID_LOAD:		    return CMD_LOAD;
		case ID_INCLUDE:	    return CMD_INCLUDE;
		case ID_GET:		    return CMD_GET;
		case ID_MAKE:		    return CMD_MAKE;
		case ID_EMU:		    return CMD_EMU;
	
		default:		    return "Unknown";
	}
}

static int cmd2id (char *name) 
{

	if (strcmp(name,CMD_EXIT) == 0) {
        	return ID_EXIT;
        } else if (strcmp(name,CMD_EXEC) == 0) {
        	return ID_EXEC;
        } else if (strcmp(name,CMD_SENDCKPD) == 0) {
      		return ID_SENDCKPD;
        } else if (strcmp(name,CMD_SET) == 0) {
		return ID_SET;
        } else if (strcmp(name,CMD_EXECSET) == 0) {
		return ID_EXECSET;
        } else if (strcmp(name,CMD_TIMER) == 0) {
		return ID_TIMER;
        } else if (strcmp(name,CMD_SEND) == 0) {
		return ID_SEND;
        } else if (strcmp(name,CMD_EXECSEND) == 0) {
		return ID_EXECSEND;
        } else if (strcmp(name,CMD_MACRO) == 0) {
		return ID_MACRO;
        } else if (strcmp(name,CMD_LOAD) == 0) {
		return ID_LOAD;
        } else if (strcmp(name,CMD_INCLUDE) == 0) {
		return ID_INCLUDE;
        } else if (strcmp(name,CMD_GET) == 0) {
		return ID_GET;
        } else if (strcmp(name,CMD_MAKE) == 0) {
		return ID_MAKE;
        } else if (strcmp(name,CMD_EMU) == 0) {
		return ID_EMU;
        } 
        return ID_UNKNOWN;
}

int cmdSet2id (const char *name) 
{
	if (name == NULL) 
        	return ID_UNKNOWN;
                
        if (strncmp(name, SET_STATUS, strlen(SET_STATUS)) == 0) {
		return ID_STATUS;
        } else if (strncmp(name, SET_TITLE, strlen(SET_TITLE)) == 0) {
		return ID_TITLE;
        } else if (strncmp(name, SET_COVER, strlen(SET_COVER)) == 0) {
		return ID_COVER;
        } else if (strncmp(name, SET_ICONS, strlen(SET_ICONS)) == 0) {
		return ID_ICONS;
        } else if (strncmp(name, SET_VOLUME, strlen(SET_VOLUME)) == 0) {
		return ID_VOLUME;
        } else if (strncmp(name, SET_SKIN, strlen(SET_SKIN)) == 0) {
		return ID_SKIN;
	} else if (strncmp(name, SET_BG, strlen(SET_BG)) == 0) {
        	return ID_BG;
        } else if (strncmp(name, SET_FG, strlen(SET_FG)) == 0) {
		return ID_FG;
        } else if (strncmp(name, SET_FONT, strlen(SET_FONT)) == 0) {
		return ID_FONT;
        } 
        return ID_UNKNOWN;
}

static void addAlias(char*old, char *new)
{
	type_alias* Al = (type_alias*) calloc(1, sizeof(type_alias));

	// Insert in head
	if (aliases == NULL) {   // first elem
		Al->next = NULL;
	} else {
		Al->next = (type_alias*) aliases;
	}
	aliases = Al;
	
	Al->key   = (char *) malloc(strlen(old)+1);
	Al->alias = (char *) malloc(strlen(new)+1);

	strcpy(Al->key,  old);
	strcpy(Al->alias,new);
}

/* preparation for i18n
static char* translateWord (const char *word) 
{
	char *trWord = (char *) malloc(strlen(word)*2+1);
	strcpy(trWord,word);
	strcat(trWord,word);
	return trWord;
}

static char* translateMenu (const char *menuString) 
{
	//printf("trMenu %d %s\n", strlen(menuString),menuString);
	char *bufPtr;
	char *menuString2 = strdup(menuString);

        char *token = strtok_r(menuString2,",",&bufPtr);
	char *trString = NULL;
	int  init = 1;
	while (token != NULL) {
		// translate token
		//printf("trMenu token = %s\n", token);fflush(stdout);
		char * trToken = translateWord(token);
		//printf("trMenu tr token = %s\n", trToken);fflush(stdout);
                
		addAlias(token,trToken);
		
		//printf("trMenu realloc %d->%d\n", 
		//           (trString == NULL ? 0 : strlen(trString)), 
		//	   (trString == NULL ? 0 : strlen(trString))+strlen(trToken)+2);fflush(stdout);

		trString = (char *) realloc(trString,(trString == NULL ? 0 : strlen(trString))+strlen(trToken)+2);
		if (init) {
			*trString = '\0';
			strcpy(trString,trToken);
			init = 0;
		} else {
			strcat(trString,trToken);
		}	
		
		free(trToken);
		token = strtok_r(NULL,",",&bufPtr);
		
		if (token != NULL) {
			strcat(trString,",");
		}
	}
	
	free(menuString2);
	//printf("trMenu %d %s\n",strlen(trString),trString);fflush(stdout);
	return trString;
}
*/

static int storeGetCmd(cmdItem* ci, char *cmd)
{
	int parseFail = 0;
        
	regex_t* regex  = (regex_t *) malloc(sizeof(regex_t));
	memset(regex, 0, sizeof(regex_t));
                
	regexpPrepare(regex, REGEX_GET);
        
	size_t no_sub = regex->re_nsub+1;	
	
        regmatch_t* result = allocRegmatch(no_sub);
	
	if (regexec(regex, cmd, no_sub, result, 0) == 0) {	// screen_size, cover_size, model
		if (result[1].rm_so >= 0) {
                	ci->descr = (char*) calloc(1, result[1].rm_eo - result[1].rm_so + 1);
                        strncpy(ci->descr, cmd + result[1].rm_so, result[1].rm_eo - result[1].rm_so);
                } else if (result[2].rm_so >= 0) {		// is_exists

			int l2 = result[2].rm_eo - result[2].rm_so;
			int l3 = result[3].rm_eo - result[3].rm_so;
			int l4 = result[4].rm_eo - result[4].rm_so;

                        ci->descr = (char*) calloc(1, l2 + l3 + l4 + 3);
                        strncpy(ci->descr,cmd + result[2].rm_so, l2);
                	strcat (ci->descr,",");
                	strncat(ci->descr,cmd + result[3].rm_so, l3);
                
                	strcat (ci->descr,",");
                	strncat(ci->descr,cmd + result[4].rm_so, l4);
		}
        } else {
                printf("storeGetCmd(): parse error\n"); 
        	parseFail = 1;
        }
        
        if (parseFail) {
        	printf("storeGetCmd(): command Get( %s ) is formed incorrectly\n", cmd); 
        }
        
        free(result); result = NULL;
        regfree(regex); free(regex); regex = NULL;
        
	return (parseFail == 0 ? EXIT_OK : EXIT_NOK);
}

static int storeMakeCmd(cmdItem* ci, char *cmd)
{
	int parseFail = 0;
        //printf ("storeMakeCmd: got >%s<\n",cmd);
        
	regex_t* regex  = (regex_t *) malloc(sizeof(regex_t));
	memset(regex, 0, sizeof(regex_t));
                
	regexpPrepare(regex, REGEX_MAKE);
        
	size_t no_sub = regex->re_nsub+1;	
	
        regmatch_t* result = allocRegmatch(no_sub);
        
	if (regexec(regex, cmd, no_sub, result, 0) == 0) {
        
		if (result[1].rm_so >= 0 && result[2].rm_so > 0) {			
                	
                        int l = result[1].rm_eo - result[1].rm_so;
                        
                        if (strncmp(cmd+result[1].rm_so,MAKE_VAR,l) == 0) {			// var

				regex_t* regex2  = (regex_t *) malloc(sizeof(regex_t));
				memset(regex2, 0, sizeof(regex_t));
				
				regexpPrepare(regex2, REGEX_SET_VR);
				size_t no_sub2 = regex2->re_nsub+1;
        
        			regmatch_t* result2 = allocRegmatch(no_sub2);
                                char* start2 = cmd + result[2].rm_so;

                                int l2 = 0;
                                
                                if (regexec(regex2, cmd + result[2].rm_so, no_sub2, result2, 0) == 0) {
                                
                                	if (result2[1].rm_so >= 0 && result2[2].rm_so >= 0) {
					 
                                        	l2 = result2[1].rm_eo - result2[1].rm_so;
                                        	int le = result2[2].rm_eo - result2[2].rm_so;
                                        	
                                         	ci->descr = (char*) calloc(1, l + l2 + 2);
                                        	strncpy(ci->descr,cmd + result[1].rm_so, l);
						
                                        	strcat (ci->descr,",");

                                       		strncat(ci->descr,start2 + result2[1].rm_so, l2);

                                        	ci->exec = (char*) calloc(1, le + 1);
                                		strncpy(ci->exec,start2 + result2[2].rm_so, le);
                                        } else {
                                        	printf("storeMakeCmd(): parse error (M V1)\n"); 
                                        	parseFail = 1;
                                 	}       
                                } else {
                                        printf("storeMakeCmd(): parse error (M V2)\n"); 
                                        parseFail = 1;
                        	}
                         	regfree(regex2);free(regex2);regex2 = NULL;
                                free(result2);result2 = NULL;

                        } else if (strncmp(cmd + result[1].rm_so, MAKE_MODE,    l) == 0) {	// mode
                        
                        	ci->descr = (char*) calloc(1, result[1].rm_eo - result[1].rm_so + 1);
                                ci->exec  = (char*) calloc(1, result[2].rm_eo - result[2].rm_so + 1);
                                strncpy(ci->descr,cmd + result[1].rm_so, result[1].rm_eo - result[1].rm_so); 
                                strncpy(ci->exec, cmd + result[2].rm_so, result[2].rm_eo - result[2].rm_so); 
                                
                                //printf ("storeMakeCmd: (2) %s %s \n",ci->descr,ci->exec);
                        
                        } else if (strncmp(cmd + result[1].rm_so, MAKE_REMOTE, l) == 0) {	   // remote 
			
                        	if (strncmp(cmd + result[2].rm_so, "on",  2) == 0 ||
                                    strncmp(cmd + result[2].rm_so, "off", 3) == 0) {
                                	ci->descr = (char*) calloc(1, result[1].rm_eo - result[1].rm_so + 1);
                                        ci->exec  = (char*) calloc(1, result[2].rm_eo - result[2].rm_so + 1);
                                        strncpy(ci->descr,cmd + result[1].rm_so, result[1].rm_eo - result[1].rm_so); 
                                        strncpy(ci->exec, cmd + result[2].rm_so, result[2].rm_eo - result[2].rm_so); 
                               		
                                        //printf ("storeMakeCmd: (3) %s %s \n",ci->descr,ci->exec);
                                
				} else {
                                        printf("storeMakeCmd(): parse error: invalide parameter in Set(remote,...)\n"); 
                                	parseFail = 1;
                                }
                        } else {
                        	parseFail = 1;
                        	printf("storeMakeCmd(): parse error: unknown subcommand (1)\n"); 
                        }
 		} else if (result[3].rm_so >= 0) {						// disconnect, flush
                						
                	ci->descr = (char*) calloc(1, result[3].rm_eo - result[3].rm_so + 1);
                	strncpy(ci->descr,cmd + result[3].rm_so, result[3].rm_eo - result[3].rm_so);
                         
                } else {	
                        parseFail = 1;
			printf("storeMakeCmd(): parse error: unknown subcommand (2)\n"); 
		}
        } else {
                printf("storeMakeCmd(): parse error\n"); 
        	parseFail = 1;
        }
        
        if (parseFail) {
        	printf("storeMakeCmd(): command Make( %s ) is formed incorrectly\n", cmd); 
        }
        
        free(result); result = NULL;
        regfree(regex); free(regex); regex = NULL;
        
	return (parseFail == 0 ? EXIT_OK : EXIT_NOK);
}

static void storeMacroCmd(cmdItem* ci, char *cmd)
{
        //printf ("storeMacroCmd: got >%s<\n",cmd);

	char *comma = index(cmd,',');
	if (comma) {
        	*comma = '\0';
                comma++;
                while (comma && (*comma == ' ' || *comma == '\t')) {
                	comma++;	
                }
        }

	deleteSpaces(&cmd,0);

	char *dsc = calloc(1, strlen(cmd) + 1);
        strcpy(dsc, cmd);
        ci->descr = dsc;
        
        if (comma && *comma != '\0') {
		char *ex = calloc(1, strlen(comma) + 1);
        	strcpy(ex, comma);
        	ci->exec = ex;
        }
}

static int storeExecAndSetCmd(cmdItem* ci, char *cmd, int execFlag)
{
	int parseFail = 0;
        
	regex_t* regex  = (regex_t *) malloc(sizeof(regex_t));
	memset(regex, 0, sizeof(regex_t));
                
	regexpPrepare(regex, REGEX_SET);
        
	size_t no_sub = regex->re_nsub+1;	
	
        //printf("storeExecAndSetCmd() %s (%d)\n", cmd, (int)no_sub); 
	
        regmatch_t* result = allocRegmatch(no_sub);
         
	if (regexec(regex, cmd, no_sub, result, 0) == 0) {	// Match it
		
                //printf("storeExecAndSetCmd() matched %d %d\n", result[1].rm_so, result[1].rm_eo);
		
                if (result[1].rm_so >= 0 && result[2].rm_so > 0) {  // subcommand , params matched
                	
                        int l = result[1].rm_eo - result[1].rm_so;
                        
                	if (strncmp(cmd+result[1].rm_so,SET_BG,l)      == 0 ||
                            strncmp(cmd+result[1].rm_so,SET_PARAM,l)   == 0 ||
                            strncmp(cmd+result[1].rm_so,SET_EFIELD,l)  == 0 ||
                            strncmp(cmd+result[1].rm_so,SET_FG,l)      == 0 ||
                            strncmp(cmd+result[1].rm_so,SET_FONT,l)    == 0 ||
                            strncmp(cmd+result[1].rm_so,SET_FSCREEN,l) == 0 ||
                            strncmp(cmd+result[1].rm_so,SET_ICONS,l)   == 0 ||
                       	    strncmp(cmd+result[1].rm_so,SET_SKIN,l)    == 0 ||
                            strncmp(cmd+result[1].rm_so,SET_STATUS,l)  == 0 ||
                            strncmp(cmd+result[1].rm_so,SET_TITLE,l)   == 0 ||
                            strncmp(cmd+result[1].rm_so,SET_VOLUME,l)  == 0 ||
                            strncmp(cmd+result[1].rm_so,SET_COVER,l)   == 0
                           ) {
                                int le = result[2].rm_eo - result[2].rm_so;
                                int le2 = 0;
                                
                            	if (!execFlag) {
                                	le2 = le + 1;
                                }
                        	ci->descr = (char*) calloc(1, l + le2 + 1);
                       		strncpy(ci->descr,cmd + result[1].rm_so, l);
                                
                            	if (execFlag) {  
                        		ci->exec = (char*) calloc(1, le + 1);
                       			strncpy(ci->exec,cmd + result[2].rm_so, le);
                                } else {
                                	strcat (ci->descr,",");
                                	strncat(ci->descr,cmd + result[2].rm_so, le);
                                }
                        } else if (strncmp(cmd+result[1].rm_so,SET_FMGR,l) == 0) {
                        
				regex_t* regex2  = (regex_t *) malloc(sizeof(regex_t));
				memset(regex2, 0, sizeof(regex_t));
				regexpPrepare(regex2, REGEX_SET_FM);
				size_t no_sub2 = regex2->re_nsub+1;
        
        			regmatch_t* result2 = allocRegmatch(no_sub2);
                                char* start2 = cmd + result[2].rm_so;
				
                                //printf("parse %s\n", cmd + result[2].rm_so);
                                
                                if (regexec(regex2, cmd + result[2].rm_so, no_sub2, result2, 0) == 0) {
                        
                        
                                	int l2 = 0;
                                
                                	if (result2[1].rm_so >= 0 && result2[2].rm_so >= 0) {  		// add|replace|select,left|right,_data_
                        
                                        	l2 = result2[1].rm_eo - result2[1].rm_so;
                                                int l3 = result2[2].rm_eo - result2[2].rm_so;
                                                
                                        	int le = result2[3].rm_eo - result2[3].rm_so;
                                        	int le2 = 0;
                                        
                                        	if (!execFlag) {
                                                	le2 = le + 1;
                                                }
                                                
                                        	ci->descr = (char*) calloc(1, l + l2 + l3 + le2 + 3);
                                        	strncpy(ci->descr,cmd + result[1].rm_so, l);

                                        	strcat (ci->descr,",");
                                       		strncat(ci->descr,start2 + result2[1].rm_so, l2);
                                        	strcat (ci->descr,",");
                                       		strncat(ci->descr,start2 + result2[2].rm_so, l3);
                                                
                                                if (execFlag) {
                                        		ci->exec = (char*) calloc(1, le + 1);
                                			strncpy(ci->exec,start2 + result2[3].rm_so, le);
                                               	} else {
                                                	strcat (ci->descr,",");
                                			strncat(ci->descr,start2 + result2[3].rm_so, le);
                                                } 

                        		} else if (result2[4].rm_so >= 0) {				// close|show
                        
                                                l2 = result2[4].rm_eo - result2[4].rm_so;
                                                ci->descr = (char*) calloc(1, l + l2 + 2);
                                                strncpy(ci->descr,cmd + result[1].rm_so, l);
                                        	strcat (ci->descr,",");
                                        	strncat(ci->descr,start2 + result2[4].rm_so, l2);

                                 	} else {
                                                printf("storeExecAndSetCmd(): parse error (F1)\n"); 
                                		parseFail = 1;
                                	}
                                } else {
                                	printf("storeExecAndSetCmd(): parse error (F2)\n"); 
                                	parseFail = 1;
                                }
                         	regfree(regex2);free(regex2); regex2 = NULL;
                                free(result2);result2 = NULL;
                         
                        } else if (strncmp(cmd+result[1].rm_so,SET_LIST,l) == 0 ||
                                   strncmp(cmd+result[1].rm_so,SET_ILIST,l) == 0 ||
                                   strncmp(cmd+result[1].rm_so,SET_TEXT,l) == 0) {
                                 
				//printf("storeExecAndSetCmd() matched2 %d %d\n", result[2].rm_so, result[2].rm_eo);
                                
				regex_t* regex2  = (regex_t *) malloc(sizeof(regex_t));
				memset(regex2, 0, sizeof(regex_t));
				regexpPrepare(regex2, REGEX_SET_TL);
				size_t no_sub2 = regex2->re_nsub+1;
        
        			regmatch_t* result2 = allocRegmatch(no_sub2);
                                char* start2 = cmd + result[2].rm_so;
				
                                //printf("parse %s\n", cmd + result[2].rm_so);
                                
                                if (regexec(regex2, cmd + result[2].rm_so, no_sub2, result2, 0) == 0) {
                                
                                	int l2 = 0;
                                
                                	if (result2[1].rm_so >= 0) {  		// fg|bg|font|select
                                        
                                        	l2 = result2[1].rm_eo - result2[1].rm_so;
                                        	int le = result2[2].rm_eo - result2[2].rm_so;
                                        	int le2 = 0;
                                        
                                        	if (!execFlag) {
                                                	le2 = le + 1;
                                                }
                                        
                                        	ci->descr = (char*) calloc(1, l + l2 + le2 + 2);
                                        	strncpy(ci->descr,cmd + result[1].rm_so, l);

                                        	strcat (ci->descr,",");

                                       		strncat(ci->descr,start2 + result2[1].rm_so, l2);
                                                
                                                if (execFlag) {
                                        		ci->exec = (char*) calloc(1, le + 1);
                                			strncpy(ci->exec,start2 + result2[2].rm_so, le);
                                               	} else {
                                                	strcat (ci->descr,",");
                                			strncat(ci->descr,start2 + result2[2].rm_so, le);
                                                } 
                                 
                                	} else if (result2[3].rm_so >= 0) {	// add|replace,_title_
                                
                                		l2 = result2[3].rm_eo - result2[3].rm_so;
                                		int l3 = result2[4].rm_eo - result2[4].rm_so;
                                        	int le = result2[5].rm_eo - result2[5].rm_so;
                                        	int le2 = 0;
                                                
                                                if (!execFlag) {
                                                	le2 = le + 1;
                                                }

                                        	ci->descr = (char*) calloc(1, l + l2 + l3 + le2 + 3);
                                        	strncpy(ci->descr,cmd + result[1].rm_so,  l);
                                        	strcat (ci->descr,",");
                                        	strncat(ci->descr,start2 + result2[3].rm_so, l2);
                                        	strcat (ci->descr,",");
                                        	strncat(ci->descr,start2 + result2[4].rm_so, l3);
                                        	
                                                if (execFlag) {
                                        		ci->exec = (char*) calloc(1, le + 1);
                               				strncpy(ci->exec,start2 + result2[5].rm_so, le);
                                                } else {
                                                	strcat (ci->descr,",");
                                                        strncat(ci->descr,start2 + result2[5].rm_so, le);
                                                }
                                	
                                        } else if (result2[6].rm_so >= 0 && result2[7].rm_so >= 0) {	// close,clear
                                        	
                                                l2 = result2[6].rm_eo - result2[6].rm_so;
                                                int l3 = result2[7].rm_eo - result2[7].rm_so;
                                                ci->descr = (char*) calloc(1, l + l2 + l3 + 3);
                                                strncpy(ci->descr,cmd + result[1].rm_so, l);
                                        	strcat (ci->descr,",");
                                        	strncat(ci->descr,start2 + result2[6].rm_so, l2);
                                        	strcat (ci->descr,",");
                                        	strncat(ci->descr,start2 + result2[7].rm_so, l3);
                                
                                        } else if (result2[8].rm_so >= 0) {	// close|clear|show
                                        	
                                                l2 = result2[8].rm_eo - result2[8].rm_so;
                                                ci->descr = (char*) calloc(1, l + l2 + 2);
                                                strncpy(ci->descr,cmd + result[1].rm_so, l);
                                        	strcat (ci->descr,",");
                                        	strncat(ci->descr,start2 + result2[8].rm_so, l2);
                                
                                	} else {
                                                printf("storeExecAndSetCmd(): parse error (T1)\n"); 
                                		parseFail = 1;
                                	}
                                } else {
                                	printf("storeExecAndSetCmd(): parse error (T2)\n"); 
                                	parseFail = 1;
                                }
                         	regfree(regex2);free(regex2);regex2 = NULL;
                                free(result2);result2 = NULL;
                                
                        } else if (strncmp(cmd+result[1].rm_so,SET_MENU,l) == 0   ||
				   strncmp(cmd+result[1].rm_so,SET_IMAGE,l) == 0) {
				   
				regex_t* regex2  = (regex_t *) malloc(sizeof(regex_t));
				memset(regex2, 0, sizeof(regex_t));
				
				if (strncmp(cmd+result[1].rm_so,SET_MENU,l) == 0) {
					regexpPrepare(regex2, REGEX_SET_MN);
				} else {
					regexpPrepare(regex2, REGEX_SET_WM);
				}
				size_t no_sub2 = regex2->re_nsub+1;
        
        			regmatch_t* result2 = allocRegmatch(no_sub2);
                                char* start2 = cmd + result[2].rm_so;
                             	
                                int l2 = 0;
                                
                                if (regexec(regex2, cmd + result[2].rm_so, no_sub2, result2, 0) == 0) {
                                
                                	if (result2[1].rm_so >= 0) {  		// add|replace or icon|window
					 
                                        	l2 = result2[1].rm_eo - result2[1].rm_so;
                                        	int le = result2[2].rm_eo - result2[2].rm_so;
                                                int le2 = 0;
                                        	
                                                if (!execFlag) {
                                                	le2 = le + 1;
                                                }
                                                
                                        	ci->descr = (char*) calloc(1, l + l2 + le2 + 2);
                                        	strncpy(ci->descr,cmd + result[1].rm_so, l);

                                        	strcat (ci->descr,",");

                                       		strncat(ci->descr,start2 + result2[1].rm_so, l2);

                                        	if (execFlag) {
                                        		ci->exec = (char*) calloc(1, le + 1);
                                			strncpy(ci->exec,start2 + result2[2].rm_so, le);
                                                } else {
                                        		strcat (ci->descr,",");
							
							/*
							// attempt to use localized menu on the phone
							char* menu = (char *) calloc(1,le+1);
                                			strncpy(menu,start2 + result2[2].rm_so, le);
							
							char *trMenu = translateMenu(menu);
							ci->descr = realloc(ci->descr,
							               l + l2 + 3 + strlen(trMenu));
						        //printf("REALLOC %d -> %d\n",strlen(ci->descr),
							//               l + l2 + le2 + 2+strlen(trMenu)-le);
                                			strcat(ci->descr,trMenu); 
							
							free(menu);
							free(trMenu);
							
							// old non-localized variant
							*/
							strncat(ci->descr,start2 + result2[2].rm_so, le);
                                                }
                                        
                                        } else if (result2[3].rm_so >= 0) {	// clear or show|remove_all|clear_cache|close|cursor|nocursor|dynamic_cursor

                                                l2 = result2[3].rm_eo - result2[3].rm_so;
                                                ci->descr = (char*) calloc(1, l + l2 + 2);
                                                strncpy(ci->descr,cmd + result[1].rm_so, l);
                                        	strcat (ci->descr,",");
                                        	strncat(ci->descr,start2 + result2[3].rm_so, l2);
                        		
                                        } else if (result2[4].rm_so >= 0) {	// set_cursor

						int le = result2[5].rm_eo - result2[5].rm_so;

                                                l2 = result2[4].rm_eo - result2[4].rm_so;
						
						int le2 = 0;
                                                if (!execFlag) {
                                                	le2 = le + 1;
                                                }
						
                                                ci->descr = (char*) calloc(1, l + l2 + le2 + 2);
                                                strncpy(ci->descr,cmd + result[1].rm_so, l);
                                        	strcat (ci->descr,",");
                                        	strncat(ci->descr,start2 + result2[4].rm_so, l2);
                        		
                                        	if (execFlag) {
                                        		ci->exec = (char*) calloc(1, le + 1);
                                			strncpy(ci->exec,start2 + result2[5].rm_so, le);
                                                } else {
                                        		strcat (ci->descr,",");
                                			strncat(ci->descr,start2 + result2[5].rm_so, le);
                                                }
                                        } else {
                                        	printf("storeExecAndSetCmd(): parse error (M1)\n"); 
                                        	parseFail = 1;
                                 	}       
                        	}
                         	regfree(regex2);free(regex2);regex2 = NULL;
                                free(result2);result2 = NULL;
                                
                 	} else {
                        	printf("storeExecAndSetCmd(): parse error (M3)\n"); 
                        	parseFail = 1;
        		}
              	} else if (result[3].rm_so >= 0) {   // vibrate,repaint matched
                
                	ci->descr = (char*) calloc(1, result[3].rm_eo - result[3].rm_so + 1);
                        strncpy(ci->descr, cmd + result[3].rm_so, result[3].rm_eo - result[3].rm_so);
		
        	} else {
                	printf("storeExecAndSetCmd(): parse error (1)\n"); 
                	parseFail = 1;
                }
        } else {
        	// ExecAndSet(_shell_commands_) goes here ????
                printf("storeExecAndSetCmd(): parse error (2)\n"); 
        	parseFail = 1;
        }
        
        if (parseFail) {
        	if (execFlag) {
                	printf("storeExecAndSetCmd(): command ExecAndSet( %s ) is formed incorrectly\n", cmd);
                } else { 
        		printf("storeExecAndSetCmd(): command Set( %s ) is formed incorrectly\n", cmd); 
                }
        }
        
        free(result); result = NULL;
        regfree(regex);free(regex);regex = NULL;
	
	return (parseFail == 1 ? EXIT_NOK : EXIT_OK);
}

static int storeTimerCmd(cmdItem* ci, char *cmd)
{
	//printf("storeTimerCmd(): %s\n",cmd); 

	regex_t* regex  = (regex_t *) malloc(sizeof(regex_t));
	memset(regex, 0, sizeof(regex_t));
	regexpPrepare(regex, REGEX_TIMER);
        
	size_t no_sub = regex->re_nsub+1;	
	
        regmatch_t* result = allocRegmatch(no_sub);
	
        int cmdIsOk = EXIT_OK;
	if (regexec(regex, cmd, no_sub, result, 0) == 0) {	// Match it

		if (result[1].rm_so >= 0 &&	// Timer(key,0,5)
		    result[2].rm_so >  0 &&
		    result[3].rm_so >  0) {
                        
                	int m = result[1].rm_eo - result[1].rm_so;
                	if (m > MAXARGLEN-1) {
                		m = MAXARGLEN-1;
                	}
                        
                	ci->descr = (char*) calloc(1, m+1);
                	strncpy(ci->descr, cmd + result[1].rm_so, m);

			ci->exec  = NULL;

                	ci->tdata = (timerData*) calloc(1, sizeof(timerData));
                        
                	char iBuff[8];
                	int l = result[2].rm_eo - result[2].rm_so;
                	if (l > 7) l = 7;
                	strncpy(iBuff,cmd + result[2].rm_so,l);
                	ci->tdata->timeout = atoi(iBuff);
                
                	l = result[3].rm_eo - result[3].rm_so;
                	if (l > 7) l = 7;
                	strncpy(iBuff,cmd + result[3].rm_so,7);
                	ci->tdata->times = atoi(iBuff);
                
                } else if (result[4].rm_so >= 0 && 	// Timer(key,cancel|pause|continue)
                           result[5].rm_so >= 0) {  

                	int l1 = result[4].rm_eo - result[4].rm_so;
                	int l2 = result[5].rm_eo - result[5].rm_so;
                	
                        ci->descr = (char*) calloc(1, l1+1);
                	strncpy(ci->descr, cmd + result[4].rm_so, l1);

                        ci->exec = (char*) calloc(1, l2+1);
                	strncpy(ci->exec, cmd + result[5].rm_so, l2);

                	ci->tdata = NULL;
                }
	} else {
        	printf("storeTimerCmd(): command %s is formed incorrectly (2)\n", cmd);
                
                // info for next few releases
        	printf("Since v4.1 syntax of Timer() command was changed. Read documentation for details.\n");
                
                cmdIsOk = EXIT_NOK;
        }

        free(result); result = NULL;
        regfree(regex);free(regex);regex = NULL;
        
	return cmdIsOk;
}
       
int storeCmds(cmdItem** cmdList, char *value) 
{
	cmdItem *curCmd = (cmdItem*) NULL;
	int cmdId;

	size_t no_sub = cmdByCmd->re_nsub+1;	
	
	//printf("storeCmds() %s (%d)\n", value, (int)no_sub); 
	
	regmatch_t* result = allocRegmatch(no_sub);

	int start = 0;
	while(regexec(cmdByCmd, value+start, no_sub, result, 0)==0) {
		
		//printf("storeCmds() next %s\n", value+start); 

		int  step  = 0;
		char *name = NULL;
		char *cmds = NULL;
	
		int u1 = 1;
		if (result[1].rm_so >= 0 && result[2].rm_so) {  // <name>(<params>) matched  
			cmds = value + start + result[2].rm_so + 1;
			 
			// Search ) which is appropriate to matched (
			int braces = 1; 	// cmd_name( 
			char *ptr = cmds;	  
			while (*ptr != '\0' && braces > 0) {
				if (*ptr == '(') {
					braces++;
				}
				if (*ptr == ')') {
					braces--;
				}
				ptr++;
			} 
			step = ptr - value - start + 1;
			if (*(ptr-1) == ')') {
				*(ptr-1) = '\0';
			}
		} else if (result[3].rm_so >= 0) {  // Exit matched
			u1 = 3;
			step = result->rm_eo;	     
		} else {
			printf("storeCmds(): Strange match\n");
			start += result->rm_eo;        
			continue;
		}
	
        	// strip spaces from tail
                if (cmds) {
        		char *p = cmds + strlen(cmds) - 1;
        		while (isspace(*p)) {
        			*p = '\0';
                		p--;
        		}  
        	}
                
		name   = value + start + result[u1].rm_so;
		*(value + start + result[u1].rm_eo) = '\0';
	
		if (name == NULL) {
			//printf("Got name >NULL<\n");
			start +=result->rm_eo;        
			continue;
		//} else {
		//	printf("Got name >%s<\n",name);
		}
	
		cmdId = cmd2id(name);
		                
		if (cmdId == ID_UNKNOWN ) {
			printf("storeCmds(): Unknown command name %s\n", name);
			//here could be params like log=... also
			start +=result->rm_eo;        
			continue;
		}
		
		/*if (cmds != NULL) {
		    printf("Got command body >%s<\n",cmds);
		} else {
		    printf("Got command body >NULL<\n");
		}*/
		
		//printf("the rest of cmd >%s<\n",value+start+step);
	
        	int cmdIsOk = EXIT_OK;
		// Insert into command list
		cmdItem* newCmd = calloc(1, sizeof(cmdItem));
	
		newCmd->type = cmdId;
		newCmd->next = NULL;
		if (cmds == NULL) {
			newCmd->descr = NULL;
		} else {
			if (cmdId == ID_SET) {
				cmdIsOk = storeExecAndSetCmd(newCmd, cmds, 0);                        	
                        } else if (cmdId == ID_EXECSET) {
				cmdIsOk = storeExecAndSetCmd(newCmd, cmds, 1);                        	
			} else if (cmdId == ID_TIMER) {
				cmdIsOk = storeTimerCmd(newCmd,cmds);
 			} else if (cmdId == ID_EXEC) {
                        	char *newDescr = calloc(1, strlen(cmds) + 1);
				strcpy(newDescr, cmds);
				newCmd->exec = newDescr;
			} else if (cmdId == ID_GET) {
				cmdIsOk = storeGetCmd(newCmd, cmds);                        	
			} else if (cmdId == ID_MAKE) {
				cmdIsOk = storeMakeCmd(newCmd, cmds);                        	
 			} else if (cmdId == ID_MACRO) {
				storeMacroCmd(newCmd, cmds);                        	
			} else if (cmdId == ID_EXECSEND) {
                        
                        	char* comma = index(cmds,',');
                                 if (comma == NULL) {
                                	cmdIsOk = EXIT_NOK;
                                } else {
                        
                        		char *dsc = calloc(1, comma - cmds + 1);
                        		char *exc = calloc(1, strlen(comma));
					strncpy(dsc, cmds, comma - cmds);
					strcpy(exc, comma + 1);
					newCmd->descr = dsc;
					newCmd->exec  = exc;
                                }
                        } else {
                        	if (cmdId == ID_SENDCKPD) {
                                	normalizeSequence(&cmds);
                                } else if (cmdId != ID_SEND){
                                	deleteSpaces(&cmds,0);
                                }
                        	char *newDescr = calloc(1, strlen(cmds) + 1);
				strcpy(newDescr, cmds);
				newCmd->descr = newDescr;
                        }
		}
		if (cmdIsOk == EXIT_OK) {
			if (curCmd == NULL) { // Insert first
				*cmdList = newCmd;
			} else {
				curCmd->next = newCmd;
			}
			curCmd = newCmd;
		} else {
                	free(newCmd);
                        if (cmds == NULL) {
                		printf("storeCmds(): command %s was not stored\n", name);
                        } else {
                		printf("storeCmds(): command %s ( %s ) was not stored\n", name,cmds);
                        }
                }
		start +=step;
		//printf("REST OF CMDS %s\n", value+start);	   
	}   // while
	free(result);result = NULL;

	return EXIT_OK;
}

static int storeKey(mode* cMode, type_key **head, char *tag, char *value) 
{
	type_key* It = NULL;
	int ret;
	
	//printf("storeKey\n");

	normalizeSequence(&tag);
        
        if(cMode && findExact(cMode, tag)) {
        	return EXIT_OK;  // do not overwrite existing items
        }

	It = (type_key*) calloc(1, sizeof(type_key));

	// Insert in head
	if ((*head) == NULL) {   // first elem
	 	It->next = NULL;
	} else {
		It->next = (type_key*) (*head);
	}
	*head = It;
	
        It->key = (char*) calloc(1, strlen(tag)+1);
	strcpy(It->key,tag);

	if (value!= NULL) {
		ret = storeCmds(&It->cmd, value);
	} else {
		It->cmd = NULL;
	}

	return EXIT_OK;
}

static char* loadLine(FILE *fp)
{ 
        int lRead   = 0;
        int sz      = 0;
        load_buf[0] = '\\';
        char* aLine = load_buf;
        
        while (aLine && sz >= 0 && sz < MAXCMDLEN - 2 && load_buf[sz] == '\\') {
        	aLine = fgets((char *) (load_buf+sz), MAXCMDLEN - sz, fp);
                lRead++;
                sz = strlen(load_buf)-2;
        }
        
        if (aLine || lRead > 1) {
        	//printf("GOT: %s\n",load_buf);
        	return load_buf;
        } 
        return NULL;
}

static int loadKeys(FILE *fp)
{
        char *tmptext;
	char *tag, *value;    

        mode *dm = findMode("default");
	if (dm == NULL) {
		// We have at least default mode
		defMode = (mode*) calloc(1, sizeof(mode));
		strcpy(defMode->name,"default");
		defMode->next = NULL;
		defMode->keys = NULL;

		modes	      = defMode;
		currentMode   = defMode;
	} 
       
	mode *curMode = currentMode;
	do {
		//printf("loadKeys -------------------------------------------------------------------------\n");
		tmptext=loadLine(fp);
	
		if (tmptext == NULL) return EXIT_NOK;
	
		tag   = NULL;
		value = NULL;
		parseKeyValue(tmptext, &tag, &value);
	
		if (tag == NULL) { 
			//printf("ERROR : incorrect command %s\n", tmptext);
			continue;
		} else if(strcmp(tag,SECTION_END_STR) == 0) {
			break;
		} else if(strcmp(tag,MODE_STR) == 0) { // mode definition
                	mode *nm = findMode(value);
                	if (nm == NULL) {	
                
				nm = (mode*) calloc(1, sizeof(mode));
				strcpy(nm->name,value);
				nm->keys = NULL;
			    
				// Insert in head
				if (modes == NULL) {   // first elem
					nm->next = NULL;
				} else {
					nm->next = (mode*) modes;
				}
				modes = nm;
                        }
			curMode = nm;
	
		} else if(strcmp(tag,MODE_END_STR) == 0) { 	// mode definition end
 			curMode  = defMode;
		} else if (tag != NULL) {		// Line with key->command definition
                	storeKey(curMode, &(curMode->keys), tag, value);
		} else {
			continue;
		}
	} while (1);

	return EXIT_OK;    
}

static void loadInternal()
{
	if (internalMode == NULL) {
		internalMode = (mode*) calloc(1, sizeof(mode));
		strcpy(internalMode->name,"_INTERNAL_");
		internalMode->keys = NULL;
	}
	
	char cmd[] = "_GET_ICON_($$)";
	char val[] = "ExecAndSet(image,icon,echo \'UF=\"$(CfgDir)/Icons/$(Index)/$(Param).png\";echo \"$(Param),$UF\"\'|bash -f -s);";
	storeKey(internalMode, &(internalMode->keys),cmd,val);
	
	char cmd2[] = "_PING_";
	char val2[] = "Get(ping)";
	storeKey(internalMode, &(internalMode->keys),cmd2,val2);
	
	/*
	char cmd3[] = "_PASSWORD_INVALID_";
	char val3[] = "Set(disconnect);Make(disconnect);Make(stop);";
	storeKey(internalMode, &(internalMode->keys),cmd3,val3);*/
}

static int loadAliases(FILE *fp)
{
        char *tmptext;
	char *key, *alias;
	    

	do {
		tmptext=loadLine(fp);
                
		if (tmptext == NULL) return EXIT_NOK;
	
		key   = NULL;
		alias = NULL;
		parseKeyValue(tmptext, &key, &alias);
	
		if (key == NULL) { 
			//printf("ERROR : incorrect alias %s\n", tmptext);
			continue;
		} else if(strcmp(key,SECTION_END_STR) == 0) {
			break;
		} else if (alias != NULL) {	// Line with key=alias definition
			addAlias(key,alias);
		} else {
			continue;
		}
	} while (1);

	return EXIT_OK;    
}

static int loadAlarms(FILE *fp)
{
        char *tmptext;
	char *tag, *value;    

	do {
		tmptext=loadLine(fp);
		
                if (tmptext == NULL) return EXIT_NOK;
	
		parseKeyValue(tmptext, &tag, &value);
	
		if (tag == NULL) { 
			continue;
		} else if(strcmp(tag,SECTION_END_STR) == 0) {
			break;
		} else if (value != NULL) {// Line with key->command definition
			storeKey(NULL, &alarms, tag, value);
		} else {
			continue;
		}
	} while (1);

	return EXIT_OK;    
}

int load_cfg(char *mfile, int isInit)
{
	FILE *fp;
	char *tag, *value, *tmptext;
	//printf("LOAD:%s\n",mfile);

	fp=fopen(mfile,"r");
	if (fp == NULL) return EXIT_NOK;

        if (keyVal == NULL) {		// init it once
		keyVal  = (regex_t *) malloc(sizeof(regex_t));
		memset(keyVal,   0, sizeof(regex_t));
		regexpPrepare(keyVal,   REGEX_KEYVAL);
	}
	
	if (cmdByCmd == NULL) { 	// init it once
		cmdByCmd = (regex_t *) malloc(sizeof(regex_t));
		memset(cmdByCmd, 0, sizeof(regex_t));
		regexpPrepare(cmdByCmd, REGEX_CMDBYCMD);
	}

	// Go through the conf file
	while(1) {
		
		tmptext=loadLine(fp);

		if (tmptext==NULL) break;
		    	
		parseKeyValue(tmptext, &tag, &value);
			
		if (tag == NULL) { 
			//printf("INFO : skip line %s\n", tmptext);
			continue;
		} else if (strcmp(tag,RETRY_STR) == 0 && isInit) {
			if (value != NULL) {
				conf.retrysecs=atoi(value);
			}	    
		} else if (strcmp(tag,DEVICE_STR) == 0 && isInit) {
			if (value != NULL) {
				strcpy(conf.device,value);
			}	    
		} else if (strcmp(tag,AUTOCONN_STR) == 0 && isInit) {
			if (value != NULL && strncmp("true",value,4)==0) {
			 	conf.autoConnect=1;
			}	    
		} else if (strcmp(tag,CHARSET_STR) == 0 && isInit) {
		    if (value != NULL) {
		    	strcpy(conf.charset,value);
		    }		
		} else if (strcmp(tag,LOG_STR) == 0 && isInit) {
			if (value != NULL) {
                        	if (strncmp("true",value,4)==0) {
					conf.log=1;
                                } else if (strncmp("debug",value,5)==0) {
                                	conf.log=2;
                                }
			}	    
		} else if (strcmp(tag,SCREEN_STR) == 0 && isInit) {
			if (value != NULL && strncmp("true",value,4)==0) {
				conf.screen=1;
			}	    
		} else if (strcmp(tag,BAUD_STR) == 0 && isInit) {
			if (value != NULL) {
				conf.baudrate = setBaudrate(atoi(value));
			}	    
		} else if (strcmp(tag,KEYS_SECTION_STR) == 0) {
			if (loadKeys(fp) != EXIT_OK) {
				return EXIT_NOK;
			} 
                        if (!isInit) {		// If we goes here from Include() command, we did all we need
                        	return EXIT_OK;
                        }
		} else if (strcmp(tag,ALIAS_SECTION_STR) == 0) {
			if (loadAliases(fp) != EXIT_OK) {
				return EXIT_NOK;
			}
		} else if (strcmp(tag,ALARMS_SECTION_STR) == 0) {
			if (loadAlarms(fp) != EXIT_OK) {
				return EXIT_NOK;
			} 
		} else if (strcmp(tag,TOMAIN_STR) == 0 && isInit) {
			if(value!= NULL) {
				strcpy(conf.toMain,value);
			}	    
		} else if (strcmp(tag,AUTOREPEAT_STR) == 0 && isInit) {
			if (value != NULL && strncmp("true",value,4)==0) {
				conf.autoRepeat=1;
			}	    
		} else if (strcmp(tag,CMER_ON_STR) == 0 && isInit) {
			if(value!= NULL) {
				strcpy(conf.cmerOn,value);
			}	    
		} else if (strcmp(tag,CMER_OFF_STR) == 0 && isInit) {
			if(value!= NULL) {
				strcpy(conf.cmerOff,value);
			}	    
		} else if (isInit) {
			if (strncmp("GuiApp",tag,6) != 0 &&
			    strncmp("GuiDescription",tag,14) != 0) {	// Do not warn about GUI meta tags
				printf("WARNING: Unknown tag in cfg file >%s<\n",tag);
			}
		}
	}

	fclose(fp);

	loadInternal();
	

	//printf("LOAD:%s OK\n",mfile);
	return EXIT_OK;
}

void freeRegexps() 
{
	if (cmdByCmd) {
		regfree(cmdByCmd); free(cmdByCmd); cmdByCmd = NULL;
	}
	if (keyVal) {
		regfree(keyVal);   free(keyVal);   keyVal = NULL;
	}
}

int init_cfg(char *path)
{
	int ret = EXIT_OK;
        
	if (path != NULL) {
		
		printf("Use configuration file %s\n",path);
                char *d = strdup(path);
                char *p = rindex(d,SEPARATOR);
                if (p != NULL) {
                 	*p = '\0';
                } else {
			*d = '\0';
		} 
                setCfgDir(d);
                free(d);
                
		ret = load_cfg(path,1);
	} else {
                setCfgDir(NULL);
		
		char cfgfile[MAXLEN];
		strcpy(cfgfile, getenv("HOME"));
		strcat(cfgfile, CFGFILE);

		printf("Search configuration file %s\n", cfgfile);
		if(load_cfg(cfgfile,1) != EXIT_OK) {
			// Try again in current dir 
			strcpy(cfgfile, ".");
			strcat(cfgfile, CFGFILE);
			
			printf("Search configuration file %s\n", cfgfile);
			ret = load_cfg(cfgfile,1);
		}
	}
        if (ret != EXIT_OK) {
        	printf("Can not find configuration file to use or incorrect content of configuration file detected.\n");	
        }
	return ret;
}

