#include <QDebug>

#include <maps_model.hh>
#include <maps_reader.hh>

MapsModel::MapsModel (QObject * parent) : QAbstractTableModel(parent) {
    // nothing more to do
}


void MapsModel::refresh (int pid) {
    // nuke all old data
    qDeleteAll(m_entries);
    m_entries.clear();

    // are we just clearing?
    if ( pid == -1 ) {
        return;
    }
    
    // parse new entries for our given pid
    if ( ! MapsReader().parse(pid, m_entries) ) {
        // failed to parse?
        qWarning() << "MapsModel::refresh: failed to parse the smaps file";
    }

    // reset the entire model
    reset();
}


const MapsEntry * MapsModel::getEntry (const QModelIndex & index) const {
    // precautions
    if ( ! index.isValid() || index.row() >= m_entries.size() ) {
        qWarning() << "MapsModel::getEntry: invalid index, row:" << index.row();
        return 0;
    }

    return  m_entries[index.row()];
}
    
 
int MapsModel::rowCount (const QModelIndex & parent) const {
    Q_UNUSED(parent);

    return m_entries.size();
}


int MapsModel::columnCount (const QModelIndex & parent) const {
    Q_UNUSED(parent);

    // always 7 columns
    return 7;
}


QVariant MapsModel::headerData (int section, Qt::Orientation orientation, int role) const {
    // don't handle stuff we don't care about
    if ( orientation != Qt::Horizontal || role != Qt::DisplayRole ) {
        return QVariant();
    }

    switch ( section ) {
    case 0: return tr("Start address"); break;
    case 1: return tr("End address"); break;
    case 2: return tr("Size"); break;
    case 3: return tr("Permissions"); break;
    case 4: return tr("Mode"); break;
    case 5: return tr("Offset"); break;
    case 6: return tr("Path"); break;
    }

    return tr("wtf?");
}


QVariant MapsModel::data (const QModelIndex & index, int role) const {
    // precautions
    if ( ! index.isValid() ) {
        return QVariant();
    }

    // we only want the display and alignment roles
    if ( role != Qt::DisplayRole && role != Qt::TextAlignmentRole) {
        return QVariant();
    }

    if ( role == Qt::TextAlignmentRole ) {
        // handle text alignment
        switch ( index.column() ) {
        case 0:
        case 1:
        case 2:
        case 5:
            // align right
            return (int)(Qt::AlignRight | Qt::AlignVCenter);
            break;

        default:
            // align left
            return (int)(Qt::AlignLeft | Qt::AlignVCenter);
            break;
        }
    }
            
    // normal textual content data wanted. find the map
    MapsEntry * entry = m_entries[index.row()];

    // which column is wanted?
    switch ( index.column() ) {
    case 0:
        //qDebug() << entry->start_address;
        return QString("%1").arg(entry->start_address, 8, 16, QChar('0') );
        break;

    case 1:
        return QString("%1").arg(entry->end_address, 8, 16, QChar('0') );
        break;

    case 2:
        return QString::number (entry->end_address - entry->start_address, 16 );
        break;

    case 3:
        return entry->permissions;
        break;

    case 4:
        return entry->getMode();
        break;

    case 5:
        return QString::number(entry->offset, 16 );
        break;

    case 6:
        return entry->path;
        break;
    }

    // we should not get this far
    qDebug() << "MapsModel::data: wtf?" << index << role;
    return QVariant("wtf?");
}

