#ifndef MAPS_MODEL_HH
#define MAPS_MODEL_HH

#include <QAbstractTableModel>
#include <QList>

#include <maps_reader.hh>

/**
 * @brief A table model that can show @e maps entries for a process.
 *
 *
 **/
class MapsModel : public QAbstractTableModel {

    Q_OBJECT

public:

    /**
     * Creates the model.
     *
     * @param parent the parent for the model.
     **/
    MapsModel (QObject * parent=0);

    /**
     * Refreshes the entire model by clearing out all data and parsing new maps entries for the
     * process indicated by @p pid. The old entries are deleted and the whole model is
     * invalidated. If @p pid is -1 then the model is simply cleared.
     *
     * @param pid the process id to show maps for.
     **/
    void refresh (int pid);

    /**
     * Returns the MapsEntry that corresponds to the given @p index. The index can be any column for
     * the entry, as only the row is checked. Returns 0 if the index is invalid.
     *
     * @param index the index to be mapped to an entry.
     *
     * @return a MapsEntry pointer or 0 if invalid.
     **/
    const MapsEntry * getEntry (const QModelIndex & index) const;
    
    /**
     * Returns the number of rows in the model, ie the number of map entries that have been parsed.
     *
     * @param parent not used.
     *
     * @return a row count.
     **/
    virtual int rowCount (const QModelIndex & parent=QModelIndex()) const;

    /**
     * Returns the number of columnsin the model. This is a fixed number.
     *
     * @param parent not used.
     *
     * @return a column count.
     **/
    virtual int columnCount (const QModelIndex & parent=QModelIndex()) const;

    /**
     * Returns a suitable header title for the given @p section, ie column. Only the display @p role
     * and the horizontal orientation are used.
     *
     * @param section the column/row for which the data is wanted.
     * @param orientation indicates a vertical or horizontal header.
     * @param role the current role for which data is wanted.
     *
     * @return the actual data coded as a variant.
     **/
    QVariant headerData (int section, Qt::Orientation orientation, int role=Qt::DisplayRole) const;

    /**
     * Returns the actual data for the cell identified by @p index and for the given @p role. Only
     * display roles are handled at the moment.
     *
     * @param index the index for which to return data.
     * @param role the current role for which data is wanted.
     *
     * @return the actual data coded as a variant.
     **/
    virtual QVariant data (const QModelIndex & index, int role=Qt::DisplayRole ) const;


protected:

    //! maps entries
    QList<MapsEntry *> m_entries;
};

#endif
