
#include <QObject>
#include <QRegExp>
#include <QFile>
#include <QTextStream>
#include <QDebug>
#include <QStringList>

#include <maps_reader.hh>

bool MapsReader::parse (int pid, QList<MapsEntry *> & entries) {
    QString line;
    MapsEntry * entry = 0;

    // assemble the file filename
    QString filename = QString("/proc/%1/smaps").arg(pid);
    
    // open our file for reading
    QFile file(filename);
    if ( ! file.open(QFile::ReadOnly) ) {
        // failed to open, so the process must be gone
        return false;
    }

    // a regexp for identifying the first special line of the format:
    // b7f1c000-b7f1e000 rw-p 00019000 08:01 278530     /lib/ld-2.7.so
    // the expression is made so that quantifiers are greedy. this is needed for the last \\s
    QRegExp info("(.{8})-(.{8})\\s+(.{4})\\s+(.{8})\\s+(..:..)\\s+(\\d+)\\s+(.*)",
                 Qt::CaseSensitive, QRegExp::RegExp2);
   
    // use a stream to read all lines
    QTextStream stream(&file);
    while( (line = stream.readLine()) != "" ) {
        // a first line?
        if ( info.exactMatch ( line ) ) {
            // we have a new entry, create it
            entry = new MapsEntry();
            entries << entry;

            // populate it with all info from our regexp
            entry->start_address = info.cap(1).toUInt(0, 16);
            entry->end_address   = info.cap(2).toUInt(0, 16);
            entry->permissions   = info.cap(3);
            entry->offset        = info.cap(4).toUInt(0, 16);
            entry->device        = info.cap(5);
            entry->inode         = info.cap(6).toUInt();
            entry->path          = info.cap(7);

            // check the path
            if ( entry->path == "" ) {
                entry->path = QObject::tr("[anonymous]");
            }
        }

        else {
            // we must have a current entry here
            Q_ASSERT(entry);

            // the rows are likely of the format:
            // Size:                  4 kB
            // so we can split the line into data
            QStringList parts = line.split(" ", QString::SkipEmptyParts);
            QString key = parts[0].simplified();

            // what did we get?
            if ( key == "Size:" ) {
                entry->size = parts[1].toUInt();
            }

            else if ( key == "Rss:" ) {
                entry->rss = parts[1].toUInt();
            }

            else if ( key == "Shared_Clean:" ) {
                entry->shared_clean = parts[1].toUInt();
            }

            else if ( key == "Shared_Dirty:" ) {
                entry->shared_dirty = parts[1].toUInt();
            }

            else if ( key == "Private_Clean:" ) {
                entry->private_clean = parts[1].toUInt();
            }

            else if ( key == "Private_Dirty:" ) {
                entry->private_dirty = parts[1].toUInt();
            }

            else if ( key == "Referenced:" ) {
                entry->referenced = parts[1].toUInt();
            }

            else {
                qWarning() << "MapsReader::parse: unknown row:" << line;
            }
        }
    }

    // debugging
    // foreach ( MapsEntry * entry, entries ) {
    //     qDebug() << entry->start_address << entry->end_address << entry->path;
    // }
    
    // all ok
    return true;
}
