
#include <QFile>
#include <QTextStream>
#include <QStringList>
#include <QRegExp>
#include <QDir>
#include <QSortFilterProxyModel>

#include <memory-monitor.hh>
#include <maps_model.hh>
#include <preferences.hh>

MemoryMonitor::MemoryMonitor (QWidget * parent) : QMainWindow(parent) {
    setupUi(this);

    // no reader yet
    m_reader = 0;

    connect ( m_processes, SIGNAL(currentIndexChanged(int)), this, SLOT(processChanged(int)) );

    // we want to know when to show realtime samples and when to show a selected sample
    connect ( m_plot, SIGNAL(showSample(Sample)), this, SLOT(plotSampleReceived(Sample)) );
    connect ( m_plot, SIGNAL(clearSample()),      this, SLOT(plotSampleCleared()) );

    // refresh on demand
    connect ( m_refresh, SIGNAL(clicked()), this, SLOT(findProcesses()) );

    // connect actions
    connect ( m_quit_action,        SIGNAL(triggered(bool)), QCoreApplication::instance(), SLOT(quit()) );
    connect ( m_preferences_action, SIGNAL(triggered(bool)),this,                          SLOT(preferences()) );
    
    // create and set a model that shows the process' maps
    m_maps_model = new MapsModel(this);

    // and a model that acts as a sorting proxy for the above model
    m_sort_model = new QSortFilterProxyModel(this);
    m_sort_model->setSourceModel(m_maps_model);
    m_maps_view->setModel ( m_sort_model );

    // we want to adjust shown data when the user clicks a row in the table
    connect ( m_maps_view->selectionModel(), SIGNAL(currentChanged(const QModelIndex &, const QModelIndex &)),
              this,                          SLOT  (mapEntryChanged(const QModelIndex &, const QModelIndex &)) );
    
    m_show_plot_sample = false;

    // find the initial processes
    findProcesses ();
}


void MemoryMonitor::processChanged (int index) {
    // has the combo been cleared?
    if ( index == -1 ) {
        return;
    }
    
    // do we have an old reader?
    if ( m_reader ) {
        delete m_reader;
        m_reader = 0;
    }

    // always clear the plot. @todo perhaps leave the plot until something else comes up?
    m_plot->clear();

    // should we monitor something else? 0 is the magical "no process" index
    if ( index == 0 ) {
        // we're done, clear all data first though
        processEnded();
        return;
    }
    
    // get the process id that matches the index
    int pid = m_process_ids[index - 1];

    // and create a new reader
    m_reader = new Reader ( pid );
    
    // we want to know when samples arrive
    connect ( m_reader, SIGNAL(newSample(Sample)), m_plot, SLOT(addSample(Sample)) );
    connect ( m_reader, SIGNAL(newSample(Sample)), this, SLOT(sampleReceived(Sample)) );
    connect ( m_reader, SIGNAL(processEnded()),    this, SLOT(processEnded()) );   

    // refresh the model to show the new pid
    m_maps_model->refresh ( pid );
}


void MemoryMonitor::mapEntryChanged (const QModelIndex & current, const QModelIndex & previous) {
    Q_UNUSED(previous);
    
    // find the current maps entry
    const MapsEntry * entry = m_maps_model->getEntry(m_sort_model->mapToSource(current) );

    // precautions
    if ( entry == 0 ) {
        // clear all labels
        m_size->clear();
        m_rss->clear();
        m_shared_clean->clear();
        m_shared_dirty->clear();
        m_private_clean->clear();
        m_private_dirty->clear();
        m_referenced->clear();
        m_device->clear();
        m_inode->clear();
    }

    else {
        // set new texts for all labels
        m_size->setText         ( QString::number( entry->size ) );
        m_rss->setText          ( QString::number( entry->rss ) );
        m_shared_clean->setText ( QString::number( entry->shared_clean ) );
        m_shared_dirty->setText ( QString::number( entry->shared_dirty ) );
        m_private_clean->setText( QString::number( entry->private_clean ) );
        m_private_dirty->setText( QString::number( entry->private_dirty ) );
        m_referenced->setText   ( QString::number( entry->referenced ) );
        m_device->setText       ( entry->device );
        m_inode->setText        ( QString::number( entry->inode ) );
    }
}


void MemoryMonitor::sampleReceived (Sample sample) {
    // are we showing a fixed plot sample?
    if ( m_show_plot_sample ) {
        return;
    }

    showSample ( sample );
}


void MemoryMonitor::processEnded () {
    // clear all labels
    m_total->setText      ( "" );
    m_resident->setText   ( "" );
    m_shared->setText     ( "" );
    m_text->setText       ( "" );
    m_library->setText    ( "" );
    m_data->setText       ( "" );
    m_stack->setText      ( "" );
    m_time->setText       ( "" );

    m_processes->setCurrentIndex ( 0 );

    // refresh the model to show nothing
    m_maps_model->refresh ( -1 );
}


void MemoryMonitor::plotSampleReceived (Sample sample) {
    // first show this new fixed sample
    showSample ( sample );

    // this flag fixes us
    m_show_plot_sample = true;
}


void MemoryMonitor::plotSampleCleared () {
    // now we can show realtime samples again
    m_show_plot_sample = false;
}


void MemoryMonitor::findProcesses () {
    // find all files in /proc
    QStringList proc_entries = QDir("/proc").entryList ( QDir::Dirs, QDir::Name );

    // a new list for combo entries
    QStringList combo_entries;
    combo_entries << tr("no process");
    
    QRegExp exp ("^\\d+$");
    foreach ( const QString & entry, proc_entries ) {
        // is this a numeric entry, ie a process?
        if ( exp.exactMatch ( entry ) ) {
            // it is an entry, get the application name
            QString name = findApplicationName ( entry.toInt() );

            // add to the combo entry texts
            combo_entries << name;
            m_process_ids << entry.toInt();
        }
    }

    // clear the combo from any old items and set the new
    m_processes->clear ();
    m_processes->addItems ( combo_entries );

    // clear all contents
    processEnded();
}


void MemoryMonitor::preferences () {
    // create and show the dialog modally
    if ( Preferences(this).exec() == QDialog::Accepted ) {
        // the dialog was accepted, take all new settings into use
        m_plot->settingsChanged();
    }
}


void MemoryMonitor::showSample (const Sample & sample) {
    // update all labels
    m_total->setText      ( QString::number( sample.values[TotalSize] / 1024 ) );
    m_resident->setText   ( QString::number( sample.values[ResidentSize] / 1024 ) );
    m_shared->setText     ( QString::number( sample.values[Shared] / 1024 ) );
    m_text->setText       ( QString::number( sample.values[Text] / 1024 ) );
    m_library->setText    ( QString::number( sample.values[Library] / 1024 ) );
    m_data->setText       ( QString::number( sample.values[Data] / 1024 ) );
    m_stack->setText      ( QString::number( sample.values[Stack] / 1024 ) );

    m_time->setText       ( sample.time.toString() );
}


QString MemoryMonitor::findApplicationName (int pid) {
    QString line;
    
    // assemble the file filename
    QString filename = QString("/proc/%1/status").arg(pid);

    // open our file for reading
    QFile file(filename);
    if ( ! file.open(QFile::ReadOnly) ) {
        // failed to open, so the process must be gone
        return tr("no process");
    }
   
    // use a stream to read all lines
    QTextStream stream(&file);
    while( (line = stream.readLine()) != "" ) {
        
        // split the line into data
        QStringList parts = line.split(QRegExp("\\s+"), QString::SkipEmptyParts);

        QString key = parts[0].simplified();
        
        // check the first part to get the data
        if ( key == "Name:" ) {
            return parts[1] + QString(" (%1)").arg(pid);
        }
    }

    // no name found?
    return tr("no process");
}

