
#ifndef MEMORY_MONITOR_HH 
#define MEMORY_MONITOR_HH 

#include <QtGui>

#include <ui_memory-monitor.h> 
#include <reader.hh>

// forwar declarations
class MapsModel;

/**
 * @brief
 *
 *
 **/
class MemoryMonitor : public QMainWindow, private Ui::MemoryMonitorBase {

    Q_OBJECT

public:

    /**
     * Constructor. 
     *
     * @param parent optional parent object.
     **/
    MemoryMonitor (QWidget * parent=0);

                                                       
protected slots:

    /**
     * Callback when the current index in the combo box with all processes has changed. Creates a new reader for
     * handling the new process. If no process was chosen then the old reader is simply destroyed. Uses the reader to
     * get memory samples. Sends samples to the internal Plot widget.
     *
     * @param index the index of the chosen item in the combobox.
     **/
    void processChanged (int index);

    /**
     * Callback for when the user has clicked on some row in the maps view. The row is identified by
     * @p current which is used to look up the proper entry and populate the labels below the table.
     *
     * @param current the index that represents the activated row.
     * @param previous the old selected row. Not used.
     **/
    void mapEntryChanged (const QModelIndex & current, const QModelIndex & previous);
    
    /**
     * Callback for when the Reader has parsed a new @p sample. The new sample is shown in the data
     * labels unless we're showing a fixed sample from the plot.
     *
     * @param the new sample.
     **/
    void sampleReceived (Sample sample);

    /**
     * Callback for when the Reader has noticed that the monitored process has terminated. Clears
     * all labels to empty default values.
     **/
    void processEnded ();

    /**
     * Callback for when the plot has sent us a fixed @p sample we should show. The @p sample data
     * is shown in the labels and no realtime samples are shown until we receive a
     * plotSampleCleared() callback.
     *
     * @param sample the fixed sample to show.
     **/
    void plotSampleReceived (Sample sample);

    /**
     * Callback for when the plot indicates that we should not longer only look at a fixed sample
     * but can update in realtime. After this call all samples from sampleReceived(Sample) are shown
     * immediately.
     **/
    void plotSampleCleared ();

    /**
     * Finds all processes on the system and builds up the combobox with the process names and pids. Stores all process
     * id:s internally for later retrieval.
     **/
    void findProcesses ();

    /**
     * Callback for the @e Preferences action in the @e File menu. Shows the user a preferences
     * dialog where settings can be configured.
     **/
    void preferences ();
    
    
protected:
    
    /**
     * Shows the data for the single @p sample in the labels in the window. The @p sample may be a
     * realtime or a fixed sample.
     *
     * @param sample the sample to show.
     **/
    void showSample (const Sample & sample);

    /**
     * Finds the name of the process by parsing the @e status file. The given @p pid is used to
     * extract the information from /proc/<pid>/status.
     *
     * @param pid the process id whose name is retrieved.
     *
     * @return the application name for the given pid.
     **/
    QString findApplicationName (int pid);
    
    //! flag indicating that we're showing a fixed sample from the plot
    bool m_show_plot_sample;

    //! a reader for reading process data
    Reader * m_reader;

    //! a list of all process ids, in the same order as in the combo box
    QList<int> m_process_ids;

    //! a model that shows the process maps
    MapsModel * m_maps_model;

    //! a proxy model for sorting
    QSortFilterProxyModel * m_sort_model;
};


#endif
