
#include <QPainter>
#include <QDebug>
#include <QStyleOption>
#include <QCursor>
#include <QMouseEvent>
#include <QPolygon>
#include <QSettings>

#include <plot.hh>


Plot::Plot (QWidget * parent) : QWidget(parent) {
    m_max_value = 0;

    settingsChanged();

    // we want tracking
    setMouseTracking(true);
}


void Plot::addSample (Sample sample) {
    m_samples << sample;

    // do we have a new max value?
    m_max_value = qMax ( sample.values[TotalSize] / 1024, m_max_value );

    // and repaint
    update();
}


void Plot::clear () {
    // get rid of all samples
    m_samples.clear();
    
    // and repaint
    update();
}


void Plot::settingsChanged () {
    QSettings settings("Smultron", "Memory monitor");

    // clear any old colors
    m_colors.clear();
    
    // populate with suitable colors
    m_colors << settings.value ("Colors/total",      QColor(Qt::black)).value<QColor>().name()
             << settings.value ("Colors/resident",   QColor(Qt::red)).value<QColor>().name()
             << settings.value ("Colors/shared",     QColor(Qt::blue)).value<QColor>().name()
             << settings.value ("Colors/text",       QColor(Qt::green)).value<QColor>().name()
             << settings.value ("Colors/library",    QColor(Qt::magenta)).value<QColor>().name()
             << settings.value ("Colors/data_color", QColor(Qt::darkCyan)).value<QColor>().name()
             << settings.value ("Colors/stack",      QColor(Qt::gray)).value<QColor>().name();

    // set the background color for the stylesheet too
    setStyleSheet ( "background-color: " + settings.value ("Colors/plot", QColor(Qt::white)).value<QColor>().name() );
}
    

void Plot::paintEvent (QPaintEvent * event) {
    int start_index, index;
    
    Q_UNUSED(event);

    QPainter painter(this);

    // allow for stylesheets to work
    QStyleOption opt;
    opt.init(this);
    style()->drawPrimitive(QStyle::PE_Widget, &opt, &painter, this);

    // any samples?
    if ( m_samples.size() == 0 ) {
        return;
    }

    // scale the max memory value. This is the value that is at y=0
    unsigned int max_value = (m_max_value / 1000) * 1000 + 1000;

    // do we have more samples than width? if so we need to scroll
    if ( m_samples.size() > width() ) {
        start_index = m_samples.size() - width();
    }

    else {
        // less samples, we need them all
        start_index = 0;
    }

    // loop over all sample types
    for ( int type = 0; type < 7; type++ ) {
        // set a suitable pen based on this type
        painter.setPen ( m_colors[type] );
    
        // always start a plot from the left edge
        int x = 0;

        // create a polygon that can hold all points
        QPolygon poly ( m_samples.size() - start_index );
        
        // now draw this type from the chosen starting point
        for ( index = start_index; index < m_samples.size(); index++ ) {
            Sample & sample = m_samples[index];
        
            // scale the value to fit the screen
            unsigned int y = scaleValue(sample.values[type] / 1024, max_value);
            
            // draw a single point
            poly.setPoint ( x, x, y );
            ++x;
        }
        
        painter.drawPolyline ( poly );

    }

    // so, is the mouse inside or not?
    if ( m_inside ) {
        // get the mouse position relative to this widget
        QPoint pos = mapFromGlobal ( QCursor::pos() );

        // the
        if ( start_index + pos.x() < m_samples.size() ) {
            // draw a vertical line
            painter.drawLine ( pos.x(), 0, pos.x(), height() - 1 );
        }
    }
}


void Plot::enterEvent (QEvent * event) {
    Q_UNUSED(event);

    m_inside = true;
}


void Plot::leaveEvent (QEvent * event) {
    Q_UNUSED(event);

    m_inside = false;

    // no need to show a fixed sample anymore
    emit clearSample();
}


void Plot::mouseMoveEvent (QMouseEvent * event) {
    int start_index;

    // precautions
    if ( event->x() < 0 || event->x() >= width() ) {
        return;
    }
    
    // do we have more samples than width? if so we need to scroll
    if ( m_samples.size() > width() ) {
        start_index = m_samples.size() - width();
    }

    else {
        // less samples, we need them all
        start_index = 0;
    }
    
    // is the mouse x inside a sample?
    if ( start_index + event->x() < m_samples.size() ) {
        // get the sample that is under the mouse and make it fixed
        emit showSample ( m_samples[start_index + event->x()] );
    }
    else {
        // no need to show a fixed sample anymore
        emit clearSample();
    }

    // scale the y coordinate to match
    int memory = static_cast<int>(static_cast<float>(height() - event->y()) / static_cast<float>(height()) * static_cast<float>(m_max_value));
    
    // set a tooltip
    setToolTip ( tr("Memory: %1 KB").arg(memory) );
}


unsigned int Plot::scaleValue (unsigned int value, unsigned int max) {
    // oh man...
    return height() - static_cast<unsigned int>((static_cast<float>(value) / static_cast<float>(max)) * height()) - 1;
}

