#ifndef PLOT_HH
#define PLOT_HH

#include <QWidget>
#include <QList>
#include <QColor>

#include <sample.hh>

/**
 * @brief Class that can handle reading from files in @e /proc.
 *
 * 
 **/
class Plot : public QWidget {

    Q_OBJECT

public:

    /**
     * Creates the plot.
     *
     * @param parent an optional parent object.
     **/
    Plot (QWidget * parent=0);

    
signals:

    /**
     * Signal emitted when the fixed @p sample should be shown by other parties. They should not show any other sample
     * until a clearSample() signal has been emitted. This signal means that the user is currently hovering the mouse
     * over the plot and wants to look at some specific sample.
     *
     * @param sample the ficed sample to show.
     **/
    void showSample (Sample sample);

    /**
     * Callback when the user no longer hovers over some specific sample and a fixed sample no longer needs to be shown.
     **/
    void clearSample ();
    
    
public slots:

    /**
     * Adds a new @p sample to the internal list of samples and triggers a repaint.
     *
     * @note the list is never cleared, so it will accumulate memory ad infinitum!
     *
     * @param sample the new sample.
     **/
    void addSample (Sample sample);

    /**
     * Clears the entire plot and all samples.
     **/
    void clear ();

    /**
     * Refreshes the used colors by reading from QSettings.
     **/
    void settingsChanged ();
    
    
protected:

    /**
     * Callback for when a timer fires. Performs a memory read.
     *
     * @param event not used.
     **/
    virtual void paintEvent (QPaintEvent * event);

    /**
     * Event handler for when the mouse enters the plot and hovers over it.
     *
     * @param event not used.
     **/
    virtual void enterEvent (QEvent * event);

    /**
     * Event handler for when the mouse leaves the plot. Emits the signal clearSample() so that other parties can know
     * that no fixed sample needs to be shown anymore.
     *
     * @param event not used.
     **/
    virtual void leaveEvent (QEvent * event);

    /**
     * Event handler for when the mouse moves when over the plot. Tracks the mouse and checks wether it currently is on
     * the plot and finds the corresponding sample. Emits a showSample(Sample) if the mouse is over the plot, ie enough
     * to the left. Emits clearSample() if the mouse is too far right and is outside the plot.
     *
     * @param event not used.
     **/
    virtual void mouseMoveEvent (QMouseEvent * event);

    /**
     * Scales @p value that represents a y position to fit inside the widget height so that the max value (0) is @p max
     * and the minimum is 0 (height() of the widget). The coordinate is thus flipped vertically.
     *
     * @param value the value to scale.
     * @param max a value that represents the max possible value, which gets y=0.
     *
     * @return a scaled y coordinate.
     **/
    unsigned int scaleValue (unsigned int value, unsigned int max);
    
    //! a list of all samples
    QList<Sample> m_samples;

    //! the max memory value seen so far
    unsigned int m_max_value;

    //! a list of pencil colors
    QList<QColor> m_colors;

    //! flag indicating wether the mouse is inside the widget or not
    bool m_inside;
};


#endif
