
#include <QString>
#include <QFile>
#include <QTextStream>
#include <QStringList>
#include <QSettings>

#include <reader.hh>

Reader::Reader (int pid, QObject * parent) : QObject(parent) {
    m_pid = pid;

    // read in the delay
    QSettings settings("Smultron", "Memory monitor");
    m_interval = settings.value ("interval", 1000).toInt();
    
    // start the timer
    m_timer_id = startTimer ( m_interval );
}


void Reader::setPid (int pid) {
    m_pid = pid;

    // is our timer stopped?
    if ( m_timer_id == 0 ) {
        // restart the timer
        m_timer_id = startTimer(1000);
    }
}


void Reader::setInterval (int interval) {
    m_interval = interval;

    // stop the old timer
    killTimer ( m_timer_id );
    
    // start the timer with our new interval
    m_timer_id = startTimer ( m_interval );
}


void Reader::timerEvent (QTimerEvent * event) {
    Q_UNUSED(event);

    Sample sample;

    // parse the "status" and "statm" files
    if ( ! parseStatus(sample) || ! parseStatm(sample) ) {
        // clean up, no more polling
        killTimer(m_timer_id);
        m_timer_id = 0;

        // tell the world that we're done
        emit processEnded();
        return;
    }

    // fill in the sample's timestamp
    sample.time = QTime::currentTime();
    
    // we have a new sample ready
    emit newSample(sample);
}


bool Reader::parseStatus (Sample & sample) {
    QString line;
    
    // assemble the file filename
    QString filename = QString("/proc/%1/status").arg(m_pid);

    // open our file for reading
    QFile file(filename);
    if ( ! file.open(QFile::ReadOnly) ) {
        // failed to open, so the process must be gone
        return false;
    }
   
    // read a single line of data
    QTextStream stream(&file);

    while( (line = stream.readLine()) != "" ) {

        // split the line into data
        QStringList parts = line.split(" ", QString::SkipEmptyParts);

        QString key = parts[0].simplified();

        // check the first part to get the data
        if ( key == "VmSize:" ) {
            sample.values[TotalSize] = parts[1].toUInt() * 1024;
        }

        else if ( key == "VmRSS:" ) {
            sample.values[ResidentSize] = parts[1].toUInt() * 1024;
        }

        else if ( key == "VmData:" ) {
            sample.values[Data] = parts[1].toUInt() * 1024;
        }

        else if ( key == "VmStk:" ) {
            sample.values[Stack] = parts[1].toUInt() * 1024;
        }
 
        else if ( key == "VmExe:" ) {
            sample.values[Text] = parts[1].toUInt() * 1024;
        }
 
        else if ( key == "VmLib:" ) {
            sample.values[Library] = parts[1].toUInt() * 1024;
        }
    }

    // all ok
    return true;
}


bool Reader::parseStatm (Sample & sample) {
    // assemble the file filename
    QString filename = QString("/proc/%1/statm").arg(m_pid);

    // open our file for reading
    QFile file(filename);
    if ( ! file.open(QFile::ReadOnly) ) {
        // failed to open, so the process must be gone
        return false;
    }

    // read a single line of data
    QTextStream stream(&file);
    QString line = stream.readLine();

    // split the line into data
    QStringList parts = line.split(" ");

    // get all data from the parts
    sample.values[Shared] = parts[2].toUInt() * 4096;

    // all ok
    return true;
}
