#ifndef READER_HH
#define READER_HH

#include <QObject>
#include <QTime>

#include <sample.hh>

/**
 * @brief Class that can handle reading from files in @e /proc.
 *
 * 
 **/
class Reader : public QObject {

    Q_OBJECT

public:

    /**
     * Creates the reader and starts the internal timer. Will parse the @e statm file for the given @pid under /proc.
     *
     * @param pid the process id to monitor.
     * @param parent an optional parent object.
     **/
    Reader (int pid, QObject * parent=0);

    /**
     * Sets a new @p pid that the reader should be watching.
     *
     * @param pid new process id.
     **/
    void setPid (int pid);

    /**
     * Returns the process id of the process that is currently being monitored.
     *
     * @return current process id.
     **/
    int getPid () const {
        return m_pid;
    }

    /**
     * Sets a new delay for the plot. By default a value from the application settings is used, but
     * if the delay changes while a reader is running it is not taken into use automatically. This
     * method will change the internal timer to use @p interval instead.
     *
     * @param interval the new delay interval in ms.
     **/
    void setInterval (int interval);
    
    
signals:

    /**
     * Signal emitted when a new memory @p sample has been extracted.
     *
     * @param sample the new sample data.
     **/
    void newSample (Sample sample);

    /**
     * Signal emitted when the monitored process has terminated.
     **/
    void processEnded ();
    
    
protected:

    /**
     * Callback for when a timer fires. Performs a memory read.
     *
     * @param event not used.
     **/
    virtual void timerEvent (QTimerEvent * event);

    /**
     * Parses the @e /proc/<pid>/status file and extracts all needed data into the passed @p sample
     * instance. If the file can not be opened the process is assume to have terminated and false is
     * returned.
     *
     * @param sample the sample where the parsed data is stored.
     *
     * @return flag indicating wether the file could be opened or not.
     **/
    bool parseStatus (Sample & sample);

    /**
     * Parses the @e /proc/<pid>/statm file and extracts all needed data into the passed @p sample
     * instance. Currently only the amount of @e shared memory is parsed. If the file can not be
     * opened the process is assume to have terminated and false is returned.
     *
     * @param sample the sample where the parsed data is stored.
     *
     * @return flag indicating wether the file could be opened or not.
     **/
    bool parseStatm (Sample & sample);

    //! the id of the used timer
    int m_timer_id;

    //! the polled process id
    int m_pid;

    //! the timer delay in ms
    int m_interval;
};


#endif
