/***************************************************************************
    smb4k_sudowriter  -  This program writes to the sudoers file. It
    belongs to the utility programs of Smb4K.
                             -------------------
    begin                : Di Jul 29 2008
    copyright            : (C) 2008 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qdir.h>
#include <qfile.h>
#include <qtextcodec.h>
#include <qstring.h>
#include <qcstring.h>
#include <qvaluelist.h>

// KDE includes
#include <kaboutdata.h>
#include <kcmdlineargs.h>
#include <kapplication.h>
#include <kdebug.h>
#include <klocale.h>
#include <kuser.h>
#include <kstandarddirs.h>

// system includes
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <pwd.h>
#include <iostream>

using namespace std;

static QFile lock_file;

static const char description[] =
  I18N_NOOP( "This program writes to the sudoers file." );

static const char authors[] =
  I18N_NOOP( "(c) 2008, Alexander Reinholdt" );

int createLockFile()
{
  // Determine the directory where to write the lock file. First, try
  // /var/lock and than /var/tmp. If that does not work either, fall
  // back to /tmp.
  QValueList<QCString> dirs;
  dirs << "/var/lock";
  dirs << "/var/tmp";
  dirs << "/tmp";

  struct stat buf;

  for ( QValueList<QCString>::ConstIterator d = dirs.begin(); d != dirs.end(); ++d )
  {
    // First check if the directory is available and writable
    if ( lstat( *d, &buf ) == -1 )
    {
      int error_number = errno;

      if ( error_number != EACCES && error_number != ENOENT )
      {
        return error_number;
      }
    }
    else
    {
      // Continue
    }

    // Get the ids of the groups the user is in and check if
    // on of them matches buf.st_gid.
    KUser user( geteuid() );
    QValueList<KUserGroup> gids = user.groups();
    gid_t sup_gid = 65534; // set this to gid 'nobody' for initialization
    bool found_gid = false;

    for ( QValueList<KUserGroup>::ConstIterator g = gids.begin(); g != gids.end(); ++g )
    {
      if ( (*g).gid() == buf.st_gid )
      {
        sup_gid = (*g).gid();
        found_gid = false;

        break;
      }
      else
      {
        continue;
      }
    }

    // Check whether we are stat'ing a directory and that the
    // user has read/write permissions.
    if ( S_ISDIR( buf.st_mode ) /* is directory */ &&
        ((buf.st_uid == getuid() && (buf.st_mode & 00600) == (S_IWUSR | S_IRUSR)) /* user */ ||
        (found_gid && buf.st_gid == sup_gid && (buf.st_mode & 00060) == (S_IWGRP | S_IRGRP)) /* group */ ||
        ((buf.st_mode & 00006) == (S_IWOTH | S_IROTH)) /* others */) )
    {
      lock_file.setName( (*d)+"/smb4k.lock" );

      break;
    }
    else
    {
      continue;
    }
  }

  if ( !lock_file.exists() )
  {
    if ( lock_file.open( IO_WriteOnly ) )
    {
      QTextStream ts( &lock_file );
      // Note: With Qt 4.3 this seems to be obsolete, we'll keep
      // it for now.
      ts.setCodec( QTextCodec::codecForLocale() );

      ts << "0" << endl;

      lock_file.close();
    }
    else
    {
      return 1;
    }
  }
  else
  {
    return 2;
  }

  return 0;
}

void removeLockFile()
{
  // Just remove the lock file. No further checking is needed.
  if ( lock_file.exists() )
  {
    lock_file.remove();
  }
  else
  {
    // Do nothing
  }
}

const QByteArray findFile( const QString &filename )
{
  QStringList paths;
  paths << "/etc";
  paths << "/usr/local/etc";

  QString canonical_path;

  for ( QStringList::Iterator p = paths.begin(); p != paths.end(); ++p )
  {
    QDir::setCurrent( *p );

    if ( QFile::exists( filename ) )
    {
      canonical_path = QDir::current().canonicalPath()+QDir::separator()+filename;

      break;
    }
    else
    {
      continue;
    }
  }

  return canonical_path.local8Bit();
}

bool checkUsers( const QCStringList &list )
{
  for ( QCStringList::ConstIterator it = list.begin(); it != list.end(); ++it )
  {
    if ( getpwnam( *it ) == NULL )
    {
      return false;
    }
    else
    {
      continue;
    }
  }

  return true;
}

int checkFile( const QByteArray &path )
{
  // Stat the file, so that we know that it is safe to
  // read from and write to it and whether we need to
  // ask for the super user's password:
  struct stat buf;

  if ( lstat( path, &buf ) == -1 )
  {
    return errno;
  }
  else
  {
    // Do nothing
  }

  // Get the ids of the groups the user is in and check if
  // on of them matches buf.st_gid.
  KUser user( geteuid() );
  QValueList<KUserGroup> gids = user.groups();
  gid_t sup_gid = 65534; // set this to gid 'nobody' for initialization
  bool found_gid = false;

  for ( QValueList<KUserGroup>::ConstIterator g = gids.begin(); g != gids.end(); ++g )
  {
    if ( (*g).gid() == buf.st_gid )
    {
      sup_gid = (*g).gid();
      found_gid = false;

      break;
    }
    else
    {
      continue;
    }
  }

  if ( !S_ISREG( buf.st_mode ) || S_ISFIFO( buf.st_mode ) || S_ISLNK( buf.st_mode ) )
  {
    return 1;
  }
  else
  {
    // Do nothing
  }

  // Check the access rights. We need to read the file.
  if ( buf.st_uid != geteuid() && !found_gid &&
       (buf.st_mode & 00004) != (S_IWOTH | S_IROTH) /* others */ )
  {
    return 2;
  }
  else
  {
    // Do nothing
  }

  return 0;
}

bool findUtilityPrograms()
{
  if ( KGlobal::dirs()->findResource( "exe", "smb4k_kill" ).isEmpty() ||
       KGlobal::dirs()->findResource( "exe", "smb4k_umount" ).isEmpty() ||
       KGlobal::dirs()->findResource( "exe", "smb4k_mount" ).isEmpty() )
  {
    return false;
  }
  else
  {
    // Do nothing
  }

  return true;
}

int main ( int argc, char *argv[] )
{
  KAboutData aboutData( "smb4k_sudowriter",
                        I18N_NOOP( "smb4k_sudowriter" ),
                        "0.1",
                        I18N_NOOP( description ),
                        KAboutData::License_GPL_V2,
                        I18N_NOOP( authors ),
                        0,
                        "http://smb4k.berlios.de",
                        "smb4k-bugs@lists.berlios.de" );

  KCmdLineArgs::init( argc, argv, &aboutData );

  static const KCmdLineOptions options[] =
  {
    { "adduser <user>",     I18N_NOOP( "Adds an user to the sudoers file" ), 0 },
    { "removeuser <user>",  I18N_NOOP( "Removes an user from the sudoers file" ), 0 },
    KCmdLineLastOption
  };

  KCmdLineArgs::addCmdLineOptions( options );
  KApplication::addCmdLineOptions();

  KApplication app( false /* no GUI */ );

  // Before doing anything else, create the lock file.
  int return_value = 0;

  if ( (return_value = createLockFile()) != 0 )
  {
    switch ( return_value )
    {
      case 1:
      {
        cerr << argv[0] << ": " << I18N_NOOP( "The lock file could not be created." ) << endl;
        break;
      }
      case 2:
      {
        cerr << argv[0] << ": " << I18N_NOOP( "Another user is currently editing the sudoers file." ) << endl;
        break;
      }
      default:
      {
        cerr << argv[0] << ": " << strerror( return_value ) << endl;
        break;
      }
    }

    cerr << argv[0] << ": " << I18N_NOOP( "Aborting." ) << endl;
    exit( EXIT_FAILURE );
  }
  else
  {
    // Do nothing
  }

  KCmdLineArgs *args = KCmdLineArgs::parsedArgs();

  // Check that everything is OK.
  QCStringList adduser    = args->getOptionList( "adduser" );
  QCStringList removeuser = args->getOptionList( "removeuser" );

  // Throw an error if no argument was provided.
  if ( adduser.isEmpty() && removeuser.isEmpty() )
  {
    KCmdLineArgs::usage( i18n( "No arguments given." ) );
    removeLockFile();
    exit( EXIT_FAILURE );
  }
  else
  {
    // Do nothing
  }

  // Check that the given users are all valid.
  if ( !checkUsers( adduser ) || !checkUsers( removeuser ) )
  {
    cerr << argv[0] << ": " << I18N_NOOP( "An invalid user name has been provided." ) << endl;
    cerr << argv[0] << ": " << I18N_NOOP( "Aborting." ) << endl;
    removeLockFile();
    exit( EXIT_FAILURE );
  }
  else
  {
    // Do nothing
  }

  // Find the sudoers file.
  QByteArray path = findFile( "sudoers" );

  if ( path.isEmpty() )
  {
    cerr << argv[0] << ": " << I18N_NOOP( "The sudoers file was not found." ) << endl;
    cerr << argv[0] << ": " << I18N_NOOP( "Aborting." ) << endl;
    removeLockFile();
    exit( EXIT_FAILURE );
  }
  else
  {
    // Do nothing
  }

  // Check that the file is regular.
  if ( (return_value = checkFile( path )) != 0 )
  {
    switch ( return_value )
    {
      case 1:
      {
        cerr << argv[0] << ": " << I18N_NOOP( "The sudoers file is irregular." ) << endl;
        break;
      }
      case 2:
      {
        cerr << argv[0] << ": " << I18N_NOOP( "Cannot access sudoers file." ) << endl;
        break;
      }
      default:
      {
        cerr << argv[0] << ": " << strerror( return_value ) << endl;
        break;
      }
    }

    cerr << argv[0] << ": " << I18N_NOOP( "Aborting." ) << endl;
    removeLockFile();
    exit( EXIT_FAILURE );
  }
  else
  {
    // Do nothing
  }

  // Check that the utility programs can actually be found.
  if ( !findUtilityPrograms() )
  {
    cerr << argv[0] << ": " << I18N_NOOP( "One or more utility programs could not be found." ) << endl;
    cerr << argv[0] << ": " << I18N_NOOP( "Aborting." ) << endl;
    removeLockFile();
    exit( EXIT_FAILURE );
  }
  else
  {
    // Do nothing
  }

  // Save the original permissions for later.
  struct stat file_stat;

  if ( stat( path, &file_stat ) == -1 )
  {
    int error_number = errno;
    cerr << argv[0] << ": " << I18N_NOOP( "Could not stat the file " ) << path << "." << endl;
    cerr << argv[0] << ": " << strerror( error_number ) << endl;
    cerr << argv[0] << ": " << I18N_NOOP( "Aborting." ) << endl;
    removeLockFile();
    exit( EXIT_FAILURE );
  }

  mode_t perms = file_stat.st_mode;

  // Temporarily give the *owner* the permission to
  // read and write to the file.
  if ( chmod( path, S_IRUSR | S_IWUSR ) == -1 )
  {
    int error_number = errno;
    cerr << argv[0] << ": " << I18N_NOOP( "Could not change file permissions for " ) << path << endl;
    cerr << argv[0] << ": " << strerror( error_number ) << endl;
    cerr << argv[0] << ": " << I18N_NOOP( "Aborting." ) << endl;
    removeLockFile();
    exit( EXIT_FAILURE );
  }
  
  // Read the contents of the file.
  QFile file( path );
  QStringList contents;

  if ( file.open( IO_ReadOnly ) )
  {
    QTextStream ts( &file );
    // Note: With Qt 4.3 this seems to be obsolete, we'll keep
    // it for now.
    ts.setCodec( QTextCodec::codecForLocale() );

    while ( !ts.atEnd() )
    {
      contents.append( ts.readLine() );
    }

    file.close();
    
    if ( chmod( path, perms ) == -1 )
    {
      int error_number = errno;
      cerr << argv[0] << ": " << I18N_NOOP( "Could not change file permissions for " ) << path << endl;
      cerr << argv[0] << ": " << strerror( error_number ) << endl;
      cerr << argv[0] << ": " << I18N_NOOP( "Aborting." ) << endl;
      removeLockFile();
      exit( EXIT_FAILURE );
    }
  }
  else
  {
    if ( chmod( path, perms ) == -1 )
    {
      int error_number = errno;
      cerr << argv[0] << ": " << I18N_NOOP( "Could not change file permissions for " ) << path << endl;
      cerr << argv[0] << ": " << strerror( error_number ) << endl;
      cerr << argv[0] << ": " << I18N_NOOP( "Aborting." ) << endl;
      removeLockFile();
      exit( EXIT_FAILURE );
    }
       
    cerr << argv[0] << ": " << file.errorString().local8Bit().data() << endl;
    removeLockFile();
    exit( EXIT_FAILURE );
  }

  // Find the beginning and the end of the entries in
  // the sudoers file:
  QStringList::Iterator begin = contents.find( "# Entries for Smb4K users." );
  QStringList::Iterator end = contents.find( "# End of Smb4K user entries." );

  bool write = false;

  // Add user(s).
  if ( !adduser.isEmpty() )
  {
    if ( contents.grep( "# Entries for Smb4K users." ).count() == 0 &&
         contents.grep( "# End of Smb4K user entries." ).count() == 0 )
    {
      // Get the hostname.
      size_t hostnamelen = 255;
      char *hn = new char[hostnamelen];

      if ( gethostname( hn, hostnamelen ) == -1 )
      {
        int error_number = errno;
        cerr << argv[0] << ": " << strerror( error_number ) << endl;
        removeLockFile();
        exit( EXIT_FAILURE );
      }
      else
      {
        // Do nothing
      }

      QString hostname( hn );
      delete [] hn;

      // Add the new entries.
      if ( !contents.last().stripWhiteSpace().isEmpty() )
      {
        contents.append( "" );
      }
      else
      {
        // Do not add empty line to the end.
      }

      QString users;
      
      for ( QCStringList::Iterator it = adduser.begin(); it != adduser.end(); ++it )
      {
        users.append( QString::fromLocal8Bit( *it, -1 ) );
        users.append( "," );
      }
      
      users.truncate( users.length() - 1 );

      contents.append( "# Entries for Smb4K users." );
      contents.append( "# Generated by Smb4K. Please do not modify!" );
      contents.append( "User_Alias\tSMB4KUSERS = "+users );
      contents.append( "Defaults:SMB4KUSERS\tenv_keep += \"PASSWD USER\"" );
      contents.append( "SMB4KUSERS\t"+hostname+" = NOPASSWD: "
                       +KGlobal::dirs()->findResource( "exe", "smb4k_kill" ) );
      contents.append( "SMB4KUSERS\t"+hostname+" = NOPASSWD: "
                       +KGlobal::dirs()->findResource( "exe", "smb4k_umount" ) );
      contents.append( "SMB4KUSERS\t"+hostname+" = NOPASSWD: "
                       +KGlobal::dirs()->findResource( "exe", "smb4k_mount" ) );
      contents.append( "# End of Smb4K user entries." );

      write = true;
    }
    else if ( contents.grep( "# Entries for Smb4K users." ).count() == 1 &&
              contents.grep( "# End of Smb4K user entries." ).count() == 1 )    
    {
      for ( QStringList::Iterator it = begin; it != end; ++it )
      {
        if ( (*it).startsWith( "User_Alias\tSMB4KUSERS" ) )
        {
          for ( QCStringList::Iterator i = adduser.begin(); i != adduser.end(); ++i )
          {
            if ( !(*it).contains( QString::fromLocal8Bit( *i, -1 ) ) )
            {
              (*it).append( ","+QString::fromLocal8Bit( *i, -1 ) );
              continue;
            }
            else
            {
              continue;
            }
          }

          write = true;
          break;
        }
        else
        {
          continue;
        }
      }
    }
    else
    {
      cerr << argv[0] << ": " << I18N_NOOP( "The Smb4K section does not conform with the required format." ) << endl;
      cerr << argv[0] << ": " << I18N_NOOP( "Aborting." ) << endl;
      removeLockFile();
      exit( EXIT_FAILURE );
    }
  }
  else
  {
    // Do nothing
  }

  // Remove user(s).
  if ( !removeuser.isEmpty() )
  {
    if ( contents.grep( "# Entries for Smb4K users." ).count() != 0 &&
         contents.grep( "# End of Smb4K user entries." ).count() != 0 )  
    {
      for ( QStringList::Iterator it = begin; it != end; ++it )
      {
        if ( (*it).startsWith( "User_Alias\tSMB4KUSERS" ) )
        {
          QString users = (*it).section( "=", 1, 1 ).stripWhiteSpace();

          if ( !users.contains( "," ) )
          {
            // In this case, there is only one user in the list. Check if
            // it is the user who requested the removal:
            for ( QCStringList::Iterator i = removeuser.begin(); i != removeuser.end(); ++i )
            {
              if ( QString::compare( users, *i ) == 0 )
              {
                contents.erase( begin, end );
                contents.erase( end );

                write = true;

                break;
              }
              else
              {
                // They are not equal: Do nothing.
                break;
              }
            }

            break;
          }
          else
          {
            // In this case there is more than one user in the list.
            // Remove the user who requested the removal:
            QStringList list = QStringList::split( ",", users, false );

            for ( QCStringList::Iterator i = removeuser.begin(); i != removeuser.end(); ++i )
            {
              if ( list.remove( QString::fromLocal8Bit( *i, -1 ) ) != 0 )
              {
                (*it).replace( users, list.join( "," ) );
                write = true;
                continue;
              }
              else
              {
                continue;
              }
            }

            break;
          }

          break;
        }
        else
        {
          continue;
        }
      }
    }
    else if ( contents.grep( "# Entries for Smb4K users." ).count() == 0 &&
              contents.grep( "# End of Smb4K user entries." ).count() == 0 )
    {
      // Do nothing
    }
    else
    {
      cerr << argv[0] << ": " << I18N_NOOP( "The Smb4K section does not conform with the required format." ) << endl;
      cerr << argv[0] << ": " << I18N_NOOP( "Aborting." ) << endl;
      removeLockFile();
      exit( EXIT_FAILURE );
    }
  }
  else
  {
    // Nothing to do.
  }

  if ( write )
  {
    // Temporarily give the *owner* the permission to
    // read and write to the file.
    if ( chmod( path, S_IRUSR | S_IWUSR ) == -1 )
    {
      int error_number = errno;
      cerr << argv[0] << ": " << I18N_NOOP( "Could not change file permissions for " ) << path << endl;
      cerr << argv[0] << ": " << strerror( error_number ) << endl;
      cerr << argv[0] << ": " << I18N_NOOP( "Aborting." ) << endl;
      removeLockFile();
      exit( EXIT_FAILURE );
    }

    if ( file.open( IO_WriteOnly ) )
    {
      QTextStream ts( &file );
      // Note: With Qt 4.3 this seems to be obsolete, we'll keep
      // it for now.
      ts.setCodec( QTextCodec::codecForLocale() );

      ts << contents.join( "\n" ) << endl;

      file.close();
    
      if ( chmod( path, perms ) == -1 )
      {
        int error_number = errno;
        cerr << argv[0] << ": " << I18N_NOOP( "Could not change file permissions for " ) << path << endl;
        cerr << argv[0] << ": " << strerror( error_number ) << endl;
        cerr << argv[0] << ": " << I18N_NOOP( "Aborting." ) << endl;
        removeLockFile();
        exit( EXIT_FAILURE );
      }    
    }
    else
    {
      if ( chmod( path, S_IRUSR | S_IWUSR ) == -1 )
      {
        int error_number = errno;
        cerr << argv[0] << ": " << I18N_NOOP( "Could not change file permissions for " ) << path << endl;
        cerr << argv[0] << ": " << strerror( error_number ) << endl;
        cerr << argv[0] << ": " << I18N_NOOP( "Aborting." ) << endl;
      }      
      
      cerr << argv[0] << ": " << file.errorString().local8Bit().data() << endl;
      removeLockFile();
      exit( EXIT_FAILURE );
    }
  }
  else
  {
    // No modifications are needed.
  }

  // File permissions were fixed above.

  args->clear();

  removeLockFile();

  app.exit( EXIT_SUCCESS );
}
