/*
 * Copyright (c) INRIA 1996-2009
 */

/*
 * Handle windows and menu bars
 *
 * Authors: I. Vatton (INRIA), S. Gully (INRIA)
 *
 */
#include "AmayaFrame.h"

#include "thot_gui.h"
#include "thot_sys.h"
#include "constmedia.h"
#include "typemedia.h"
#include "appdialogue.h"
#include "dialog.h"

#include "AmayaParams.h"
#include "appdialogue_wx_f.h"

#include "application.h"
#include "dialog.h"
#include "document.h"
#include "message.h"
#include "libmsg.h"

#define MAX_ARGS	20
#define MaxEquivLen     400

#undef THOT_EXPORT
#define THOT_EXPORT extern
#include "font_tv.h"
#include "edit_tv.h"
#include "modif_tv.h"
#include "select_tv.h"
#include "frame_tv.h"
#include "units_tv.h"
#include "appdialogue_tv.h"
#include "appevents_tv.h"

#include "absboxes_f.h"
#include "actions_f.h"
#include "appdialogue_f.h"
#include "appdialogue_wx_f.h"
#include "applicationapi_f.h"
#include "boxlocate_f.h"
#include "boxparams_f.h"
#include "boxselection_f.h"
#include "buildboxes_f.h"
#include "callback_f.h"
#include "context_f.h"
#include "creationmenu_f.h"
#include "dialogapi_f.h"
#include "dictionary_f.h"
#include "displaybox_f.h"
#include "displayview_f.h"
#include "editcommands_f.h"
#include "selectionapi_f.h"
#include "textcommands_f.h"
#include "font_f.h"
#include "inites_f.h"

#define MENU_VAL_LENGTH 1000

#include "input_f.h" 
#include "appli_f.h"
#include "profiles_f.h"
#include "thotmsg_f.h"

#include "glwindowdisplay.h"
#include "gltimer.h"

/* function generated by the compiler app */
extern void TteLoadApplications ( void );
//extern ThotBool     WithMessages;	/* partage avec le module dialog.c */
//extern ThotPixmap   image;
extern int          appArgc;
extern char       **appArgv;
extern int          iString;
typedef void        (*Thot_ActionProc) ();
typedef struct _CallbackCTX *PtrCallbackCTX;

typedef struct _CallbackCTX
{
  Thot_ActionProc     callbackProc;
  int                 callbackSet;
  PtrCallbackCTX      callbackNext;
}
CallbackCTX;

static PtrCallbackCTX FirstCallbackAPI;
static int          FreeMenuAction;
static ThotBool     g_DoingAction = FALSE;

void InitClue (ThotWidget toplevel);


/*----------------------------------------------------------------------
  TteInitMenuActions alloue la table des actions.                    
  ----------------------------------------------------------------------*/
void TteInitMenus (char *name, int number)
{
  char               *profile = NULL;
  int                 i;

  /* Initialisation du  contexte serveur */
  InitDocContexts ();
  /* Initialise le dialogue */
  servername = NULL;
  if (appArgc > 2)
    {
      i = 1;
      while (i < appArgc - 1)
        if (!strcmp (appArgv[i], "-display"))
          /* the display name is the following argument */
          servername = appArgv[++i];
        else if (!strcmp (appArgv[i], "-profile"))
          /* the profile name is the following argument */
          profile = appArgv[++i];
        else
          i++;
    }
  /* Init the profile table */
  Prof_InitTable (profile);
  TtaInitDialogue (servername, &app_cont);
  /* Definition de la procedure de retour des dialogues */
  TtaDefineDialogueCallback ((Proc)ThotCallback);
  Dict_Init ();
  ThotInitDisplay (name, 0, 0);
   
  /* reserve Thot entries */
  TtaGetReferencesBase (MAX_ThotMenu);

  /* Il faut ajouter les actions internes liees a la structure */
  number += MAX_INTERNAL_CMD;

  MaxMenuAction = number;
  MenuActionList = (Action_Ctl *) TtaGetMemory (number * sizeof (Action_Ctl));
  /* initialisation des equilalents clavier et validation de l'action */
  for (FreeMenuAction = 0; FreeMenuAction < MAX_INTERNAL_CMD; FreeMenuAction++)
    {
      MenuActionList[FreeMenuAction].ActionEquiv = NULL;
      for (i = 0; i < MAX_DOCUMENTS; i++)
        MenuActionList[FreeMenuAction].ActionActive[i] = TRUE;
    }
  for (i = FreeMenuAction; i < MaxMenuAction; i++)
    {
      MenuActionList[i].ActionName = "";
      MenuActionList[i].Call_Action = (Proc) NULL;
      MenuActionList[i].ActionEquiv = NULL;
    }

  /* Initialisation des actions internes obligatoires */
  MenuActionList[0].ActionName = "TtcInsertChar";	/* action InsertChar() */
  MenuActionList[0].Call_Action = (Proc) NULL;

  MenuActionList[CMD_DeletePrevChar].ActionName = "TtcDeletePreviousChar";
  MenuActionList[CMD_DeletePrevChar].Call_Action = (Proc) NULL;
  MenuActionList[CMD_DeletePrevChar].ActionEquiv = TtaStrdup ("BackSpace");

  MenuActionList[CMD_DeleteSelection].ActionName = "TtcDeleteSelection";
  MenuActionList[CMD_DeleteSelection].Call_Action = (Proc) NULL;
  MenuActionList[CMD_DeleteSelection].ActionEquiv = TtaStrdup ("Delete");

  MenuActionList[CMD_PreviousChar].ActionName = "TtcPreviousChar";
  MenuActionList[CMD_PreviousChar].Call_Action = (Proc) TtcPreviousChar;

  MenuActionList[CMD_NextChar].ActionName = "TtcNextChar";
  MenuActionList[CMD_NextChar].Call_Action = (Proc) TtcNextChar;

  MenuActionList[CMD_PreviousLine].ActionName = "TtcPreviousLine";
  MenuActionList[CMD_PreviousLine].Call_Action = (Proc) TtcPreviousLine;

  MenuActionList[CMD_NextLine].ActionName = "TtcNextLine";
  MenuActionList[CMD_NextLine].Call_Action = (Proc) TtcNextLine;

  MenuActionList[CMD_PreviousWord].ActionName = "TtcPreviousWord";
  MenuActionList[CMD_PreviousWord].Call_Action = (Proc) TtcPreviousWord;

  MenuActionList[CMD_NextWord].ActionName = "TtcNextWord";
  MenuActionList[CMD_NextWord].Call_Action = (Proc) TtcNextWord;

  MenuActionList[CMD_BeginningOfLine].ActionName = "TtcStartOfLine";
  MenuActionList[CMD_BeginningOfLine].Call_Action = (Proc) TtcStartOfLine;

  MenuActionList[CMD_SelBeginningOfLine].ActionName = "TtcSelStartOfLine";
  MenuActionList[CMD_SelBeginningOfLine].Call_Action = (Proc) TtcSelStartOfLine;

  MenuActionList[CMD_SelEndOfLine].ActionName = "TtcSelEndOfLine";
  MenuActionList[CMD_SelEndOfLine].Call_Action = (Proc) TtcSelEndOfLine;


  MenuActionList[CMD_EndOfLine].ActionName = "TtcEndOfLine";
  MenuActionList[CMD_EndOfLine].Call_Action = (Proc) TtcEndOfLine;

  MenuActionList[CMD_ParentElement].ActionName = "TtcParentElement";
  MenuActionList[CMD_ParentElement].Call_Action = (Proc) TtcParentElement;

  MenuActionList[CMD_PreviousElement].ActionName = "TtcPreviousElement";
  MenuActionList[CMD_PreviousElement].Call_Action = (Proc) TtcPreviousElement;

  MenuActionList[CMD_NextElement].ActionName = "TtcNextElement";
  MenuActionList[CMD_NextElement].Call_Action = (Proc) TtcNextElement;

  MenuActionList[CMD_ChildElement].ActionName = "TtcChildElement";
  MenuActionList[CMD_ChildElement].Call_Action = (Proc) TtcChildElement;

  MenuActionList[CMD_PageUp].ActionName = "TtcPageUp";
  MenuActionList[CMD_PageUp].Call_Action = (Proc) TtcPageUp;
  MenuActionList[CMD_PageUp].ActionEquiv = TtaStrdup ("Prior");

  MenuActionList[CMD_PageDown].ActionName = "TtcPageDown";
  MenuActionList[CMD_PageDown].Call_Action = (Proc) TtcPageDown;
  MenuActionList[CMD_PageDown].ActionEquiv = TtaStrdup ("Next");

  MenuActionList[CMD_PageTop].ActionName = "TtcPageTop";
  MenuActionList[CMD_PageTop].Call_Action = (Proc) TtcPageTop;
  MenuActionList[CMD_PageTop].ActionEquiv = TtaStrdup ("Home");

  MenuActionList[CMD_PageEnd].ActionName = "TtcPageEnd";
  MenuActionList[CMD_PageEnd].Call_Action = (Proc) TtcPageEnd;
  MenuActionList[CMD_PageEnd].ActionEquiv = TtaStrdup ("End");

  MenuActionList[CMD_LineUp].ActionName = "TtcLineUp";
  MenuActionList[CMD_LineUp].Call_Action = (Proc) TtcLineUp;
  MenuActionList[CMD_LineUp].ActionEquiv = TtaStrdup ("C Left");

  MenuActionList[CMD_LineDown].ActionName = "TtcLineDown";
  MenuActionList[CMD_LineDown].Call_Action = (Proc) TtcLineDown;
  MenuActionList[CMD_LineDown].ActionEquiv = TtaStrdup ("C Down");

  MenuActionList[CMD_ScrollRight].ActionName = "TtcScrollRight";
  MenuActionList[CMD_ScrollRight].Call_Action = (Proc) TtcScrollRight;
  MenuActionList[CMD_ScrollRight].ActionEquiv = TtaStrdup ("C Right");

  MenuActionList[CMD_ScrollLeft].ActionName = "TtcScrollLeft";
  MenuActionList[CMD_ScrollLeft].Call_Action = (Proc) TtcScrollLeft;
  MenuActionList[CMD_ScrollLeft].ActionEquiv = TtaStrdup ("C Left");

  MenuActionList[CMD_CreateElement].ActionName = "TtcCreateElement";
  MenuActionList[CMD_CreateElement].Call_Action = (Proc) NULL;

  MenuActionList[CMD_LineBreak].ActionName = "TtcInsertLineBreak";
  MenuActionList[CMD_LineBreak].Call_Action = (Proc) TtcInsertLineBreak;
  MenuActionList[CMD_LineBreak].ActionEquiv = TtaStrdup ("C Return");

  MenuActionList[CMD_CopyToClipboard].ActionName = "TtcCopyToClipboard";
  MenuActionList[CMD_CopyToClipboard].Call_Action = (Proc) TtcCopyToClipboard;

  MenuActionList[CMD_PasteFromClipboard].ActionName = "TtcPasteFromClipboard";
  MenuActionList[CMD_PasteFromClipboard].Call_Action = (Proc) NULL;

  MenuActionList[CMD_PreviousSelChar].ActionName = "TtcPreviousSelChar";
  MenuActionList[CMD_PreviousSelChar].Call_Action = (Proc) TtcPreviousSelChar;

  MenuActionList[CMD_NextSelChar].ActionName = "TtcNextSelChar";
  MenuActionList[CMD_NextSelChar].Call_Action = (Proc) TtcNextSelChar;

  MenuActionList[CMD_PreviousSelLine].ActionName = "TtcPreviousSelLine";
  MenuActionList[CMD_PreviousSelLine].Call_Action = (Proc) TtcPreviousSelLine;

  MenuActionList[CMD_NextSelLine].ActionName = "TtcNextSelLine";
  MenuActionList[CMD_NextSelLine].Call_Action = (Proc) TtcNextSelLine;

  MenuActionList[CMD_PreviousSelWord].ActionName = "TtcPreviousSelWord";
  MenuActionList[CMD_PreviousSelWord].Call_Action = (Proc) TtcPreviousSelWord;

  MenuActionList[CMD_NextSelWord].ActionName = "TtcNextSelWord";
  MenuActionList[CMD_NextSelWord].Call_Action = (Proc) TtcNextSelWord;

  MenuActionList[CMD_Copy].ActionName = "TtcCopySelection";
  MenuActionList[CMD_Copy].Call_Action = (Proc) TtcCopySelection;

  MenuActionList[CMD_Paste].ActionName = "TtcPaste";
  MenuActionList[CMD_Paste].Call_Action = (Proc) TtcPaste;

  MenuActionList[CMD_CutSelection].ActionName = "TtcCutSelection";
  MenuActionList[CMD_CutSelection].Call_Action = (Proc) TtcCutSelection;

  MenuActionList[CMD_SelPageUp].ActionName = "TtcPreviousSelPage";
  MenuActionList[CMD_SelPageUp].Call_Action = (Proc) TtcPreviousSelPage;

  MenuActionList[CMD_SelPageDown].ActionName = "TtcNextSelPage";
  MenuActionList[CMD_SelPageDown].Call_Action = (Proc) TtcNextSelPage;
}

/*----------------------------------------------------------------------
  TtaGetMenuActionNumber returns the number of actions
  ----------------------------------------------------------------------*/
int TtaGetMenuActionNumber()
{
  return FreeMenuAction;
}

/*----------------------------------------------------------------------
  FreeMenus frees all contextes allocated by the menu manager
  ----------------------------------------------------------------------*/
void FreeMenus ()
{
  PtrAction           pAction, aNext;
  Menu_Ctl           *ptrmenu, *mNext;
  Item_Ctl           *ptrItem;
  PtrCallbackCTX      ctxCallback;
  int                 i;

  /* free menu actions */
  if (MenuActionList)
    {
      for (i = 0; i < FreeMenuAction; i++)
        {
          TtaFreeMemory (MenuActionList[i].ActionEquiv);
          MenuActionList[i].ActionEquiv = NULL;
        }
      for (i = MAX_INTERNAL_CMD; i < FreeMenuAction; i++)
        {
//          TtaFreeMemory (MenuActionList[i].ActionName);
          MenuActionList[i].ActionName = NULL;
        }
      TtaFreeMemory (MenuActionList);
      MenuActionList = NULL;
    }
  FreeMenuAction = 0;

  /* free menu contexts allocated for standard documents*/
  ptrmenu = DocumentMenuList;
  DocumentMenuList = NULL;
  while (ptrmenu)
    {
      ptrItem = ptrmenu->ItemsList;
      for (i = 0; i < ptrmenu->ItemsNb; i++)
        if (ptrItem[i].ItemType == 'M')
          {
            /* free a submenu */
            TtaFreeMemory (ptrItem[i].SubMenu->ItemsList);
            TtaFreeMemory (ptrItem[i].SubMenu);
          }
	  
      /* free the items list */
      TtaFreeMemory (ptrItem);
      /* free the menu context */
      mNext = ptrmenu->NextMenu;
      TtaFreeMemory (ptrmenu);
      ptrmenu = mNext;
    }
    
  /* free actions */
  pAction = ActionList;
  ActionList = NULL;
  while (pAction != NULL)
    {
      aNext = pAction->ActNext;
      TtaFreeMemory (pAction);
      pAction = aNext;
    }

  /* free callback contexts */
  ctxCallback = FirstCallbackAPI;
  while (ctxCallback)
    {
      FirstCallbackAPI = ctxCallback->callbackNext;
      TtaFreeMemory (ctxCallback);
      ctxCallback = FirstCallbackAPI;
    }
}


/*----------------------------------------------------------------------
  TteAddMenuAction ajoute une nouvelle action dans la table des      
  actions d'interface.                                            
  ----------------------------------------------------------------------*/
void TteAddMenuAction (const char *actionName, Proc procedure, ThotBool state)
{
  char               *ptr;
  int                 lg;
  int                 i;

  if (actionName == NULL/* || !Prof_BelongTable (actionName)*/)
    return;			/* pas de nom d'action declare */

  lg = strlen (actionName);
  if (FreeMenuAction < MaxMenuAction && lg != 0)
    {
      /* Alloue une chaine de caractere pour le nom de l'action */
      ptr = (char *)TtaGetMemory (lg + 1);
      strcpy (ptr, actionName);
      MenuActionList[FreeMenuAction].ActionName = ptr;
      MenuActionList[FreeMenuAction].Call_Action = procedure;
      MenuActionList[FreeMenuAction].ActionEquiv = NULL;
      /* Cette nouvelle action n'est active pour aucune frame */
      for (i = 0; i < MAX_DOCUMENTS; i++)
        MenuActionList[FreeMenuAction].ActionActive[i] = state;
      FreeMenuAction++;
    }
}


/*----------------------------------------------------------------------
  FindMenuAction returns the entry that describes the menu action.
  ----------------------------------------------------------------------*/
int FindMenuAction (const char *actionName)
{
  int                 i;

  for (i = 0; i < MaxMenuAction; i++)
    {
      if (!strcmp (actionName, MenuActionList[i].ActionName))
        return (i);
    }
  return (i);
}

/*----------------------------------------------------------------------
  FindMenuItemIDFromMenuAction returns the menu item id corresponding to the action id.
  notice: only used in WX version
  ----------------------------------------------------------------------*/
int FindMenuItemIDFromMenuAction (Menu_Ctl * ptrmenu, int action_id)
{
  int        menu_id = -1;
  Item_Ctl * ptritem = NULL;
  int        item_nb = 0;
  char       item_type = ' ';

  if (!ptrmenu)
    ptrmenu = DocumentMenuList;

  /* loop on menus */
  while ( (menu_id == -1) && ptrmenu )
    {
      ptritem = ptrmenu->ItemsList;
      item_nb = 0;

      /* loop on menu items */
      while ( (menu_id == -1) && (item_nb < ptrmenu->ItemsNb) )
        {
          item_type = ptritem[item_nb].ItemType;
          switch (item_type)
            {
            case 'B':
            case 'T':
              if (ptritem[item_nb].ItemAction == action_id)
                menu_id = ptritem[item_nb].ItemID;
              break;

            case 'S':
              break;

            case 'M':
              menu_id = FindMenuItemIDFromMenuAction(ptritem[item_nb].SubMenu, action_id);
              break;

            default:
              wxASSERT_MSG(FALSE, _T("The menu item type is not supported."));
              break;
            }
          item_nb++;
        }
      
      ptrmenu = ptrmenu->NextMenu;
    }
  return menu_id;
}

/*----------------------------------------------------------------------
  FindMenuActionFromMenuItemID returns the action id corresponding to the menu item id.
  notice: only used in WX version
  ----------------------------------------------------------------------*/
int FindMenuActionFromMenuItemID (Menu_Ctl * ptrmenu, int item_id)
{
  int        action_id = -1;
  Item_Ctl * ptritem = NULL;
  int        item_nb = 0;
  char       item_type = ' ';

  if (!ptrmenu)
    ptrmenu = DocumentMenuList;

  /* loop on menus */
  while ( (action_id == -1) && ptrmenu )
    {
      ptritem = ptrmenu->ItemsList;
      item_nb = 0;

      /* loop on menu items */
      while ( (action_id == -1) && (item_nb < ptrmenu->ItemsNb) )
        {
          item_type = ptritem[item_nb].ItemType;
          switch (item_type)
            {
            case 'B':
            case 'T':
              if (ptritem[item_nb].ItemID == item_id)
                action_id = ptritem[item_nb].ItemAction;
              break;

            case 'S':
              break;

            case 'M':
              action_id = FindMenuActionFromMenuItemID(ptritem[item_nb].SubMenu, item_id);
              break;

            default:
              wxASSERT_MSG(FALSE, _T("The menu item type is not supported."));
              break;
            }
          item_nb++;
        }
      
      ptrmenu = ptrmenu->NextMenu;
    }
  return action_id;
}


/*----------------------------------------------------------------------
  TtaExecuteMenuActionFromActionId execute the corresponding menu action.
  When force is TRUE the action is called even if it's not active.
  ----------------------------------------------------------------------*/
void TtaExecuteMenuActionFromActionId (int action_id, Document doc,
                                       View view, ThotBool force)
{
  int                 frame_id;
  // prevent recursive call
  if (g_DoingAction)
    return;
  g_DoingAction = TRUE;

  UserErrorCode = 0;
  if (doc == 0 || view == 0 || action_id <= 0)
    TtaError (ERR_invalid_parameter);
  else
    {
      frame_id = GetWindowNumber (doc, view);
      if (frame_id == 0)
	{
	  g_DoingAction = FALSE;
	  return;
	}
     if (action_id > 0 && action_id < MaxMenuAction &&
         (MenuActionList[action_id].ActionActive[doc] || force) &&
         MenuActionList[action_id].Call_Action)
       {
         if (!SelPosition ||
             (strcmp (MenuActionList[action_id].ActionName, "TtcDeletePreviousChar") &&
              strcmp (MenuActionList[action_id].ActionName, "TtcDeleteSelection")))
           CloseTextInsertion ();

         // redirect focus to the canvas because when an action is done 
         // it's more probable that the user wants to type some characteres after executing the action
         TtaRedirectFocus();
         g_DoingAction = FALSE;
         (*(Proc2)MenuActionList[action_id].Call_Action) ((void *)doc, (void *)view);
       }
     else
       TtaRedirectFocus();
    }
  g_DoingAction = FALSE;
}

/*----------------------------------------------------------------------
  TteZeroMenu signale qu'il n'y a pas de menu dans ce type de        
  fenentre.                                                       
  ----------------------------------------------------------------------*/
void TteZeroMenu ()
{
}

/*----------------------------------------------------------------------
  TtaIsActionActive
  Returns TRUE if the function is available for that document
  ----------------------------------------------------------------------*/
ThotBool TtaIsActionActive (char *name, Document doc)
{
  int      result = 0;

  if (doc == 0 || name == NULL ||  LoadedDocument[doc - 1] == NULL)
    TtaError (ERR_invalid_parameter);
  else
    result = Prof_BelongDoctype (name, LoadedDocument[doc -1]->DocProfile, FALSE);
  return (result != 0);
}

/*----------------------------------------------------------------------
  TtaIsActionAvailable
  Returns TRUE if the function is available
  ----------------------------------------------------------------------*/
ThotBool TtaIsActionAvailable (const char *name)
{
  int      result = 0;

  result = Prof_BelongTable (name);
  return (result != 0);
}

/*----------------------------------------------------------------------
  TtaExecuteMenuAction execute the corresponding menu action.
  ----------------------------------------------------------------------*/
void TtaExecuteMenuAction (const char *actionName, Document doc, View view,
                           ThotBool force)
{
  int                 action_id;

  /* verifie le parametre document */
  if (doc == 0 || view == 0 || actionName == NULL)
    TtaError (ERR_invalid_parameter);
  else
    {
      action_id = FindMenuAction (actionName);
      TtaExecuteMenuActionFromActionId (action_id, doc, view, force);
    }
}

/*----------------------------------------------------------------------
  TteAddMenu ajoute un nouveau menu pour le schema donne. Si le      
  nom de schema est Null, il s'agit des menus pris par defaut.    
  ----------------------------------------------------------------------*/
void TteAddMenu (int view, int menuID, int itemsNumber, const char *menuName)
{
  Menu_Ctl           *ptrmenu = NULL;
  Menu_Ctl           *newmenu;
  Item_Ctl           *ptr;
  int                 i;

  /* Creation du nouveau menu */
  newmenu = (Menu_Ctl *) TtaGetMemory (sizeof (Menu_Ctl));
  newmenu->MenuID = menuID;
  newmenu->MenuView = view;
  newmenu->ItemsNb = itemsNumber;
  /* Enregistre les menus actifs */
  newmenu->MenuAttr = FALSE;
  newmenu->MenuSelect = FALSE;
  newmenu->MenuHelp = FALSE;
  newmenu->MenuContext = FALSE;
  newmenu->MenuDocContext = FALSE;
  if (!strcmp (menuName, "MenuAttribute"))
    newmenu->MenuAttr = TRUE;
  else if (!strcmp (menuName, "MenuSelection"))
    newmenu->MenuSelect = TRUE;
  else if (!strcmp (menuName, "MenuHelp"))
    newmenu->MenuHelp = TRUE;
  else if (!strcmp (menuName, "MenuContext"))
    newmenu->MenuContext = TRUE;
  else if (!strcmp (menuName, "MenuDocContext"))
    newmenu->MenuDocContext = TRUE;
   
  /* creation et initialisation de la table des items */
  ptr = (Item_Ctl *)TtaGetMemory (itemsNumber * sizeof (Item_Ctl));
  for (i = 0; i < itemsNumber; i++)
    {
      ptr[i].ItemID = -1;
      ptr[i].ItemAction = -1;
      ptr[i].ItemType = SPACE;
    }
  newmenu->ItemsList = ptr;
  newmenu->NextMenu = NULL;

  /* Chainage du nouveau menu aux autres menus existants */
  /* il s'agit d'un des menus pris par defaut */
  if (DocumentMenuList == NULL)
    {
      DocumentMenuList = newmenu;
      ptrmenu = NULL;
    }
  else
    ptrmenu = DocumentMenuList;	/* simple ajout du menu dans une liste */

  if (ptrmenu != NULL)
    {
      /* Ajout du nouveau menu en fin de liste */
      while (ptrmenu->NextMenu != NULL)
        ptrmenu = ptrmenu->NextMenu;
      ptrmenu->NextMenu = newmenu;
    }
}


/*----------------------------------------------------------------------
  TteAddSubMenu ajoute un sous-menu pour le schema donne.            
  ----------------------------------------------------------------------*/
void TteAddSubMenu( int menuID,	int itemID, int itemsNumber )
{
  Menu_Ctl           *ptrmenu;
  Menu_Ctl           *newmenu;
  Item_Ctl           *ptr, *ptrItem;
  int                 i, j;

  /* Recherche la bonne liste de menus */
  ptrmenu = NULL;
  /* il s'agit d'un des menus pris par defaut */
  if (DocumentMenuList != NULL)
    ptrmenu = DocumentMenuList;

  /* Recherche le bon menu */
  while (ptrmenu != NULL && menuID != ptrmenu->MenuID)
    ptrmenu = ptrmenu->NextMenu;

  if (ptrmenu != NULL)
    {
      /* recherche l'item dans le menu */
      ptrItem = ptrmenu->ItemsList;
      j = 0;
      while (j < ptrmenu->ItemsNb && ptrItem[j].ItemType != SPACE)
        j++;
      if (j < ptrmenu->ItemsNb)
        {
          /* Creation du sous-menu */
          newmenu = (Menu_Ctl *) TtaGetMemory (sizeof (Menu_Ctl));
          newmenu->MenuID = 0;
          newmenu->MenuView = 0;
          newmenu->ItemsNb = itemsNumber;
          newmenu->MenuAttr = FALSE;
          newmenu->MenuSelect = FALSE;
          newmenu->MenuHelp = FALSE;
          newmenu->MenuContext = FALSE;
          newmenu->MenuDocContext = FALSE;

          /* creation et initialisation de la table des items */
          ptr = (Item_Ctl *) TtaGetMemory (itemsNumber * sizeof (Item_Ctl));
          for (i = 0; i < itemsNumber; i++)
            {
              ptr[i].ItemID = -1;
              ptr[i].ItemAction = -1;
              ptr[i].ItemType = SPACE;
            }
          newmenu->ItemsList = ptr;
          newmenu->NextMenu = NULL;
          /* relie le sous-menu a l'item */
          ptrItem[j].SubMenu = newmenu;
          ptrItem[j].ItemID = itemID;
          ptrItem[j].ItemType = 'M';
        }
    }
}


/*----------------------------------------------------------------------
  TteAddMenuItem ajoute une nouvel item dans un menu.                
  ----------------------------------------------------------------------*/
void TteAddMenuItem (int menuID, int subMenu, int itemID, const char *actionName, char itemType, const char * iconName)
{
  Menu_Ctl           *ptrmenu;
  Item_Ctl           *ptr;
  int                 i;

  /* Recherche la bonne liste de menus */
  ptrmenu = NULL;
  /* il s'agit d'un des menus pris par defaut */
  if (DocumentMenuList != NULL)
    ptrmenu = DocumentMenuList;
       
  /* Recherche le menu */
  while (ptrmenu != NULL && menuID != ptrmenu->MenuID)
    ptrmenu = ptrmenu->NextMenu;
       
  if (ptrmenu != NULL && subMenu != -1)
    {
      /* Recherche l'entree du sous-menu dans le menu */
      i = 0;
      ptr = ptrmenu->ItemsList;
      while (i < ptrmenu->ItemsNb && (ptr[i].ItemID != subMenu))
        i++;
      if (i < ptrmenu->ItemsNb)
        ptrmenu = ptr[i].SubMenu;
      else
        /* on n'a pas trouve le sous-menu */
        return;
    }
 

  /* checks if the item is present in the user profile */
  if (actionName == NULL || Prof_BelongTable (actionName))
    { 
      /* ajoute l'item dans le menu */
      i = 0;
      ptr = ptrmenu->ItemsList;
      while (i < ptrmenu->ItemsNb && ptr[i].ItemType != SPACE)
        i++;
      if (i < ptrmenu->ItemsNb)
        {
          /* Remove the separaror if the previous element was one */
          if (! (itemType == 'S' && (i == 0 || ptr[i-1].ItemType == 'S')))
            {
              ptr[i].ItemID = itemID;
              ptr[i].ItemType = itemType;
              if (actionName != NULL)
                ptr[i].ItemAction = FindMenuAction (actionName);
              if (iconName != NULL)
                strcpy(ptr[i].ItemIconName, iconName);
              else
                ptr[i].ItemIconName[0] = EOS;
            }
          else
            /* Remove separator */
            ptrmenu->ItemsNb--;
        }
    }
  else
    {
      /* removes the entry */
      ptrmenu->ItemsNb--;
    }       
}


/*----------------------------------------------------------------------
  TteOpenMainWindow opens the application main window.
 
  Parameters:
  name: the name to be displayed as the title of the main window.
  ----------------------------------------------------------------------*/
void TteOpenMainWindow (char *name)
{
  /* Creation de la fenetre principale */
  UserErrorCode = 0;
  InitTranslations (name);
  /* no external action declared at that time */
  ActionList = NULL;
  TteLoadApplications ();
}


/*----------------------------------------------------------------------
  TtaAddTextZone

  Adds a new textual command in a document view.
  This function must specify a valid view of a valid document.
  Parameters:
  doc: the concerned document.
  view: the concerned view.
  label: label of the new entry.
  procedure: procedure to be executed when the new entry is changed by the
  user.
  listUrl gives URLs that will be displayed in the combobox.
  ----------------------------------------------------------------------*/
int TtaAddTextZone (Document doc, View view, char *label,
                    ThotBool editable, void (*procedure) (), 
                    char *listUrl)
{
  int            frame, ret;
  ThotWidget     w;

  UserErrorCode = 0;
  w = 0;
  ret = 0;
  /* verifie le parametre document */
  if (doc == 0 && view == 0)
    TtaError (ERR_invalid_parameter);
  else
    {
      frame = GetWindowNumber (doc, view);
      if (frame == 0 || frame > MAX_FRAME)
        TtaError (ERR_invalid_parameter);
      else if (FrameTable[frame].WdFrame)
          {
            TtaSetURLBar( frame, listUrl, procedure );
            ret = 1;
          }
    }
  /* force la mise a jour de la fenetre */
  TtaHandlePendingEvents ();
  return ret;
}

/*----------------------------------------------------------------------
  TtaSetTextZone

  Sets the text in text-zone in a document view.
  This function must specify a valid view of a valid document.
  Parameters:
  doc: identifier of the document.
  view: identifier of the view.
  listUrl gives URLs that will be displayed in the combobox.
  ----------------------------------------------------------------------*/
void TtaSetTextZone (Document doc, View view, char *listUrl)
{
  int            frame;

  UserErrorCode = 0;
  /* verifie le parametre document */
  if (doc == 0 && view == 0)
    TtaError (ERR_invalid_parameter);
  else if (listUrl)
    {
      frame = GetWindowNumber (doc, view);
      if (frame == 0 || frame > MAX_FRAME)
        TtaError (ERR_invalid_parameter);
      else if (FrameTable[frame].WdFrame)
        {
          TtaSetURLBar( frame, listUrl, NULL );	      
        }
    }
}


/*----------------------------------------------------------------------
  Evenement sur une frame document.                              
  ----------------------------------------------------------------------*/
void DrawingInput (int *w, int frame, int *infos)
{
}


/*----------------------------------------------------------------------
  BuildMenus builds or rebuilds frame menus.
  The parameter RO is TRUE when only ReadOnly functions are accepted
  ----------------------------------------------------------------------*/
void TtaUpdateMenus (Document doc, View view, ThotBool RO)
{
  Menu_Ctl           *ptrmenu, *ptrsmenu;
  Item_Ctl           *ptr, *sptr;
  int                 profile, action, i, j, m = 0, res;

  if (doc)
    {
      ptrmenu = DocumentMenuList;
      profile = TtaGetDocumentProfile (doc);
      while (ptrmenu)
        {
          m++;
          /* skip menus that concern another view */
          if (ptrmenu->MenuID != 0 /* skip menu File */ &&
              !ptrmenu->MenuAttr &&
              !ptrmenu->MenuSelect &&
              !ptrmenu->MenuContext &&
              !ptrmenu->MenuDocContext &&
              !ptrmenu->MenuHelp &&
              (ptrmenu->MenuView == 0 || ptrmenu->MenuView == view) &&
              Prof_ShowMenu (ptrmenu))
            {
              ptr = ptrmenu->ItemsList;
              i = 0;
              while (i < ptrmenu->ItemsNb)
                {
                  action = ptr[i].ItemAction;
                  if (action == -1)
                    ;	/* separator */
                  else if (ptr[i].ItemType == 'M')
                    {
                      j = 0;
                      ptrsmenu = ptr[i].SubMenu;
                      sptr = ptrsmenu->ItemsList;
                      while (j < ptrsmenu->ItemsNb)
                        {
                          action = sptr[j].ItemAction;
                          if (action == -1)
                            ;	/* separator */
                          else
                            {
                              res = Prof_BelongDoctype (MenuActionList[action].ActionName, profile, RO);
                              if (res == 1)
                                MenuActionList[action].ActionActive[doc] = TRUE;
                              else if (res == 0)
                                MenuActionList[action].ActionActive[doc] = FALSE;
                            }
                          j++;
                        }
                    }
                  else
                    {
                      res = Prof_BelongDoctype (MenuActionList[action].ActionName, profile, RO);
                      if (res == 1)
                         MenuActionList[action].ActionActive[doc] = TRUE;
                       else if (res == 0)
                         MenuActionList[action].ActionActive[doc] = FALSE;
                     }
                  i++;
                }
              // refresh that menu
              TtaRefreshTopMenuStats (doc, m);
            }
          ptrmenu = ptrmenu->NextMenu;
        }
    }
}


/*----------------------------------------------------------------------
  MakeFrame
  Create a frame at position X,Y and dimensions width,height (if >0).
  - name gives the title of the window.
  - schema gives the schema name of the current document.
  - view is the schema view number.
  - doc gives the document index.
  - withMenu is TRUE when menus are displayed
  - withButton is TRUE when buttons are displayed
  Returns:
  - volume: the number of characters that can be displayed in the window.
  - the allocated window index or 0.
  ----------------------------------------------------------------------*/
int  MakeFrame (char *schema, int view, char *name, int X, int Y,
                int width, int height, int *volume, int doc,
                ThotBool withMenu, ThotBool withButton)
{
  /* this is for none gui compilation */
  return 0;
}

/*----------------------------------------------------------------------
  TtaDisableScrollbars  : Disable scrollbars for this view                 
  ----------------------------------------------------------------------*/
void TtaDisableScrollbars (Document doc, View view)
{
  int frame;

  frame = GetWindowNumber (doc, view);
  FrameTable[frame].WdScrollH = NULL;
  FrameTable[frame].WdScrollV = NULL;
#ifdef _GL
  FrameTable[frame].Scroll_enabled = FALSE;
#endif /* _GL */
}
/*----------------------------------------------------------------------
  Si l'entree existe :
  WX: detruit seulement la AmayaFrame correspondante ainsi que son canvas opengl (AmayaCanvas),
  mais ne detruit pas les menus car ils sont associes au document.                                             
  Ferme la fenetre, detruit le fichier et libere l'entree.      
  Libere toutes les boites allouees a la fenetre.
  ----------------------------------------------------------------------*/
void DestroyFrame (int frame)
{
  ThotFrame           w;
  
#ifdef _GL
  /* set the frame context active to avoid mesa warnings */
  GL_prepare (frame);
  GL_DestroyFrame (frame);
#endif /* _GL */
  
  CloseTextInsertion ();
#ifdef _GL
  /* free animate blocks */
  if (FrameTable[frame].Animated_Boxes)
    {
      FreeAnimatedBoxes ((Animated_Cell *)FrameTable[frame].Animated_Boxes);
      FrameTable[frame].Animated_Boxes = NULL;
    }
#endif /* _GL */

  w = FrameTable[frame].WdFrame;
  if (w != 0)
    {
      /* don't destroy frame menu on WX because menus are specific to the document */
      /* with WX, never really delete the widgets */
      /* keep it alive in order to reuse it for the next frame */
	  if (TtaIsDocumentSelected (FrameTable[frame].FrDoc) &&
		  FrameTable[frame].FrView == 1)
		TtaUnselect (FrameTable[frame].FrDoc);
	  if (ActiveFrame == frame)
		ActiveFrame = 0;

      /* Elimine les evenements ButtonRelease, DestroyNotify, FocusOut */
      ClearConcreteImage (frame);
      ThotFreeFont (frame);	/* On libere les polices de caracteres utilisees */
      TtaDetachFrame( frame );
      TtaDestroyFrame( frame );
      FrameTable[frame].WdFrame = 0;
    }
  FrameTable[frame].FrDoc = 0;
  TtaHandlePendingEvents();
}


/*----------------------------------------------------------------------
  GetMenu_Ctl returns the pointer to the right menu context. 
  ----------------------------------------------------------------------*/
static Menu_Ctl *GetMenu_Ctl (int frame, int menu)
{  
  int                 i;
  Menu_Ctl           *ptrmenu;
  ptrmenu = DocumentMenuList;
  i = 0;
  while (i != menu && ptrmenu != NULL)
    {
      ptrmenu = ptrmenu->NextMenu;
      i++;
    }
  return (ptrmenu);
}


/*----------------------------------------------------------------------
  FindMenu returns the menu index and its context if this menu is
  displayed in this specific frame or -1.
  ----------------------------------------------------------------------*/
int FindMenu (int frame, int menuID, Menu_Ctl ** ctxmenu)
{
  Menu_Ctl           *ptrmenu;
  int                 m;

  /* Look for the menu */
  m = 1;			/* menu index */
  /* look for that menu in the menu list */
  ptrmenu = DocumentMenuList;
  while (ptrmenu != NULL && menuID != ptrmenu->MenuID)
    {
      m++;
      ptrmenu = ptrmenu->NextMenu;
    }
   
  *ctxmenu = ptrmenu;
  if (ptrmenu == NULL)
    /* menu not found */
    return (-1);
  else if (ptrmenu->MenuView != 0 &&
           ptrmenu->MenuView != FrameTable[frame].FrView)
    /* menu found but that frame is not concerned */
    return (-1);
  else
    /* ok */
    return (m);
}


/*----------------------------------------------------------------------
  FindItemMenu returns all information concerning that dialogue entry
  in this specific frame:
  - menu index or -1
  - submenu index or 0
  - item index or 0
  - action index or -1
  ----------------------------------------------------------------------*/
static void FindItemMenu (int frame, int menuID, int itemID, int *menu,
                          int *submenu, int *item, int *action)
{
  Menu_Ctl           *ptrmenu, *ptrsmenu;
  Item_Ctl           *ptr;
  int                 i, j, max;
  int                 m, sm;
  int                 entry, sentry;             
  ThotBool            found;

  j = 0;
  i = 0;
  entry = 0;
  sm = 0;
  sentry = 0;
  /* look for that menu in the menu list */
  m = FindMenu (frame, menuID, &ptrmenu);
  found = (m != -1);
  if (found)
    {
      /* search that item in the item list or in submenus */
      ptr = ptrmenu->ItemsList;
      found = FALSE;
      max = ptrmenu->ItemsNb;
      ptrsmenu = NULL;
      while (ptrmenu && !found)
        {
          while (i < max && !found)
            {
              j = ptr[i].ItemAction;
              if (j == -1)
                {
                  i++;	/* separator */
                  entry++;
                }
              else if (ptr[i].ItemID == itemID)
                {
                  /* the entry is found */
                  found = TRUE;
                  if (ptr[i].ItemType == 'M')
                    /* it doesn't match an action */
                    j = -1;
                }
              else if (ptr[i].ItemType == 'M')
                {
                  if (ptr[i].SubMenu->ItemsNb == 0)
                    i++;
                  else
                    {
                      /* search in that submenu */
                      sm = i + 1;
                      sentry = entry + 1;
                      ptrsmenu = ptr[i].SubMenu;
                      i = 0;
                      entry = 0;
                      ptr = ptrsmenu->ItemsList;
                      max = ptrsmenu->ItemsNb;
                    }
                }
              else
                {
                  /* it's not that one */
                  i++;
                  entry++;
                }
            }
	   
          /* do we close the search in a submenu? */
          if (!found && ptrsmenu != NULL)
            {
              /* continue the search in the menu */
              i = sm;
              sm = 0;
              entry = sentry;
              sentry = 0;
              ptrsmenu = NULL;
              ptr = ptrmenu->ItemsList;
              max = ptrmenu->ItemsNb;
            }
          else
            /* we close the menu search and this itemID doesn't exist */
            ptrmenu = NULL;
        }
    }

  /* do we sucess? */
  if (found)
    {
      /* yes */
      *menu = m;
      *submenu = sm;
      *item = i;
      *action = j;
    }
  else
    {
      /* no */
      *menu = -1;
      *submenu = 0;
      *item = 0;
      *action = -1;
    }
}

/*----------------------------------------------------------------------
  SwitchUndo enables (on=TRUE) or disables (on=FALSE) the Undo
  entry in all document frames.
  ----------------------------------------------------------------------*/
void SwitchUndo (PtrDocument pDoc, ThotBool on)
{  
  if (pDoc == NULL)
    {
      for (int doc_id = 1; doc_id < MAX_DOCUMENTS; doc_id++)
        {
          pDoc = LoadedDocument[doc_id-1];
          if (pDoc)
            SwitchUndo(pDoc, on);
        }
    }
  else
    { 
      Document document  = IdentDocument(pDoc);
      int      window_id = TtaGetDocumentWindowId( document, -1 );

      /* do nothing if the document doesn't have a window parent */
      if ( window_id <= 0 )
        return;

      int      item_id   = WindowTable[window_id].MenuItemUndo;
      int      action    = FindMenuActionFromMenuItemID(NULL, item_id);
      
      if ( action > 0 &&
           action < MaxMenuAction &&
           MenuActionList[action].ActionActive[document] != on)
        {
          MenuActionList[action].ActionActive[document] = on;
          TtaRefreshMenuItemStats( document, NULL, item_id );
        }
    }
}

/*----------------------------------------------------------------------
  SwitchRedo enables (on=TRUE) or disables (on=FALSE) the Redo
  entry in all document frames.
  ----------------------------------------------------------------------*/
void SwitchRedo (PtrDocument pDoc, ThotBool on)
{
  if (pDoc == NULL)
    {
      for (int doc_id = 1; doc_id < MAX_DOCUMENTS; doc_id++)
        {
          pDoc = LoadedDocument[doc_id-1];
          if (pDoc)
            SwitchRedo(pDoc, on);
        }
    }
  else
    {
      Document document  = IdentDocument(pDoc);
      int      window_id = TtaGetDocumentWindowId( document, -1 );

      /* do nothing if the document doesn't have a window parent */
      if ( window_id <= 0 )
        return;

      int      item_id   = WindowTable[window_id].MenuItemRedo;
      int      action    = FindMenuActionFromMenuItemID(NULL, item_id);
      
      if ( action > 0 &&
           action < MaxMenuAction &&
           MenuActionList[action].ActionActive[document] != on)
        {
          MenuActionList[action].ActionActive[document] = on;
          TtaRefreshMenuItemStats( document, NULL, item_id );
        }
    }
}

/*----------------------------------------------------------------------
  SwitchPaste enables (on=TRUE) or disables (on=FALSE) the Paste
  entry in all frames.
  ----------------------------------------------------------------------*/
void SwitchPaste (PtrDocument pDoc, ThotBool on)
{
  if (pDoc == NULL)
    {
      for (int doc_id = 1; doc_id < MAX_DOCUMENTS; doc_id++)
        {
          pDoc = LoadedDocument[doc_id-1];
          if (pDoc)
            SwitchPaste(pDoc, on);
        }
    }
  else
    {
      Document document  = IdentDocument(pDoc);
      int      window_id = TtaGetDocumentWindowId( document, -1 );

      /* do nothing if the document doesn't have a window parent */
      if ( window_id <= 0 )
        return;

      int      item_id   = WindowTable[window_id].MenuItemPaste;
      int      action    = FindMenuActionFromMenuItemID(NULL, item_id);
      
      if ( action > 0 &&
           action < MaxMenuAction &&
           MenuActionList[action].ActionActive[document] != on)
        {
          MenuActionList[action].ActionActive[document] = on;
          TtaRefreshMenuItemStats( document, NULL, item_id );
        }
    }
}

/*----------------------------------------------------------------------
  TtaEnableAction enable/disable the given action.
  ----------------------------------------------------------------------*/
void TtaEnableAction( Document document, const char * action_name, ThotBool enable )
{
  int      action_id = FindMenuAction(action_name);
  int        item_id = FindMenuItemIDFromMenuAction(NULL, action_id);

  wxASSERT( action_id > 0 && action_id < MaxMenuAction );
  
  if ( action_id > 0 &&
       action_id < MaxMenuAction &&
       MenuActionList[action_id].ActionActive[document] != enable)
    {
      MenuActionList[action_id].ActionActive[document] = enable;
      TtaRefreshMenuItemStats( document, NULL, item_id );
    }    
}

/*----------------------------------------------------------------------
  TtaSetMenuOff desactive le menu (1 a n) de la vue du document ou   
  de la fenetre principale (document = 0, view = 0).                 
  ----------------------------------------------------------------------*/
void TtaSetMenuOff (Document document, View view, int menuID)
{
  int                 menu;
  int                 frame;
  PtrDocument         pDoc = LoadedDocument[document-1];
  Menu_Ctl*           ptrmenu;
  
  if (document == 0 || view == 0)
    frame = 0;
  else
    frame = GetWindowNumber (document, view);
  
  /* Check parameters */
  if (frame > MAX_FRAME)
    return;
  else if ((FrameTable[frame].WdFrame) == 0)
    return;
  
  menu = FindMenu (frame, menuID, &ptrmenu);
  if (menu != -1)
    {
      menu--;
      if (pDoc->EnabledMenus[menu])
        {
          pDoc->EnabledMenus[menu] = FALSE;
          TtaRefreshTopMenuStats( document, menu );
        }
    }	  
}

/*----------------------------------------------------------------------
  TtaSetMenuOn reactive le menu (1 a n) de la vue du document ou     
  de la fenetre principale (document = 0, view = 0).                 
  ----------------------------------------------------------------------*/
void TtaSetMenuOn (Document document, View view, int menuID)
{
  int                 menu;
  int                 frame;
  PtrDocument         pDoc = LoadedDocument[document-1];  
  Menu_Ctl*           ptrmenu;

  if (document == 0 || view == 0)
    frame = 0;
  else
    frame = GetWindowNumber (document, view);
  
  /* Check parameters */
  if (frame > MAX_FRAME)
    return;
  else if ((FrameTable[frame].WdFrame) == 0)
    return;
  menu = FindMenu (frame, menuID, &ptrmenu);
  if (menu != -1)
    {
      menu--;
      if (!pDoc->EnabledMenus[menu])
        {
          pDoc->EnabledMenus[menu] = TRUE;
          TtaRefreshTopMenuStats( document, menu );
        }
    }
}


/*----------------------------------------------------------------------
  TtaSetToggleItem positionne l'item du menu de la vue du document   
  ou de la fenetre principale (document = 0, view = 0).   
  ----------------------------------------------------------------------*/
void TtaSetToggleItem (Document document, View view, int menuID, int itemID, ThotBool on)
{
  int                 frame;
  int                 menu, submenu;
  int                 item, action;

  /* Check parameters */
  if (menuID == 0 || itemID == 0)
    return;
  if (document == 0 || view == 0)
    frame = 0;
  else
    frame = GetWindowNumber (document, view);

  if (frame > MAX_FRAME)
    return;
  else if ((FrameTable[frame].WdFrame) == 0)
    return;

  /* Search the menu, submenu and item */
  FindItemMenu (frame, menuID, itemID, &menu, &submenu, &item, &action);
  if (menu >= 0 && item >= 0)
    {
      menu--;
      if ( action != -1 && MenuActionList[action].ActionToggle[document] != on )
        MenuActionList[action].ActionToggle[document] = on;
      /* refresh the item even if the action state didn't change because when clicking on the menu item
       * the WX toolkit will performe a toggle action and update automaticaly the menu item so
       * to be sure just force the refresh */
      TtaRefreshMenuItemStats( document, NULL, itemID );
    }
}


/*----------------------------------------------------------------------
  TtaSetItemOff desactive l'item actionName de la vue du document  
  ou de la fenetre principale (document = 0, view = 0).   
  ----------------------------------------------------------------------*/
void  TtaSetItemOff (Document document, View view, int menuID, int itemID)
{
  int                 frame;
  int                 menu, submenu;
  int                 item;
  int                 action;

  /* Check parameters */
  if (document == 0 || view == 0)
    frame = 0;
  else
    frame = GetWindowNumber (document, view);
   
  if (frame > MAX_FRAME)
    return;
  else if ((FrameTable[frame].WdFrame) == 0)
    return;
  /* Search the menu, submenu and item */
  FindItemMenu (frame, menuID, itemID, &menu, &submenu, &item, &action);
  if (action > 0 && action < MaxMenuAction /*&&
      MenuActionList[action].ActionActive[document]*/)
    {
    /* the entry is found and is active */
      MenuActionList[action].ActionActive[document] = FALSE;
      if (menu > 0)
        TtaRefreshMenuItemStats( document, NULL, itemID );
    }
}


/*----------------------------------------------------------------------
  TtaSetItemOn active l'item actionName de la vue du document      
  ou de la fenetre principale (document = 0, view = 0).   
  ----------------------------------------------------------------------*/
void  TtaSetItemOn (Document document, View view, int menuID, int itemID)
{
  int                 frame;
  int                 menu, submenu;
  int                 item;
  int                 action;

  /* Si les parametres sont invalides */
  if (document == 0 || view == 0)
    frame = 0;
  else
    frame = GetWindowNumber (document, view);
  if (frame > MAX_FRAME)
    return;
  else if ((FrameTable[frame].WdFrame) == 0)
    return;
  /* Recherche les bons indices de menu, sous-menu et item */
  FindItemMenu (frame, menuID, itemID, &menu, &submenu, &item, &action);
  if (action > 0 && action < MaxMenuAction &&
      !MenuActionList[action].ActionActive[document])
    {
      /* the entry is found and is not active */
      MenuActionList[action].ActionActive[document] = TRUE;
      if (menu > 0)
        TtaRefreshMenuItemStats( document, NULL, itemID );
    }
}


/*----------------------------------------------------------------------
  TtaSetCallback

  Specifies the procedure that is called when the user activates a set of forms
  and/or menus created by the application.
  The parameter set gives the number of forms and/or menus created by the 
  application managed by this porcedure.
  This function must be called before processing any event, only if the
  application uses the Dialogue tool kit for generating specific forms or menus.
  This function replaces the DefineCallback function in the Dialogue tool kit.
  This procedure is called with three parameters: the menu or form reference,
  data type, and data value.

  void callbakProcedure(reference, datatype, data)
  int reference;
  int datatype;
  char *data; 

  Parameter:
  callbakProcedure: the application callback procedure.
  set: the number of forms and/or menus managed.
  Return:
  reference of the first form or menu.
  ----------------------------------------------------------------------*/
int TtaSetCallback (void (*callbakProcedure) (), int set)
{
  PtrCallbackCTX      ctxCallback;

  UserErrorCode = 0;
  if (FirstCallbackAPI == NULL)
    {
      /* le premier bloc de callback */
      FirstCallbackAPI = (PtrCallbackCTX) TtaGetMemory (sizeof (CallbackCTX));
      ctxCallback = FirstCallbackAPI;
    }
  else
    {
      ctxCallback = FirstCallbackAPI;
      while (ctxCallback->callbackNext != NULL)
        ctxCallback = ctxCallback->callbackNext;
      ctxCallback->callbackNext = (PtrCallbackCTX) TtaGetMemory (sizeof (CallbackCTX));
      ctxCallback = ctxCallback->callbackNext;
    }

  ctxCallback->callbackProc = callbakProcedure;
  ctxCallback->callbackSet = set;
  ctxCallback->callbackNext = NULL;
  return (TtaGetReferencesBase (set));
}


/*----------------------------------------------------------------------
  ThotCallback ge`re tous les retours du dialogue de Thot.        
  ----------------------------------------------------------------------*/
void ThotCallback (int ref, int typedata, char *data)
{
  Document            document;
  View                view;
  PtrCallbackCTX      ctxCallback;
  Menu_Ctl           *ptrmenu;
  int                 frame, item;
  int                 menu, base;
  int                 menuThot;
  int                 action, i, j;

  /* Termine l'insertion courante s'il y en a une */
  CloseTextInsertion ();

  if (ref >= MAX_ThotMenu)
    {
      if (FirstCallbackAPI == NULL)
        return;		/* pas de callback definis */
      else
        {
          /* recherche le bon callback */
          ctxCallback = FirstCallbackAPI;
          base = MAX_ThotMenu;
          while (ref >= base + ctxCallback->callbackSet)
            {
              if (ctxCallback->callbackNext == NULL)
                return;
              else
                {
                  base = base + ctxCallback->callbackSet;
                  ctxCallback = ctxCallback->callbackNext;
                }
            }
          (*(Proc3)ctxCallback->callbackProc) ((void *)ref, (void *)typedata, (void *)data);
        }
    }
  else if (ref < MAX_LocalMenu)
    /*** Action interne et optionnelle de l''eur ***/
    switch (ref)
      {
      case NumMenuInsert:
        (*(Proc3)ThotLocalActions[T_rcinsertpaste]) (
                                                     (void *)TRUE,
                                                     (void *)FALSE,
                                                     (void *)((long int) data + 1) );
        break;
      case NumMenuPaste:
        (*(Proc3)ThotLocalActions[T_rcinsertpaste]) (
                                                     (void *)FALSE,
                                                     (void *)TRUE,
                                                     (void *)((long int) data + 1));
        break;
      case NumMenuInclude:
        (*(Proc3)ThotLocalActions[T_rcinsertpaste]) (
                                                     (void *)FALSE,
                                                     (void *)FALSE,
                                                     (void *)((long int) data + 1));
        break;
      case NumMenuElChoice:
        (*(Proc2)ThotLocalActions[T_rchoice]) (
                                               (void *)((long int) data + 1),
                                               (void *)NULL);
        break;
      case NumSelectNatureName:
        (*(Proc2)ThotLocalActions[T_rchoice]) (
                                               (void *)0,
                                               (void *)data);
        break;
	
      case NumMenuAttrRequired:
      case NumMenuAttrNumNeeded:
      case NumMenuAttrTextNeeded:
      case NumMenuAttrEnumNeeded:
        (*(Proc3)ThotLocalActions[T_rattrreq]) (
                                                (void *)ref,
                                                (void *)((long int) data),
                                                (void *)data);
        break;
      case NumMenuAttr:
      case NumMenuAttrNumber:
      case NumMenuAttrText:
      case NumMenuAttrEnum:
        (*(Proc3)ThotLocalActions[T_rattrval]) (
                                                (void *)ref,
                                                (void *)((long int) data),
                                                (void *)data);
        break;
	
      case NumSelectLanguage:
        (*(Proc3)ThotLocalActions[T_rattrlang]) (
                                                 (void *)ref,
                                                 (void *)0,
                                                 (void *)data);
        break;
      case NumFormLanguage:
      case NumMenuAlphaLanguage:
        (*(Proc3)ThotLocalActions[T_rattrlang]) (
                                                 (void *)ref,
                                                 (void *)((long int) data),
                                                 (void *)NULL);
        break;
      case NumFormClose:
        (*(Proc3)ThotLocalActions[T_rconfirmclose]) (
                                                     (void *)ref,
                                                     (void *)typedata,
                                                     (void *)data);
        break;
	
      case NumFormPresChar:
      case NumFormPresFormat:
      case NumFormPresGraphics:
      case NumFormColors:
      case NumMenuCharFamily:
      case NumMenuCharFontStyle:
      case NumMenuCharFontWeight:
      case NumMenuCharFontSize:
      case NumMenuUnderlineType:
      case NumMenuUnderlineWeight:
      case NumMenuAlignment:
      case NumZoneRecess:
      case NumMenuRecessSense:
      case NumZoneLineSpacing:
      case NumMenuLineSpacing:
      case NumMenuStrokeStyle:
      case NumZoneStrokeWeight:
      case NumToggleWidthUnchanged:
      case NumTogglePatternUnchanged:
        (*(Proc3)ThotLocalActions[T_present]) (
                                               (void *)ref,
                                               (void *)(long int) data,
                                               (void *)NULL);
        break;
      case NumSelectPattern:
      case NumSelectForegroundColor:
      case NumSelectBackgroundColor:
        (*(Proc3)ThotLocalActions[T_present]) (
                                               (void *)ref,
                                               (void *)0,
                                               (void *)data);
        break;
      case NumFormPresentStandard:
      case NumMenuPresentStandard:
        (*(Proc2)ThotLocalActions[T_presentstd]) (
                                                  (void *)ref,
                                                  (void *)((long int) data));
        break;
      case NumFormSearchText:
      case NumMenuReplaceMode:
      case NumToggleUpperEqualLower:
      case NumMenuSearchNature:
        /* sous-menu mode de remplacement */
        (*(Proc3)ThotLocalActions[T_searchtext]) (
                                                  (void *)ref,
                                                  (void *)((long int) data),
                                                  (void *)NULL);
        break;
      case NumZoneTextSearch:
      case NumZoneTextReplace:
      case NumSelTypeToSearch:
      case NumSelAttributeToSearch:
        /* zone de saisie du texte de remplacement */
        (*(Proc3)ThotLocalActions[T_searchtext]) (
                                                  (void *)ref,
                                                  (void *)0,
                                                  (void *)data);
        break;
      case NumMenuOrSearchText:
        (*(Proc2)ThotLocalActions[T_locatesearch]) (
                                                    (void *)ref,
                                                    (void *)((long int) data));
        break;  
      case NumFormElemToBeCreated:
        CallbackElemToBeCreated (ref, (long int)data, NULL);
        break;
      case NumSelectElemToBeCreated:
        CallbackElemToBeCreated (ref, 0, data);
        break;

      default:
        if (ref >= NumMenuAttrName && ref <= NumMenuAttrName + MAX_ITEM)
          /* retour du menu des attributs */
          {
            (*(Proc3)ThotLocalActions[T_rattr]) (
                                                 (void *)ref,
                                                 (void *)((long int) data),
                                                 (void *)ActiveFrame);
          }
        break;
      }
  else
    {
      /*** Action attachee au retour du dialoque de l'application ***/
      /* Calcule les indices menu, item et frame */
      /* ref = (((item+1) * MAX_MENU + menu) * MAX_ITEM) + frame + MAX_LocalMenu */
      j = ref - MAX_LocalMenu;
      i = j / MAX_ITEM;
      frame = j - (i * MAX_ITEM);	/* reste de la division */
      item = i / MAX_MENU;
      menu = i - (item * MAX_MENU);	/* reste de la division */
      if (frame == 0)
        {
          document = 0;
          view = 0;
        }
      else
        {
          FrameToView (frame, &document, &view);
          if (document == 0)
            return;
          int         window_id = TtaGetDocumentWindowId( document, -1 );
          menuThot = FindMenu (frame, WindowTable[window_id].MenuAttr, &ptrmenu) - 1;
          if (menu == menuThot)
            {
              /* traitement du menu attribut */
              (*(Proc3)ThotLocalActions[T_rattr]) (
                                                   (void *)ref,
                                                   (void *)((long int) data),
                                                   (void *)frame);
              return;
            }
        }

      /* Call the right action */
      ptrmenu = GetMenu_Ctl (frame, menu);
      action = 0;
      if (ptrmenu)
        {
          if (item != 0)
            {
              item--;
              if (item < ptrmenu->ItemsNb && ptrmenu->ItemsList)
                {
                  for (i = 0; i <= item; i++)
                    {
                      action = ptrmenu->ItemsList[i].ItemAction;
                      if (ptrmenu->ItemsList[i].ItemType == 'M' &&
                          ptrmenu->ItemsList[i].SubMenu->ItemsNb == 0)
                        item++;
                    }
                  ptrmenu = ptrmenu->ItemsList[item].SubMenu;
                }
            }
        }

      if (ptrmenu)
        {
          item = (long int) data;
          if (item < ptrmenu->ItemsNb && ptrmenu->ItemsList != NULL)
            {
              for (i = 0; i <= item; i++)
                {
                  action = ptrmenu->ItemsList[i].ItemAction;
                  if (ptrmenu->ItemsList[i].ItemType == 'M' &&
                      ptrmenu->ItemsList[i].SubMenu->ItemsNb == 0)
                    item++;
                }
            }
          /*action = GetActionItem(frame, menu, (int)data); */
          if (action > 0 && action < MaxMenuAction)
            /* l'action existe et le menu est actif */
            if (MenuActionList[action].ActionActive[document])
                {
                  if (MenuActionList[action].Call_Action)
                    (*(Proc2)MenuActionList[action].Call_Action) ((void *)document, (void *)view);
                }
        }
    }
  
  // to be sure the focus is not lost
  TtaCheckLostFocus();
}
