/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * gelide
 * Copyright (C) 2008 Juan Ángel Moreno Fernández
 *
 * gelide is free software.
 *
 * You can redistribute it and/or modify it under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * gelide is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with gelide.  If not, see <http://www.gnu.org/licenses/>
 */

#include "cmprodatfile.hpp"

t_ReservedWord g_words[] = {
	// Tokens para los bloques del dat
	TK_CLRMAMEPRO,		"clrmamepro",
	TK_EMULATOR,		"emulator",
	TK_GAME,			"game",
	TK_ROM,				"rom",
	TK_DISK,			"disk",
	TK_BIOSSET,			"biosset",
	// Tokens en el bloque clrmamepro y comunes
	TK_NAME,			"name",
	TK_DESCRIPTION,		"description",
	TK_CATEGORY,		"category",
	TK_VERSION,			"version",
	TK_AUTHOR,			"author",
	// Tokens para el bloque game
	TK_YEAR,			"year",
	TK_MANUFACTURER,	"manufacturer",
	TK_CLONEOF,			"cloneof",
	// Tokens para los bloques rom y disk
	TK_SIZE,			"size",
	TK_CRC,				"crc",
	// Tokens para los parentesis
	TK_LPAR, 			"(",
	TK_RPAR, 			")"
};

CCMProDatFile::CCMProDatFile(void):CDatFile(){
	m_words = g_words;
}

CCMProDatFile::CCMProDatFile(const Glib::ustring& p_file):CDatFile(){
	m_words = g_words;

	this->load(p_file);
}

CCMProDatFile::~CCMProDatFile(void){
}


bool CCMProDatFile::load(const Glib::ustring& p_file){
	CToken l_token;
	CDatGame* l_game = NULL;

	if(p_file == "")
		return false;
	// Intentamos iniciar el tokenizador antes de hacer cambios en el dat
	if(m_tokenizer.initFromFile(p_file)){
		if(m_loaded)
			this->clear();
		// Iniciamos variables
		m_loaded = true;
		m_has_header = false;
		m_name = "";
		m_description = "";
		m_category = "";
		m_version = "";
		m_author = "";
		m_emulator_name = "";
		m_emulator_version = "";

		m_tokenizer.setReservedWords(m_words, NUM_TK_WORDS);
		while(m_tokenizer.hasMoreTokens()){
			m_tokenizer.nextToken(l_token);
			switch(l_token.getType()){
				case TK_CLRMAMEPRO:
					m_has_header = true;
					this->parseClrmameproBlock();
					break;
				case TK_EMULATOR:
					m_has_header = true;
					this->parseEmulatorBlock();
					break;
				case TK_GAME:
					l_game = this->parseGameBlock();
					if(l_game != NULL){
						m_games_list.push_back(l_game);
					}
					break;
				default:
					skipBlock();
			}
		}
		m_games = m_games_list.size();
		return true;
	}
	return false;
}

bool CCMProDatFile::loadHeader(const Glib::ustring& p_file){
	CToken l_token;

	if(p_file == "")
		return false;
	// Intentamos iniciar el tokenizador antes de hacer cambios en el dat
	if(m_tokenizer.initFromFile(p_file)){
		if(m_loaded)
			this->clear();
		// Iniciamos variables
		m_loaded = false;
		m_has_header = false;
		m_games = 0;
		m_name = "";
		m_description = "";
		m_category = "";
		m_version = "";
		m_author = "";
		m_emulator_name = "";
		m_emulator_version = "";

		m_tokenizer.setReservedWords(m_words, NUM_TK_WORDS);
		while(m_tokenizer.hasMoreTokens()){
			m_tokenizer.nextToken(l_token);
			switch(l_token.getType()){
				case TK_CLRMAMEPRO:
					m_has_header = true;
					this->parseClrmameproBlock();
					break;
				case TK_EMULATOR:
					m_has_header = true;
					this->parseEmulatorBlock();
					break;
				case TK_GAME:
					m_games++;
				default:
					skipBlock();
			}
		}
		return true;
	}
	return false;
}

void CCMProDatFile::parseClrmameproBlock(void){
	CToken l_token;

	// Leemos el paréntesis izquierdo
	m_tokenizer.nextToken(l_token);
	if(l_token.getType() == TK_LPAR){
		while(m_tokenizer.hasMoreTokens() && l_token.getType() != TK_RPAR){
			m_tokenizer.nextToken(l_token);
			switch(l_token.getType()){
				case TK_NAME:
					m_tokenizer.nextToken(l_token);
					m_name = l_token.getString();
					break;
				case TK_DESCRIPTION:
					m_tokenizer.nextToken(l_token);
					m_description = l_token.getString();
					break;
				case TK_CATEGORY:
					m_tokenizer.nextToken(l_token);
					m_category = l_token.getString();
					break;
				case TK_VERSION:
					m_tokenizer.nextToken(l_token);
					m_version = l_token.getString();
					break;
				case TK_AUTHOR:
					m_tokenizer.nextToken(l_token);
					m_author = l_token.getString();
					break;
				default:
					 // En la cabecera hay pares id valor saltamos lo que no
					 // nos interesa
					 //
					if(l_token.getType() != TK_RPAR){
						m_tokenizer.nextToken(l_token);
					}
			}
		}
	}
}

void CCMProDatFile::parseEmulatorBlock(void){
	CToken l_token;

	// Leemos el paréntesis izquierdo
	m_tokenizer.nextToken(l_token);
	if(l_token.getType() == TK_LPAR){
		while(m_tokenizer.hasMoreTokens() && l_token.getType() != TK_RPAR){
			m_tokenizer.nextToken(l_token);
			switch(l_token.getType()){
				case TK_NAME:
					m_tokenizer.nextToken(l_token);
					m_emulator_name = l_token.getString();
					break;
				case TK_VERSION:
					m_tokenizer.nextToken(l_token);
					m_emulator_version = l_token.getString();
					break;
				default:
					if(l_token.getType() != TK_RPAR){
						m_tokenizer.nextToken(l_token);
					}
			}
		}
	}
}

CDatGame* CCMProDatFile::parseGameBlock(void){
	CToken l_token;
	CDatGame * l_game;
	bool l_has_rom = false;

	// Leemos el paréntesis izquierdo
	m_tokenizer.nextToken(l_token);
	if(l_token.getType() == TK_LPAR){
		l_game = new CDatGame();
		while(m_tokenizer.hasMoreTokens() && (l_token.getType() != TK_RPAR)){
			m_tokenizer.nextToken(l_token);
			switch(l_token.getType()){
				case TK_NAME:
					m_tokenizer.nextToken(l_token);
					l_game->setName(l_token.getString());
					break;
				case TK_DESCRIPTION:
					m_tokenizer.nextToken(l_token);
					l_game->setDescription(l_token.getString());
					break;
				case TK_YEAR:
					m_tokenizer.nextToken(l_token);
					l_game->setYear(l_token.getValue());
					break;
				case TK_MANUFACTURER:
					m_tokenizer.nextToken(l_token);
					l_game->setManufacturer(l_token.getString());
					break;
				case TK_CLONEOF:
					m_tokenizer.nextToken(l_token);
					l_game->setCloneof(l_token.getString());
					break;
				// Solamente leemos la primera rom o disk
				case TK_ROM:
					if(!l_has_rom){
						l_game->setRom(parseRomBlock(DATROM_ROM));
						l_has_rom = true;
					}
					else{
						skipBlock();
					}
					break;
				case TK_DISK:
					if(!l_has_rom){
						l_game->setRom(parseRomBlock(DATROM_DISK));
						l_has_rom = true;
					}
					else{
						skipBlock();
					}
					break;
				case TK_BIOSSET:
					skipBlock();
					break;
				default:
					if(l_token.getType() != TK_RPAR){
						m_tokenizer.nextToken(l_token);
					}
			}
		}
		return l_game;
	}
	return NULL;
}

CDatRom* CCMProDatFile::parseRomBlock(unsigned int p_type){
	CToken l_token;
	CDatRom* l_rom;

	// Leemos el paréntesis izquierdo
	m_tokenizer.nextToken(l_token);
	if(l_token.getType() == TK_LPAR){
		l_rom = new CDatRom(p_type);
		while(m_tokenizer.hasMoreTokens() && l_token.getType() != TK_RPAR){
			m_tokenizer.nextToken(l_token);
			switch(l_token.getType()){
				case TK_NAME:
					m_tokenizer.nextToken(l_token);
					l_rom->setName(l_token.getString());
					break;
				case TK_SIZE:
					m_tokenizer.nextToken(l_token);
					l_rom->setSize(l_token.getValue());
					break;
				case TK_CRC:
					m_tokenizer.nextToken(l_token);
					l_rom->setCrc(l_token.getString());
					break;
				default:
					if(l_token.getType() != TK_RPAR){
						m_tokenizer.nextToken(l_token);
					}
			}
		}
		return l_rom;
	}
	return NULL;
}

void CCMProDatFile::skipBlock(void){
	CToken l_token;
	int l_cont = 1;

	// Leemos el paréntesis izquierdo
	m_tokenizer.nextToken(l_token);
	if(l_token.getType() == TK_LPAR){
		while(m_tokenizer.hasMoreTokens() && (l_cont > 0)){
			m_tokenizer.nextToken(l_token);
			if(l_token.getType() == TK_RPAR){
				l_cont--;
			}
			if(l_token.getType() == TK_LPAR){
				l_cont++;
			}
		}
	}
}


