/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * gelide
 * Copyright (C) 2008 Juan Ángel Moreno Fernández
 *
 * gelide is free software.
 *
 * You can redistribute it and/or modify it under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * gelide is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with gelide.  If not, see <http://www.gnu.org/licenses/>
 */

#include <fstream>
#include <stdlib.h>
#include <string.h>
#include "tokenizer.hpp"

CToken::CToken(const int p_type, const unsigned int p_value, char* p_string){
	m_type = p_type;
	m_value = p_value;
	m_string = p_string;
}

CToken::~CToken(){
}

int CToken::getType(void){
	return m_type;
}

unsigned int CToken::getValue(void){
	return m_value;
}

char* CToken::getString(void){
	return m_string;
}

void CToken::setType(const int p_type){
	m_type = p_type;
}

void CToken::setValue(const unsigned int p_value){
	m_value = p_value;
}

void CToken::setString(char* p_string){
	m_string = p_string;
}
/*---------------------------------------------------------------------------*/

CTokenizer::CTokenizer(void){
	m_delimiters = NULL;
	m_delimiters_count = 0;
	m_words = NULL;
	m_words_count = 0;
	m_buff = NULL;
	m_buff_pos = NULL;
	m_init = false;
}

CTokenizer::~CTokenizer(){
	if(m_buff)
		delete[] m_buff;
}

void CTokenizer::setReservedWords(t_ReservedWord p_words[],	const unsigned int p_count){
	m_words = p_words;
	m_words_count = p_count;
}

void CTokenizer::setDelimiters(char* p_delimiters, const unsigned int p_count){
	m_delimiters = p_delimiters;
	m_delimiters_count = p_count;
}

bool CTokenizer::initFromFile(const Glib::ustring& p_file){
	std::ifstream m_file;
	int l_fsize;
	int l_readed;

	// Nos aseguramos de liberar la memoria y hacer nulos los punteros
	if(m_buff)
		delete[] m_buff;
	m_buff = NULL;
	m_buff_pos = NULL;
	m_init = false;

	m_file.open(p_file.data());
	// Comprobamos si la apertura fue correcta
	if(!m_file.good()){
		m_file.close();
		return false;
	}
	// Obtenemos el tamaño del fichero
	m_file.seekg (0, std::ios::end);
	l_fsize = m_file.tellg();
	m_file.seekg (0, std::ios::beg);
	// Reservamos la memoria necesaria
	m_buff = new char[l_fsize + 1];
	if(!m_buff){
		m_file.close();
		return false;
	}
	// Cargamos el contenido completo del fichero en memoria
	m_file.read (m_buff, l_fsize);
	l_readed = m_file.gcount();
	m_file.close();
	// Reiniciamos el puntero de lectura
	m_buff_pos = m_buff;
	m_buff[l_readed] = '\0';
	m_init = true;
	findNextToken();
	return true;
}

bool CTokenizer::initFromString(const Glib::ustring& p_str){
	std::ifstream m_file;
	int l_ssize;
	int l_readed;

	// Nos aseguramos de liberar la memoria y hacer nulos los punteros
	if(m_buff)
		delete[] m_buff;
	m_buff = NULL;
	m_buff_pos = NULL;
	m_init = false;

	// Obtenemos el tamaño de la cadena
	l_ssize = p_str.size();
	// Reservamos la memoria necesaria
	m_buff = new char[l_ssize + 1];
	if(!m_buff){
		m_file.close();
		return false;
	}
	// Cargamos el contenido completo de la cadena en el buffer
	l_readed = p_str.copy(m_buff, l_ssize, 0);
	// Reiniciamos el puntero de lectura
	m_buff_pos = m_buff;
	m_buff[l_readed] = '\0';
	m_init = true;
	findNextToken();
	return true;
}

void CTokenizer::nextToken(CToken& p_token){
	if(!m_init)
		return;
	// Valor de control para token
	p_token.setType(TK_NOTK);
	p_token.setValue(-1);
	p_token.setString(NULL);
	while(p_token.getType() == TK_NOTK){
		switch(*m_buff_pos){
		// Fin de fichero
		case '\0':
			p_token.setType(TK_EOF);
			p_token.setValue(0);
			// Hacemos que apunte a "/0", para evitar segfaults
			p_token.setString(m_buff_pos);
			break;
		// Comienzo de una cadena de caracteres
		case '"':
			// Obtenemos la cadena
			p_token.setType(TK_STR);
			p_token.setValue(0);
			p_token.setString(this->getString());
			break;
		default:
			// Llegados a este punto, como minimo será un identificador
			p_token.setType(TK_ID);
			p_token.setString(this->getIdentifier());
			// Comprobamos si se trata de un número
			if(checkNum(p_token))
				break;
			// Chequeamos palabras reservadas
			checkReservedWords(p_token);
		}
		// Posicionamos al comienzo del siguiente token
		findNextToken();

	}
}

bool CTokenizer::hasMoreTokens(void){
	return *m_buff_pos != '\0';
}

bool CTokenizer::findNextToken(void){
	while(isDelimiter())
		m_buff_pos++;
	if(*m_buff_pos == '\0')
		return false;
	else
		return true;
}

bool CTokenizer::isDelimiter(void){
	unsigned int l_ind = 0;

	if( (*m_buff_pos == ' ') ||		// Espacios
		(*m_buff_pos == '\t') ||	// Tabuladores
		(*m_buff_pos == '\r') ||	// Retorno de carro
		(*m_buff_pos == '\n'))		// Salto de línea*/
		return true;

	if(m_delimiters){
		// Buscamos nuestra cadena hasta agotar todos los separadores
		while((l_ind < m_delimiters_count) && (*m_buff_pos != m_delimiters[l_ind]))
			l_ind++;
		// Comprobamos si se encontró alguno
		if(l_ind < m_delimiters_count)
			return true;
	}
	return false;
}

void CTokenizer::checkReservedWords(CToken& p_token){
	char* l_string;
	unsigned int l_ind = 0;

	// Cargamos la cadena para evitar multiples llamadas a la función
	l_string = p_token.getString();
	if(m_words){
		// Buscamos la cadena hasta agotar todas las palabras reservadas
		while((l_ind < m_words_count) && (strcmp(m_words[l_ind].m_name,	l_string) != 0))
			l_ind++;
		// Comprobamos si se encontró alguna
		if(l_ind < m_words_count)
			p_token.setType(m_words[l_ind].m_type);
	}
}

bool CTokenizer::checkNum(CToken& p_token){
	char* l_string = NULL;
	char* l_ind = NULL;

	l_string = p_token.getString();
	l_ind = l_string;
	// Avanzamos mientras tengamos números
	while(isdigit((int) *l_ind))
		l_ind++;
	// Si no se avanza, no tenemos ningún número
	if(l_ind == l_string)
		return false;
	if(*l_ind == '\0'){
		p_token.setType(TK_NUM);
		p_token.setValue(atoi(l_string));
		return true;
	}
	return false;
}

char* CTokenizer::getString(void){
	char* l_string;

	// Saltamos las comillas iniciales y colocamos el inicio de cadena
	m_buff_pos++;
	l_string = m_buff_pos;

	// Ignoramos caracteres escapados iniciales
	while(*m_buff_pos == '\\')
		m_buff_pos += 2;

	// Consideramos que la cadena finaliza en " o final de línea
	while( (*m_buff_pos != '"') && (*m_buff_pos != '\0') ){
		m_buff_pos++;
		// Ignoramos caracteres escapados
		while(*m_buff_pos == '\\')
			m_buff_pos += 2;

	}

	// Si no llegamos a eof, marcamos el fin de la cadena y avanzamos
	// para evitar un eof erroneo en la siguiente lectura
	if(*m_buff_pos == '"'){
		*m_buff_pos = '\0';
		m_buff_pos++;
	}

	return l_string;
}

char* CTokenizer::getIdentifier(void){
	char* l_string;
	// Asignamos el puntero de la cadena en el comienzo del identificador
	l_string = m_buff_pos;
	while((!isDelimiter()) && (*m_buff_pos != '\0'))
		m_buff_pos++;

	// Si no llegamos a eof, marcamos el fin de la cadena y avanzamos
	// para evitar un eof erroneo en la siguiente lectura
	if(*m_buff_pos != '\0'){
		*m_buff_pos = '\0';
		m_buff_pos++;
	}
	return l_string;
}

