/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * gelide
 * Copyright (C) 2008 Juan Ángel Moreno Fernández
 *
 * gelide is free software.
 *
 * You can redistribute it and/or modify it under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * gelide is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with gelide.  If not, see <http://www.gnu.org/licenses/>
 */

#ifndef _TOKENIZER_HPP_
#define _TOKENIZER_HPP_

#include "gelide.hpp"
#include <glibmm/ustring.h>

// Definición de constantes internas
#define TOKENIZER_RES_WORD_SIZE	25		// Tamaño máximo de las palabras
										// reservadas

// Tokens genericos devueltos por el tokenizador. Se definen negativos, para
// utilizar los positivos con las palabras reservadas
#define TK_NOTK			-1		// Token Notoken
#define TK_NUM			-2		// Token número
#define TK_ID			-3		// Token identificador
#define TK_STR			-4		// Token cadena
#define	TK_EOF			-5		// Token final de fichero
#define TK_UNK			-6		// Token desconocido

// Tipo para especificar las palabras reservadas
typedef struct
{
	int m_type;								// Tipo de token
	char m_name[TOKENIZER_RES_WORD_SIZE];	// Identificador de la palabra
}t_ReservedWord;

/*------------------------------------------------------
	Encapsula a un bloque representativo de un conjunto
	especial de bloques de texto.
--------------------------------------------------------*/
class CToken
{
public:
	// Constructor y destructor
	CToken(const int p_type = 0, const unsigned int p_value = -1,
		   char* p_string = NULL);
	~CToken();

	// Establece el tipo del token
	void setType(const int p_type = 0);

	// Establece el valor numérico del token
	void setValue(const unsigned int p_value = -1);

	// Establece el valor textual del token
	void setString(char* p_string = NULL);

	// Devuelve el tipo de token
	int getType(void);

	//Devuelve el valor numérico del token
	unsigned int getValue(void);

	//Devuelve el valor textual del token
	char* getString(void);

private:
	int m_type;					// Representa el tipo de token
	unsigned int m_value;		// Representa el valor numérico del token
	char* m_string;		// Representa el valor textual del token
};

/*------------------------------------------------------
	Clase encargada del particionamiento de de un fichero
	de texto en bloques fundamentales (tokens) de un
	conjunto dado como parámetro y teniendo en cuenta unos
	delimitadores determinados.
--------------------------------------------------------*/
class CTokenizer
{
public:
	// Constructor y destructor
	CTokenizer(void);
	~CTokenizer();

	// Establece las palabras reservadas que identificará el tokenizador
	void setReservedWords(t_ReservedWord p_reserved[], const unsigned int p_count);

	// Establece los delimitadores adicionales, por defecto el
	// tokenizador interpreta los siguientes ' ', """, '\t', '\r', '\n'
	void setDelimiters(char* p_delimiters, const unsigned int p_count);

	// Inicializa el tokenizador sobre un fichero determinado
	bool initFromFile(const Glib::ustring& p_file);

	// Inicializa el tokenizador sobre una cadena determinada
	bool initFromString(const Glib::ustring& p_str);

	// Devuelve el siguiente token del fichero
	void nextToken(CToken& p_token);

	// True si quedan tokens por leer en el fichero
	bool hasMoreTokens(void);

private:
	// Establece el puntero de lectura del buffer sobre el siguiente token
	bool findNextToken(void);

	// Búsca en las palabras reservadas un token y lo rellena si lo encuentra
	void checkReservedWords(CToken& p_token);

	// Comprueba si el puntero de lectura del buffer está sobre un delimitador
	bool isDelimiter(void);

	// Comprueba si token dado dado es un número
	bool checkNum(CToken& p_token);

	// Obtiene una cadena ".." del buffer del tokenizador
	char* getString(void);

	// Obtiene un identificador del buffer del tokenizador
	char* getIdentifier(void);

	bool m_init;						// Indica si el tokenizador se inicio
	char* m_buff;						// Buffer del tokenizador
	char* m_buff_pos;					// Puntero de lectura del buffer
	char* m_delimiters;					// Lista de delimitadores adicionales
	unsigned int m_delimiters_count;	// Número de delimitadores adicionales
	t_ReservedWord* m_words;			// Lista de palabras reservadas
	unsigned int m_words_count;			// Número de palabras reservadas
};

#endif // _TOKENIZER_HPP_
