/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * gelide
 * Copyright (C) 2008 Juan Ángel Moreno Fernández
 *
 * gelide is free software.
 *
 * You can redistribute it and/or modify it under the terms of the
 * GNU General Public License, as published by the Free Software
 * Foundation; either version 3 of the License, or (at your option)
 * any later version.
 *
 * gelide is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with gelide.  If not, see <http://www.gnu.org/licenses/>
 */

//#include <fstream>
#include <algorithm>
#include <sstream>
#include <gtkmm/main.h>
#include <gtkmm/stock.h>
#include <gtkmm/messagedialog.h>
#include "dialog_gamelist_generator.hpp"
#include "../core/cmprodatfile.hpp"
//#include "../gelide-utils.hpp"


CDialogGamelistGenerator::CDialogGamelistGenerator(void):
	Gtk::Dialog(),
	m_button_close(Gtk::Stock::CLOSE)
{
	// Obtenemos las instancias
	m_config = CConfig::getInstance();
	m_smanager = CSystemManager::getInstance();

	m_system = 0;

	// Configuración visual del dialogo
	this->set_title( _("Generating games list") );
	this->set_resizable(false);
	this->set_size_request(400, -1);
	this->set_modal(true);
	this->set_border_width(5);

	// Configuración de las etiquetas
	m_label_description.set_markup((Glib::ustring) "<b><big>" + _("Generating games list") + "</big></b>");
	m_label_description.set_alignment(0,0.5);
	m_label_description.set_line_wrap(true);
	//"Espere un momento mientras se analiza el fichero dat y el directorio de roms del sistema..."
	m_label_info.set_label(_("Please wait while system's dat file and roms directory are processed..."));
	m_label_info.set_alignment(0,0.5);
	m_label_info.set_line_wrap(true);
	m_label_status.set_label(_("Initializing."));
	m_label_status.set_alignment(0,0.5);
	m_label_status.set_line_wrap(false);
	m_label_status.set_ellipsize(Pango::ELLIPSIZE_END);
	m_label_status.set_padding(0, 5);

	// Configuración de la barra de progreso
	m_progressbar.set_pulse_step(0.005);

	// Configuración del cuerpo principal
	m_vbox_body.pack_start(m_label_description, true, true);
	m_vbox_body.pack_start(m_label_info, true, true);
	m_vbox_body.pack_start(m_progressbar, true, true);
	m_vbox_body.pack_start(m_label_status, true, true);
	m_vbox_body.set_spacing(5);

	// Configuración del boton close
	m_button_close.signal_clicked().connect( sigc::mem_fun(*this,
		&CDialogGamelistGenerator::onCloseClicked) );

	// Añadimos el notebook y los botones al dialogo
	this->get_vbox()->set_spacing(5);
	this->get_vbox()->pack_start(m_vbox_body, false, false);
	this->get_action_area()->pack_start(m_button_close);
	this->get_action_area()->set_layout(Gtk::BUTTONBOX_END);

	// Establecemos el diálogo por defecto en el centro del padre
	this->set_position(Gtk::WIN_POS_CENTER_ON_PARENT);
	// Cargamos la configuración
	loadConfig();

	// Mostramos todos los widgets
	this->show_all_children();
}

CDialogGamelistGenerator::~CDialogGamelistGenerator(){
	saveConfig();
}

void CDialogGamelistGenerator::setSystem(CSystem* p_system){
	m_system = p_system;
}

int CDialogGamelistGenerator::run(void){
	Gtk::MessageDialog l_dialog(*this, _("Are you sure you want to create a new list?"), false, Gtk::MESSAGE_WARNING, Gtk::BUTTONS_OK_CANCEL);

	assert(m_system);

	//"Si continua se generará un nuevo listado de juegos\npara el sistema, perdiendo las propiedades\nestablecidas en los juegos del listado actual."
	l_dialog.set_secondary_text( _("If you continue, a new games list will be created for the current system. Current games's properties will be lost"));
	if(l_dialog.run() == Gtk::RESPONSE_CANCEL){
		return Gtk::RESPONSE_CANCEL;
	}
	l_dialog.hide();
	m_button_close.set_sensitive(false);
	this->show();
	// Esperamos a que se oculte el dialog
	while (Gtk::Main::events_pending())
		Gtk::Main::iteration();
	generateGameList();
	m_button_close.set_sensitive(true);
	return Gtk::Dialog::run();
}

int CDialogGamelistGenerator::run(CSystem* p_system){
	setSystem(p_system);
	return this->run();
}

void CDialogGamelistGenerator::loadConfig(void){
	int l_x, l_y;

	// Obtenemos la posición almacenada
	m_config->getGameListGeneratorDialogPos(l_x, l_y);
	if((l_x!= -1) && (l_y != -1))
		this->move(l_x, l_y);
}

void CDialogGamelistGenerator::saveConfig(void){
	int l_x, l_y;

	// Guardamos la posición del dialogo
	this->get_position(l_x, l_y);
	m_config->setGameListGeneratorDialogPos(l_x, l_y);
}

bool CDialogGamelistGenerator::qSortCompare(CGame* a, CGame* b){
	return a->getName() < b->getName();
}

int CDialogGamelistGenerator::bSearch(const std::vector<CGame*>& p_list, const Glib::ustring& p_key)
{
	int l_top = p_list.size()-1;
	int l_bottom = 0;
	int l_center;
	int l_compare;
	// Pasamos la clave a minusculas antes de comenzar
	Glib::ustring l_key = p_key.lowercase();

	while (l_bottom <= l_top){
		l_center = (l_top + l_bottom)/2;
		l_compare = l_key.compare(p_list[l_center]->getName().lowercase());
		if (l_compare == 0)
			return l_center;
		else{
			if (l_compare < 0)
				l_top = l_center - 1;
			else
				l_bottom = l_center + 1;
			}
	}
	return -1;
}

bool CDialogGamelistGenerator::generateGameList(void){
	CGame* l_game;
	// Datfile y un iterador sobre la lista de juegos del dat
	CCMProDatFile l_dat;
	std::list<CDatGame*>::iterator l_dat_iter;
	// Listado de juegos final y un iterador sobre ella
	std::vector<CGame*> l_dat_games;
	std::vector<CGame*> l_games;
	std::vector<CGame*>::iterator l_games_iter;
	// Iterador de directorio para acceder a los ficheros
	Glib::Dir::iterator l_dir_iter;
	// Path completo del fichero y nombre del juego
	Glib::ustring l_file;
	Glib::ustring l_name;
	// Indice para busqueda de juegos
	int l_find;
	std::stringstream l_str;

	GELIDE_DEBUG("Generating list for system " << m_system->getName().data() << "...");
	// Cargamos el dat
	m_progressbar.set_fraction(0.05);
	m_label_status.set_label(_("Loading dat file..."));
	while (Gtk::Main::events_pending())
		Gtk::Main::iteration();

	l_dat.load(m_system->getDatFile());
	// Establecemos la info del dat en el sistema
	m_system->setDatName(l_dat.getName());
	m_system->setDatDescription(l_dat.getDescription());
	m_system->setDatCategory(l_dat.getCategory());
	m_system->setDatVersion(l_dat.getVersion());
	m_system->setDatAuthor(l_dat.getAuthor());
	m_system->setDatGames(l_dat.getGames());

	// Generamos el listado a partir del dat
	m_progressbar.set_fraction(0.10);
	m_label_status.set_label(_("Generating initial list..."));
	while (Gtk::Main::events_pending())
		Gtk::Main::iteration();
	for(l_dat_iter = l_dat.getList().begin();
		l_dat_iter != l_dat.getList().end(); l_dat_iter++){
		l_game = new CGame();
		//l_game->setSystemId(getId());
		l_game->setName((*l_dat_iter)->getName());
		l_game->setYear((*l_dat_iter)->getYear());
		l_game->setDescription((*l_dat_iter)->getDescription());
		l_game->setManufacturer((*l_dat_iter)->getManufacturer());
		l_game->setRomCrc((*l_dat_iter)->getRom()->getCrc());
		// Marcamos el juego como no desconocido
		l_game->setUnknown(false);
		l_dat_games.push_back(l_game);
	}

	m_progressbar.set_fraction(0.15);
	m_label_status.set_label(_("Ordering initial list..."));
	while (Gtk::Main::events_pending())
		Gtk::Main::iteration();
	std::sort(l_dat_games.begin(), l_dat_games.end(), CDialogGamelistGenerator::qSortCompare);

	// Cargamos los ficheros
	m_progressbar.set_fraction(0.20);
	m_label_status.set_label(_("Checking files..."));
	while (Gtk::Main::events_pending())
		Gtk::Main::iteration();
	try{
		Glib::Dir l_dir(m_system->getRomsDir());
		for (l_dir_iter = l_dir.begin(); l_dir_iter != l_dir.end(); l_dir_iter++){
			l_file = m_system->getRomsDir() + "/" + (*l_dir_iter);
			if(!Glib::file_test(l_file.c_str(), Glib::FILE_TEST_IS_DIR)){
				l_name = (*l_dir_iter);
				// TODO: Añadir filtrado de extensiones conocidas para el sistema
				l_name = l_name.rfind('.') == std::string::npos ? l_name : l_name.substr(0, l_name.rfind('.'));
				m_progressbar.pulse();
				m_label_status.set_label(_("Checking: ") + l_name);
				while (Gtk::Main::events_pending())
					Gtk::Main::iteration();
				l_find = bSearch(l_dat_games, l_name);
				if(l_find >= 0){
					l_dat_games[l_find]->setWorking(true);
					l_dat_games[l_find]->setAvailable(true);
					l_dat_games[l_find]->setPath(l_file);
				}
				else{
					//CHECKME: ¿que pasas con la puntuación?
					l_game = new CGame();
					//l_game->setSystemId(getId());
					l_game->setName(l_name);
					l_game->setDescription(l_name);
					l_game->setPath(l_file);
					l_game->setWorking(true);
					l_game->setAvailable(true);
					l_game->setUnknown(true);
					l_games.push_back(l_game);
				}
			}
		}
	}
	catch (Glib::Error& l_exception)
	{
		GELIDE_WARNING("Reading directory (" << l_exception.what().c_str() << "...");
	}

	// Guardamos el listado y liberamos la memoria
	m_progressbar.set_fraction(0.80);
	m_label_status.set_label(_("Saving list..."));
	while (Gtk::Main::events_pending())
		Gtk::Main::iteration();

	// Limpiamos la lista de juegos
	GELIDE_DEBUG("System " << m_system->getName().data() << ": Loading games...");
	// Vaciamos la lista de juegos
	m_system->getGameList()->clear();
	// Añadimos los juegos del dat
	for (l_games_iter = l_dat_games.begin(); l_games_iter != l_dat_games.end(); l_games_iter++){
		m_system->addGame(*(*l_games_iter));
		// Eliminamos el juego
		delete *l_games_iter;
	}
	l_dat_games.clear();
	// Añadimos los juegos que no son del dat
	for (l_games_iter = l_games.begin(); l_games_iter != l_games.end(); l_games_iter++){
		m_system->addGame(*(*l_games_iter));
		// Eliminamos el juego
		delete *l_games_iter;
	}
	l_games.clear();

	m_progressbar.set_fraction(1);
	//l_str << _("End. ") << m_system->getGames() << _(" added games.");
	l_str << Glib::ustring::compose(_("End. %1 added games."), m_system->getGames());
	m_label_status.set_label(l_str.str());
	while (Gtk::Main::events_pending())
		Gtk::Main::iteration();
	return true;
}

void CDialogGamelistGenerator::onCloseClicked(void){
	this->response(Gtk::RESPONSE_CLOSE);
}

bool CDialogGamelistGenerator::on_delete_event(GdkEventAny* p_event){
	return ! m_button_close.is_sensitive();
}


