%expect 1

%{
#include <malloc.h>
#include <stdlib.h>
#include <regex.h>
#include <string.h>

#include "command.h"
#include "lirccd.h"

#define YYERROR_VERBOSE

extern int yycurline;
extern unsigned char *yycurfile;

void yyerror(const char *err);

static struct instruction *pushnull(void)
{
    struct value *val;
    struct instruction *new;
    
    val = newval();
    new = inspush(val);
    relval(val);
    return(new);
}

static struct instruction *pusharraystart(void)
{
    struct value *val;
    struct instruction *new;
    
    (val = newval())->type = VAL_ARRAYSTART;
    new = inspush(val);
    relval(val);
    return(new);
}

static struct instruction *pushint(int num)
{
    struct value *val;
    struct instruction *new;
    
    val = valmakeint(newval(), num);
    new = inspush(val);
    relval(val);
    return(new);
}

static struct instruction *pushfloat(double num)
{
    struct value *val;
    struct instruction *new;
    
    val = newval();
    val->type = VAL_FLOAT;
    val->val.flnum = num;
    new = inspush(val);
    relval(val);
    return(new);
}

static struct instruction *pushstr(unsigned char *str)
{
    struct value *val;
    struct instruction *new;
    
    val = valmakestr(newval(), str);
    new = inspush(val);
    relval(val);
    return(new);
}

%}

%union {
    int num;
    unsigned char *str;
    struct instruction *ins;
    struct binding *binding;
}

%token <str> INT
%token <str> FLOAT
%token <str> STRING
%token <str> ID
%token FUNCTION
%token RETURN
%token VAR
%token BIND
%token SUBTREE
%token RECURSIVE
%token FINAL
%token GLOBAL
%token IF
%token ELSE
%token FOR
%token WHILE
%token NULLTOK
%token T_INT
%token T_FLOAT
%token T_STRING
%token T_ARRAY
%right '=' AUTOADD AUTOSUB
%left LOGOR
%left LOGAND
%left EQ
%left NE
%left LT
%left GT
%left LE
%left GE
%left '+' '-'
%left '*' '/'
%nonassoc AUTOINC AUTODEC '!'
%nonassoc '(' ')' '[' ']'

%type <num> bindflags
%type <ins> expr arglist invarglist call makearray
%type <ins> stmt multistmt ifstmt whilestmt forstmt returnstmt varstmt
%type <ins> lvarlist declarglist
%type <binding> binding bindings

%%

file:  | file directive;

bindflags: 	{$$ = 0;}
		| bindflags FINAL {$$ = $1 | BIND_STOP;}
		| bindflags RECURSIVE {$$ = $1 | BIND_RECURSIVE;}
;

binding:	BIND bindflags STRING stmt
		{
		    unsigned char *p;
		    if((p = strchr($3, ':')) == NULL)
		    {
			$$ = newbinding(".*", $3, $4);
		    } else {
			*(p++) = 0;
			$$ = newbinding($3, p, $4);
		    }
		    $$->flags = $2;
		    free($3);
		}
		| BIND bindflags STRING SUBTREE '{' bindings '}'
		{
		    unsigned char *p;
		    if((p = strchr($3, ':')) == NULL)
		    {
			$$ = newbinding(".*", $3, NULL);
		    } else {
			*(p++) = 0;
			$$ = newbinding($3, p, NULL);
		    }
		    addbinding(&$$->child, $6);
		    free($3);
		}
		| BIND bindflags STRING SUBTREE '{' stmt bindings '}'
		{
		    unsigned char *p;
		    if((p = strchr($3, ':')) == NULL)
		    {
			$$ = newbinding(".*", $3, $6);
		    } else {
			*(p++) = 0;
			$$ = newbinding($3, p, $6);
		    }
		    addbinding(&$$->child, $7);
		    free($3);
		}
;

bindings:	binding {$$ = $1;}
		| bindings binding
		{
		    addbinding(&$1, $2);
		    $$ = $1;
		}
;

globaldecl:	ID
		{
		    struct value *val;
		    val = newval();
		    newsym($1, val);
		    relval(val);
		    free($1);
		}
		| ID '=' expr
		{
		    struct value *val;
		    run($3);
		    freecode($3);
		    val = pop();
		    checkstack();
		    newsym($1, val);
		    relval(val);
		    free($1);
		}
;

varlist:	globaldecl
		| varlist ',' globaldecl
;

declarglist:	ID
		{
		    $$ = insconcat(
			pushstr($1),
			insmake("MAKEVAR"),
			insmake("SWAP"),
			insmake("STORE"),
			insmake("DROP"),
			NULL);
		    free($1);
		}
		| declarglist ',' ID
		{
		    $$ = insconcat(
			pushstr($3),
			insmake("MAKEVAR"),
			insmake("SWAP"),
			insmake("STORE"),
			insmake("DROP"),
			$1,
			NULL);
		    free($3);
		}
;

functiondecl:	FUNCTION ID '(' ')' stmt
		{
		    newfunc($2, insconcat(
				$5,
				pushnull(),
				insmake("RETURN"),
				NULL));
		    free($2);
		}
		| FUNCTION ID '(' declarglist ')' stmt
		{
		    newfunc($2, insconcat(
				insmake("BEGINSCOPE"),
				$4,
				$6,
				pushnull(),
				insmake("ENDSCOPE"),
				insmake("RETURN"),
				NULL));
		    free($2);
		}
;

directive:	functiondecl
		| GLOBAL varlist ';'
		| binding	{addbinding(&bindings, $1);}
;

stmt:	';' {$$ = insmake("NOOP");}
	| expr ';' {$$ = insconcat($1, insmake("DROP"), NULL);}
	| ifstmt {$$ = $1;}
	| whilestmt {$$ = $1;}
	| forstmt {$$ = $1;}
	| returnstmt {$$ = $1;}
	| varstmt {$$ = $1;}
	| '{' multistmt '}' {$$ = insconcat(insmake("BEGINSCOPE"), $2, insmake("ENDSCOPE"), NULL);}
;

multistmt:	stmt {$$ = $1;}
		| multistmt stmt {$$ = insconcat($1, $2, NULL);}
;

ifstmt: IF '(' expr ')' stmt
	{struct instruction *ins;
	ins = insmake("NOOP");
	$$ = insconcat(
	    $3,
	    insmake("NOT"),
	    pushint(label(ins)),
	    insmake("SWAP"),
	    insmake("CJUMP"),
	    $5,
	    ins,
	    NULL);
	}
	| IF '(' expr ')' stmt ELSE stmt
	{struct instruction *ins;
	ins = insmake("NOOP");
	$$ = insconcat(
	    $3,
	    insmake("NOT"),
	    pushint(label($7)),
	    insmake("SWAP"),
	    insmake("CJUMP"),
	    $5,
	    pushint(label(ins)),
	    insmake("JUMP"),
	    $7,
	    ins,
	    NULL);
	}
;

whilestmt: WHILE '(' expr ')' stmt
	{struct instruction *ins;
	int lab;
	ins = insmake("NOOP");
	lab = label($3);
	$$ = insconcat(
	    $3,
	    insmake("NOT"),
	    pushint(label(ins)),
	    insmake("SWAP"),
	    insmake("CJUMP"),
	    $5,
	    pushint(lab),
	    insmake("JUMP"),
	    ins,
	    NULL);
	};

forstmt: FOR '(' expr ';' expr ';' expr ')' stmt
	{struct instruction *ins;
	int lab;
	ins = insmake("NOOP");
	lab = label($5);
	$$ = insconcat(
	    $3,
	    insmake("DROP"),
	    $5,
	    insmake("NOT"),
	    pushint(label(ins)),
	    insmake("SWAP"),
	    insmake("CJUMP"),
	    $9,
	    $7,
	    insmake("DROP"),
	    pushint(lab),
	    insmake("JUMP"),
	    ins,
	    NULL);
	};

returnstmt: RETURN '(' expr ')' ';'
	{
	    $$ = insconcat(
		$3,
		insmake("RETURN"),
		NULL);
	};

lvarlist:	ID {$$ = insconcat(pushstr($1), insmake("MAKEVAR"), insmake("DROP"), NULL); free($1);}
		| lvarlist ',' ID {$$ = insconcat(pushstr($3), insmake("MAKEVAR"), insmake("DROP"), NULL); free($3);}
;

varstmt: VAR lvarlist ';' {$$ = $2;};

arglist: 	expr {$$ = $1;}
		| arglist ',' expr {$$ = insconcat($1, $3, NULL);}
;

invarglist: 	expr {$$ = $1;}
		| invarglist ',' expr {$$ = insconcat($3, $1, NULL);}
;

call:	ID '(' ')'
	{
	    if(findcommand($1) != NULL)
	    {
		$$ = insmake($1);
	    } else {
		$$ = insconcat(
		    pushstr($1),
		    insmake("CALL"),
		    NULL);
	    }
	    free($1);
	}
	| ID '(' arglist ')'
	{
	    if(findcommand($1) != NULL)
	    {
		$$ = insconcat($3, insmake($1), NULL);
	    } else {
		$$ = insconcat(
		    $3,
		    pushstr($1),
		    insmake("CALL"),
		    NULL);
	    }
	    free($1);
	}
;

makearray:	T_ARRAY '(' ')'
		{
		    $$ = insconcat(
			pusharraystart(),
			insmake("ARRAY"),
			NULL);
		}
		| T_ARRAY '(' invarglist ')'
		{
		    $$ = insconcat(
			pusharraystart(),
			$3,
			insmake("ARRAY"),
			NULL);
		}
;

expr:	NULLTOK {$$ = pushnull();}
	| INT {$$ = pushint(atoi($1)); free($1);}
	| FLOAT {$$ = pushfloat(atof($1)); free($1);}
	| STRING {$$ = pushstr($1); free($1);}
	| ID {$$ = insconcat(pushstr($1), insmake("LOAD"), NULL); free($1);}
	| makearray {$$ = $1;}
	| call {$$ = $1;}
	| expr '+' expr {$$ = insconcat($1, $3, insmake("ADD"), NULL);}
	| expr '-' expr {$$ = insconcat($1, $3, insmake("SUB"), NULL);}
	| expr '*' expr {$$ = insconcat($1, $3, insmake("MUL"), NULL);}
	| expr '/' expr {$$ = insconcat($1, $3, insmake("DIV"), NULL);}
	| expr '=' expr {$$ = insconcat($1, $3, insmake("STORE"), NULL);}
	| expr AUTOADD expr {$$ = insconcat($1, insmake("DUP"), $3, insmake("ADD"), insmake("STORE"), NULL);}
	| expr AUTOSUB expr {$$ = insconcat($1, insmake("DUP"), $3, insmake("SUB"), insmake("STORE"), NULL);}
	| expr EQ expr {$$ = insconcat($1, $3, insmake("EQ"), NULL);}
	| expr NE expr {$$ = insconcat($1, $3, insmake("NE"), NULL);}
	| expr GT expr {$$ = insconcat($1, $3, insmake("GT"), NULL);}
	| expr LT expr {$$ = insconcat($1, $3, insmake("LT"), NULL);}
	| expr GE expr {$$ = insconcat($1, $3, insmake("GE"), NULL);}
	| expr LE expr {$$ = insconcat($1, $3, insmake("LE"), NULL);}
	| '!' expr {$$ = insconcat($2, insmake("NOT"), NULL);}
	| expr LOGOR expr
        {
	    struct instruction *ins;
	    ins = insmake("NOOP");
	    $$ = insconcat(
		$1,
		insmake("DUP"),
		pushint(label(ins)),
		insmake("SWAP"),
		insmake("CJUMP"),
		insmake("DROP"),
		$3,
		ins,
		NULL);
	}
	| expr LOGAND expr
        {
	    struct instruction *ins;
	    ins = insmake("NOOP");
	    $$ = insconcat(
		$1,
		insmake("DUP"),
		insmake("NOT"),
		pushint(label(ins)),
		insmake("SWAP"),
		insmake("CJUMP"),
		insmake("DROP"),
		$3,
		ins,
		NULL);
	}
	| AUTOINC expr {$$ = insconcat($2, insmake("INC"), NULL);}
	| AUTODEC expr {$$ = insconcat($2, insmake("DEC"), NULL);}
	| expr AUTOINC {$$ = insconcat($1, insmake("DUP"), insmake("SWAP"), insmake("INC"), insmake("DROP"), NULL);}
	| expr AUTODEC {$$ = insconcat($1, insmake("DUP"), insmake("SWAP"), insmake("DEC"), insmake("DROP"), NULL);}
	| expr '[' expr ']' {$$ = insconcat($1, $3, insmake("INDEX"), NULL);}
	| '(' expr ')' {$$ = $2;}
	| '(' T_INT ')' expr {$$ = insconcat($4, insmake("MAKEINT"), NULL);}
	| '(' T_FLOAT ')' expr {$$ = insconcat($4, insmake("MAKEFLOAT"), NULL);}
	| '(' T_STRING ')' expr {$$ = insconcat($4, insmake("MAKESTRING"), NULL);}
	| '(' T_ARRAY ')' expr {$$ = insconcat($4, insmake("MAKEARRAY"), NULL);}
;

%%

void yyerror(const char *err)
{
    printf("%i in %s: %s\n", yycurline, yycurfile, err);
    exit(1);
}
