/* NVTV simple API wrapper for client backend
 *
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: nvtv_simple.c,v 1.7 2004/04/20 16:01:36 dthierbach Exp $
 *
 * Contents:
 *
 * Initial coding by Matthias Hopf <mat@mshopf.de>
 * Extracted by Mattias Eriksson <snaggen@acc.umu.se>
 *
 */

#ifdef HAVE_CONFIG_H
#include "simple_config.h"
#endif

#include "local.h" /* before everything else */

#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <syslog.h>


#include "back_client.h"
#include "nvtv_simple.h"
#include "resize.h"

#ifdef HAVE_RANDR
#include <X11/Xlib.h>
#include <X11/extensions/Xrandr.h>
#include <X11/extensions/Xrender.h>
#include <gdk/gdkx.h>
#endif

#undef NVTV_DEBUG 

#ifdef NVTV_DEBUG
#define DPRINTF(x...) g_message(x)
#else
#define DPRINTF(x...)
#endif

/*
 * PRIVATE
 */

BackCardPtr        back_card   = 0;
BackAccessPtr      back_access = 0;

static int          current_type, current_width, current_height;
static double       current_fps;

static TVRegs       old_tvregs;
static TVSettings   old_settings;

static int          tvmode_enabled = 1;
static int          was_enabled = 0;
#ifdef HAVE_RANDR
static int          xrandr_available = 0;
static int          autoresize_enabled = 0;
static int          xrandr_current = -1;
#endif
static TVSystem  opt_system  = TV_SYSTEM_PAL;
static TVConnect opt_connect = CONNECT_NONE;

static TVMode *modes_pal = NULL;
static int size_modes_pal = 0;
static TVMode *modes_ntsc =NULL;
static int size_modes_ntsc = 0;

static double opt_aspect  = 4.0 / 3.0;


/* Try to connect to nvtvd server */
static void tvmode_connect() {
    CardPtr main_card_list, main_card = NULL;

    if (was_enabled) {
        DPRINTF ("nvtv_simple: Already initialized!\n");
        return;
    }
    if (back_client_avail ()) {
        main_card_list = back_client_init ();
    } else {
        DPRINTF ("nvtv_simple: Nvtvd not detected, make sure nvtvd is running.\n");
        /* Handle well in some way... */
        return;
    }

    if (back_access) {
        //back_access->closeCard ();
    }

    if (!main_card_list) {
        DPRINTF ("nvtv_simple: No supported video card found.\n");
        /* Handle well in some way... */
        return;
    }

    /* Just pick the first, here we might want to pick an alternative card... */
    main_card = main_card_list;

    if (!main_card) {
        DPRINTF ("nvtv_simple: No supported video card found at specified address.\n"); 
        /* Handle well in some way... */
        return;
    }


    if (back_access) {
		DPRINTF ("nvtv_simple: 8\n");
        back_access->openCard (main_card);
        DPRINTF("nvtv_simple: Using card %s for tvout\n", main_card->name);
        was_enabled = 1;
    } else {
        DPRINTF("nvtv_simple: cannot connect to nvtvd - no TV mode switching available\n");
    }
	DPRINTF ("nvtv_simple: 9\n");

}


/* Disconnect from server */
static void tvmode_disconnect () {
    back_access->closeCard ();
    back_access = 0;
    was_enabled = 0;
}


/* Save current CRT and TV register configuration */
static void tvmode_savestate () {
    back_card->getSettings(&old_settings);
    back_card->getMode (&old_tvregs);
#ifdef HAVE_RANDR
    if (xrandr_available) {
      xrandr_current = nvtv_randr_resize_get_current();  
    }
#endif
}


/* Restore CRT and TV register configuration */
static void tvmode_restorestate () {
    back_card->setModeSettings (&old_tvregs,&old_settings); 

    DPRINTF("nvtv_simple: Old mode saved!");

    current_type = NVTV_SIMPLE_TV_OFF;
#ifdef HAVE_RANDR
    if (xrandr_available) {
        nvtv_randr_restore(xrandr_current);
    }
#endif
}


/* Set CRT and TV registers to given TV-Out configuration */
static void tvmode_settvstate (int width, int height) {
    TVSettings settings;
    TVMode     mode;
    TVMode     *modes;
    int         size;
    int        found = 0;
    int        tmp_fps;
    int i; 

#ifdef HAVE_RANDR
    int j;
    XRRScreenConfiguration *xr_screen_conf;
    XRRScreenSize *xr_sizes;
    SizeID xr_current_size;
    int xr_nsize;
    Rotation xr_rotations;
    Rotation xr_current_rotation;
    Status status;
    
      /* Getting the current info */
    XLockDisplay (GDK_DISPLAY());
    xr_screen_conf = XRRGetScreenInfo
        (GDK_DISPLAY(), GDK_ROOT_WINDOW());
    xr_current_size = XRRConfigCurrentConfiguration
        (xr_screen_conf, &xr_current_rotation);
    xr_sizes = XRRConfigSizes (xr_screen_conf, &xr_nsize);
    xr_rotations = XRRConfigRotations (xr_screen_conf,
            &xr_current_rotation);
    XUnlockDisplay (GDK_DISPLAY());
#endif 
    
    /* TODO: do that at initialization and save possible combinations */
    /* Find supported TV mode */
    if (opt_system == TV_SYSTEM_NTSC) {
        modes = modes_ntsc;
        size = size_modes_ntsc;
        tmp_fps = 29.97;
    } else {
        modes = modes_pal;
        tmp_fps = 25;
        size = size_modes_pal;
    }

#ifdef HAVE_RANDR
    /* First try to find a perfect match for XRandr */
    DPRINTF("Looking for a perfect match\n");
    for (i=0; i < size; i++) {
        
        DPRINTF("Trying mode %dx%d against requested %dx%d\n", modes[i].spec.res_x, modes[i].spec.res_y, width, height);
         
        if ((width <= modes[i].spec.res_x) && (height <= modes[i].spec.res_y) &&
                (back_card->findByOverscan (opt_system, modes[i].spec.res_x, modes[i].spec.res_y, 10, 5, &mode))) {
            
            DPRINTF("Found a candidate %dx%d\n",  modes[i].spec.res_x,  modes[i].spec.res_y);
            
            for (j = 0; j < xr_nsize; j++)
            {
                /*
                fprintf(stderr, "Match against %dx%d\n",  xr_sizes[j].width, xr_sizes[j].height,  modes[i].spec.res_y);
                */
                if ((xr_sizes[j].width ==  mode.spec.res_x) && (xr_sizes[j].height == mode.spec.res_y)) {
                current_width  = mode.spec.res_x;
                current_height = mode.spec.res_y;
                current_fps = tmp_fps;
                DPRINTF("Perfect match found %dx%d\n",mode.spec.res_x,mode.spec.res_y);
                found++;
                }
            }
        }
    }
    if (!found && autoresize_enabled) {
        /* Could use some error handling. If no xrandr mode is found we 
         * can't to the switch. */
        return;
    }
#endif
	DPRINTF("Still searching\n");

    /* If no perfect match exists, fallback to the ordinary solution */
    if (!found) {
        DPRINTF("NO Perfect match found\n");
        for (i=0; i < size; i++) {
            /*
               printf ("Trying mode %dx%d %f %f\n", modes[i].spec.res_x, modes[i].spec.res_y, modes[i].spec.hoc, modes[i].spec.voc);
             */
            if ((width <= modes[i].spec.res_x) && (height <= modes[i].spec.res_y) &&
                    (back_card->findByOverscan (opt_system, modes[i].spec.res_x, modes[i].spec.res_y, 10, 5, &mode))) {
                current_width  = mode.spec.res_x;
                current_height = mode.spec.res_y;
                current_fps = tmp_fps;
                found++;
                DPRINTF("Non-perfect match found %dx%d\n",mode.spec.res_x,mode.spec.res_y);
            }
        }
    }
    /* Switch to mode */
	DPRINTF("Going to switch!\n");
    if (found) {
        back_card->getSettings (&settings);
        if (opt_connect > CONNECT_NONE) {
            settings.connector = opt_connect;
        } else {
            settings.connector = CONNECT_BOTH;
        }

        mode.regs.devFlags = DEV_MONITOR | DEV_TELEVISION;
        settings.flags |= TV_DESC_DUALVIEW;    

/* FIXME Note: This is a hack which forces always DUALVIEW, even if the
 * monitor is not able to handle it. Not sure if this should be in.
 * - Dirk
 */
        back_card->setModeSettings (&mode.regs, &settings);

        DPRINTF("nvtv_simple: Trying to use mode %i x %i\n",current_width,current_height);
        current_type = NVTV_SIMPLE_TV_ON;
#ifdef HAVE_RANDR
        if (autoresize_enabled) {
            nvtv_randr_resize(current_height,current_width);
        }
#endif
    } else {
        DPRINTF("nvtv_simple: cannot find any valid TV mode - TV output disabled\n");
    }
}

static void init_modes() {
    int i;

    size_modes_pal = back_card->listModes(TV_SYSTEM_PAL, &modes_pal);

    size_modes_ntsc = back_card->listModes(TV_SYSTEM_NTSC, &modes_ntsc);

    /* 
       for (i=0; i < size_modes_pal; i++) {
       printf ("Modes PAL %dx%d %f %f\n", modes_pal[i].spec.res_x, modes_pal[i].spec.res_y, modes_pal[i].spec.hoc, modes_pal[i].spec.voc);
       }

       for (i=0; i < size_modes_pal; i++) {
       printf ("Modes NTSC %dx%d %f %f\n", modes_ntsc[i].spec.res_x, modes_ntsc[i].spec.res_y, modes_ntsc[i].spec.hoc, modes_ntsc[i].spec.voc);
       }
     */

}

/*
 * PUBLIC
 */

/* Set to NVTV_SIMPLE_TV_OFF or NVTV_SIMPLE_TV_ON */
int nvtv_simple_switch (int type, int width, int height) {

    if(tvmode_enabled) {

        /* 
         * Wasn't initialized
         */
        if(!was_enabled) {
            nvtv_simple_init();
        }
        if (back_card) {
            DPRINTF("nvtv_simple: switching to %s\n", type ? "TV" : "default");

            switch (type) {
                case NVTV_SIMPLE_TV_OFF:
                    tvmode_restorestate ();
                    break;
                case NVTV_SIMPLE_TV_ON:
                    tvmode_settvstate (width, height);
                    break;
                default:
                    DPRINTF("nvtv_simple: illegal type for switching\n");
                    tvmode_restorestate ();
            }
        } else {
            DPRINTF("nvtv_simple: not connected to nvtvd for switching\n");
        }

    }
    return current_type;
}

int nvtv_simple_get_state()
{
    return current_type;
}


/* Addapt (maximum) output size to visible area and set pixel aspect and fps */
void nvtv_simple_size (int *width, int *height,
        double *pixelratio, double *fps) {

    if(tvmode_enabled) {

        switch (current_type) {
            case NVTV_SIMPLE_TV_ON:
                /*
                if (width  && *width > current_width)
                */
                if (width)
                    *width  = current_width;
                /*
                if (height && *height > current_height)
                */
                if (height)
                    *height = current_height;
                if (pixelratio)
                    *pixelratio = ((double) current_width / current_height) / opt_aspect;
                if (fps)
                    *fps = current_fps;
                break;
        }

    }
}


/* Connect to nvtvd server if possible and store settings */
int  nvtv_simple_init() {
#ifdef HAVE_RANDR
    gdk_init (NULL,NULL);
   
    xrandr_available = nvtv_randr_resize_init();
#endif
    tvmode_connect ();
    if (back_card) {
        tvmode_savestate ();
        init_modes();
        return 1;
    }

    return 0;
}

int nvtv_enable_autoresize (int enable) {
#ifdef HAVE_RANDR
    if (xrandr_available && enable) {
        autoresize_enabled = 1;
    } else {
        autoresize_enabled = 0;
    }
    return (autoresize_enabled);
#else
    return (0); 
#endif
}


int nvtv_simple_is_available() {
	return (was_enabled&&tvmode_enabled);
}

int nvtv_simple_enable(int use_tvmode) {

    tvmode_enabled = 0;
    if (use_tvmode) {
        if (was_enabled || nvtv_simple_init()) {
            tvmode_enabled = 1;
        } 
    }

    return (tvmode_enabled);
}

/* FIXME Note: There is already an enumeration of TV systems in 
 * tvchip.h, I am not sure if this should really be duplicated again.
 * - Dirk
 */

void nvtv_simple_set_tvsystem(nvtv_simple_tvsystem system) {

    switch (system) {
        case NVTV_SIMPLE_TVSYSTEM_PAL:
            opt_system  = TV_SYSTEM_PAL;
            break;
        case NVTV_SIMPLE_TVSYSTEM_NTSC:
            opt_system  = TV_SYSTEM_NTSC;
            break;
        default:
            opt_system  = TV_SYSTEM_PAL;
    }
}

/* Restore old CRT and TV registers and close nvtvd connection */
void nvtv_simple_exit () {

    if(tvmode_enabled || was_enabled) {
        if (back_card) {
            tvmode_restorestate ();
            tvmode_disconnect ();
        }
        if (modes_pal != NULL) {
            free(modes_pal);
        }
        if (modes_ntsc != NULL) {
            free(modes_ntsc);
        }
    }
}

/* -------- Error handling stolen from nvtvd (with small adjustments) -------- */

void raise_msg (int class, char *format, ...)
{
	static char buf[1024];
	static int pos = 0;
	int delta;
	va_list args;

	va_start(args, format);
	delta = vsnprintf(buf + pos, sizeof(buf) - pos, format, args);
	va_end(args);
	if (delta == -1 || pos + delta > sizeof(buf) - 1) {
		pos = sizeof(buf);
	} else {
		pos += delta;
	}
	if (class == MSG_DEBUG_NL) return;
	pos = 0;
#ifdef NVTV_DEBUG
	switch (class) {
		case MSG_ABORT:
			DPRINTF (stderr, "Fatal: ");
			break;
		case MSG_WARNING:
			DPRINTF (stderr, "Warning: ");
			break;
	}
	DPRINTF (stderr, "%s\n", buf);
#else
	switch (class) {
		case MSG_ABORT:
			syslog (LOG_CRIT, "%.500s", buf);
			break;
		case MSG_WARNING:
			syslog (LOG_WARNING, "%.500s", buf);
			break;
		case MSG_INFO:
			syslog (LOG_INFO, "%.500s", buf);
			break;
		case MSG_DEBUG:
			syslog (LOG_DEBUG, "%.500s", buf);
			break;
		default:
		case MSG_ERROR:
			syslog (LOG_ERR, "%.500s", buf);
			break;
	}
#endif
	if (class == MSG_ABORT) exit (1);
}
