#include <glib-object.h>

#include "common.h"

#include <gmyth/gmyth_uri.h>
#include <gmyth/gmyth_backendinfo.h>
#include <gmyth/gmyth_livetv.h>
#include <gmyth/gmyth_file.h>
#include <gmyth/gmyth_file_transfer.h>
#include <gmyth/gmyth_file_local.h>
#include <gmyth/gmyth_common.h>

#define URI_DEFAULT	"myth://192.168.3.165:6543/livetv?channel=9"

static          gboolean
test_livetv_setup(gchar * uri)
{
    GMythLiveTV    *livetv = NULL;
    GMythFile      *file = NULL;
    gchar          *channel_name = NULL;
    gboolean        ret = TRUE;
    gboolean        live_tv = FALSE;

    if (NULL == uri)
        uri = g_strdup(URI_DEFAULT);

    GMythURI       *gmyth_uri = gmyth_uri_new_with_value(uri);

    GMythBackendInfo *backend_info = gmyth_backend_info_new_with_uri(uri);
    live_tv = gmyth_uri_is_livetv(gmyth_uri);
    if (live_tv) {
        livetv = gmyth_livetv_new(backend_info);

        gchar          *ch = gmyth_uri_get_channel_name(gmyth_uri);
        if (ch != NULL)
            channel_name = ch;

        if (channel_name != NULL) {
            if (gmyth_livetv_channel_name_setup(livetv, channel_name) ==
                FALSE) {
                g_debug("LiveTV setup felt down on error.");
                ret = FALSE;
                goto init_failed;
            }
        } else {
            if (gmyth_livetv_setup(livetv) == FALSE) {
                g_debug("LiveTV setup felt down on error");
                ret = FALSE;
                goto init_failed;
            }
        }

        file = GMYTH_FILE(gmyth_livetv_create_file_transfer(livetv));

        if (NULL == file) {
            g_debug("[LiveTV] FileTransfer equals to NULL");
            ret = FALSE;
            goto init_failed;
        }

        /*
         * Check if the file is local to this specific client renderer 
         */
        if (gmyth_uri_is_local_file(gmyth_uri))
            ret = gmyth_file_local_open(GMYTH_FILE_LOCAL(file));
        else
            ret =
                gmyth_file_transfer_open(GMYTH_FILE_TRANSFER(file),
                                         livetv->uri !=
                                         NULL ? gmyth_uri_get_path(livetv->
                                                                   uri) :
                                         livetv->proginfo->pathname->str);

        if (!ret) {
            g_debug
                ("Error: couldn't open the FileTransfer from LiveTV source!");
            goto init_failed;
        }
    } else {

        /*
         * Check if the file is local to this specific client renderer,
         * and tries to open a local connection 
         */
        if (gmyth_uri_is_local_file(gmyth_uri)) {
            g_debug("Opening local file connection to download...");
            file = GMYTH_FILE(gmyth_file_local_new(backend_info));
            ret = gmyth_file_local_open(GMYTH_FILE_LOCAL(file));
        } else {
            g_debug("Opening remote file connection to download...");
            file = GMYTH_FILE(gmyth_file_transfer_new(backend_info));
            ret = gmyth_file_transfer_open(GMYTH_FILE_TRANSFER(file), uri);
        }

    }                           /* if (else) - recorded FileTransfer */

    if (NULL == file) {
        g_debug("FileTransfer is NULL");
        ret = FALSE;
        goto init_failed;
    }
    g_debug("uri = %s", uri);

    if (ret == FALSE) {
        g_debug
            ("MythTV FileTransfer request failed when setting up socket connection!");
        goto init_failed;
    }

    g_debug("MythTV FileTransfer filesize = %lld",
            gmyth_file_get_filesize(file));

  init_failed:
    if (livetv != NULL)
        g_object_unref(livetv);

    if (file != NULL)
        g_object_unref(file);

    // if ( uri != NULL )
    // g_free( uri );

    if (gmyth_uri != NULL)
        g_object_unref(gmyth_uri);

    // if ( backend_info != NULL )
    // g_object_unref( backend_info );

    return ret;

}

gint
main(gint args, const gchar ** argv)
{
    gboolean        ret;

    g_type_init();

    g_thread_init(NULL);

    fprintf(stdout, SYNC_STRING);
    fflush(NULL);
    getchar();

    if (args > 1)
        ret = test_livetv_setup(argv[1]);
    else
        ret = test_livetv_setup(NULL);

    if (!ret)
        g_debug("Error when running LiveTV setup test script!");
    else
        g_debug("LiveTV setup test script finished with success.");

    return (0);
}
