// Copyright (C) 2002 Neil Stevens <neil@hakubi.us>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
// AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name(s) of the author(s) shall not be
// used in advertising or otherwise to promote the sale, use or other dealings
// in this Software without prior written authorization from the author(s).

#include <kbuttonbox.h>
#include <kdebug.h>
#include <keditcl.h>
#include <klineeditdlg.h>
#include <klistview.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <liblighthawk/configpage.h>
#include <qhbox.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qvbox.h>

#include "awaymessagedialog.h"

QString LightHawk::AwayMessageDialog::getAwayMessage(Data &data, const QString &caption, bool *ok, QWidget *parent)
{
	AwayMessageDialog dialog(data, parent);
	if(!caption.isNull())
		dialog.setCaption(caption);

	bool ret = dialog.exec();
	if(ok) *ok = ret;

	if(ret)
		return dialog.message();
	else
		return QString::null;
}

LightHawk::AwayMessageDialog::AwayMessageDialog(Data &d, QWidget *p)
	: KDialogBase(KDialogBase::Plain, QString::null, Ok | Cancel, Ok, p, 0, true, true)
	, data(d)
{
	QWidget *page = plainPage();

	QVBox *box1 = new QVBox(page);
	box1->setMargin(0);
	box1->setSpacing(spacingHint());
	new QLabel(i18n("Name"), box1);

	QHBox *listBox = new QHBox(box1);
	listBox->setMargin(0);
	listBox->setSpacing(spacingHint());

	KButtonBox *listBBox = new KButtonBox(listBox, Qt::Vertical);
	addButton = listBBox->addButton(i18n("Add"));
	removeButton = listBBox->addButton(i18n("Remove"));
	renameButton = listBBox->addButton(i18n("Rename"));
	listBBox->layout();

	awayList = new KListView(listBox);
	awayList->addColumn(i18n("Name"));

	QVBox *editBox = new QVBox(page);
	editBox->setMargin(0);
	editBox->setSpacing(spacingHint());
	new QLabel(i18n("Message"), editBox);
	away = new KEdit(editBox);
	away->setWordWrap(KEdit::WidgetWidth);

	QHBoxLayout *layout = new QHBoxLayout(page, marginHint(), spacingHint());
	layout->addWidget(box1);
	layout->addWidget(editBox);

	connect(awayList, SIGNAL(selectionChanged(QListViewItem *)), this, SLOT(awayListSelectionChanged(QListViewItem *)));
	connect(away, SIGNAL(textChanged(void)), this, SLOT(awayTextChanged(void)) );
	connect(renameButton, SIGNAL(clicked(void)), this, SLOT(renameMessage(void)));
	connect(addButton, SIGNAL(clicked(void)), this, SLOT(addMessage(void)));
	connect(removeButton, SIGNAL(clicked(void)), this, SLOT(removeMessage(void)));

	awayMessages = data.entryMap("Away Messages");

	if(awayMessages.empty())
	{
		awayMessages.insert(i18n("Default"), i18n("I do really enjoy the default away message of Light Hawk.  It is nice and original!"));
		awayMessages.insert(i18n("Baseball"), i18n("I'm off watching baseball.  Go Dodgers!"));
	}

	for(QMap<QString, QString>::Iterator i = awayMessages.begin(); i != awayMessages.end(); ++i)
		(void)new KListViewItem(awayList, i.key());

	awayList->setSelected(awayList->firstChild(), true);
}

QString LightHawk::AwayMessageDialog::message(void)
{
	return away->text();
}

void LightHawk::AwayMessageDialog::accept(void)
{
	data.deleteGroup("Away Messages", true);
	data.setGroup("Away Messages");
	for(QMap<QString, QString>::Iterator i = awayMessages.begin(); i != awayMessages.end(); ++i)
		data.writeEntry(i.key(), i.data());

	KDialogBase::accept();
}

void LightHawk::AwayMessageDialog::awayListSelectionChanged(QListViewItem *item)
{
	if(!item)
	{
		enableButtonOK(false);
		removeButton->setEnabled(false);
		renameButton->setEnabled(false);
		away->setEnabled(false);
		away->setText(QString::null);
	}
	else
	{
		enableButtonOK(true);
		removeButton->setEnabled(awayList->childCount() > 1);
		renameButton->setEnabled(true);
		away->setEnabled(true);
		away->setText(awayMessages[item->text(0)]);
	}
}

void LightHawk::AwayMessageDialog::awayTextChanged(void)
{
	awayMessages[awayList->currentItem()->text(0)] = away->text();
}

QString LightHawk::AwayMessageDialog::validateAwayMessageName(const QString &name)
{
	QString trial = name;
	int i = 1;
	while(awayMessages.contains(trial))
	{
		i++;
		trial = i18n("%1 (%2)").arg(name).arg(i);
	}
	return trial;
}

void LightHawk::AwayMessageDialog::renameMessage(void)
{
	QListViewItem *item = awayList->currentItem();
	bool ok;
	QString oldName = item->text(0);
	QString newName = KLineEditDlg::getText(i18n("Enter a new name for the message:"), oldName, &ok, this);
	if(!ok) return;

	if(newName == oldName) return;

	newName = validateAwayMessageName(newName);

	QString message = awayMessages[oldName];
	awayMessages.remove(oldName);
	awayMessages[newName] = message;
	item->setText(0, newName);
	away->setFocus();
}

void LightHawk::AwayMessageDialog::addMessage(void)
{
	bool ok;
	QString oldName = i18n("My Away Message");
	QString newName = KLineEditDlg::getText(i18n("Enter a name for the new away message:"), oldName, &ok, this);
	if(!ok) return;

	newName = validateAwayMessageName(newName);

	KListViewItem *item = new KListViewItem(awayList, newName);
	awayMessages[newName] = i18n("I do really enjoy the default away message of Light Hawk.  It is nice and original!");
	awayList->setSelected(item, true);
	away->selectAll();
	away->setFocus();
}

void LightHawk::AwayMessageDialog::removeMessage(void)
{
	QListViewItem *item = awayList->currentItem();
	if(KMessageBox::Yes == KMessageBox::warningYesNo(0, i18n("Are you sure you want to delete %1?").arg(item->text(0))))
	{
		awayMessages.remove(item->text(0));
		delete item;
		awayList->setSelected(awayList->firstChild(), true);
	}
}

#include "awaymessagedialog.moc"
// arch-tag: awaymessagedialog.cpp
