// Copyright (C) 2003 Neil Stevens <neil@hakubi.us>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
// AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name(s) of the author(s) shall not be
// used in advertising or otherwise to promote the sale, use or other dealings
// in this Software without prior written authorization from the author(s).

#include <kconfigdata.h>
#include <kdebug.h>
#include <qregexp.h>
#include "configpage.h"

struct LightHawk::Data::PrivateData
{
	PrivateData(void)
	{
		groupRE = QString("^([^/]*)/(.*)");
	}

	QRegExp groupRE;
	QMap<QString, QString> map;
};

LightHawk::Data::Data(void)
{
	d = new PrivateData;
}

LightHawk::Data::~Data(void)
{
	delete d;
	d = 0;
}

QStringList LightHawk::Data::groupList(void) const
{
	QStringList groups;
	for(QMap<QString, QString>::Iterator i = d->map.begin(); i != d->map.end(); ++i)
	{
		int ret = d->groupRE.search(i.key());
		if(ret == -1 && !groups.contains(""))
			groups.append("");
		else if(!groups.contains(d->groupRE.cap(1)))
			groups.append(d->groupRE.cap(1));
	}
	return groups;
}

QMap<QString, QString> LightHawk::Data::entryMap(const QString &group) const
{
	QMap<QString, QString> groupMap;
	for(QMap<QString, QString>::Iterator i = d->map.begin(); i != d->map.end(); ++i)
	{
		if(0 == i.key().find(QString("%1/").arg(group)))
		{
			d->groupRE.search(i.key());
			groupMap[d->groupRE.cap(2)] = i.data();
		}
	}
	return groupMap;
}

void LightHawk::Data::reparseConfiguration(void)
{
	// nothing
}

QMap<QString, QString> LightHawk::Data::rawMap(void) const
{
	return d->map;
}


void LightHawk::Data::setRawMap(const QMap<QString, QString> &map)
{
	d->map = map;
}


KEntryMap LightHawk::Data::internalEntryMap(const QString &group) const
{
	QMap<QString, QString> map = entryMap(group);
	KEntryMap entryMap;

	for(QMap<QString, QString>::Iterator i = map.begin(); i != map.end(); ++i)
	{
		KEntryKey key(group.utf8(), i.key().utf8());
		KEntry entry;
		entry.mValue = i.data().utf8();
		entryMap[key] = entry;
	}
	return entryMap;
}

KEntryMap LightHawk::Data::internalEntryMap(void) const
{
	KEntryMap entryMap;

	for(QMap<QString, QString>::Iterator i = d->map.begin(); i != d->map.end(); ++i)
	{
		QString group;
		int ret = d->groupRE.search(i.key());
		if(ret == -1)
			group = "";
		else
			group = d->groupRE.cap(1);

		KEntryKey key(group.utf8(), i.key().utf8());
		KEntry entry;
		entry.mValue = i.data().utf8();
		entryMap[key] = entry;
	}
	return entryMap;
}

namespace
{
QString keyForEntry(const QString &group, const QString &key)
{
	if(group.isEmpty() || group == "<default>")
		return key;
	else
		return group + "/" + key;
}
}

void LightHawk::Data::putData(const KEntryKey &key, const KEntry &entry, bool)
{
	d->map[keyForEntry(QString::fromUtf8(key.mGroup), QString::fromUtf8(key.c_key))] = QString::fromUtf8(entry.mValue);
}

KEntry LightHawk::Data::lookupData(const KEntryKey &key) const
{
	KEntry entry;
	QString stringKey = keyForEntry(QString::fromUtf8(key.mGroup), QString::fromUtf8(key.c_key));

	if(!d->map.contains(stringKey))
		entry.mValue = QCString(0);
	else
		entry.mValue = d->map[stringKey].utf8();

	/*
	kdDebug() << "Data::lookupData for " << key.mGroup << " " << key.c_key
	          << " returns " << entry.mValue
	          << " ( map key is " << stringKey << " ) " << endl;
	*/

	return entry;
}

bool LightHawk::Data::internalHasGroup(const QCString &cstring) const
{
	QString group = QString::fromUtf8(cstring);
	for(QMap<QString, QString>::Iterator i = d->map.begin(); i != d->map.end(); ++i)
	{
		int ret = d->groupRE.search(i.key());
		if(ret != -1 && d->groupRE.cap(1) == group)
			return true;
	}
	return false;
}

LightHawk::ConfigPage::ConfigPage(QWidget *parent, const char *name)
	: KCModule(parent, name)
{
	new Data;
}

#include "configpage.moc"
// arch-tag: liblighthawk/configpage.cpp
