// Copyright (C) 2003 Neil Stevens <neil@hakubi.us>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
// AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name(s) of the author(s) shall not be
// used in advertising or otherwise to promote the sale, use or other dealings
// in this Software without prior written authorization from the author(s).

#include <kdebug.h>
#include <kurl.h>
#include <zinv/message.h>
#include <zinv/types.h>
#include <qdom.h>
#include <qfile.h>
#include <qtextstream.h>
#include "jid.h"
#include "logmanager.h"

QString LightHawk::LogManager::encodeLogFilename(const QString &before)
{
	QString after = KURL::encode_string(before);
	after.replace('/', "%2F");
	return after;
}

LightHawk::LogManager::LogManager(QObject *parent, const char *name)
	: QObject(parent, name)
	, isLogging(false)
{
}

LightHawk::LogManager::~LogManager(void)
{
	closeFiles();
}

void LightHawk::LogManager::enableLogging(bool b)
{
	isLogging = b;
	if(!isLogging)
		closeFiles();
}

bool LightHawk::LogManager::loggingEnabled(void)
{
	return isLogging;
}

void LightHawk::LogManager::setLogLocation(const QString &path)
{
	if(logPath != path)
	{
		closeFiles();
		logPath = path;
	}
}

QString LightHawk::LogManager::logLocation(void)
{
	return logPath;
}

void LightHawk::LogManager::closeFiles(void)
{
	chatFileMap.clear();
}


void LightHawk::LogManager::messageIn(const Message &m)
{
	message(m, m.from());
}

void LightHawk::LogManager::messageOut(const Message &m, const Jid &jid)
{
	Message msg = m;
	msg.setFrom(jid);
	message(msg, msg.to());
}

void LightHawk::LogManager::message(const Message &message, const Jid &other)
{
	QFile *file = getFile(message, other);
	if(!file || !file->isOpen())
	{
		// TODO: act/warn on error
		delete file;
		return;
	}

	// TODO: add xml header
	QDomDocument doc;
	doc.appendChild( const_cast<Message &>(message).toXml(&doc, true));

	QTextStream stream(file);
	stream.setEncoding(QTextStream::UnicodeUTF8);
	stream << doc.toString();
	stream << endl;

	delete file;
}

QString LightHawk::LogManager::nameForMessage(const Message &message, const Jid &other)
{
	Jid strippedOther = other;
	strippedOther.setResource(QString::null);

	QString type = message.type();
	if(type.isEmpty()) type = "normal";
	QString path = type + "-" +
	               QString::number(message.timeStamp().toTime_t()) + "-" +
	               strippedOther.userHost();
	path = encodeLogFilename(path);
	path = logPath + path;
	//kdDebug() << "Log file: " << path << endl;
	return path;
}

QFile *LightHawk::LogManager::getFile(const Message &message, const Jid &other)
{
	QFile *file;
	if(message.type() == "chat" ||
	   message.type() == "groupchat" ||
	   message.type() == "headline")
	{
		Jid strippedOther = other;
		strippedOther.setResource(QString::null);

		if(!chatFileMap.contains(strippedOther))
			chatFileMap.insert(strippedOther, FileMap());

		if(!chatFileMap[strippedOther].contains(message.type()))
			chatFileMap[strippedOther].insert(message.type(), nameForMessage(message, other));

		file = new QFile(chatFileMap[strippedOther][message.type()]);
	}
	else
	{
		file = new QFile(nameForMessage(message, other));
	}

	file->open(IO_WriteOnly | IO_Append);
	// TODO: warn on error
	return file;
}

#include "logmanager.moc"
// arch-tag: logmanager.cpp
