// Copyright (C) 2002 Neil Stevens <neil@hakubi.us>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
// AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name(s) of the author(s) shall not be
// used in advertising or otherwise to promote the sale, use or other dealings
// in this Software without prior written authorization from the author(s).

#include <kaction.h>
#include <kapplication.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kdeversion.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kglobal.h>
#include <kkeydialog.h>
#include <klineeditdlg.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#include <knotifyclient.h>
#include <knotifydialog.h>
#include <kpassdlg.h>
#include <kpopupmenu.h>
#include <kprogress.h>
#include <kpushbutton.h>
#include <kstandarddirs.h>
#include <kstatusbar.h>
#include <kstdaction.h>
#include <ktoolbarbutton.h>
#include <liblighthawk/messagewindow.h>
#include <liblighthawk/utils.h>
#include <qwidgetlist.h>
#include <zinv/client.h>
#include <zinv/stream.h>
#include <zinv/tasks.h>
#include <zinv/types.h>

#include "application.h"
#include "awaymessagedialog.h"
#include "configdialog.h"
#include "configmanager.h"
#include "logmanager.h"
#include "mainwindow.h"
#include "messagewindowmanager.h"
#include "profiledialog.h"
#include "rosteritemdialog.h"
#include "rosterlistview.h"
#include "systemtray.h"

LightHawk::MainWindow::MainWindow(const QString &p)
	: KMainWindow(0, QString("MainWindow-%1").arg(p).local8Bit().data())
	, profile(p)
	, client(*new XMPP::Client(this))
	, configManager(*new ConfigManager(profile, this, "ConfigManager"))
	, logManager(*new LogManager(this, "LogManager"))
	, windowManager(*new MessageWindowManager(configManager.config(), client, this, "Manager"))
	, connectDialog(0)
	, configDialog(0)
{
	setStandardToolBarMenuEnabled(true);

	client.setName("Client");

	connect(&client, SIGNAL(error(const StreamError &)), this, SLOT(clientError(const StreamError &)));
	connect(&client, SIGNAL(connected()), this, SLOT(connectHandshake()));
	connect(&client, SIGNAL(handshaken()), this, SLOT(connectAuthenticate()));
	connect(&client, SIGNAL(authFinished(bool, int, const QString &)), this, SLOT(connectGetRoster(bool, int, const QString &)));
	connect(&client, SIGNAL(rosterRequestFinished(bool, int, const QString &)), this, SLOT(connectComplete(bool, int, const QString &)));
	connect(&client, SIGNAL(subscription(const Jid &, const QString &)), this, SLOT(subscriptionReceived(const Jid &, const QString &)));
	connect(&client, SIGNAL(messageReceived(const Message &)), this, SLOT(receiveMessage(const Message &)));

	connect(&windowManager, SIGNAL(addContact(const Jid &)), this, SLOT(addContact(const Jid &)));
	connect(&windowManager, SIGNAL(presence(const Jid &, const QString &)), this, SLOT(sendPresence(const Jid &, const QString &)));
	connect(&windowManager, SIGNAL(message(const Message &)), this, SLOT(sendMessage(const Message &)));
	connect(this, SIGNAL(preferencesChanged(void)), &windowManager, SIGNAL(applyPreferences(void)));
	connect(&client, SIGNAL(resourceAvailable(const Jid &, const Resource &)), &windowManager, SLOT(resourceAvailable(const Jid &, const Resource &)));
	connect(&client, SIGNAL(resourceUnavailable(const Jid &, const Resource &)), &windowManager, SLOT(resourceUnavailable(const Jid &, const Resource &)));

	connect(&client, SIGNAL(messageReceived(const Message &)), &logManager, SLOT(messageIn(const Message &)));

	connect(this, SIGNAL(connectSuccess(void)), this, SLOT(setPresenceInternal()));

	KStdAction::quit(this, SLOT(fileQuit()), actionCollection());
	KStdAction::preferences(this, SLOT(settingsPreferences()), actionCollection());
	KStdAction::keyBindings(this, SLOT(settingsKeys()), actionCollection());
	KStdAction::configureNotifications(this, SLOT(settingsNotifications(void)), actionCollection());
	(void)new KAction(i18n("&Available"), 0, this, SLOT(available()), actionCollection(), "available");
	(void)new KAction(i18n("Free to &chat"), 0, this, SLOT(chat()), actionCollection(), "chat");
	(void)new KAction(i18n("A&way..."), 0, this, SLOT(away()), actionCollection(), "away");
	(void)new KAction(i18n("E&xtended away..."), 0, this, SLOT(xa()), actionCollection(), "xa");
	(void)new KAction(i18n("&Do not disturb..."), 0, this, SLOT(dnd()), actionCollection(), "dnd");
	(void)new KAction(i18n("&Invisible"), 0, this, SLOT(invisible()), actionCollection(), "invisible");
	(void)new KAction(i18n("&Unavailable"), 0, this, SLOT(unavailable()), actionCollection(), "unavailable");
	rosterAddAction = new KAction(i18n("&Add Contact..."), 0, this, SLOT(rosterAdd()), actionCollection(), "roster_addcontact");
	rosterAddServiceAction = new KAction(i18n("Add &Service..."), 0, this, SLOT(rosterAddService()), actionCollection(), "roster_addservice");
	rosterComposeAction = new KAction(i18n("Compose &Message"), 0, this, SLOT(rosterCompose()), actionCollection(), "roster_composemessage");
	rosterOpenChatAction = new KAction(i18n("Open &Chat"), 0, this, SLOT(rosterOpenChat()), actionCollection(), "roster_openchat");
	rosterEditAction = new KAction(i18n("&Edit Contact"), 0, this, SLOT(rosterEdit()), actionCollection(), "roster_edit");
	rosterRemoveAction = new KAction(i18n("&Remove Contact"), 0, this, SLOT(rosterRemove()), actionCollection(), "roster_remove");
	groupOpenChatAction = new KAction(i18n("&Join Group Chat"), 0, this, SLOT(groupOpenChat()), actionCollection(), "message_groupchat");
	toolsSendXMLAction = new KAction(i18n("&Send XML..."), 0, this, SLOT(toolsSendXML(void)), actionCollection(), "tools_sendxml");
	(void)new KAction(i18n("P&rofiles..."), 0, this, SLOT(profiles()), actionCollection(), "profiles");

	createGUI();

	listMenu = (KPopupMenu *)guiFactory()->container("tree", this);
	listMenu->insertTitle("Howdy Doody", -1, 0);

	// Done this way to associate icons with every item
	toolBar("presenceToolBar")->insertButton(DesktopIcon("lighthawk"), 0);
	toolBar("presenceToolBar")->getButton(0)->setPopup((QPopupMenu *)guiFactory()->container("presence", this));
	toolBar("presenceToolBar")->setItemAutoSized(0, true);

	KConfig &config = *KGlobal::config();

	listView = new RosterListView(configManager.config(), client, this);
	connect(this, SIGNAL(connectSuccess(void)), listView, SLOT(fill(void)));
	connect(listView, SIGNAL(contextMenu(const RosterItem &, const QPoint &)), this, SLOT(rosterContextMenu(const RosterItem &, const QPoint &)));
	connect(listView, SIGNAL(selectionChanged(void)), this, SLOT(rosterSelectionChanged(void)));
	connect(listView, SIGNAL(executed(const RosterItem &)), this, SLOT(rosterExecuted(const RosterItem &)));

	setCentralWidget(listView);

	tray = new SystemTray("tray", this, "Tray");
	unavailable();

	applyMainWindowSettings(&config, "MainWindow");
	configManager.readData(*KGlobal::config());
	applyPreferences();

	rosterSelectionChanged();

	Data &data = configManager.config();
	data.setGroup("");
	if(data.readBoolEntry("autoAvailable", false)) available();
}

LightHawk::MainWindow::~MainWindow(void)
{
	configManager.writeData(*KGlobal::config());
}

void LightHawk::MainWindow::closeWindowsForProfile(const QString &profile)
{
	QWidgetList *windowList = kapp->topLevelWidgets();
	for(QWidgetListIt i(*windowList); *i; ++i)
	{
		MainWindow *window = dynamic_cast<MainWindow *>(*i);
		if(window && window->profile == profile)
			window->close();
	}
	delete windowList;
}

void LightHawk::MainWindow::closeEvent(QCloseEvent *e)
{
	// kdDebug() << "MainWindow closeEvent" << endl;
	unavailable();
	saveMainWindowSettings(KGlobal::config(), "MainWindow");
	mainWindowWorkaround(this);
	KMainWindow::closeEvent(e);
}

void LightHawk::MainWindow::saveProperties(KConfig *config)
{
	config->writeEntry("LHProfile", profile);
	KMainWindow::saveProperties(config);
}

void LightHawk::MainWindow::applyPreferences(void)
{
	// TODO: make more stuff configurable

	Data &data = configManager.config();
	data.setGroup("");

	setCaption(data.readEntry("profileName"));
	tray->setTitle(SmallIcon("lighthawk"), data.readEntry("profileName"));

	QString me = data.readEntry("jid") + "/" + data.readEntry("resource");
	QString subpath = "lighthawk/" + LogManager::encodeLogFilename(me) + "/";
	logManager.setLogLocation(KGlobal::dirs()->saveLocation("data", subpath, true));

	tray->show();

	data.setGroup("Roster");
	listView->setOnlineOnly(data.readBoolEntry("showOnlineOnly"));

	client.setOSName("WashuuOS");
	client.setClientName("telnet");
	client.setClientVersion("4.2BSD");

	data.setGroup("Message");
	logManager.enableLogging(data.readBoolEntry("logging"));

	// update priority
	setPresenceInternal();

	emit preferencesChanged();

	configManager.writeData(*KGlobal::config());
}

void LightHawk::MainWindow::setPresence(const QString &show, const QString &longShow,
                                        const QString &status, bool invisible)
{
	//kdDebug() << "setPresence " << show << " " << longShow << " " << status << endl;
	setPresenceShow = show;
	setPresenceLongShow = longShow;
	setPresenceStatus = status;
	setPresenceInvisible = invisible;
	setPresenceInternal();
}

void LightHawk::MainWindow::setPresenceInternal(void)
{
	// Connect if necessary
	if(setPresenceShow != "unavailable" && !client.isAuthenticated())
	{
		// already trying to connect?
		if(connectDialog) return;

		connectBegin();
		return;
	}

	tray->setPixmap(trayIconForStatus(XMPP::Status(setPresenceShow, setPresenceStatus)));

	toolBar("presenceToolBar")->setButtonIcon(0, setPresenceShow);
	if(!setPresenceStatus.length())
		toolBar("presenceToolBar")->getButton(0)->setText(setPresenceLongShow);
	else
		toolBar("presenceToolBar")->getButton(0)->setText(setPresenceLongShow + ": " + setPresenceStatus);

	toolBar("presenceToolBar")->repaint();

	if(setPresenceShow == "unavailable")
	{
		if(client.isActive())
		{
			KNotifyClient::event("Disconnect");
			client.close();
		}
		rosterAddAction->setEnabled(false);
		rosterAddServiceAction->setEnabled(false);
		groupOpenChatAction->setEnabled(false);
		toolsSendXMLAction->setEnabled(false);
		listView->clear();
	}
	else
	{
		Data &data = configManager.config();
		data.setGroup("");
		int priority = data.readNumEntry("priority");

		XMPP::Status s;
		if(setPresenceInvisible)
		{
			s.setIsInvisible(true);
		}
		else
		{
			s.setIsAvailable(true);
			s.setShow(setPresenceShow);
		}
		s.setStatus(setPresenceStatus);
		s.setPriority(priority);
		client.setPresence(s);
		rosterAddAction->setEnabled(true);
		rosterAddServiceAction->setEnabled(true);
		groupOpenChatAction->setEnabled(true);
		toolsSendXMLAction->setEnabled(true);
	}
}

void LightHawk::MainWindow::connectBegin(void)
{
	// Don't start up twice
	if(connectDialog) return;

	connectDialog = new KProgressDialog(this, 0, i18n("Connecting"), QString::null);
	connectDialog->setAllowCancel(true);
	connectDialog->progressBar()->setRange(0, 4);
	connectDialog->setLabel(i18n("Contacting Host"));

	connect(connectDialog, SIGNAL(cancelClicked()), this, SLOT(connectCancel()));

	// Get virtual host from jid
	Data &data = configManager.config();
	data.setGroup("");

	XMPP::Jid jid = data.readEntry("jid");
	client.connectToHost(data.readEntry("host"), data.readNumEntry("port"), jid.host());
}

void LightHawk::MainWindow::connectHandshake(void)
{
	connectDialog->progressBar()->advance(1);
	connectDialog->setLabel(i18n("Handshaking"));
}

void LightHawk::MainWindow::connectAuthenticate(void)
{
	connectDialog->progressBar()->advance(1);
	connectDialog->setLabel(i18n("Authenticating"));

	// Get user from jid
	configManager.config().setGroup("");
	XMPP::Jid jid = configManager.config().readEntry("jid");
	QString resource = configManager.config().readEntry("resource");

	// Get password
	QString password = configManager.config().readEntry("password");
	if(!configManager.config().readBoolEntry("savePassword"))
	{
		KPasswordDialog passDialog(KPasswordDialog::Password, false, false, this);
		passDialog.setPrompt(i18n("Password for %1?").arg(jid.userHost()));
		if(passDialog.exec() != KPasswordDialog::Accepted)
		{
			connectCancel();
			return;
		}
		password = passDialog.password();
	}

	client.authDigest(jid.user(), password, resource);
}

void LightHawk::MainWindow::connectGetRoster(bool success, int code, const QString &message)
{
	if(!success)
	{
		KMessageBox::error(this, i18n("Code %1: %2").arg(code).arg(message));
		connectCancel();
		return;
	}

	connectDialog->progressBar()->advance(1);
	connectDialog->setLabel(i18n("Requesting Roster"));
	client.rosterRequest();
}

void LightHawk::MainWindow::connectComplete(bool success, int code, const QString &message)
{
	if(!success)
	{
		KMessageBox::error(this, i18n("Code %1: %2").arg(code).arg(message));
		connectCancel();
		return;
	}

	KNotifyClient::event("Connect");

	delete connectDialog;
	connectDialog = 0;
	emit connectSuccess();
}

void LightHawk::MainWindow::connectCancel(void)
{
	connectDialog->deleteLater();
	connectDialog = 0;
	client.close();
}

void LightHawk::MainWindow::clientError(const XMPP::StreamError &error)
{
	connectCancel();
	unavailable();
	KMessageBox::error(this, error.toString());
}

void LightHawk::MainWindow::rosterContextMenu(const XMPP::RosterItem &item, const QPoint &p)
{
	listMenu->removeItemAt(0);
	listMenu->insertTitle(item.jid().userHost(), -1, 0);
	listMenu->popup(p);
}

void LightHawk::MainWindow::rosterSelectionChanged(void)
{
	const bool enable = listView->selectedItem() != 0;
	rosterEditAction->setEnabled(enable);
	rosterComposeAction->setEnabled(enable);
	rosterOpenChatAction->setEnabled(enable);
	rosterRemoveAction->setEnabled(enable);
}

void LightHawk::MainWindow::rosterExecuted(const XMPP::RosterItem &item)
{
	MessageWindow *window = windowManager.messageWindow(currentJid(), item.jid(), "chat");
	if(!window) return;
	window->show();
}

void LightHawk::MainWindow::subscriptionReceived(const XMPP::Jid &jid, const QString &type)
{
	if(type == "subscribe")
	{
		const QString message = i18n("Do you wish to authorize %1 to subscribe to your presence?")
		                         .arg(jid.userHost());
		if(KMessageBox::Yes == KMessageBox::warningYesNo(this, message))
			client.sendSubscription(jid, "subscribed");
		else
			client.sendSubscription(jid, "unsubscribed");
		// TODO: request subscription at the same time
		// TODO: add to roster if not already there
	}
	else if(type == "unsubscribed")
	{
		KMessageBox::information(this, i18n("Subscription to %1 has been revoked.").arg(jid.userHost()));
	}
	else if(type == "subscribed")
	{
		KMessageBox::information(this, i18n("Subscription to %1 has been approved.").arg(jid.userHost()));
	}
}

void LightHawk::MainWindow::addContact(const XMPP::Jid &)
{
	// TODO: addContact()
}

void LightHawk::MainWindow::sendPresence(const XMPP::Jid &, const QString &)
{
	// TODO: sendPresence()
}

XMPP::Jid LightHawk::MainWindow::currentJid(void)
{
	if(client.isActive())
	{
		return client.jid();
	}
	else
	{
		configManager.config().setGroup("");
		return XMPP::Jid(configManager.config().readEntry("jid") + "/" + configManager.config().readEntry("resource"));
	}
}

void LightHawk::MainWindow::sendMessage(const XMPP::Message &m)
{
	logManager.messageOut(m, currentJid());
	client.sendMessage(m);
}

void LightHawk::MainWindow::receiveMessage(const XMPP::Message &m)
{
	MessageWindow *window;
	XMPP::LiveRoster::ConstIterator i = client.roster().find(m.from());
	if(i != client.roster().end())
		window = windowManager.messageWindow(currentJid(), (*i).jid(), m.type());
	else
		window = windowManager.messageWindow(currentJid(), m.from(), m.type());

	if(!window) return;

	window->messageIn(m);
}

void LightHawk::MainWindow::fileQuit(void)
{
	close();
}

void LightHawk::MainWindow::settingsPreferences(void)
{
	if(!configDialog)
	{
		configDialog = new ConfigDialog(configManager, this);
		connect(configDialog, SIGNAL(applyClicked(void)), this, SLOT(applyPreferences(void)));
		connect(configDialog, SIGNAL(okClicked(void)), this, SLOT(applyPreferences(void)));
		connect(configDialog, SIGNAL(finished(void)), this, SLOT(configFinished(void)));
	}

	configDialog->show();
}

void LightHawk::MainWindow::configFinished(void)
{
	configDialog = 0;
}

void LightHawk::MainWindow::settingsKeys(void)
{
	KKeyDialog::configureKeys(actionCollection(), xmlFile());
}

void LightHawk::MainWindow::settingsNotifications(void)
{
	KNotifyDialog::configure(this);
}

void LightHawk::MainWindow::profiles(void)
{
	static_cast<Application *>(kapp)->newInstance();
}

void LightHawk::MainWindow::groupOpenChat(void)
{
	// TODO: groupOpenChat()
}

void LightHawk::MainWindow::rosterCompose(void)
{
	// compare with rosterOpenChat
	XMPP::RosterItem item = listView->rosterItem(listView->selectedItem());
	MessageWindow *window = windowManager.composer(currentJid(), item.jid());

	if(!window) return;

	window->show();
}

void LightHawk::MainWindow::rosterOpenChat(void)
{
	// compare with rosterOpen
	XMPP::RosterItem item = listView->rosterItem(listView->selectedItem());
	MessageWindow *window = windowManager.messageWindow(currentJid(), item.jid(), "chat");

	if(!window) return;

	window->show();
}

void LightHawk::MainWindow::rosterEdit(void)
{
	RosterItemDialog dlg(client, listView->rosterItem(listView->selectedItem()), this);
	dlg.exec();
}

void LightHawk::MainWindow::rosterAdd(void)
{
	KLineEditDlg dlg(i18n("What JID would you like to add to your roster?"), QString::null, this);
	if(dlg.exec())
	{
		// TODO: let user name it
		XMPP::JT_Roster *task = new XMPP::JT_Roster(client.rootTask());
		task->set(dlg.text(), dlg.text(), QStringList());
		task->go(true);
	}
}

void LightHawk::MainWindow::rosterRemove(void)
{
	XMPP::RosterItem item = listView->rosterItem(listView->currentItem());
	QString message(i18n("Are you sure you wish to remove %1 from your roster?"));
	if( KMessageBox::Yes == KMessageBox::questionYesNo(this, message.arg(item.jid().userHost())) )
	{
		XMPP::JT_Roster *task = new XMPP::JT_Roster(client.rootTask());
		task->remove(item.jid());
		task->go(true);
	}
}

void LightHawk::MainWindow::rosterAddAgent(void)
{
	// TODO rosterAddAgent()
}

void LightHawk::MainWindow::rosterAddService(void)
{
	// TODO rosterAddService()
}

void LightHawk::MainWindow::toolsSendXML(void)
{
	KLineEditDlg dlg(i18n("<qt>What XML do you want to send to the server?  This could break your connection so press Cancel if you're not sure of what you are doing.</qt>"), QString::null, this);
	if(dlg.exec())
	{
		client.send(dlg.text());
	}
}

void LightHawk::MainWindow::available(void)
{
	setPresence("available", i18n("Available"));
}

void LightHawk::MainWindow::chat(void)
{
	setPresence("chat", i18n("Free to Chat"));
}

void LightHawk::MainWindow::away(void)
{
	bool ok;
	QString message = AwayMessageDialog::getAwayMessage(configManager.config(),
	                                                    i18n("Away Message"), &ok);

	if(ok)
	{
		configManager.writeData(*KGlobal::config());
		setPresence("away", i18n("Away"), message);
	}
}

void LightHawk::MainWindow::xa(void)
{
	bool ok;
	QString message = AwayMessageDialog::getAwayMessage(configManager.config(),
	                                                    i18n("Extended Away Message"),
	                                                    &ok);

	if(ok)
	{
		configManager.writeData(*KGlobal::config());
		setPresence("xa", i18n("Extended Away"), message);
	}
}

void LightHawk::MainWindow::dnd(void)
{
	bool ok;
	QString message = AwayMessageDialog::getAwayMessage(configManager.config(),
	                                                    i18n("Do Not Disturb Message"),
	                                                    &ok);

	if(ok)
	{
		configManager.writeData(*KGlobal::config());
		setPresence("dnd", i18n("Do not disturb"), message);
	}
}

void LightHawk::MainWindow::invisible(void)
{
	setPresence("invisible", i18n("Invisible"), QString::null, true);
}

void LightHawk::MainWindow::unavailable(void)
{
	setPresence("unavailable", i18n("Unavailable"));
}

#include "mainwindow.moc"
// arch-tag: mainwindow.cpp
