// Copyright (C) 2002 Neil Stevens <neil@hakubi.us>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
// AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name(s) of the author(s) shall not be
// used in advertising or otherwise to promote the sale, use or other dealings
// in this Software without prior written authorization from the author(s).

#include <kdebug.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kparts/part.h>
#include <kparts/componentfactory.h>
#include <kservice.h>
#include <kuserprofile.h>
#include <zinv/message.h>
#include <zinv/types.h>
#include <qwidget.h>

#include "jid.h"
#include <liblighthawk/messagewindow.h>
#include "messagewindowmanager.h"

LightHawk::MessageWindowManager::MessageWindowManager(Data &d, XMPP::Client &client,
                                                      QWidget *p, const char *n)
	: QWidget(p, n)
	, data(d)
	, c(client)
{
	messageWindows.setAutoDelete(true);
	messageWindows.insert("normal", new Map);
	messageWindows.insert("chat", new Map);
	messageWindows.insert("groupchat", new Map);
	messageWindows.insert("headline", new Map);
	messageWindows.insert("error", new Map);
	messageWindows.insert("composer", new Map);
}

LightHawk::MessageWindowManager::~MessageWindowManager()
{
	// close message windows
	for(DictIterator i(messageWindows); i.current(); ++i)
	{
		for(Map::Iterator j = i.current()->begin(); j != i.current()->end(); ++j)
		{
			disconnect(j.data());
			j.data()->close();
		}
	}
}

void LightHawk::MessageWindowManager::messageWindowClosing(MessageWindow *w)
{
	messageWindows[w->type()]->remove(w->peer());
	if(w->type() == "groupchat")
		emit presence(w->peer(), "unavailable");
}

LightHawk::Data &LightHawk::MessageWindowManager::config(void)
{
	return data;
}

XMPP::Client &LightHawk::MessageWindowManager::client(void)
{
	return c;
}

LightHawk::MessageWindow *LightHawk::MessageWindowManager::messageWindow(const XMPP::Jid &from, const XMPP::Jid &to, const QString &t)
{
	QString type = t;
	if(type.isEmpty())
		type = "normal";
	if(type == "composer")
		return 0;
	
	return messageWindowInternal(from, to, type, true);
}

LightHawk::MessageWindow *LightHawk::MessageWindowManager::composer(const XMPP::Jid &from, const XMPP::Jid &to)
{
	return messageWindowInternal(from, to, "composer", false);
}

QString LightHawk::MessageWindowManager::nameForJid(const Jid &jid)
{
	const XMPP::LiveRoster &roster = client().roster();
	XMPP::LiveRoster::ConstIterator rosterIterator = roster.find(jid, false);

	if(rosterIterator != roster.end())
		return (*rosterIterator).name();
	else
		return jid.userHost();
}

bool LightHawk::MessageWindowManager::resourceForJid(const Jid &jid, XMPP::Resource &resource)
{
	const XMPP::LiveRoster &roster = client().roster();
	XMPP::LiveRoster::ConstIterator rosterIterator = roster.find(jid, false);

	if(rosterIterator != roster.end())
	{
		// kdDebug() << "Found " << jid.userHost() << " in roster" << endl;
		const XMPP::ResourceList &list = (*rosterIterator).resourceList();
		XMPP::ResourceList::ConstIterator resourceIterator = list.find(jid.resource());
		if(resourceIterator != list.end())
		{
			resource = (*resourceIterator);
			return true;
		}
	}
	return false;
}

LightHawk::MessageWindow *LightHawk::MessageWindowManager::messageWindowInternal(const XMPP::Jid &from, const XMPP::Jid &to, const QString &type, bool reuse)
{
	//kdDebug() << "Manager: Getting window of type " << type << " for " << to.full() << endl;
	Map &map = (*messageWindows[type]);

	if(reuse)
	{
		// The obvious case
		if(map.contains(to))
			return map[to];

		// OK, now this is tricky.  If the user opens a message to another user, he
		// likely didn't care what resource the other person is using.  So what we
		// do is leave it blank, then fill it in the first time we get a reply
		// from that person.
		XMPP::Jid modifiedTo = to;
		modifiedTo.setResource(QString::null);
		for(Map::Iterator i = map.begin(); i != map.end(); ++i)
		{
			if(i.key() == modifiedTo)
			{
				kdDebug() << "Manager: SETTING window from " << i.key().full() << " to " << to.full() << endl;

				MessageWindow *w = i.data();
				QString toName = to.userHost();
				w->setPeer(to, nameForJid(to));

				XMPP::Resource resource;
				if(resourceForJid(to, resource))
					w->resourceUpdate(to, resource);

				map.remove(i.key());
				map.insert(to, w);
				return w;
			}
		}
	}

	//kdDebug() << "Manager: Creating window for " << to.full() << endl;

	// TODO: overrides
	const QString servicetype = QString("LightHawk/%1").arg(type);
	const QString generic = "LightHawk/MessageWindow";
	KService::Ptr service = KServiceTypeProfile::preferredService(servicetype, generic);
	if(!service)
	{
		kdDebug() << "KTrader failed" << endl;
		KMessageBox::error(0, i18n("Unable to find message window plugin."));
		return 0;
	}

	MessageWindow *w = KParts::ComponentFactory::createInstanceFromService<LightHawk::MessageWindow>(service, this);
	if(!w)
	{
		kdDebug() << "ComponentFactory failed" << endl;
		KMessageBox::error(0, i18n("Unable to load message window plugin."));
		return 0;
	}

	connect(w, SIGNAL(closing(MessageWindow *)), this, SLOT(messageWindowClosing(MessageWindow *)));
	connect(w, SIGNAL(messageOut(const Message &)), this, SIGNAL(message(const Message &)));
	connect(w, SIGNAL(addContact(const Jid &)), this, SIGNAL(addContact(const Jid &)));
	connect(this, SIGNAL(applyPreferences(void)), w, SLOT(applyPreferences(void)));
	if(reuse)
		map.insert(to, w);

	QString fromName = from.userHost();
	QString toName = to.userHost();
	w->setMyself(from, nameForJid(from));
	w->setPeer(to, nameForJid(to));
	w->setType(type);

	XMPP::Resource resource;

	if(resourceForJid(from, resource))
		w->resourceUpdate(from, resource);

	if(resourceForJid(to, resource))
		w->resourceUpdate(to, resource);

	w->show();

	return w;
}

void LightHawk::MessageWindowManager::resourceAvailable(const Jid &jid, const Resource &resource)
{
	Jid search(jid.full());
	search.setResource(resource.name());
	// kdDebug() << "MessageWindowManager: " << search.full() << " available" << endl;

	for(DictIterator i(messageWindows); *i; ++i)
	{
		// TODO: special handling for groupchat
		// or just send all resources to all jids that match
		if((**i).contains(search))
		{
			MessageWindow *window = (**i)[search];
			window->resourceUpdate(jid, resource);
		}
	}
}

void LightHawk::MessageWindowManager::resourceUnavailable(const Jid &jid, const Resource &resource)
{
	Jid search(jid.full());
	search.setResource(resource.name());
	// kdDebug() << "MessageWindowManager: " << search.full() << " unavailable" << endl;

	for(DictIterator i(messageWindows); *i; ++i)
	{
		// TODO: special handling for groupchat
		// or just send all resources to all jids that match
		if((**i).contains(search))
		{
			MessageWindow *window = (**i)[search];
			window->resourceUpdate(jid, resource);
		}
	}
}

#include "messagewindowmanager.moc"
// arch-tag: messagewindowmanager.cpp
