// Copyright (C) 2002 Neil Stevens <neil@hakubi.us>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
// AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name(s) of the author(s) shall not be
// used in advertising or otherwise to promote the sale, use or other dealings
// in this Software without prior written authorization from the author(s).

#include <kdebug.h>
#include <keditcl.h>
#include <kfontdialog.h>
#include <kiconloader.h>
#include <kglobal.h>
#include <kglobalsettings.h>
#include <klineedit.h>
#include <klocale.h>
#include <krestrictedline.h>
#include <qcheckbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qvbox.h>
#include "pages.h"

LightHawk::IdentityPage::IdentityPage(QWidget *parent, const char *name)
	: ConfigPage(parent, name)
{
	profileEdit = new KLineEdit(this);
	jidEdit = new KLineEdit(this);
	hostEdit = new KLineEdit(this);
	portEdit = new KLineEdit(this);
	resourceEdit = new KLineEdit(this);
	priorityEdit = new KRestrictedLine(this);
	priorityEdit->setValidChars("0123456789");
	passwordEdit = new KLineEdit(this);
	passwordEdit->setEchoMode(QLineEdit::Password);
	savePasswordBox = new QCheckBox(i18n("Save Password"), this);
	autoAvailableBox = new QCheckBox(i18n("Go available automatically"), this);

	QVBoxLayout *layout = new QVBoxLayout(this, 0, KDialog::spacingHint());
	QHBoxLayout *profileLayout = new QHBoxLayout(0, 0, KDialog::spacingHint());
	QHBoxLayout *jidLayout = new QHBoxLayout(0, 0, KDialog::spacingHint());
	QHBoxLayout *hostLayout = new QHBoxLayout(0, 0, KDialog::spacingHint());
	QHBoxLayout *resourceLayout = new QHBoxLayout(0, 0, KDialog::spacingHint());
	QHBoxLayout *passwordLayout = new QHBoxLayout(0, 0, KDialog::spacingHint());
	layout->addLayout(profileLayout);
	layout->addLayout(jidLayout);
	layout->addLayout(hostLayout);
	layout->addLayout(resourceLayout);
	layout->addLayout(passwordLayout);
	layout->addWidget(savePasswordBox);
	layout->addWidget(autoAvailableBox);
	layout->addStretch();
	profileLayout->addWidget(new QLabel(i18n("Profile:"), this));
	profileLayout->addWidget(profileEdit);
	jidLayout->addWidget(new QLabel(i18n("JID:"), this));
	jidLayout->addWidget(jidEdit);
	hostLayout->addWidget(new QLabel(i18n("Host:"), this));
	hostLayout->addWidget(hostEdit);
	hostLayout->addWidget(new QLabel(i18n("Port:"), this));
	hostLayout->addWidget(portEdit);
	resourceLayout->addWidget(new QLabel(i18n("Resource:"), this));
	resourceLayout->addWidget(resourceEdit);
	resourceLayout->addWidget(new QLabel(i18n("Priority:"), this));
	resourceLayout->addWidget(priorityEdit);
	passwordLayout->addWidget(new QLabel(i18n("Password:"), this));
	passwordLayout->addWidget(passwordEdit);

	connect(savePasswordBox, SIGNAL(toggled(bool)), this, SLOT(updateSavePassword()));
}

void LightHawk::IdentityPage::apply(Data &data)
{
	data.setGroup("");
	profileEdit->setText(data.readEntry("profileName", i18n("New Profile")));
	// TODO: default it to the profile name if it's a jid
	jidEdit->setText(data.readEntry("jid", "me@localhost"));
	hostEdit->setText(data.readEntry("host", "localhost"));
	portEdit->setText(QString::number(data.readNumEntry("port", 5222)));
	resourceEdit->setText(data.readEntry("resource", "Light Hawk"));
	priorityEdit->setText(QString::number(data.readNumEntry("priority", 6)));
	passwordEdit->setText(data.readEntry("password"));
	savePasswordBox->setChecked(data.readBoolEntry("savePassword", true));
	autoAvailableBox->setChecked(data.readBoolEntry("autoAvailable", false));
	updateSavePassword();
}

void LightHawk::IdentityPage::createResult(Data &data)
{
	data.setGroup("");
	data.writeEntry("profileName", profileEdit->text());
	data.writeEntry("jid", jidEdit->text());
	data.writeEntry("host", hostEdit->text());
	data.writeEntry("port", portEdit->text().toInt());
	data.writeEntry("resource", resourceEdit->text());
	data.writeEntry("priority", priorityEdit->text().toInt());
	data.writeEntry("password", savePasswordBox->isChecked() ? passwordEdit->text() : QString::null);
	data.writeEntry("savePassword", savePasswordBox->isChecked());
	data.writeEntry("autoAvailable", autoAvailableBox->isChecked());
}

QString LightHawk::IdentityPage::pageName(void)
{
	return i18n("Identity");
}

QString LightHawk::IdentityPage::pageHeader(void)
{
	return i18n("Identity");
}

QPixmap LightHawk::IdentityPage::pagePixmap(void)
{
	return DesktopIcon("identity");
}

void LightHawk::IdentityPage::updateSavePassword(void)
{
	if(savePasswordBox->isChecked())
	{
		passwordEdit->setEnabled(true);
	}
	else
	{
		passwordEdit->setEnabled(false);
		passwordEdit->clear();
	}
}

LightHawk::MessagePage::MessagePage(QWidget *parent, const char *name)
	: ConfigPage(parent, name)
{
	loggingBox = new QCheckBox(i18n("Log all messages"), this);
	timestampingBox = new QCheckBox(i18n("Show timestamps"), this);
	composerFontLabel = new QLabel(i18n("Composer Font"), this);
	viewerFontLabel = new QLabel(i18n("Viewer Font"), this);
	fixedViewerFontLabel = new QLabel(i18n("Fixed Viewer Font"), this);

	QPushButton *composerFontButton = new QPushButton(i18n("Change..."), this);
	connect(composerFontButton, SIGNAL(clicked()), this, SLOT(selectComposerFont()));
	QPushButton *viewerFontButton = new QPushButton(i18n("Change..."), this);
	connect(viewerFontButton, SIGNAL(clicked()), this, SLOT(selectViewerFont()));
	QPushButton *fixedViewerFontButton = new QPushButton(i18n("Change..."), this);
	connect(fixedViewerFontButton, SIGNAL(clicked()), this, SLOT(selectFixedViewerFont()));

	QHBoxLayout *composerLayout = new QHBoxLayout(0, 0, KDialog::spacingHint());
	composerLayout->addWidget(composerFontLabel, 1);
	composerLayout->addWidget(composerFontButton, 1);

	QHBoxLayout *viewerLayout = new QHBoxLayout(0, 0, KDialog::spacingHint());
	viewerLayout->addWidget(viewerFontLabel, 1);
	viewerLayout->addWidget(viewerFontButton, 1);

	QHBoxLayout *fixedViewerLayout = new QHBoxLayout(0, 0, KDialog::spacingHint());
	fixedViewerLayout->addWidget(fixedViewerFontLabel, 1);
	fixedViewerLayout->addWidget(fixedViewerFontButton, 1);

	QVBoxLayout *layout = new QVBoxLayout(this, 0, KDialog::spacingHint());
	layout->addWidget(loggingBox);
	layout->addWidget(timestampingBox);
	layout->addLayout(composerLayout);
	layout->addLayout(viewerLayout);
	layout->addLayout(fixedViewerLayout);
	layout->addStretch();
}

void LightHawk::MessagePage::apply(Data &data)
{
	data.setGroup("Message");
	loggingBox->setChecked(data.readBoolEntry("logging", true));
	timestampingBox->setChecked(data.readBoolEntry("timestamping", true));

	QFont font = KGlobalSettings::generalFont();
	composerFontLabel->setFont(data.readFontEntry("composerFont", &font));
	viewerFontLabel->setFont(data.readFontEntry("viewerFont", &font));

	font = KGlobalSettings::fixedFont();
	fixedViewerFontLabel->setFont(data.readFontEntry("fixedViewerFont", &font));
}

void LightHawk::MessagePage::createResult(Data &data)
{
	data.setGroup("Message");
	data.writeEntry("logging", loggingBox->isChecked());
	data.writeEntry("timestamping", timestampingBox->isChecked());
	data.writeEntry("composerFont", composerFontLabel->font());
	data.writeEntry("viewerFont", viewerFontLabel->font());
	data.writeEntry("fixedViewerFont", fixedViewerFontLabel->font());
}

QString LightHawk::MessagePage::pageName(void)
{
	return i18n("Messages");
}

QString LightHawk::MessagePage::pageHeader(void)
{
	return i18n("Messages");
}

QPixmap LightHawk::MessagePage::pagePixmap(void)
{
	return DesktopIcon("message");
}

void LightHawk::MessagePage::selectComposerFont(void)
{
	QFont font = composerFontLabel->font();
	if(KFontDialog::getFont(font))
		composerFontLabel->setFont(font);
}

void LightHawk::MessagePage::selectViewerFont(void)
{
	QFont font = viewerFontLabel->font();
	if(KFontDialog::getFont(font))
		viewerFontLabel->setFont(font);
}

void LightHawk::MessagePage::selectFixedViewerFont(void)
{
	QFont font = fixedViewerFontLabel->font();
	if(KFontDialog::getFont(font, true))
		fixedViewerFontLabel->setFont(font);
}

LightHawk::RosterPage::RosterPage(QWidget *parent, const char *name)
	: ConfigPage(parent, name)
{
	onlineOnlyBox = new QCheckBox(i18n("Show only online contacts"), this);

	QVBoxLayout *layout = new QVBoxLayout(this, 0, KDialog::spacingHint());
	layout->addWidget(onlineOnlyBox);
	layout->addStretch();
}

void LightHawk::RosterPage::apply(Data &data)
{
	data.setGroup("Roster");
	onlineOnlyBox->setChecked(data.readBoolEntry("showOnlineOnly", false));
}

void LightHawk::RosterPage::createResult(Data &data)
{
	data.setGroup("Roster");
	data.writeEntry("showOnlineOnly", onlineOnlyBox->isChecked());
}

QString LightHawk::RosterPage::pageName(void)
{
	return i18n("Roster");
}

QString LightHawk::RosterPage::pageHeader(void)
{
	return i18n("Roster");
}

QPixmap LightHawk::RosterPage::pagePixmap(void)
{
	return DesktopIcon("list");
}

#include "pages.moc"
// arch-tag: pages.cpp
