// Copyright (C) 2002 Neil Stevens <neil@hakubi.us>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
// AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name(s) of the author(s) shall not be
// used in advertising or otherwise to promote the sale, use or other dealings
// in this Software without prior written authorization from the author(s).

#include <kaction.h>
#include <kapplication.h>
#include <kdebug.h>
#include <kdialog.h>
#include <kgenericfactory.h>
#include <kglobal.h>
#include <kglobalsettings.h>
#include <khtml_part.h>
#include <khtmlview.h>
#include <kiconloader.h>
#include <kio/global.h>
#include <kio/job.h>
#include <kkeydialog.h>
#include <klocale.h>
#include <kmenubar.h>
#include <kmessagebox.h>
#include <kparts/plugin.h>
#include <knotifyclient.h>
#include <kpopupmenu.h>
#include <kstandarddirs.h>
#include <kstdaction.h>
#include <kstatusbar.h>
#include <kstringhandler.h>
#include <ktextedit.h>
#include <liblighthawk/utils.h>
#include <qfile.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qregexp.h>
#include <qtextcodec.h>
#include <qtextstream.h>
#include <qvbox.h>

#include "messagewindow.h"

K_EXPORT_COMPONENT_FACTORY(lighthawk_kitchat, KGenericFactory<KitChat::MessageWindow>);

KitChat::MessageWindow::MessageWindow(QWidget *parent, const char *name, const QStringList &args)
	: LightHawk::MessageWindow(parent, name, args)
	, lineBreak(false)
	, useSound(true)
	, useFixedFont(false)
	, receivedResource(false)
{
	config().setGroup("Message");
	isTimestamping = config().readBoolEntry("timestamping");

	// Define Actions
	aClear = new KAction(i18n("C&lear"), 0, this, SLOT(clear(void)), actionCollection(), "clear");
	aClose = KStdAction::close(this, SLOT(close(void)), actionCollection());
	KStdAction::keyBindings(this, SLOT(settingsKeys()), actionCollection());
	aTimestamping = new KToggleAction(i18n("&Timestamping"), 0, this, SLOT(timestampingPressed(void)), actionCollection(), "timestamping");
	aSound = new KToggleAction(i18n("&Sound"), 0, this, SLOT(soundPressed(void)), actionCollection(), "sound");
	aFixedFont = new KToggleAction(i18n("Use &Fixed Font"), 0, this, SLOT(fixedFontPressed(void)), actionCollection(), "fixedfont");

	aTimestamping->setChecked(isTimestamping);
	aSound->setChecked(useSound);
	aSound->setChecked(useFixedFont);

	statusText = new QLabel(statusBar());
	statusIcon = new QLabel(statusBar());
	statusBar()->addWidget(statusText, 0, true);
	statusBar()->addWidget(statusIcon, 0, true);
	statusBar()->show();

	// the stuff in the middle
	centralWidget = new QFrame(this);
	QVBoxLayout *layout = new QVBoxLayout(centralWidget);
	layout->setMargin(0);
	layout->setSpacing(KDialog::spacingHint());
	setCentralWidget(centralWidget);

	htmlpart = new KHTMLPart(centralWidget, 0, this);
	connect(htmlpart->browserExtension(), SIGNAL(openURLRequest(const KURL &, const KParts::URLArgs &)), this, SLOT(urlClicked(const KURL &)));
	htmlpart->setEncoding("utf8", true);
	htmlpart->setAutoloadImages(false);
	htmlpart->setJScriptEnabled(false);
	htmlpart->setJavaEnabled(false);
	htmlpart->setPluginsEnabled(false);
	htmlpart->show();
	clear();

	setXMLFile("kitchatui.rc");
	createGUI(htmlpart);

	edit = new KTextEdit(centralWidget);
	connect(edit, SIGNAL(returnPressed()), this, SLOT(send()) );

	layout->addWidget(htmlpart->view(), 20, 0);
	layout->addWidget(edit, 5, 0);

	centralWidget->setMinimumHeight(KDialog::marginHint() * 2 + KDialog::spacingHint() + htmlpart->view()->minimumHeight() + edit->minimumHeight());
	resize(600, 300);
	
	XMPP::Status status("unavailable", "unavailable");
	statusIcon->setPixmap(LightHawk::trayIconForStatus(status));
	setIcon(LightHawk::smallIconForStatus(status));

	statusText->setText(QString::null);

	setFocusProxy(edit);
	htmlpart->view()->viewport()->setFocusProxy(edit);

	edit->installEventFilter(this);
	htmlpart->view()->installEventFilter(this);

	applyPreferences();
}

KitChat::MessageWindow::~MessageWindow(void)
{
}

void KitChat::MessageWindow::applyPreferences(void)
{
	QFont font;
	LightHawk::Data &data = config();
	data.setGroup("Message");
	font = data.readFontEntry("composerFont", &font);
	edit->setFont(font);

	initializeHTML();
}

void KitChat::MessageWindow::messageIn(const Message &message)
{
	if(useSound)
		KNotifyClient::event("Message Received");
	//kdDebug() << "messageIn: from: " << message.from().full() << " to: " << message.from().full() << endl;
	updateHTML(message);
}

void KitChat::MessageWindow::resourceUpdate(const Jid &jid, const Resource &resource)
{
	if(resource.name() != hisJid.resource()) return;

	kdDebug() << "update status: " << resource.status().status() << " show: " << resource.status().show() << endl;
	statusIcon->setPixmap(LightHawk::trayIconForStatus(resource.status()));
	statusText->setText(resource.status().status());
	setIcon(LightHawk::smallIconForStatus(resource.status()));

	if(!receivedResource)
	{
		receivedResource = true;
		lastResourceAvailable = resource.status().isAvailable();;
		return;
	}

	if(lastResourceAvailable == resource.status().isAvailable()) return;
	lastResourceAvailable = resource.status().isAvailable();

	// add HTML wrapping paper to the message
	QString html = "<div>";
	html += QString("<p><span class=\"timestamp\">%1</span> ")
	        .arg(LightHawk::htmlEscape(KGlobal::locale()->formatDateTime(QDateTime::currentDateTime())));
	// kdDebug() << "updateHTML: from: " << message.from().full() << "  myJid: " << myJid.full() << endl;
	Jid from = jid;
	from.setResource(resource.name());

	if(resource.status().isAvailable())
	{
		html += QString("%2 is now available.")
		        .arg(LightHawk::htmlEscape(from.full()));
	}
	else
	{
		html += QString("%2 is not available anymore.")
		        .arg(LightHawk::htmlEscape(from.full()));
	}
	html += QString("</p></div>");
	conversationBuffer += html;
	htmlpart->write(html);

	// scroll to the new text, and beat khtmlpart until it cooperates
	// TODO: merge with updateHTML
	KHTMLView *view = htmlpart->view();
	view->layout();
	view->ensureVisible(0, view->contentsHeight());
	view->updateContents(view->contentsX(), view->contentsY(),
	                     view->contentsWidth(), view->contentsHeight());
}

void KitChat::MessageWindow::jidChanged(void)
{
	initializeHTML();
}

void KitChat::MessageWindow::typeChanged(void)
{
}

void KitChat::MessageWindow::setTimestamping(bool stamp)
{
	isTimestamping = stamp;
	aTimestamping->setChecked(isTimestamping);
}

void KitChat::MessageWindow::setUsingFixedFont(bool ff)
{
	useFixedFont = ff;
	aFixedFont->setChecked(useFixedFont);
}

void KitChat::MessageWindow::resizeEvent(QResizeEvent *)
{
	if(htmlpart && htmlpart->view())
		htmlpart->view()->ensureVisible(0, htmlpart->view()->contentsHeight());
}

void KitChat::MessageWindow::keyPressEvent(QKeyEvent *e)
{
	if(e->key() == Qt::Key_Shift)
	{
		edit->grabKeyboard();
		lineBreak = true;
	}
}

void KitChat::MessageWindow::keyReleaseEvent(QKeyEvent *e)
{
	if(e->key() == Qt::Key_Shift)
	{
		edit->releaseKeyboard();
		lineBreak = false;
	}
}

bool KitChat::MessageWindow::eventFilter(QObject *, QEvent *e)
{
	if(e->type() == QEvent::KeyPress)
	{
		QKeyEvent &event = *static_cast<QKeyEvent *>(e);
		if(event.state() == Qt::ShiftButton && event.key() == Qt::Key_PageUp)
		{
			htmlpart->view()->scrollBy(0, -(htmlpart->view()->visibleHeight() * 8 / 10));
			return true;
		}
		else if(event.state() == Qt::ShiftButton && event.key() == Qt::Key_PageDown)
		{
			htmlpart->view()->scrollBy(0, htmlpart->view()->visibleHeight() * 8 / 10);
			return true;
		}
	}
	return false;
}

void KitChat::MessageWindow::send(void)
{
	// Don't send on these
	if(lineBreak) return;

	edit->doKeyboardAction(QTextEdit::ActionBackspace);
	QString text = edit->text();

	// don't send empty texts
	if(text.stripWhiteSpace().isEmpty())
	{
		edit->clear();
		return;
	}

	edit->clear();

	text.replace(QRegExp("\n$"), "");

	Message message;
	message.setTo(hisJid);
	message.setBody(text);
	message.setType(replyMessageType);
	message.setTimeStamp(QDateTime::currentDateTime());
	//kdDebug() << "sendMessage: to: " << message.from().full() << "  myJid: " << myJid.full() << endl;
	emit messageOut(message);
	updateHTML(message, myJid);

	// press on
	if(useSound)
		KNotifyClient::event("Message Sent");
}

void KitChat::MessageWindow::clear(void)
{
	conversationBuffer = QString::null;
	initializeHTML();
}

void KitChat::MessageWindow::timestampingPressed(void)
{
	setTimestamping(!isTimestamping);
	initializeHTML();
}

void KitChat::MessageWindow::fixedFontPressed(void)
{
	setUsingFixedFont(!useFixedFont);
	initializeHTML();
}

void KitChat::MessageWindow::soundPressed(void)
{
	useSound = !useSound;
}

void KitChat::MessageWindow::initializeHTML(void)
{
	htmlpart->begin();
	htmlpart->write(QString("<html><head><style type=\"text/css\">"));

	// Read stylesheet into the document
	QString stylesheet = locate("data", "lighthawk/kitchat/stylesheet");
	QFile file(stylesheet);
	file.open(IO_ReadOnly);
	QTextStream stream(&file);
	stream.setEncoding(QTextStream::Latin1);
	QString sheet = stream.read();

	QFont font;
	config().setGroup("Message");
	if(useFixedFont)
		font = config().readFontEntry("fixedViewerFont", &font);
	else
		font = config().readFontEntry("viewerFont", &font);
	QString fontString = LightHawk::htmlFont(font);
	sheet.replace("font: MAIN;\n", fontString);
	if(isTimestamping)
		sheet.replace("TIMESTAMP;\n", QString::null);
	else
		sheet.replace("TIMESTAMP;\n", "display: none;\n");
	htmlpart->write(sheet);
	htmlpart->write(QString("</style></head><title>%1</title><body>").arg(hisJid.userHost()));
	file.close();

	// write the buffer, if any
	if(conversationBuffer.length())
		htmlpart->write(conversationBuffer);

	KHTMLView *view = htmlpart->view();
	view->layout();
	view->ensureVisible(0, view->contentsHeight());
	view->updateContents(view->contentsX(), view->contentsY(),
	                     view->contentsWidth(), view->contentsHeight());
}

void KitChat::MessageWindow::updateHTML(const Message &message, const Jid &jid)
{
	// add HTML wrapping paper to the message
	QString html = "<div><p>";
	html += QString("<span class=\"timestamp\">%1</span> ")
	        .arg(LightHawk::htmlEscape(KGlobal::locale()->formatDateTime(message.timeStamp())));
	// kdDebug() << "updateHTML: from: " << message.from().full() << "  myJid: " << myJid.full() << endl;
	Jid from;
	if(!jid.isValid())
		from = message.from().full();
	else
		from = jid.full();
	html += QString("<span class=\"%1\">%2:</span> ")
	        .arg(LightHawk::htmlEscape(from.compare(myJid, true) ? "sender-me" : "sender-notme"))
	        .arg(LightHawk::htmlEscape(from.full()));
	// TODO: use rich version if available
	QString body = LightHawk::htmlEscape(message.body(false));
	if(body.contains('\n') > 0)
		body.prepend('\n');

	body.replace('\n', "<br/>");
	body = KStringHandler::tagURLs(body);
	html += body;
	html += QString("</p></div>");
	conversationBuffer += html;
	htmlpart->write(html);
	// scroll to the new text, and beat khtmlpart until it cooperates
	KHTMLView *view = htmlpart->view();
	view->layout();
	view->ensureVisible(0, view->contentsHeight());
	view->updateContents(view->contentsX(), view->contentsY(),
	                     view->contentsWidth(), view->contentsHeight());
}

void KitChat::MessageWindow::urlClicked(const KURL &url)
{
	kapp->invokeBrowser(url.url());
}

void KitChat::MessageWindow::insertText(const QString &string)
{
	edit->insert(string);
}

void KitChat::MessageWindow::settingsKeys(void)
{
	KKeyDialog dialog(true, this);
	dialog.insert(actionCollection());
	for(QPtrListIterator<KXMLGUIClient> i(*childClients()); *i; ++i)
		dialog.insert((*i)->actionCollection());
	dialog.configure();
}

void KitChat::MessageWindow::slotSetStatusBarText(const QString &)
{
	// Swallow it since khtml is too verbose for me
}

#include "messagewindow.moc"
// arch-tag: plugins/kitchat/messagewindow.cpp
