// Copyright (C) 2002 Neil Stevens <neil@hakubi.us>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
// AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name(s) of the author(s) shall not be
// used in advertising or otherwise to promote the sale, use or other dealings
// in this Software without prior written authorization from the author(s).

#include <kapplication.h>
#include <kbuttonbox.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kdialog.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <klineeditdlg.h>
#include <klistview.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qregexp.h>
#include <qstringlist.h>

#include "configmanager.h"
#include "mainwindow.h"
#include "profiledialog.h"

QString LightHawk::ProfileDialog::defaultProfile(void)
{
	KConfig &config = *KGlobal::config();
	config.setGroup("Profiles");
	return config.readEntry("Default Profile");
};

LightHawk::ProfileDialog::ProfileDialog(QWidget *parent, const char *name)
	: KDialogBase(parent, name, true, i18n("Account Profiles"), KDialogBase::Cancel)
{
	QFrame *mainBox = new QFrame(this);
	setMainWidget(mainBox);

	profileList = new KListView(mainBox);
	profileList->addColumn(i18n("Profile"));
	connect(profileList, SIGNAL(selectionChanged(QListViewItem *)), this, SLOT(selectionChanged(QListViewItem *)));
	KButtonBox *buttonBox = new KButtonBox(mainBox, Qt::Vertical);
	openButton = buttonBox->addButton(i18n("Open"));
	setButton = buttonBox->addButton(i18n("Set Default"));
	addButton = buttonBox->addButton(i18n("Add New..."));
	deleteButton = buttonBox->addButton(i18n("Delete"));
	buttonBox->layout();

	QHBoxLayout *layout = new QHBoxLayout(mainBox, 0, KDialog::spacingHint());
	layout->addWidget(profileList, 1);
	layout->addWidget(buttonBox, 0);

	connect(openButton, SIGNAL(clicked()), this, SLOT(openClicked()) );
	connect(setButton, SIGNAL(clicked()), this, SLOT(setClicked()) );
	connect(addButton, SIGNAL(clicked()), this, SLOT(addClicked()) );
	connect(deleteButton, SIGNAL(clicked()), this, SLOT(deleteClicked()) );
	
	selectionChanged(0);
	updateList();

	// Yes, I set nostalgic sizes
	setMinimumSize(320, 240);
}

const QString &LightHawk::ProfileDialog::profileSelected(void)
{
	return items[profileList->selectedItem()];
}

void LightHawk::ProfileDialog::openClicked(void)
{
	// if there is just one profile set it as default
	if(!profileList->firstChild()->nextSibling())
	{
		profileList->setSelected(profileList->firstChild(), true);
		setClicked();
	}
	accept();
}

void LightHawk::ProfileDialog::setClicked(void)
{
	KConfig &config = *KGlobal::config();
	config.setGroup("Profiles");
	config.writeEntry("Default Profile", profileSelected());
	config.sync();
	updateList();
}

void LightHawk::ProfileDialog::addClicked(void)
{
	KLineEditDlg lineDialog(i18n("Enter a name for the profile."), i18n("Add name here"), this);
	if(lineDialog.exec() == KLineEditDlg::Accepted && lineDialog.text().length())
	{
		QString name = lineDialog.text();

		// Get next available profile
		KConfig &config = *KGlobal::config();
		int i = -1;
		QString group;
		do
		{
			++i;
			group = ConfigManager::groupForProfile(QString::number(i));
		} while(config.groupList().contains(group));

		// save it
		config.setGroup(group);
		config.writeEntry("profileName", name);
		config.sync();

		updateList();
	}
}

void LightHawk::ProfileDialog::deleteClicked(void)
{
	QListViewItem *item = profileList->selectedItem();
	QString profile = items[item];
	QString message = i18n("Are you sure you want to remove profile \"%1\"?")
	                  .arg(item->text(0));

	if(KMessageBox::Yes == KMessageBox::warningYesNo(this, message))
	{
		MainWindow::closeWindowsForProfile(profile);

		KConfig &config = *KGlobal::config();

		// Unset as default if it's the default
		if(defaultProfile() == profile)
		{
			config.setGroup("Profiles");
			config.writeEntry("Default Profile", QString::null);
		}

		// now to delete the thing itself
		const QString group = ConfigManager::groupForProfile(profile);
		config.setGroup(group);
		const QMap<QString, QString> map = config.entryMap(group);
		for(QMapConstIterator<QString, QString> i = map.begin(); i != map.end(); ++i)
			config.writeEntry(i.key(), QString::null);

		config.sync();
	}
	updateList();
}

void LightHawk::ProfileDialog::updateList(void)
{
	const QString selected = items[profileList->selectedItem()];

	profileList->clear();
	items.clear();
	KConfig &config = *KGlobal::config();

	const QStringList groups = config.groupList();
	const QRegExp prefix = QString("^Profile - ");
	for(QStringList::ConstIterator i = groups.begin(); i != groups.end(); ++i)
	{
		const QString group = *i;
		config.setGroup(group);
		const QString name = config.readEntry("profileName");

		if((group.find(prefix) == 0) && (name.length()))
		{
			QString profile = group;
			profile.replace(prefix, "");

			KListViewItem *item = new KListViewItem(profileList, name);

			if(profile == defaultProfile())
				item->setPixmap(0, SmallIcon("defaultprofile"));
			else
				item->setPixmap(0, SmallIcon("available"));

			// clear() destroys the selection, so we restore it here
			if(profile == selected)
				profileList->setSelected(item, true);

			items[item] = profile;
		}
	}
	selectionChanged(profileList->selectedItem());
}

void LightHawk::ProfileDialog::selectionChanged(QListViewItem *item)
{
	// Only let them open when one is selected to open
	openButton->setEnabled(item != 0);
	// Only let them Set Default when a non-default one is selected
	setButton->setEnabled(item && items[item] != defaultProfile());
	// Don't let them delete the last one
	deleteButton->setEnabled(profileList->firstChild() && profileList->firstChild()->nextSibling());
}

#include "profiledialog.moc"
// arch-tag: profiledialog.cpp
