// Copyright (C) 2002 Neil Stevens <neil@hakubi.us>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
// AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name(s) of the author(s) shall not be
// used in advertising or otherwise to promote the sale, use or other dealings
// in this Software without prior written authorization from the author(s).

#include <kdebug.h>
#include <kiconloader.h>
#include <klocale.h>
#include <knotifyclient.h>
#include <liblighthawk/configpage.h>
#include <liblighthawk/utils.h>
#include <qheader.h>
#include <zinv/client.h>
#include <zinv/stream.h>
#include <zinv/types.h>
#include "rosterlistview.h"

LightHawk::RosterListView::RosterListView(Data &d, XMPP::Client &c, QWidget *parent, const char *name)
	: KListView(parent, name)
	, client(c)
	, data(d)
	, showOnlineOnly(false)
{
	setAllColumnsShowFocus(true);
	setRootIsDecorated(true);

	connect(this, SIGNAL(contextMenu(KListView *, QListViewItem *, const QPoint &)), this, SLOT(passOnContextMenu(KListView *, QListViewItem *, const QPoint &)));
	connect(this, SIGNAL(executed(QListViewItem *)), this, SLOT(passOnExecuted(QListViewItem *)));

	connect(&client, SIGNAL(rosterItemAdded(const RosterItem &)), this, SLOT(rosterItemAdded(const RosterItem &)));
	connect(&client, SIGNAL(rosterItemRemoved(const RosterItem &)), this, SLOT(rosterItemRemoved(const RosterItem &)));
	connect(&client, SIGNAL(rosterItemUpdated(const RosterItem &)), this, SLOT(rosterItemUpdated(const RosterItem &)));
	connect(&client, SIGNAL(resourceAvailable(const Jid &, const Resource &)), this, SLOT(resourceAvailable(const Jid &, const Resource &)));
	connect(&client, SIGNAL(resourceUnavailable(const Jid &, const Resource &)), this, SLOT(resourceUnavailable(const Jid &, const Resource &)));

	connect(header(), SIGNAL(indexChange(int, int, int)), this, SLOT(saveCurrentLayout(void)));
	connect(header(), SIGNAL(clicked(int)), this, SLOT(saveCurrentLayout(void)));
}

void LightHawk::RosterListView::setOnlineOnly(bool b)
{
	if(b != showOnlineOnly)
	{
		showOnlineOnly = b;
		fill();
	}
}

bool LightHawk::RosterListView::onlineOnly(void)
{
	 return showOnlineOnly;
}

XMPP::RosterItem LightHawk::RosterListView::rosterItem(QListViewItem *i)
{
	return itemMap[i];
}

void LightHawk::RosterListView::clear(void)
{
	setSelected(selectedItem(), false);

	KListView::clear();
	itemMap.clear();

	// remove all columns
	while(header()->count() > 0)
		removeColumn(0);
}

void LightHawk::RosterListView::saveCurrentLayout(void)
{
	if(header()->count())
		saveLayout(reinterpret_cast<KConfig *>(&data), "RosterListView");
}

void LightHawk::RosterListView::fill(void)
{
	clear();
	if(!client.isAuthenticated()) return;

	addColumn(i18n("Roster"), -1);
	addColumn(i18n("ID"), -1);
	addColumn(i18n("Resource"), -1);
	addColumn(i18n("Status"), -1);

	// get convenient reference to information
	const XMPP::LiveRoster &roster = client.roster();

	// walk the list of roster members
	for(XMPP::LiveRoster::ConstIterator i = roster.begin(); i != roster.end(); ++i)
		rosterItemAdded(*i);

	restoreLayout(reinterpret_cast<KConfig *>(&data), "RosterListView");
	header()->adjustHeaderSize();
}

QListViewItem *LightHawk::RosterListView::findTreeGroup(const QString &name)
{
	for(QListViewItemIterator i(this); i.current(); i++)
	{
		// TODO: check case sensitivity
		if(i.current()->text(0) == name && !i.current()->parent()) return i.current();
	}

	// group not found, so add it
	KListViewItem *groupItem = new KListViewItem(this, name);
	groupItem->setOpen(true);
	groupItem->setSelectable(false);
	groupItem->setExpandable(true);
	return groupItem;
}

namespace
{
QIconSet statusIconForResourceIterator(const XMPP::ResourceList &list, const XMPP::ResourceList::ConstIterator &resourceIterator)
{
	XMPP::Status status = (*resourceIterator).status();
	// kdDebug() << "Getting icon for " << status.show() << endl;
	if(resourceIterator == list.end())
		return LightHawk::smallIconForStatus(XMPP::Status("unavailable", "unavailable"));
	else
		return LightHawk::smallIconForStatus(status);
}
}

void LightHawk::RosterListView::rosterItemAdded(const XMPP::RosterItem &item)
{
	QStringList groups = item.groups();
	if(!groups.size())
	{
		groups << i18n("Unfiled");
	}

	XMPP::LiveRoster::ConstIterator liveItemIterator = client.roster().find(item.jid());
	if(liveItemIterator == client.roster().end())
	{
		kdDebug() << "No item found for " << item.jid().full() << endl;
		return;
	}

	XMPP::LiveRosterItem liveItem = (*liveItemIterator);
	const XMPP::ResourceList resourceList = liveItem.resourceList();

	if(showOnlineOnly && !resourceList.count())
		return;

	/*
	{
	XMPP::Resource resource = *resourceList.priority();
	kdDebug() << "Status of " << item.jid().full() << ":"  << resource.status().status() << endl;
	kdDebug() << "\t priority " << resource.status().priority() << endl;
	kdDebug() << "\t show " << resource.status().show() << endl;
	kdDebug() << "\t timeStamp " << resource.status().timeStamp().toString() << endl;
	kdDebug() << "\t available " << resource.status().isAvailable() << endl;
	kdDebug() << "\t away " << resource.status().isAway() << endl;
	kdDebug() << "\t invisible " << resource.status().isInvisible() << endl;
	kdDebug() << "\t hasError " << resource.status().hasError() << endl;
	}
	*/

	// set main status icon using the priority resource
	QPixmap statusIcon = statusIconForResourceIterator(resourceList, resourceList.priority()).pixmap();

	// walk the list of groups the member has, adding the item to every group
	for(QStringList::Iterator i = groups.begin(); i != groups.end(); ++i)
	{
		QListViewItem *group = findTreeGroup(*i);
		KListViewItem *mainItem = new KListViewItem(group, QString::null, item.jid().full(), (*resourceList.priority()).name(), (*resourceList.priority()).status().status());
		mainItem->setPixmap(0, statusIcon);

		XMPP::RosterItem copy = item;
		XMPP::Jid jid = copy.jid();
		jid.setResource((*resourceList.priority()).name());
		copy.setJid(jid);
		itemMap.insert(mainItem, copy);

		// add sub-items for each resource if there is more than one
		if(resourceList.count() > 1)
		{
			for(XMPP::ResourceList::ConstIterator j = resourceList.begin(); j != resourceList.end(); ++j)
			{
				if((*j).name() == (*resourceList.priority()).name()) continue;

				KListViewItem *resourceItem = new KListViewItem(mainItem, QString::null, QString::null, (*j).name(), (*j).status().status());
				resourceItem->setPixmap(0, statusIconForResourceIterator(resourceList, j).pixmap());

				XMPP::RosterItem copy = item;
				XMPP::Jid jid = copy.jid();
				jid.setResource((*j).name());
				copy.setJid(jid);
				itemMap.insert(resourceItem, copy);
			}
		}
	}
}

void LightHawk::RosterListView::rosterItemUpdated(const XMPP::RosterItem &)
{
	//kdDebug() << "rosterItemUpdated " << item.jid().full() << endl;
	fill();
}

void LightHawk::RosterListView::resourceAvailable(const XMPP::Jid &, const XMPP::Resource &)
{
	// TODO: check if it was already marked available
	KNotifyClient::event("Message Sent");
	//kdDebug() << "resourceAvailable " << jid.full() << endl;
	fill();
}

void LightHawk::RosterListView::resourceUnavailable(const XMPP::Jid &, const XMPP::Resource &)
{
	// TODO: check if it was already marked unavailable
	KNotifyClient::event("Message Received");
	//kdDebug() << "resourceUnavailable " << jid.full() << endl;
	fill();
}

void LightHawk::RosterListView::rosterItemRemoved(const XMPP::RosterItem &item)
{
	QValueList<QListViewItem *> list;
	for(QMapIterator<QListViewItem *, XMPP::RosterItem> i = itemMap.begin(); i != itemMap.end(); ++i)
		if(i.data().jid().full() == item.jid().full())
			list.append(i.key());
	
	for(QValueListIterator<QListViewItem *> i = list.begin(); i != list.end(); ++i)
	{
		delete *i;
		itemMap.remove(*i);
	}
}

void LightHawk::RosterListView::passOnContextMenu(KListView *, QListViewItem *i, const QPoint &p)
{
	if(!itemMap.contains(i))
		emit contextMenu(p);
	else
		emit contextMenu(itemMap[i], p);
}

void LightHawk::RosterListView::passOnExecuted(QListViewItem *i)
{
	if(itemMap.contains(i))
		emit executed(itemMap[i]);
}

#include "rosterlistview.moc"
// arch-tag: rosterlistview.cpp
