/*
 * stream.cpp - handles a Jabber XML stream
 * Copyright (C) 2001, 2002  Justin Karneges
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <kdebug.h>
#include <kextendedsocket.h>
#include <qfileinfo.h>
#include <qlibrary.h>
#include <qptrqueue.h>
#include <qregexp.h>
#include <qtimer.h>

#include "stream.h"
#include "xmlfilter.h"

using namespace XMPP;



//----------------------------------------------------------------------------
// StreamError
//----------------------------------------------------------------------------

//! \brief Creates StreamError object
//!
//! \param _type - type of error
//! \param _string - error message for that type
//! \param _isWarning - if this is a warning or an error
StreamError::StreamError(int _type, const QString &_string, bool _isWarning)
{
	v_type = _type;
	v_string = _string;
	v_isWarning = _isWarning;
}

//! \brief Returns if this is a warning or error
//! 
//! \return true: if warning\n
//!   false: if error
bool StreamError::isWarning() const
{
	return v_isWarning;
}

//! \brief Returns the error/warrning type.
int StreamError::type() const
{
	return v_type;
}

//! \brief Returns the error/warning information.
const QString & StreamError::details() const
{
	return v_string;
}

//! \brief creates a warning/error string for you.
//!
//! This function is here for your convenience. This will return a neatly
//! formated error/warrning message.
QString StreamError::toString() const
{
	QString str;
	if(isWarning())
		str += Stream::tr("Warning");
	else
		str += Stream::tr("Error");
	str += ": ";
	enum { DNS, Refused, Timeout, Socket, Disconnected, Handshake, Unknown };
	switch(type()) {
		case DNS:
			str += Stream::tr("DNS"); break;
		case Refused:
			str += Stream::tr("Connection Refused"); break;
		case Timeout:
			str += Stream::tr("Connection Timeout"); break;
		case Socket:
			str += Stream::tr("Socket"); break;
		case Disconnected:
			str += Stream::tr("Disconnected"); break;
		case Handshake:
			str += Stream::tr("Handshake"); break;
		case Unknown:
		default:
			break;
	}
	if(!details().isEmpty()) {
		str += ": ";
		str += details();
	}

	return str;
}


//----------------------------------------------------------------------------
// Stream
//----------------------------------------------------------------------------
//! \if _hide_doc_
class Stream::StreamPrivate
{
public:
	StreamPrivate() {}

	XmlFilter xml;

	KExtendedSocket *sock;

	QTimer *noopTimer;
	bool isActive, isConnected, isHandshaken, closing;
	QString host, realhost;
	int port;

	bool http_inHeader;

	StreamError err;
	int noop_time;

	QString id;

	QPtrQueue<QDomElement> in;
};
//! \endif

//! \class XMPP::Stream stream.h
//! \brief Makes tcp connection to host/port you specify.
//!
//! \fn void XMPP::Stream::connected()
//! \brief Signals when the connction has been made.

//! \fn void XMPP::Stream::handshaken()
//! \brief Signals when the handshakeing process has finished.

//! \fn void XMPP::Stream::error(const StreamError &)
//! \brief Signals when there was an error in the stream.

//! \fn void XMPP::Stream::receivePacket(QDomElement &)
//! \brief Signals when you recive something through the stream.

//! \brief Create Stream object.
//!
//! \param QObject - pass on to QObject
Stream::Stream(QObject *par)
:QObject(par)
{
	d = new StreamPrivate;

	d->closing = false;
	d->sock = 0;
	d->isActive = d->isConnected = d->isHandshaken = false;

	d->noop_time = 0;
	d->noopTimer = new QTimer;
	connect(d->noopTimer, SIGNAL(timeout()), SLOT(doNoop()));

	d->in.setAutoDelete(true);


	connect(&d->xml, SIGNAL(packetReady(const QDomElement &)), SLOT(xml_packetReady(const QDomElement &)));
	connect(&d->xml, SIGNAL(handshake(bool, const QString &)), SLOT(xml_handshake(bool, const QString &)));
}

//! \brief Destroys Stream object
Stream::~Stream()
{
	close();

	delete d->noopTimer;

	delete d;
}

//! \brief Tells you if the connection is busy.
//!
//! \return true: is busy\n
//!   false: nothings going on
bool Stream::isActive() const
{
	return d->isActive;
}

//! \brief Tells you if the connection is establised.
bool Stream::isConnected() const
{
	return d->isConnected;
}

//! \brief Tells you if the handshakeing process has been completed.
bool Stream::isHandshaken() const
{
	return d->isHandshaken;
}

//! \brief Closes socket connection.
//!
//! This function will disconnect and clean this object out.
void Stream::close()
{
	//printf("stream: closing...\n");
	if(!d->isActive || d->closing)
	{
		return;
	}
	else if(!d->sock)
	{
		QTimer::singleShot(0, this, SLOT(afterClose()));
	}
	else if(d->sock->socketStatus() == KExtendedSocket::lookupInProgress)
	{
		d->sock->cancelAsyncLookup();
		QTimer::singleShot(0, this, SLOT(afterClose()));
	}
	else if(d->sock && d->sock->socketStatus() == KExtendedSocket::connected)
	{
		if(d->isHandshaken)
			sendString("</stream:stream>\n");

		d->closing = true;

		int bytesLeft = d->sock->bytesToWrite();
		//printf("stream: bytesToWrite: %d\n", bytesLeft);

		d->sock->close();

		if(bytesLeft == 0)
			QTimer::singleShot(0, this, SLOT(afterClose()));
		else {
			//printf("stream: socket shutting down...\n");
		}
	}
	else
	{
		cleanup();
	}
}

void Stream::afterClose()
{
	//printf("stream: socket closed.\n");
	cleanup();
	closeFinished();
}

void Stream::cleanup()
{
	//printf("stream: cleanup()\n");
	delete d->sock;
	d->sock = 0;

	if(d->isConnected)
		d->xml.reset();

	d->noopTimer->stop();
	d->isActive = d->isConnected = d->isHandshaken = false;
	d->closing = false;
}

//! \brief Sets the host to connect to.
//!
//! \param _host - Sets the host you want to connect to.
//! \param _port - Sets the port to connect.
//! \param _virtual - If the server name differ from _host, specify here.
void Stream::connectToHost(const QString &_host, int _port, const QString &_virtualHost)
{
	if(d->isActive)
		return;

	d->host = _host;

	if(_port == -1) 
		d->port = 5222;
	else
		d->port = _port;

	if(_virtualHost.isEmpty())
		d->realhost = _host;
	else
		d->realhost = _virtualHost;

	d->isActive = true;

	QString str;
	str = d->host;

	d->sock = new KExtendedSocket(QString::null, 0, KExtendedSocket::inputBufferedSocket);
	connect(d->sock, SIGNAL(lookupFinished(int)), SLOT(sock_lookupFinished(int)));
	d->sock->setAddress(str, d->port);
	d->sock->startAsyncLookup();
	kdDebug() << "Looking up " << str << endl;
}

//! \brief Continue After Warning.
void Stream::continueAfterWarning()
{
}

void Stream::sock_lookupFinished(int)
{
	if(d->sock->status() == KExtendedSocket::error)
	{
		d->err = StreamError(StreamError::DNS);

		// process error later
		QTimer::singleShot(0, this, SLOT(delayedProcessError()));
	}
	else
	{
		connect(d->sock, SIGNAL(closed(int)), SLOT(sock_disconnected(int)));
		connect(d->sock, SIGNAL(connectionFailed(int)),  SLOT(sock_disconnected(int)));
		connect(d->sock, SIGNAL(bytesWritten(int)), SLOT(sock_bytesWritten(int)));
		connect(d->sock, SIGNAL(connectionSuccess()), SLOT(sock_connected()));

		connect(d->sock, SIGNAL(readyRead()), SLOT(sock_readyRead()));
		d->sock->enableRead(true);

		kdDebug() << "Connecting to " << d->sock->host() << " " <<  d->port << endl;
		d->sock->startAsyncConnect();
	}
}

//! \brief Return stream id.
QString Stream::id() const
{
	return d->id;
}

//! \brief Set ping interval.
//!
//! This section will set the interval to send new lines to keep the conenction
//! alive.
void Stream::setNoopTime(int mills)
{
	d->noop_time = mills;

	if(!d->isHandshaken)
		return;

	if(d->noop_time == 0) {
		d->noopTimer->stop();
		return;
	}

	d->noopTimer->start(d->noop_time);
}

//! \brief Send xml to host.
void Stream::sendPacket(const QDomElement &e)
{
	sendString(elemToString(e));
}

//! \brief send String to host.
void Stream::sendString(const QCString &str)
{
	if(d->isConnected)
		d->sock->writeBlock(str, str.length());
}

void Stream::sock_connected()
{
	// kdDebug() << "Connected.  Socket Status: " << d->sock->socketStatus() << endl;
	d->isConnected = true;
	d->xml.begin();
	connected();
	startHandshake();
}

void Stream::sock_disconnected(int i)
{
	if(d->closing)
	{
		kdDebug() << "Disconnected: closing " << i << endl;
	}
	else if(i == KBufferedIO::delayed)
	{
		kdDebug() << "Disconnected: delayed " << i << endl;
		afterClose();
	}
	else
	{
		kdDebug() << "Disconnected: error " << i << endl;
	
		// TODO
		/*
		if(x == Socket::ErrConnectionRefused)
			d->err = StreamError(StreamError::Refused);
		else if(x == Socket::ErrHostNotFound)
			d->err = StreamError(StreamError::DNS);
		else if(x == Socket::ErrSocketRead)
			d->err = StreamError(StreamError::Socket);
		else
			d->err = StreamError(StreamError::Timeout);
		*/
	
		// process error later
		QTimer::singleShot(0, this, SLOT(delayedProcessError()));
	}
}

void Stream::sock_readyRead()
{
	int size;
	QByteArray buf;

	size = d->sock->bytesAvailable();
	if(!size) return;
	//kdDebug() << "readyRead for " << size << endl;
	buf.resize(size);
	d->sock->readBlock(buf.data(), size);

	processIncomingData(buf);
}

void Stream::sock_bytesWritten(int)
{
	// kdDebug() << "stream: wrote=" << x << " left=" << d->sock->bytesToWrite() << endl;
}

void Stream::startHandshake()
{
	// Start the handshake
	QCString str;
	str.sprintf("<stream:stream to=\"%s\" xmlns=\"jabber:client\" xmlns:stream=\"http://etherx.jabber.org/streams\">\n", encodeXML(d->realhost).data());
	sendString(str);
}

void Stream::processIncomingData(const QByteArray &buf)
{
	d->xml.putIncomingXmlData(buf);
}

void Stream::delayedProcessError()
{
	if(!d->err.isWarning())
		close();
	error(d->err);
}

void Stream::delayedProcessReceived()
{
	// process chunks
	while(!d->in.isEmpty()) {
		QDomElement *e = d->in.dequeue();
		receivePacket(*e);
		delete e;
	}
}

void Stream::delayedProcessHandShake()
{
	d->isHandshaken = true;

	setNoopTime(d->noop_time);

	handshaken();
}

void Stream::doNoop()
{
	if(d->isHandshaken)
		sendString("\n");
}

void Stream::xml_packetReady(const QDomElement &e)
{
	d->in.enqueue(new QDomElement(e));
	QTimer::singleShot(0, this, SLOT(delayedProcessReceived()));
}

void Stream::xml_handshake(bool ok, const QString &id)
{
	if(!ok) {
		d->err = StreamError(StreamError::Handshake);

		// process error later
		QTimer::singleShot(0, this, SLOT(delayedProcessError()));
		return;
	}

	d->id = id;

	// process handshake later
	QTimer::singleShot(0, this, SLOT(delayedProcessHandShake()));
}

//! \brief Set the XML string.
QCString Stream::encodeXML(const QString &_str)
{
	QString str = _str;

	str.replace(QRegExp("&"), "&amp;");
	str.replace(QRegExp("<"), "&lt;");
	str.replace(QRegExp(">"), "&gt;");
	str.replace(QRegExp("\""), "&quot;");
	str.replace(QRegExp("'"), "&apos;");

	return str.utf8();
}

//! \brief Convert QDomElemnt to QCString.
QCString Stream::elemToString(const QDomElement &e)
{
	QString out;
	QTextStream ts(&out, IO_WriteOnly);
	e.save(ts, 0);
	return out.utf8();
}
#include "stream.moc"
// arch-tag: zinv/stream.cpp
