/*
 * tasks.cpp - basic tasks
 * Copyright (C) 2001, 2002  Justin Karneges
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include"tasks.h"
#include"util.h"
#include"xmlcommon.h"
#include"stream.h"
#include"types.h"
#include"vcard.h"

#include<qregexp.h>
#include<qvaluelist.h>

using namespace XMPP;


static QString lineEncode(QString str)
{
	str.replace(QRegExp("\\\\"), "\\\\");   // backslash to double-backslash
	str.replace(QRegExp("\\|"), "\\p");     // pipe to \p
	str.replace(QRegExp("\n"), "\\n");      // newline to \n
	return str;
}

static QString lineDecode(const QString &str)
{
	QString ret;

	for(unsigned int n = 0; n < str.length(); ++n) {
		if(str.at(n) == '\\') {
			++n;
			if(n >= str.length())
				break;

			if(str.at(n) == 'n')
				ret.append('\n');
			if(str.at(n) == 'p')
				ret.append('|');
			if(str.at(n) == '\\')
				ret.append('\\');
		}
		else {
			ret.append(str.at(n));
		}
	}

	return ret;
}

static Roster xmlReadRoster(const QDomElement &q, bool push)
{
	Roster r;

	for(QDomNode n = q.firstChild(); !n.isNull(); n = n.nextSibling()) {
		QDomElement i = n.toElement();
		if(i.isNull())
			continue;

		if(i.tagName() == "item") {
			RosterItem item;
			item.fromXml(i);

			if(push)
				item.setIsPush(true);

			r += item;
		}
	}

	return r;
}


//----------------------------------------------------------------------------
// JT_Auth
//----------------------------------------------------------------------------
JT_Auth::JT_Auth(Task *parent)
:Task(parent)
{
}

JT_Auth::~JT_Auth()
{
}

void JT_Auth::plain(const QString &user, const QString &pass, const QString &resource)
{
	to = client()->host();
	iq = createIQ(doc(), "set", to.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:auth");
	iq.appendChild(query);
	query.appendChild(textTag(doc(), "username", user));
	query.appendChild(textTag(doc(), "password", pass));
	query.appendChild(textTag(doc(), "resource", resource));
}

void JT_Auth::digest(const QString &user, const QString &pass, const QString &resource)
{
	to = client()->host();
	iq = createIQ(doc(), "set", to.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:auth");
	iq.appendChild(query);
	query.appendChild(textTag(doc(), "username", user));
	query.appendChild(textTag(doc(), "digest", SHA1::digest(client()->stream().id() + pass)) );
	query.appendChild(textTag(doc(), "resource", resource));
}

void JT_Auth::onGo()
{
	send(iq);
}

bool JT_Auth::take(const QDomElement &x)
{
	//if(x.attribute("from") != to || x.attribute("id") != id())
	if(x.attribute("id") != id())
		return false;

	if(x.attribute("type") == "result") {
		// login success
		setSuccess();
	}
	else {
		// login fail
		setError(x);
	}

	return true;
}


//----------------------------------------------------------------------------
// JT_Register
//----------------------------------------------------------------------------
class JT_Register::Private
{
public:
	Private() {}

	Form form;
	Jid jid;
	int type;
};

JT_Register::JT_Register(Task *parent)
:Task(parent)
{
	d = new Private;
	d->type = -1;
}

JT_Register::~JT_Register()
{
	delete d;
}

void JT_Register::reg(const QString &user, const QString &pass)
{
	d->type = 0;
	to = client()->host();
	iq = createIQ(doc(), "set", to.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:register");
	iq.appendChild(query);
	query.appendChild(textTag(doc(), "username", user));
	query.appendChild(textTag(doc(), "password", pass));
}

void JT_Register::changepw(const QString &pass)
{
	d->type = 1;
	to = client()->host();
	iq = createIQ(doc(), "set", to.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:register");
	iq.appendChild(query);
	query.appendChild(textTag(doc(), "username", client()->user()));
	query.appendChild(textTag(doc(), "password", pass));
}

void JT_Register::unreg(const Jid &j)
{
	d->type = 2;
	to = j.isEmpty() ? client()->host() : j.full();
	iq = createIQ(doc(), "set", to.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:register");
	iq.appendChild(query);
	query.appendChild(doc()->createElement("remove"));
}

void JT_Register::getForm(const Jid &j)
{
	d->type = 3;
	to = j;
	iq = createIQ(doc(), "get", to.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:register");
	iq.appendChild(query);
}

void JT_Register::setForm(const Form &form)
{
	d->type = 4;
	to = form.jid();
	iq = createIQ(doc(), "set", to.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:register");
	iq.appendChild(query);

	// key?
	if(!form.key().isEmpty())
		query.appendChild(textTag(doc(), "key", form.key()));

	// fields
	for(Form::ConstIterator it = form.begin(); it != form.end(); ++it) {
		const FormField &f = *it;
		query.appendChild(textTag(doc(), f.realName(), f.value()));
	}
}

const Form & JT_Register::form() const
{
	return d->form;
}

void JT_Register::onGo()
{
	send(iq);
}

bool JT_Register::take(const QDomElement &x)
{
	Jid from = x.attribute("from");
	if(!from.isEmpty() && !to.compare(from))
		return false;

	if(x.attribute("id") != id())
		return false;

	if(x.attribute("type") == "result") {
		if(d->type == 3) {
			d->form.clear();
			d->form.setJid(from);

			QDomElement q = queryTag(x);
			for(QDomNode n = q.firstChild(); !n.isNull(); n = n.nextSibling()) {
				QDomElement i = n.toElement();
				if(i.isNull())
					continue;

				if(i.tagName() == "instructions")
					d->form.setInstructions(tagContent(i));
				else if(i.tagName() == "key")
					d->form.setKey(tagContent(i));
				else {
					FormField f;
					if(f.setType(i.tagName())) {
						f.setValue(tagContent(i));
						d->form += f;
					}
				}
			}
		}

		setSuccess();
	}
	else
		setError(x);

	return true;
}


//----------------------------------------------------------------------------
// JT_Roster
//----------------------------------------------------------------------------
class JT_Roster::Private
{
public:
	Private() {}

	Roster roster;
	QValueList<QDomElement> itemList;
};

JT_Roster::JT_Roster(Task *parent)
:Task(parent)
{
	type = -1;
	d = new Private;
}

JT_Roster::~JT_Roster()
{
	delete d;
}

void JT_Roster::get()
{
	type = 0;
	//to = client()->host();
	iq = createIQ(doc(), "get", to.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:roster");
	iq.appendChild(query);
}

void JT_Roster::set(const Jid &jid, const QString &name, const QStringList &groups)
{
	type = 1;
	//to = client()->host();
	QDomElement item = doc()->createElement("item");
	item.setAttribute("jid", jid.full());
	if(!name.isEmpty())
		item.setAttribute("name", name);
	for(QStringList::ConstIterator it = groups.begin(); it != groups.end(); ++it)
		item.appendChild(textTag(doc(), "group", *it));
	d->itemList += item;
}

void JT_Roster::remove(const Jid &jid)
{
	type = 1;
	//to = client()->host();
	QDomElement item = doc()->createElement("item");
	item.setAttribute("jid", jid.full());
	item.setAttribute("subscription", "remove");
	d->itemList += item;
}

void JT_Roster::onGo()
{
	if(type == 0)
		send(iq);
	else if(type == 1) {
		//to = client()->host();
		iq = createIQ(doc(), "set", to.full(), id());
		QDomElement query = doc()->createElement("query");
		query.setAttribute("xmlns", "jabber:iq:roster");
		iq.appendChild(query);
		for(QValueList<QDomElement>::ConstIterator it = d->itemList.begin(); it != d->itemList.end(); ++it)
			query.appendChild(*it);
		send(iq);
	}
}

const Roster & JT_Roster::roster() const
{
	return d->roster;
}

QString JT_Roster::toString() const
{
	if(type != 1)
		return "";

	QDomElement i = doc()->createElement("request");
	i.setAttribute("type", "JT_Roster");
	for(QValueList<QDomElement>::ConstIterator it = d->itemList.begin(); it != d->itemList.end(); ++it)
		i.appendChild(*it);
	return lineEncode(QString::fromUtf8(Stream::elemToString(i)));
}

bool JT_Roster::fromString(const QString &str)
{
	QDomDocument *dd = new QDomDocument;
	if(!dd->setContent(lineDecode(str).utf8()))
		return false;
	QDomElement e = doc()->importNode(dd->documentElement(), true).toElement();
	delete dd;

	if(e.tagName() != "request" || e.attribute("type") != "JT_Roster")
		return false;

	type = 1;
	//to = client()->host();
	d->itemList.clear();
	for(QDomNode n = e.firstChild(); !n.isNull(); n = n.nextSibling()) {
		QDomElement i = n.toElement();
		if(i.isNull())
			continue;
		d->itemList += i;
	}

	return true;
}

bool JT_Roster::take(const QDomElement &x)
{
	if(x.attribute("id") != id())
		return false;

	// get
	if(type == 0) {
		if(x.attribute("type") == "result") {
			QDomElement q = queryTag(x);
			d->roster = xmlReadRoster(q, false);
			setSuccess();
		}
		else {
			setError(x);
		}

		return true;
	}
	// set
	else if(type == 1) {
		if(x.attribute("type") == "result")
			setSuccess();
		else
			setError(x);

		return true;
	}
	// remove
	else if(type == 2) {
		setSuccess();
		return true;
	}

	return false;
}


//----------------------------------------------------------------------------
// JT_PushRoster
//----------------------------------------------------------------------------
JT_PushRoster::JT_PushRoster(Task *parent)
:Task(parent)
{
}

JT_PushRoster::~JT_PushRoster()
{
}

bool JT_PushRoster::take(const QDomElement &e)
{
	// must be an iq-set tag
	if(e.tagName() != "iq" || e.attribute("type") != "set")
		return false;

	if(queryNS(e) != "jabber:iq:roster")
		return false;

	roster(xmlReadRoster(queryTag(e), true));

	return true;
}


//----------------------------------------------------------------------------
// JT_Presence
//----------------------------------------------------------------------------
JT_Presence::JT_Presence(Task *parent)
:Task(parent)
{
	type = -1;
}

JT_Presence::~JT_Presence()
{
}

void JT_Presence::pres(const Status &s)
{
	type = 0;

	tag = doc()->createElement("presence");
	if(!s.isAvailable()) {
		tag.setAttribute("type", "unavailable");
		if(!s.status().isEmpty())
			tag.appendChild(textTag(doc(), "status", s.status()));
	}
	else {
		if(s.isInvisible())
			tag.setAttribute("type", "invisible");

		if(!s.show().isEmpty())
			tag.appendChild(textTag(doc(), "show", s.show()));
		if(!s.status().isEmpty())
			tag.appendChild(textTag(doc(), "status", s.status()));

		tag.appendChild( textTag(doc(), "priority", QString("%1").arg(s.priority()) ) );

		if(!s.xsigned().isEmpty()) {
			QDomElement x = textTag(doc(), "x", s.xsigned());
			x.setAttribute("xmlns", "jabber:x:signed");
			tag.appendChild(x);
		}
	}
}

void JT_Presence::pres(const Jid &to, const Status &s)
{
	pres(s);
	tag.setAttribute("to", to.full());
}

void JT_Presence::sub(const Jid &to, const QString &subType)
{
	type = 1;

	tag = doc()->createElement("presence");
	tag.setAttribute("to", to.full());
	tag.setAttribute("type", subType);
}

void JT_Presence::onGo()
{
	send(tag);
	setSuccess();
}


//----------------------------------------------------------------------------
// JT_PushPresence
//----------------------------------------------------------------------------
JT_PushPresence::JT_PushPresence(Task *parent)
:Task(parent)
{
}

JT_PushPresence::~JT_PushPresence()
{
}

bool JT_PushPresence::take(const QDomElement &e)
{
	if(e.tagName() != "presence")
		return false;

	Jid j(e.attribute("from"));
	Status p;

	if(e.hasAttribute("type")) {
		QString type = e.attribute("type");
		if(type == "unavailable") {
			p.setIsAvailable(false);
		}
		else if(type == "error") {
			QString str = "";
			int code = 0;
			getErrorFromElement(e, &code, &str);
			p.setError(code, str);
		}
		else {
			subscription(j, type);
			return true;
		}
	}

	QDomElement tag;
	bool found;

	tag = findSubTag(e, "status", &found);
	if(found)
		p.setStatus(tagContent(tag));
	tag = findSubTag(e, "show", &found);
	if(found)
		p.setShow(tagContent(tag));
	tag = findSubTag(e, "priority", &found);
	if(found)
		p.setPriority(tagContent(tag).toInt());

	for(QDomNode n = e.firstChild(); !n.isNull(); n = n.nextSibling()) {
		QDomElement i = n.toElement();
		if(i.isNull())
			continue;

		if(i.tagName() == "x" && i.attribute("xmlns") == "jabber:x:delay") {
			if(i.hasAttribute("stamp")) {
				QDateTime d;
				stamp2TS(i.attribute("stamp"), &d);
				p.setTimeStamp(d);
			}
		}
		else if(i.tagName() == "x" && i.attribute("xmlns") == "gabber:x:music:info") {
			QDomElement t;
			bool found;
			QString title, state;

			t = findSubTag(i, "title", &found);
			if(found)
				title = tagContent(t);
			t = findSubTag(i, "state", &found);
			if(found)
				state = tagContent(t);

			if(!title.isEmpty() && state == "playing")
				p.setSongTitle(title);
		}
		else if(i.tagName() == "x" && i.attribute("xmlns") == "jabber:x:signed") {
			p.setXSigned(tagContent(i));
		}
	}

	presence(j, p);

	return true;
}


//----------------------------------------------------------------------------
// JT_Message
//----------------------------------------------------------------------------
JT_Message::JT_Message(Task *parent, const Message &msg)
:Task(parent)
{
	m = msg;
}

JT_Message::~JT_Message()
{
}

void JT_Message::onGo()
{
	send(m.toXml(doc()));
	setSuccess();
}


//----------------------------------------------------------------------------
// JT_PushMessage
//----------------------------------------------------------------------------
JT_PushMessage::JT_PushMessage(Task *parent)
:Task(parent)
{
}

JT_PushMessage::~JT_PushMessage()
{
}

bool JT_PushMessage::take(const QDomElement &e)
{
	if(e.tagName() != "message")
		return false;

	Message m;
	if(!m.fromXml(e))
		return false;

	message(m);
	return true;
}


//----------------------------------------------------------------------------
// JT_GetServices
//----------------------------------------------------------------------------
JT_GetServices::JT_GetServices(Task *parent)
:Task(parent)
{
}

void JT_GetServices::get(const Jid &j)
{
	agentList.clear();

	jid = j;
	iq = createIQ(doc(), "get", jid.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:agents");
	iq.appendChild(query);
}

const AgentList & JT_GetServices::agents() const
{
	return agentList;
}

void JT_GetServices::onGo()
{
	send(iq);
}

bool JT_GetServices::take(const QDomElement &x)
{
	if(x.attribute("id") != id())
		return false;

	if(x.attribute("type") == "result") {
		QDomElement q = queryTag(x);

		// agents
		for(QDomNode n = q.firstChild(); !n.isNull(); n = n.nextSibling()) {
			QDomElement i = n.toElement();
			if(i.isNull())
				continue;

			if(i.tagName() == "agent") {
				AgentItem a;

				a.setJid(Jid(i.attribute("jid")));

				QDomElement tag;
				bool found;

				tag = findSubTag(i, "name", &found);
				if(found)
					a.setName(tagContent(tag));
				tag = findSubTag(i, "register", &found);
				if(found)
					a.setCanRegister(true);
				tag = findSubTag(i, "search", &found);
				if(found)
					a.setCanSearch(true);
				tag = findSubTag(i, "groupchat", &found);
				if(found)
					a.setCanGroupchat(true);
				tag = findSubTag(i, "transport", &found);
				if(found)
					a.setIsTransport(true);

				agentList += a;
			}
		}

		setSuccess(true);
	}
	else {
		setError(x);
	}

	return true;
}


//----------------------------------------------------------------------------
// JT_VCard
//----------------------------------------------------------------------------
class JT_VCard::Private
{
public:
	Private() {}

	QDomElement iq;
	Jid jid;
	VCard vcard;
};

JT_VCard::JT_VCard(Task *parent)
:Task(parent)
{
	type = -1;
	d = new Private;
}

JT_VCard::~JT_VCard()
{
	delete d;
}

void JT_VCard::get(const Jid &_jid)
{
	type = 0;
	d->jid = _jid;
	d->iq = createIQ(doc(), "get", d->jid.userHost(), id());
	QDomElement v = doc()->createElement("vCard");
	v.setAttribute("xmlns", "vcard-temp");
	v.setAttribute("version", "2.0");
	v.setAttribute("prodid", "-//HandGen//NONSGML vGen v1.0//EN");
	d->iq.appendChild(v);
}

const Jid & JT_VCard::jid() const
{
	return d->jid;
}

const VCard & JT_VCard::vcard() const
{
	return d->vcard;
}

void JT_VCard::set(const VCard &card)
{
	type = 1;
	d->vcard = card;
	d->iq = createIQ(doc(), "set", "", id());
	d->iq.appendChild(card.toXml(doc()) );
}

void JT_VCard::onGo()
{
	send(d->iq);
}

bool JT_VCard::take(const QDomElement &x)
{
	if(x.attribute("id") != id())
		return false;

	if(x.attribute("type") == "result") {
		if(type == 0) {
			bool found = false;
			QDomElement q = findSubTag(x, "vcard", &found);
			if(found) {
				if(d->vcard.fromXml(q)) {
					setSuccess();
					return true;
				}
			}

			setError(0, tr("No VCard available"));
			return true;
		}
		else {
			setSuccess();
			return true;
		}
	}
	else {
		setError(x);
	}

	return true;
}


//----------------------------------------------------------------------------
// JT_Search
//----------------------------------------------------------------------------
class JT_Search::Private
{
public:
	Private() {}

	Jid jid;
	Form form;
	QValueList<SearchResult> resultList;
};

JT_Search::JT_Search(Task *parent)
:Task(parent)
{
	d = new Private;
	type = -1;
}

JT_Search::~JT_Search()
{
	delete d;
}

void JT_Search::get(const Jid &jid)
{
	type = 0;
	d->jid = jid;
	iq = createIQ(doc(), "get", d->jid.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:search");
	iq.appendChild(query);
}

void JT_Search::set(const Form &form)
{
	type = 1;
	d->jid = form.jid();
	iq = createIQ(doc(), "set", d->jid.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:search");
	iq.appendChild(query);

	// key?
	if(!form.key().isEmpty())
		query.appendChild(textTag(doc(), "key", form.key()));

	// fields
	for(Form::ConstIterator it = form.begin(); it != form.end(); ++it) {
		const FormField &f = *it;
		query.appendChild(textTag(doc(), f.realName(), f.value()));
	}
}

const Form & JT_Search::form() const
{
	return d->form;
}

const QValueList<SearchResult> & JT_Search::results() const
{
	return d->resultList;
}

void JT_Search::onGo()
{
	send(iq);
}

bool JT_Search::take(const QDomElement &x)
{
	Jid from = x.attribute("from");
	if(!from.isEmpty() && !d->jid.compare(from))
		return false;

	if(x.attribute("id") != id())
		return false;

	if(x.attribute("type") == "result") {
		if(type == 0) {
			d->form.clear();
			d->form.setJid(from);

			QDomElement q = queryTag(x);
			for(QDomNode n = q.firstChild(); !n.isNull(); n = n.nextSibling()) {
				QDomElement i = n.toElement();
				if(i.isNull())
					continue;

				if(i.tagName() == "instructions")
					d->form.setInstructions(tagContent(i));
				else if(i.tagName() == "key")
					d->form.setKey(tagContent(i));
				else {
					FormField f;
					if(f.setType(i.tagName())) {
						f.setValue(tagContent(i));
						d->form += f;
					}
				}
			}
		}
		else {
			d->resultList.clear();

			QDomElement q = queryTag(x);
			for(QDomNode n = q.firstChild(); !n.isNull(); n = n.nextSibling()) {
				QDomElement i = n.toElement();
				if(i.isNull())
					continue;

				if(i.tagName() == "item") {
					SearchResult r(Jid(i.attribute("jid")));

					QDomElement tag;
					bool found;

					tag = findSubTag(i, "nick", &found);
					if(found)
						r.setNick(tagContent(tag));
					tag = findSubTag(i, "first", &found);
					if(found)
						r.setFirst(tagContent(tag));
					tag = findSubTag(i, "last", &found);
					if(found)
						r.setLast(tagContent(tag));
					tag = findSubTag(i, "email", &found);
					if(found)
						r.setEmail(tagContent(tag));

					d->resultList += r;
				}
			}
		}
		setSuccess();
	}
	else {
		setError(x);
	}

	return true;
}


//----------------------------------------------------------------------------
// JT_ClientVersion
//----------------------------------------------------------------------------
JT_ClientVersion::JT_ClientVersion(Task *parent)
:Task(parent)
{
}

void JT_ClientVersion::get(const Jid &jid)
{
	j = jid;
	iq = createIQ(doc(), "get", j.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:version");
	iq.appendChild(query);
}

void JT_ClientVersion::onGo()
{
	send(iq);
}

bool JT_ClientVersion::take(const QDomElement &x)
{
	Jid from = x.attribute("from");
	if(!j.compare(from) || x.attribute("id") != id())
		return false;

	if(x.attribute("type") == "result") {
		bool found;
		QDomElement q = queryTag(x);
		QDomElement tag;
		tag = findSubTag(q, "name", &found);
		if(found)
			v_name = tagContent(tag);
		tag = findSubTag(q, "version", &found);
		if(found)
			v_ver = tagContent(tag);
		tag = findSubTag(q, "os", &found);
		if(found)
			v_os = tagContent(tag);

		setSuccess();
	}
	else {
		setError(x);
	}

	return true;
}

const Jid & JT_ClientVersion::jid() const
{
	return j;
}

const QString & JT_ClientVersion::name() const
{
	return v_name;
}

const QString & JT_ClientVersion::version() const
{
	return v_ver;
}

const QString & JT_ClientVersion::os() const
{
	return v_os;
}


//----------------------------------------------------------------------------
// JT_ClientTime
//----------------------------------------------------------------------------
/*JT_ClientTime::JT_ClientTime(Task *parent, const Jid &_j)
:Task(parent)
{
	j = _j;
	iq = createIQ("get", j.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:time");
	iq.appendChild(query);
}

void JT_ClientTime::go()
{
	send(iq);
}

bool JT_ClientTime::take(const QDomElement &x)
{
	if(x.attribute("id") != id())
		return FALSE;

	if(x.attribute("type") == "result") {
		bool found;
		QDomElement q = queryTag(x);
		QDomElement tag;
		tag = findSubTag(q, "utc", &found);
		if(found)
			stamp2TS(tagContent(tag), &utc);
		tag = findSubTag(q, "tz", &found);
		if(found)
			timezone = tagContent(tag);
		tag = findSubTag(q, "display", &found);
		if(found)
			display = tagContent(tag);

		setSuccess(TRUE);
	}
	else {
		setError(getErrorString(x));
		setSuccess(FALSE);
	}

	return TRUE;
}
*/


//----------------------------------------------------------------------------
// JT_ServInfo
//----------------------------------------------------------------------------
JT_ServInfo::JT_ServInfo(Task *parent)
:Task(parent)
{
}

JT_ServInfo::~JT_ServInfo()
{
}

bool JT_ServInfo::take(const QDomElement &e)
{
	if(e.tagName() != "iq" || e.attribute("type") != "get")
		return false;

	QString ns = queryNS(e);
	if(ns == "jabber:iq:version") {
		QDomElement iq = createIQ(doc(), "result", e.attribute("from"), e.attribute("id"));
		QDomElement query = doc()->createElement("query");
		query.setAttribute("xmlns", "jabber:iq:version");
		iq.appendChild(query);
		query.appendChild(textTag(doc(), "name", client()->clientName()));
		query.appendChild(textTag(doc(), "version", client()->clientVersion()));
		query.appendChild(textTag(doc(), "os", client()->OSName()));
		send(iq);
		return true;
	}
	//else if(ns == "jabber:iq:time") {
	//	QDomElement iq = createIQ("result", e.attribute("from"), e.attribute("id"));
	//	QDomElement query = doc()->createElement("query");
	//	query.setAttribute("xmlns", "jabber:iq:time");
	//	iq.appendChild(query);
	//	QDateTime local = QDateTime::currentDateTime();
	//	QDateTime utc = local.addSecs(-getTZOffset() * 3600);
	//	QString str = getTZString();
	//	query.appendChild(textTag("utc", TS2stamp(utc)));
	//	query.appendChild(textTag("tz", str));
	//	query.appendChild(textTag("display", QString("%1 %2").arg(local.toString()).arg(str)));
	//	send(iq);
	//	return TRUE;
	//}

	return false;
}


//----------------------------------------------------------------------------
// JT_Gateway
//----------------------------------------------------------------------------
JT_Gateway::JT_Gateway(Task *parent)
:Task(parent)
{
	type = -1;
}

void JT_Gateway::get(const Jid &jid)
{
	type = 0;
	v_jid = jid;
	iq = createIQ(doc(), "get", v_jid.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:gateway");
	iq.appendChild(query);
}

void JT_Gateway::set(const Jid &jid, const QString &prompt)
{
	type = 1;
	v_jid = jid;
	v_prompt = prompt;
	iq = createIQ(doc(), "set", v_jid.full(), id());
	QDomElement query = doc()->createElement("query");
	query.setAttribute("xmlns", "jabber:iq:gateway");
	iq.appendChild(query);
	query.appendChild(textTag(doc(), "prompt", v_prompt));
}

void JT_Gateway::onGo()
{
	send(iq);
}

Jid JT_Gateway::jid() const
{
	return v_jid;
}

QString JT_Gateway::desc() const
{
	return v_desc;
}

QString JT_Gateway::prompt() const
{
	return v_prompt;
}

bool JT_Gateway::take(const QDomElement &x)
{
	if(x.attribute("id") != id())
		return false;

	if(x.attribute("type") == "result") {
		if(type == 0) {
			QDomElement query = queryTag(x);
			bool found;
			QDomElement tag;
			tag = findSubTag(query, "desc", &found);
			if(found)
				v_desc = tagContent(tag);
			tag = findSubTag(query, "prompt", &found);
			if(found)
				v_prompt = tagContent(tag);
		}
		else {
			QDomElement query = queryTag(x);
			bool found;
			QDomElement tag;
			tag = findSubTag(query, "prompt", &found);
			if(found)
				v_prompt = tagContent(tag);
		}

		setSuccess();
	}
	else {
		setError(x);
	}

	return true;
}
#include "tasks.moc"
// arch-tag: zinv/tasks.cpp
