/*
 * types.h - classes for handling Jabber datatypes
 * Copyright (C) 2001, 2002  Justin Karneges
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef JABBER_TYPES_H
#define JABBER_TYPES_H

#include<qstring.h>
#include<qstringlist.h>
#include<qvaluelist.h>
#include<qdatetime.h>
#include"jid.h"

class QDomElement;
class QDomDocument;


namespace XMPP
{
	class Subscription
	{
	public:
		enum SubType { None, To, From, Both, Remove };

		Subscription(SubType type=None);

		int type() const;

		QString toString() const;
		bool fromString(const QString &);

	private:
		SubType value;
	};

	class Status
	{
	public:
		Status(const QString &show="", const QString &status="", int priority=5, bool available=true);
		~Status();

		int priority() const;
		const QString & show() const;
		const QString & status() const;
		QDateTime timeStamp() const;
		const QString & xsigned() const;
		bool isAvailable() const;
		bool isAway() const;
		bool isInvisible() const;
		bool hasError() const;
		int errorCode() const;
		const QString & errorString() const;

		const QString & songTitle() const;

		void setPriority(int);
		void setShow(const QString &);
		void setStatus(const QString &);
		void setTimeStamp(const QDateTime &);
		void setXSigned(const QString &);
		void setIsAvailable(bool);
		void setIsInvisible(bool);
		void setError(int, const QString &);

		void setSongTitle(const QString &);

	private:
		int v_priority;
		QString v_show, v_status, v_xsigned;
		QDateTime v_timeStamp;
		bool v_isAvailable;
		bool v_isInvisible;

		// gabber song extension
		QString v_songTitle;

		int ecode;
		QString estr;

		class Private;
		Private *d;
	};

	class Resource
	{
	public:
		Resource(const QString &name="", const Status &s=Status());
		~Resource();

		const QString & name() const;
		int priority() const;
		const Status & status() const;

		void setName(const QString &);
		void setStatus(const Status &);

	private:
		QString v_name;
		Status v_status;

		class ResourcePrivate *d;
	};

	class ResourceList : public QValueList<Resource>
	{
	public:
		ResourceList();
		~ResourceList();

		ResourceList::Iterator find(const QString &);
		ResourceList::Iterator priority();

		ResourceList::ConstIterator find(const QString &) const;
		ResourceList::ConstIterator priority() const;

	private:
		class ResourceListPrivate *d;
	};

	class RosterItem
	{
	public:
		RosterItem(const Jid &jid="");
		virtual ~RosterItem();

		const Jid & jid() const;
		const QString & name() const;
		const QStringList & groups() const;
		const Subscription & subscription() const;
		const QString & ask() const;
		bool isPush() const;
		bool inGroup(const QString &) const;

		virtual void setJid(const Jid &);
		void setName(const QString &);
		void setGroups(const QStringList &);
		void setSubscription(const Subscription &);
		void setAsk(const QString &);
		void setIsPush(bool);
		bool addGroup(const QString &);
		bool removeGroup(const QString &);

		QDomElement toXml(QDomDocument *) const;
		bool fromXml(const QDomElement &);

	private:
		Jid v_jid;
		QString v_name;
		QStringList v_groups;
		Subscription v_subscription;
		QString v_ask;
		bool v_push;

		class RosterItemPrivate *d;
	};

	class Roster : public QValueList<RosterItem>
	{
	public:
		Roster();
		~Roster();

		Roster::Iterator find(const Jid &);
		Roster::ConstIterator find(const Jid &) const;

	private:
		class RosterPrivate *d;
	};

	class AgentItem
	{
	public:
		AgentItem()
		{
			v_reg = v_search = v_gc = v_transport = false;
		}

		const Jid & jid() const { return v_jid; }
		const QString & name() const { return v_name; }
		bool canRegister() const { return v_reg; }
		bool canSearch() const { return v_search; }
		bool canGroupchat() const { return v_gc; }
		bool isTransport() const { return v_transport; }

		void setJid(const Jid &j) { v_jid = j; }
		void setName(const QString &n) { v_name = n; }
		void setCanRegister(bool b) { v_reg = b; }
		void setCanSearch(bool b) { v_search = b; }
		void setCanGroupchat(bool b) { v_gc = b; }
		void setIsTransport(bool b) { v_transport = b; }

	private:
		Jid v_jid;
		QString v_name;
		bool v_reg, v_search, v_gc, v_transport;
	};

	class AgentList : public QValueList<AgentItem>
	{
	public:
		AgentList() : QValueList<AgentItem>() {}
	};

	class FormField
	{
	public:
		enum { username, nick, password, name, first, last, email, address, city, state, zip, phone, url, date, misc };
		FormField(const QString &type="", const QString &value="");
		~FormField();

		int type() const;
		QString fieldName() const;
		QString realName() const;
		bool isSecret() const;
		const QString & value() const;
		void setType(int);
		bool setType(const QString &);
		void setValue(const QString &);

	private:
		int tagNameToType(const QString &) const;
		QString typeToTagName(int) const;

		int v_type;
		QString v_value;

		class Private;
		Private *d;
	};

	class Form : public QValueList<FormField>
	{
	public:
		Form(const Jid &j="");
		~Form();

		Jid jid() const;
		QString instructions() const;
		QString key() const;
		void setJid(const Jid &);
		void setInstructions(const QString &);
		void setKey(const QString &);

	private:
		Jid v_jid;
		QString v_instructions, v_key;

		class Private;
		Private *d;
	};

	class SearchResult
	{
	public:
		SearchResult(const Jid &jid="");
		~SearchResult();

		const Jid & jid() const;
		const QString & nick() const;
		const QString & first() const;
		const QString & last() const;
		const QString & email() const;

		void setJid(const Jid &);
		void setNick(const QString &);
		void setFirst(const QString &);
		void setLast(const QString &);
		void setEmail(const QString &);

	private:
		Jid v_jid;
		QString v_nick, v_first, v_last, v_email;
	};
}

#endif
// arch-tag: zinv/types.h
