/*
 * Copyright (C) 2001, 2002  Justin Karneges
 *                           Akito Nozaki
 *                           Hideaki Omuro
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include"util.h"

#include<qstring.h>

/****************************************************************************
  SHA1 - from a public domain implementation by Steve Reid (steve@edmweb.com)
****************************************************************************/

#define rol(value, bits) (((value) << (bits)) | ((value) >> (32 - (bits))))
#define blk(i) (block->l[i&15] = rol(block->l[(i+13)&15]^block->l[(i+8)&15]^block->l[(i+2)&15]^block->l[i&15],1))

/* (R0+R1), R2, R3, R4 are the different operations used in SHA1 */
#define R0(v,w,x,y,z,i) z+=((w&(x^y))^y)+blk0(i)+0x5A827999+rol(v,5);w=rol(w,30);
#define R1(v,w,x,y,z,i) z+=((w&(x^y))^y)+blk(i)+0x5A827999+rol(v,5);w=rol(w,30);
#define R2(v,w,x,y,z,i) z+=(w^x^y)+blk(i)+0x6ED9EBA1+rol(v,5);w=rol(w,30);
#define R3(v,w,x,y,z,i) z+=(((w|x)&y)|(w&x))+blk(i)+0x8F1BBCDC+rol(v,5);w=rol(w,30);
#define R4(v,w,x,y,z,i) z+=(w^x^y)+blk(i)+0xCA62C1D6+rol(v,5);w=rol(w,30);


SHA1::SHA1()
{
   int wordSize;

   qSysInfo(&wordSize, &bigEndian);
}

unsigned long SHA1::blk0(Q_UINT32 i)
{
   if(bigEndian)
      return block->l[i];
   else
      return (block->l[i] = (rol(block->l[i],24)&0xFF00FF00) | (rol(block->l[i],8)&0x00FF00FF));
}

// Hash a single 512-bit block. This is the core of the algorithm.
void SHA1::transform(Q_UINT32 state[5], unsigned char buffer[64])
{
   Q_UINT32 a, b, c, d, e;

   block = (CHAR64LONG16*)buffer;

   // Copy context->state[] to working vars
   a = state[0];
   b = state[1];
   c = state[2];
   d = state[3];
   e = state[4];

   // 4 rounds of 20 operations each. Loop unrolled.
   R0(a,b,c,d,e, 0); R0(e,a,b,c,d, 1); R0(d,e,a,b,c, 2); R0(c,d,e,a,b, 3);
   R0(b,c,d,e,a, 4); R0(a,b,c,d,e, 5); R0(e,a,b,c,d, 6); R0(d,e,a,b,c, 7);
   R0(c,d,e,a,b, 8); R0(b,c,d,e,a, 9); R0(a,b,c,d,e,10); R0(e,a,b,c,d,11);
   R0(d,e,a,b,c,12); R0(c,d,e,a,b,13); R0(b,c,d,e,a,14); R0(a,b,c,d,e,15);
   R1(e,a,b,c,d,16); R1(d,e,a,b,c,17); R1(c,d,e,a,b,18); R1(b,c,d,e,a,19);
   R2(a,b,c,d,e,20); R2(e,a,b,c,d,21); R2(d,e,a,b,c,22); R2(c,d,e,a,b,23);
   R2(b,c,d,e,a,24); R2(a,b,c,d,e,25); R2(e,a,b,c,d,26); R2(d,e,a,b,c,27);
   R2(c,d,e,a,b,28); R2(b,c,d,e,a,29); R2(a,b,c,d,e,30); R2(e,a,b,c,d,31);
   R2(d,e,a,b,c,32); R2(c,d,e,a,b,33); R2(b,c,d,e,a,34); R2(a,b,c,d,e,35);
   R2(e,a,b,c,d,36); R2(d,e,a,b,c,37); R2(c,d,e,a,b,38); R2(b,c,d,e,a,39);
   R3(a,b,c,d,e,40); R3(e,a,b,c,d,41); R3(d,e,a,b,c,42); R3(c,d,e,a,b,43);
   R3(b,c,d,e,a,44); R3(a,b,c,d,e,45); R3(e,a,b,c,d,46); R3(d,e,a,b,c,47);
   R3(c,d,e,a,b,48); R3(b,c,d,e,a,49); R3(a,b,c,d,e,50); R3(e,a,b,c,d,51);
   R3(d,e,a,b,c,52); R3(c,d,e,a,b,53); R3(b,c,d,e,a,54); R3(a,b,c,d,e,55);
   R3(e,a,b,c,d,56); R3(d,e,a,b,c,57); R3(c,d,e,a,b,58); R3(b,c,d,e,a,59);
   R4(a,b,c,d,e,60); R4(e,a,b,c,d,61); R4(d,e,a,b,c,62); R4(c,d,e,a,b,63);
   R4(b,c,d,e,a,64); R4(a,b,c,d,e,65); R4(e,a,b,c,d,66); R4(d,e,a,b,c,67);
   R4(c,d,e,a,b,68); R4(b,c,d,e,a,69); R4(a,b,c,d,e,70); R4(e,a,b,c,d,71);
   R4(d,e,a,b,c,72); R4(c,d,e,a,b,73); R4(b,c,d,e,a,74); R4(a,b,c,d,e,75);
   R4(e,a,b,c,d,76); R4(d,e,a,b,c,77); R4(c,d,e,a,b,78); R4(b,c,d,e,a,79);

   // Add the working vars back into context.state[]
   state[0] += a;
   state[1] += b;
   state[2] += c;
   state[3] += d;
   state[4] += e;

   // Wipe variables
   a = b = c = d = e = 0;
}

// SHA1Init - Initialize new context
void SHA1::init(SHA1_CONTEXT* context)
{
   // SHA1 initialization constants
   context->state[0] = 0x67452301;
   context->state[1] = 0xEFCDAB89;
   context->state[2] = 0x98BADCFE;
   context->state[3] = 0x10325476;
   context->state[4] = 0xC3D2E1F0;
   context->count[0] = context->count[1] = 0;
}

// Run your data through this
void SHA1::update(SHA1_CONTEXT* context, unsigned char* data, Q_UINT32 len)
{
   Q_UINT32 i, j;

   j = (context->count[0] >> 3) & 63;
   if((context->count[0] += len << 3) < (len << 3))
      context->count[1]++;

   context->count[1] += (len >> 29);

   if((j + len) > 63) {
      memcpy(&context->buffer[j], data, (i = 64-j));
      transform(context->state, context->buffer);
      for ( ; i + 63 < len; i += 64) {
         transform(context->state, &data[i]);
      }
      j = 0;
   }
   else i = 0;
   memcpy(&context->buffer[j], &data[i], len - i);
}

// Add padding and return the message digest
void SHA1::final(unsigned char digest[20], SHA1_CONTEXT* context)
{
   Q_UINT32 i, j;
   unsigned char finalcount[8];

   for (i = 0; i < 8; i++) {
      finalcount[i] = (unsigned char)((context->count[(i >= 4 ? 0 : 1)]
      >> ((3-(i & 3)) * 8) ) & 255);  // Endian independent
   }
   update(context, (unsigned char *)"\200", 1);
   while ((context->count[0] & 504) != 448) {
      update(context, (unsigned char *)"\0", 1);
   }
   update(context, finalcount, 8);  // Should cause a transform()
   for (i = 0; i < 20; i++) {
      digest[i] = (unsigned char) ((context->state[i>>2] >> ((3-(i & 3)) * 8) ) & 255);
   }

   // Wipe variables
   i = j = 0;
   memset(context->buffer, 0, 64);
   memset(context->state, 0, 20);
   memset(context->count, 0, 8);
   memset(&finalcount, 0, 8);
}


/* static */ QString SHA1::digest(QString in)
{
   SHA1_CONTEXT context;
   unsigned char digest[20];

   SHA1 s;
   QCString data = in.utf8();

   s.init(&context);
   s.update(&context, (unsigned char *)data.data(), (unsigned int)data.length());
   s.final(digest, &context);

   QString out;
   for(int n = 0; n < 20; ++n) {
      QString str;
      str.sprintf("%02x", digest[n]);
      out.append(str);
   }

   return out;
}


// base64
QByteArray Base64::encode(const QByteArray &s)
{
   int i;
   int len = s.size();
   char tbl[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
   int a, b, c;

   QByteArray p((len+2)/3*4);
   int at = 0;
   for( i = 0; i < len; i += 3 ) {
      a = ((unsigned char)s[i] & 3) << 4;
      if(i + 1 < len) {
         a += (unsigned char)s[i + 1] >> 4;
         b = ((unsigned char)s[i + 1] & 0xF) << 2;
         if(i + 2 < len) {
            b += (unsigned char)s[i + 2] >> 6;
            c = (unsigned char)s[i + 2] & 0x3F;
         }
         else
            c = 64;
      }
      else
         b = c = 64;

      p[at++] = tbl[(unsigned char)s[i] >> 2];
      p[at++] = tbl[a];
      p[at++] = tbl[b];
      p[at++] = tbl[c];
   }
   return p;
}

QByteArray Base64::decode(const QByteArray &s)
{
   // return value
   QByteArray p;

   // -1 specifies invalid
   // 64 specifies eof
   // everything else specifies data

   char tbl[] = {
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,62,-1,-1,-1,63,
      52,53,54,55,56,57,58,59,60,61,-1,-1,-1,64,-1,-1,
      -1, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,
      15,16,17,18,19,20,21,22,23,24,25,-1,-1,-1,-1,-1,
      -1,26,27,28,29,30,31,32,33,34,35,36,37,38,39,40,
      41,42,43,44,45,46,47,48,49,50,51,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
      -1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,
   };

   // this should be a multiple of 4
   int len = s.size();

   if(len % 4)
      return p;

   p.resize(len / 4 * 3);

   int i;
   int at = 0;

   int a, b, c, d;

   for( i = 0; i < len; i += 4 ) {
      a = tbl[s[i]];
      b = tbl[s[i + 1]];
      c = tbl[s[i + 2]];
      d = tbl[s[i + 3]];
      if((a == 64 || b == 64) || (a < 0 || b < 0 || c < 0 || d < 0)) {
         p.resize(0);
         return p;
      }
      p[at++] = ((a & 0x3F) << 2) | ((b >> 4) & 0x03);
      p[at++] = ((b & 0x0F) << 4) | ((c >> 2) & 0x0F);
      p[at++] = ((c & 0x03) << 6) | ((d >> 0) & 0x3F);
   }

   if(c & 64)
      p.resize(at - 2);
   if(d & 64)
      p.resize(at - 1);

   return p;
}

QString Base64::arrayToString(const QByteArray &a)
{
   QByteArray b = encode(a);
   QCString c;
   c.resize(b.size()+1);
   memcpy(c.data(), b.data(), b.size());
   return QString::fromLatin1(c);
}

QByteArray Base64::stringToArray(const QString &s)
{
	QCString c = s.latin1();
	QByteArray b = c;
	b.resize(b.size()-1); // take off the null
	return decode(b);
}

// arch-tag: zinv/util.cpp
