/*
 * kmtail.cpp
 *
 * Copyright (C) 2008 %{AUTHOR} <%{EMAIL}>
 */
#include "kmtail.h"
#include "kmconf.h"
#include "kmtailview.h"

#include <QtGui/QDropEvent>
#include <QtGui/QPainter>
#include <QtGui/QPrinter>
#include <QDragEnterEvent>
#include <QDropEvent>
#include <qdebug.h>
#include <qlabel.h>
#include <qtimer.h>
#include <qclipboard.h>

#include <kconfigdialog.h>
#include <kfiledialog.h>
#include <kstatusbar.h>
#include <kapplication.h>
#include <kurl.h>
#include <kmenubar.h>
#include <kfontdialog.h>
#include <kmessagebox.h>

#include <kaction.h>
#include <kactioncollection.h>
#include <kstandardaction.h>

#include <KDE/KLocale>
#include <ktoolbar.h>

#include <stdlib.h>

using namespace std;

KmTail::KmTail(): KXmlGuiWindow(),
      mpView(new KmTailView(this)),
			fTB( "filterToolBar" )

{
	setCaption( i18n( "multiple Tail" ), false );

	// accept dnd
	setAcceptDrops( true );

	// tell the KXmlGuiWindow that this is indeed the main widget
	setCentralWidget( mpView );

	// check the turbotail program is installed
	checkTail();

	// then, setup our actions
	setupActions();

	// add a status bar
	mpSBtext1 = new QLabel( i18n("Ready"), this );
	mpSBtext1->setFrameShape( QFrame::StyledPanel );
	mpSBtext1->setFrameShadow( QFrame::Sunken );
	statusBar()->addWidget(mpSBtext1);
	statusBar()->show();
	statusBar()->setToolTip( i18n("Maximum buffer size set at ") + QString::number(MAXLINES));

	// a call to KXmlGuiWindow::setupGUI() populates the GUI
	// with actions, using KXMLGUI.
	// It also applies the saved mainwindow settings, if any, and ask the
	// mainwindow to automatically save settings if changed: window size,
	// toolbar position, icon size, etc.
	//setupGUI( Keys | StatusBar | Save | Create, "/usr/src/dev/kde4/KmTail/src/kmtailui.rc" );
	
	setupGUI();
	
	// menu option to hide menu!
	mpMenuHide->setCheckable( true );
	if ( menuBar()->isVisible() )
		mpMenuHide->setChecked( true );

	// simple high-lighting system
	defCritTags << "critical";
	defErrTags  << "error" << "SYSERR" << "refused";
	defWarnTags << "warning" << "kernel" << "timeout" << "timed out";

	// read saved options
	mUsrFont = false;
	mConfig = KGlobal::config();
	mpGenGroup = new KConfigGroup( mConfig, "General" );
	readOptions();

	connect( mpView, SIGNAL(tabCloseRequested(int)), this, SLOT(closeTab(int)) );
	connect( mpView, SIGNAL(currentChanged(int)), this, SLOT(changedTab(int)) );
	connect( mpView->locTabBar(), SIGNAL(tabMoved(int, int)), this, SLOT(moveElems(int, int)) );
	
	if ( mFiles.count() > 1  )
		mpView->setCurrentIndex( 0 );                    // view first tab
	else if ( mFiles.count() == 1  )
		changedTab( 0 );																 // update status		
}

KmTail::~KmTail()
{
	saveOptions();

	for ( int k = 0; k < mFiles.count(); ++k )
		closeTab( k );
}

void KmTail::setupActions()
{
	KStandardAction::open(this, SLOT(fileNew()), actionCollection());
	KStandardAction::close(this, SLOT(fileClose()), actionCollection());
	KStandardAction::quit(kapp, SLOT(closeAllWindows()), actionCollection());

	KStandardAction::preferences(this, SLOT(confOptions()), actionCollection());

	mpFileReload = new KAction( KIcon("view-refresh"), i18n("&Reload"), this );
	actionCollection()->addAction( QLatin1String("reload"), mpFileReload );
	mpFileReload->setShortcut( Qt::Key_F5 );
	connect( mpFileReload, SIGNAL(triggered(bool)), this, SLOT(fileReload()) );

	mpMenuHide = new KAction( KIcon("show-menu"), i18n("Show &Menu"), this );
	actionCollection()->addAction( QLatin1String("menuhide"), mpMenuHide );
	mpMenuHide->setShortcut( Qt::CTRL+Qt::Key_M );
	connect( mpMenuHide, SIGNAL(triggered(bool)), this, SLOT(menuHide()) );

	// context menu for log entries
	QAction *action;
	mpPopup = new QMenu( mpView );
	mpPopup->addAction( mpMenuHide );
  mpPopup->addSeparator();
  action = mpPopup->addAction( KIcon("view-refresh"), QString(i18n("&Reload")) );
	action->setData( "reload" );
	action->setShortcut( Qt::Key_F5 );	
  mpPopup->addSeparator();
  action = mpPopup->addAction( KIcon("edit-select-all"), QString(i18n("Select &All")) );
	action->setData( "selectall" );
	action->setShortcut( Qt::CTRL+Qt::Key_A );	
  action = mpPopup->addAction( KIcon("edit-copy"), QString(i18n("&Copy")) );
	action->setData( "copy" );
	action->setShortcut( Qt::CTRL+Qt::Key_C );	

	// extra toolbar for filter
	mpLnEdit = new KLineEdit( this );
	mpLnEdit->setClearButtonShown( true );
	mpLnEdit->setClickMessage( i18n("Filter log entries") );
	mpLnEdit->setToolTip( i18n("Enter a search argument and press Enter") );
	
	mpEditIcon = new KAction( KIcon("view-filter"), "", this );
	mpEditIcon->setToolTip( i18n("View filter toolbar") );
	actionCollection()->addAction( "filter_icon", mpEditIcon );
	
	mpEdit = new KAction( i18n(""), this );
 	mpEdit->setDefaultWidget( mpLnEdit );
	actionCollection()->addAction( "filter_edit", mpEdit );
	connect( mpLnEdit, SIGNAL(returnPressed(QString)), this, SLOT(editDone(QString)) );
	connect( mpLnEdit, SIGNAL(clearButtonClicked()), this, SLOT(clearEdit()) ); 

	toolBar( fTB )->show();
}

void KmTail::fileNew()
{
	// get a new log file
	QString lf = KFileDialog::getOpenFileName( KUrl("/var/log") );
	if ( !lf.isEmpty() )
		addFile( lf );
}

void KmTail::fileClose()
{
	closeTab( mpView->currentIndex() );
}


void KmTail::saveOptions()
{
	mpGenGroup->writeEntry( "files", mFileNms );
	mpGenGroup->writeEntry( "item-font", mFont.toString() );
	mpGenGroup->writeEntry( "crit-tags", mCritTags );
	mpGenGroup->writeEntry( "warn-tags", mWarnTags );
	mpGenGroup->writeEntry( "err-tags", mErrTags );
	mpGenGroup->writeEntry( "crit-col", mRedB.color().name() );
	mpGenGroup->writeEntry( "err-col", mOrangeB.color().name() );
	mpGenGroup->writeEntry( "warn-col", mYellowB.color().name() );
	mpGenGroup->sync();
}

void KmTail::readOptions()
{
	QStringList files;
	QString font;

  font = mpGenGroup->readEntry( "item-font", "" );
	if ( !font.isEmpty() )
	{	
		mFont.fromString( font );
		mUsrFont = true;
	}	
	
  files = mpGenGroup->readEntry( "files", QStringList() );
	if ( !files.isEmpty() )
	{	
		for ( int k = 0; k < files.count(); ++k )
			addFile( files[k] );
	}	
	
	mCritTags = mpGenGroup->readEntry( "crit-tags", defCritTags ); 
	mErrTags  = mpGenGroup->readEntry( "err-tags", defErrTags ); 
	mWarnTags = mpGenGroup->readEntry( "warn-tags", defWarnTags ); 

	QString crit_col, err_col, warn_col;
	crit_col = mpGenGroup->readEntry( "crit-col", "#FF0000" ); 	// red
	err_col  = mpGenGroup->readEntry( "err-col", "#FFAA00" ); 	// orange
	warn_col = mpGenGroup->readEntry( "warn-col", "#FFFF00" ); 	// yellow
	
	mRedB.setColor( crit_col );
	mYellowB.setColor( warn_col );
	mOrangeB.setColor( err_col );
}

void KmTail::confOptions()
{
	KmConf opts( this );
	if ( opts.exec() == KDialog::Accepted )
	{
		saveOptions();
		fileReload();
	}	
}


void KmTail::addFile( const QString &rLogfile )
{
	KmFile lfile;

	lfile.mpTW = new QTreeWidget;
	lfile.mpTW->setRootIsDecorated( false );
	lfile.mpTW->setAllColumnsShowFocus( true );
	lfile.mpTW->setAlternatingRowColors( true );
	lfile.mpTW->setContextMenuPolicy( Qt::CustomContextMenu );
	lfile.mpTW->setColumnCount( 1 );
	lfile.mpTW->setUniformRowHeights( true );
	lfile.mpTW->setHeaderHidden( true );
	lfile.mpTW->setSelectionMode( QAbstractItemView::ExtendedSelection );
	//lfile.mpTW->setStyleSheet("* { font-family: monospace; }");
		

  connect( lfile.mpTW, SIGNAL(customContextMenuRequested(const QPoint &)),
     	     this, SLOT(rmbItem(const QPoint&)) );
	
	int p = rLogfile.lastIndexOf( '/' );
	lfile.mName = rLogfile.mid( p+1 );
	lfile.mFileNm = rLogfile;
	if ( lfile.mName.right( 4 ) == ".log" )
		lfile.mName.chop( 4 );
	mpView->addTab( lfile.mpTW, lfile.mName );
	
	startProc( lfile );																// start tail process
	
	mFiles << lfile;
	mFileNms << rLogfile;
	
	mpView->setCurrentWidget( lfile.mpTW ); 
	if ( mFiles.count() == 1  )
		changedTab( 0 );																 // update status		
}

void KmTail::startProc( KmFile &lfile )
{
	lfile.mpProc = new KProcess( this );	
	lfile.mpProc->setOutputChannelMode( KProcess::SeparateChannels );

	connect( lfile.mpProc, SIGNAL(readyReadStandardOutput()), this, SLOT(readStdOut()) );
	connect( lfile.mpProc, SIGNAL(readyReadStandardError()), this, SLOT(readStdErr()) );

	*lfile.mpProc << mTail << "--retry" << "--lines=" + QString::number(MAXLINES) 
								<< "--follow=name" << lfile.mFileNm;
	lfile.mpProc->start();
	
	if ( !lfile.mpProc->waitForStarted() )
	{
		qDebug() << "error: " << mTail.toAscii() << " " << lfile.mFileNm.toAscii() << "failed to start" << "\n";
	}	
}


void KmTail::readStdOut()
{
  char buf[4096]; int k; 
	bool found = false;
	
	for ( k = 0; k < mFiles.count(); ++k )
	{
		if ( mFiles[k].mpProc == QObject::sender() ) 
		{
			found = true;
			break;
		}	
	}	
	
	if ( !found ) return; 
	
	mFiles[k].mpProc->setReadChannel( QProcess::StandardOutput );
	while ( true )
	{	
		qint64 l = mFiles[k].mpProc->readLine( buf, sizeof(buf) );
		if ( l < 1 ) break; 
		
		QString line = QString( buf );
		
		if ( mFiles[k].mPartialLine )
		{	
			line.truncate( l-1 );
			mFiles[k].mLine += line;
			mFiles[k].mPartialLine = false;
		}	
		else if ( buf[l-1] != '\n' )
		{
			mFiles[k].mLine = line;
			mFiles[k].mPartialLine = true;
			return;                         // wait for rest of line to arrive
		}	
		else
		{	
			line.truncate( l-1 );
			mFiles[k].mLine = line;
		}	
		
		QTreeWidgetItem *pItem = new QTreeWidgetItem( mFiles[k].mpTW );
		pItem->setText( 0, mFiles[k].mLine );
		
		if ( mUsrFont )
			pItem->setFont( 0, mFont );
		
		if ( !mFiles[k].mFilter.isEmpty() )
			if ( pItem->text( 0 ).indexOf( mFiles[k].mFilter, 0, Qt::CaseInsensitive ) == -1 )
				pItem->setHidden( true );
		
		if ( !highLightItem( pItem, mCritTags, mRedB ) )
			if ( !highLightItem( pItem, mErrTags, mOrangeB ) )
				highLightItem( pItem, mWarnTags, mYellowB );
			
		line = mFiles[k].mLine;
		for ( int incr = 80; incr < line.length(); incr += 80 )
		{
			int p = line.indexOf( ' ', incr );
			if ( p > -1 ) 
				line.replace( p, 1, '\n' );
			else 
			{
				p = line.indexOf( ',', incr );
				if ( p > -1 )
					line.insert( p, '\n');
				else
					line.insert( incr, '\n' );
			}	
		}	
		
		pItem->setToolTip( 0, line );
	}
	
	while ( mFiles[k].mpTW->topLevelItemCount() > MAXLINES )
		delete mFiles[k].mpTW->takeTopLevelItem( 0 );
	
	if ( k == mpView->currentIndex() )
		changedTab( k );																// update visible status bar
	else
		mFiles[k].mpTW->scrollToBottom();
}

void KmTail::readStdErr()
{
  char buf[4096]; int k;
	bool openfail = false;
	
	for ( k = 0; k < mFiles.count(); ++k )
	{
		if ( mFiles[k].mpProc == QObject::sender() ) break;
	}	
		
	mFiles[k].mpProc->setReadChannel( QProcess::StandardError );
	while ( true )
	{	
		qint64 l = mFiles[k].mpProc->readLine( buf, sizeof(buf) );
		if ( l > 0 ) 
		{
			//QTreeWidgetItem *pItem = new QTreeWidgetItem( mFiles[k].mpTW );
			QString line( QString(buf).left( l-1 ) );
			//pItem->setText( 0, line );
			qDebug() << line;
			if ( line == "fstat64 failed" )
				openfail = true;                                // file cannot be opened
		}	
		else
			break;
	}
	
	if ( openfail )
	{
		KMessageBox::sorry( this, mFileNms[k] + i18n(" cannot be opened.") );
		// closeTab( k );               hangs app!
	}	
	else // delay reload while log is being compressed, moved, service restarted...
	{ 
		mFiles[k].mpTimer = new QTimer( this );
		mFiles[k].mpTimer->setSingleShot( true );;
		connect( mFiles[k].mpTimer, SIGNAL(timeout()), this, SLOT(reloadTimeout()) );
		mFiles[k].mpTimer->start( 4000 );
	}
}

void KmTail::reloadTimeout()
{
  int k;
	
	for ( k = 0; k < mFiles.count(); ++k )
	{
		if ( mFiles[k].mpTimer == QObject::sender() ) break;
	}	

	delete mFiles[k].mpTimer;
	fileReload( k );
}

void KmTail::closeTab( int k )
{
	mFiles[k].mpProc->close();
	mpView->removeTab( k );
	delete mFiles[k].mpTW;
	delete mFiles[k].mpProc;
	mFiles.removeAt( k );
	mFileNms.removeAt( k );
	
	k = mpView->currentIndex();									// where are we?
	changedTab( k );														// update status bar
}

void KmTail::changedTab( int k )
{
	if ( (k > -1) & (k < mFiles.count()) )
	{	
		mFiles[k].mpTW->scrollToBottom();
		mpSBtext1->setText( mFileNms[k] + ":   " + 
				QString::number( mFiles[k].mpTW->topLevelItemCount() ) + " lines." );
		mpLnEdit->setText( mFiles[k].mFilter );
	}
	else if ( k == -1 )
		mpSBtext1->setText( "Ready" );
}

void KmTail::fileReload( int k )
{
	if ( k == -1 )
	  k = mpView->currentIndex();	

	mFiles[k].mpProc->close();
	mFiles[k].mpTW->clear();
	delete mFiles[k].mpProc;
	
	startProc( mFiles[k] );
}

void KmTail::menuHide()
{
	if ( menuBar()->isVisible() )
	{
		menuBar()->hide();
		mpMenuHide->setChecked( false );
	}	
	else
	{
		menuBar()->show();
		mpMenuHide->setChecked( true );
	}	
}

void KmTail::dragEnterEvent( QDragEnterEvent *event )
{
	if ( event->mimeData()->hasUrls() )
		event->acceptProposedAction();
}

void KmTail::dropEvent( QDropEvent *event )
{
	KUrl url( event->mimeData()->text() );
	
	if ( url.scheme() == "file" )
	{	
		event->acceptProposedAction();
		addFile( url.toLocalFile() );
	}	
}

void KmTail::rmbItem( const QPoint &point )
{
	QAction *action;
	int k = mpView->currentIndex();
	
	action = mpPopup->exec( mFiles[k].mpTW->viewport()->mapToGlobal(point) );    
	
	if ( action == 0 )
		return;
	else if ( action->data().toString() == "reload" )
		fileReload( k );
	else if ( action->data().toString() == "selectall" )
		selectAllItems( k );
	else if ( action->data().toString() == "copy" )
		copySelection( k );
}

void KmTail::selectAllItems( int k )
{
	mFiles[k].mpTW->selectAll();
}

void KmTail::copySelection( int k )
{
	QList<QTreeWidgetItem *> selRows = mFiles[k].mpTW->selectedItems();
	QString selText;
	
	for ( int j = 0; j< selRows.count(); ++j )
		selText += selRows[j]->text( 0 ) + '\n';

	QClipboard *clipboard = QApplication::clipboard();
	clipboard->setText( selText );
}

void KmTail::setFont()
{
	for ( int k = 0; k < mFiles.count(); ++k )
	{
		for ( int j = 0; j < mFiles[k].mpTW->topLevelItemCount(); ++j )
			mFiles[k].mpTW->topLevelItem( j )->setFont( 0, mFont );
	}	

	mUsrFont = true;
}

void KmTail::checkTail()
{
  mTail = whereIs( TAIL );
	if ( mTail.isEmpty() )
	{
/*		KMessageBox::detailedError( this, "Turbotail program not found!",
			"Turbotail is not installed. Please download from\n\n"
			"  http://www.vanheusden.com/turbotail/turbotail-0.3.tgz \n\n"
			"or install from your distro.\nFalling back to GNU tail.");*/
		
		qDebug() << "turbotail not found - falling back to GNU tail";	
		mTail = whereIs( "tail" );	
	}		
}

QString KmTail::whereIs( const QString &bin )
{
	QString tmp, loc; char *ch;
	char line[128]; 	  
	FILE *fp;  

	fp = popen( QString("whereis -b " + bin).toAscii(), "r" );
	ch = fgets( line, sizeof( line ), fp );
	tmp = line;
	loc = tmp.section( ' ', 1, 1 ).trimmed();
  pclose(fp);        
	return loc;         
}
                              
void KmTail::editDone( QString arg )
{
	if ( arg.isEmpty() ) return;
 
	int k = mpView->currentIndex();
	mFiles[k].mFilter = arg;
 
	for ( int j = 0; j < mFiles[k].mpTW->topLevelItemCount(); ++j )
	{
		if ( mFiles[k].mpTW->topLevelItem( j )->text( 0 ).indexOf( arg, 0, Qt::CaseInsensitive ) == -1 )
			mFiles[k].mpTW->topLevelItem( j )->setHidden( true );
	}
}

void KmTail::clearEdit()
{
	int k = mpView->currentIndex();
	mFiles[k].mFilter.clear();
 
	for ( int j = 0; j < mFiles[k].mpTW->topLevelItemCount(); ++j )
	{
		if ( mFiles[k].mpTW->topLevelItem( j )->isHidden() )
			mFiles[k].mpTW->topLevelItem( j )->setHidden( false );
	}
	
	mFiles[k].mpTW->scrollToBottom();
}

bool KmTail::highLightItem( QTreeWidgetItem *pItem , QStringList &tags , QBrush &brush )
{
	bool found = false;
	
	for ( int j = 0; j < tags.count(); ++j )
	{
		if ( pItem->text( 0 ).indexOf( tags[j], 0, Qt::CaseInsensitive ) >= 0 )
		{	
			pItem->setForeground( 0, brush );
			found = true;
			break;
		}
	}	
	
	return found;
}

void KmTail::moveElems( int from, int to )
{
	mFiles.move( from, to );
	mFileNms.move( from, to );
}

#include "kmtail.moc"
