/*
 * This file is part of QTGZManager, an open-source GUI for Slackware pkgtools.
 * Copyright (C) 2006  Alexandre Albuquerque Arnt
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact e-mail : Alexandre Albuquerque Arnt <aaarnt@gmail.com>
 * Program URL   : http://jtgzmanager.sf.net
 *
 */

#include <iostream>
#include "mainwindowimpl.h"
#include "package.h"
#include "tvpackagesitemdelegate.h"
#include "unixcommand.h"
#include "argumentlist.h"
#include <QtGui>
#include <QFutureWatcher>

using namespace QtConcurrent;

MainWindowImpl::MainWindowImpl( QWidget * parent, Qt::WFlags f) 
:	QMainWindow(parent, f), 
ctn_APPLICATION_NAME("QTGZManager 1.0 - Beta5"),
ctn_LABEL_TREEVIEW_PACKAGES(tr("0 Packages in Directory")),
m_PackagesInDirFontSize(9),
m_InstalledPackagesFontSize(9),
m_DirectoryFontSize(8),
m_TodoFontSize(9),
m_PkgListFontSize(9),
m_ToolTipFontSize(9),
ctn_TODO_INSTALL(tr("To be installed")),
ctn_TODO_REMOVE(tr("To be removed")),
ctn_TODO_DOWNGRADE(tr("To be downgraded")),
ctn_TODO_UPGRADE(tr("To be upgraded")),
ctn_TODO_REINSTALL(tr("To be reinstalled")),
ctn_ICON_INTERNAL_ERROR(QIcon(":/resources/images/internalerror.png")),
ctn_ICON_INFERIOR(QIcon(":/resources/images/inferiorversion_red.png")),
ctn_ICON_SUPERIOR(QIcon(":/resources/images/superiorversion.png")),
ctn_ICON_OTHER_VERSION(QIcon(":/resources/images/agent.png")),
ctn_ICON_INSTALLED(QIcon(":/resources/images/installed.png")),
ctn_ICON_NOT_INSTALLED(QIcon(":/resources/images/notinstalled.png")),
ctn_ICON_FOLDER(QIcon(":/resources/images/folder.png")),
ctn_ICON_BINARY(QIcon(":/resources/images/binary.png")),
ctn_ICON_FROZEN(QIcon(":/resources/images/tgz_frozen.png")),
ctn_ICON_UNFROZEN(QIcon(":/resources/images/tgz4.png")),
ctn_ICON_RPM(QIcon(":/resources/images/rpm.png"))
{
  QFont font("Verdana");
  font.setPointSize(m_ToolTipFontSize);
  QToolTip::setFont(font);
  setStyleSheet("QMenu { font: 11px; }");

  m_fw = new QFutureWatcher<QStringList>(this);
  m_argList = new ArgumentList();

  QFile::remove(ctn_TEMP_ACTIONS_FILE);
  m_unixCommand = 0;
  m_reallyWannaClose = false;
  m_frozenPkgList = FrozenPkgListSingleton::instance();
	m_clickedTab = -1;  
  m_findDialog = 0;
	setupUi( this );	  
  setWindowTitle(ctn_APPLICATION_NAME);
  dockPackages->setWindowTitle(ctn_LABEL_TREEVIEW_PACKAGES);

  QIcon icon(":/resources/images/QTGZManager.png");
	setWindowIcon(icon);    
  m_psw = new QFileSystemWatcher(QStringList() << ctn_PACKAGES_DIR, this);
  m_fsw.addPath( QDir::homePath() );
  InstalledPkgListSingleton::instance()->setFileSystemWatcher(m_psw);
  setMinimumSize(QSize(640, 480));

  initializeActions();
  initializeMenuBar();
  initializeToolBar();
  initializePackageTreeView();
  initializeInstalledPackagesTreeView();
	initializeTodoTreeView();

  m_textForMoreInformation = tr("<small>For more information about QTGZManager visit: %1</small>").
      arg("<a href=\"http://jtgzmanager.sf.net\" target=\"_blank\">http://jtgzmanager.sf.net</a>");

  twTODO->setTabPosition(QTabWidget::North);
  twTODO->setUsesScrollButtons(true);
	twTODO->setElideMode(Qt::ElideNone);
	m_tabBar = twTODO->findChild<QTabBar*>();
  m_tabBar->setObjectName("tabBar");
	m_tabBar->installEventFilter(this);
  m_tabBar->setFont(font);
  m_btnCloseTab = new QToolButton( twTODO );
  m_btnCloseTab->setAutoRaise( true );
  m_btnCloseTab->setIcon(QIcon(":/resources/images/close.png"));
  twTODO->setCornerWidget( m_btnCloseTab );
  m_btnCloseTab->setToolTip(tr("Close tab"));
  m_btnCloseTab->setEnabled( false );
  m_btnCloseTab->show();
  m_lblStatus = new QLabel( this );
  m_lblStatus->setFrameStyle( QFrame::NoFrame );
  m_statusbar = new QSimpleStatusBar( this );
  setStatusBar( m_statusbar );
  statusBar()->addPermanentWidget( m_lblStatus, true );
  m_savedGeometry = this->saveGeometry();
  m_systemTrayIcon = new QSystemTrayIcon( icon, this );
  m_systemTrayIcon->setToolTip( ctn_APPLICATION_NAME );
  m_systemTrayIcon->show();
  m_systemTrayIconMenu = new QMenu( this );
  m_actionAbout = new QAction( this );
  m_actionAbout->setIcon(QIcon((":/resources/images/QTGZManager.png")));
  m_actionAbout->setText( tr("About..."));
  m_systemTrayIconMenu->addAction( m_actionAbout );
  m_systemTrayIconMenu->addAction( actionExit );
  m_systemTrayIcon->setContextMenu( m_systemTrayIconMenu );

  connect(tvDir, SIGNAL(clicked(const QModelIndex)), this, SLOT(changeDir()));
	connect(tvDir, SIGNAL(activated(const QModelIndex)), tvDir, SIGNAL(clicked(const QModelIndex)));
	connect(tvDir, SIGNAL(clicked(const QModelIndex)), this, SLOT(showPackagesInDirectory()));	
	connect( tvPackage->header(), SIGNAL( sectionClicked ( int )), 
			this, SLOT( headerViewPackageList_click( int ) ) );
	connect( tvInstalledPackages->header(), SIGNAL( sectionClicked ( int )), 
			this, SLOT( headerViewInstalledPackageList_click( int ) ) );
	connect(tvPackage, SIGNAL(clicked(const QModelIndex)), this, SLOT(selectInstalledPackage()));
  connect(tvDir, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(execContextMenuDirectories(QPoint)));
	connect(tvTODO, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(execContextMenutvTODO(QPoint)));
	connect(tvPackage, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(execContextMenuPackages(QPoint)));
  connect(tvPackage, SIGNAL(doubleClicked(QModelIndex)), this, SLOT(showPackageContent()));
  connect(tvInstalledPackages, SIGNAL(customContextMenuRequested(QPoint)),
		this, SLOT(execContextMenuInstalledPackages(QPoint)));
  connect(tvInstalledPackages, SIGNAL(doubleClicked(QModelIndex)), this, SLOT(showPackageContent()));
  connect ( m_psw, SIGNAL(directoryChanged ( const QString &)),
		this, SLOT(fileSystemWatcher_packageDirectoryChanged ( const QString &) ));
  connect ( &m_fsw, SIGNAL(directoryChanged ( const QString &)),
    this, SLOT(fileSystemWatcher_directoryChanged ( const QString &) ));
  connect(m_modelTodo, SIGNAL(rowsInserted ( const QModelIndex , int, int )),
		this, SLOT(verifyStateTODO ( const QModelIndex , int, int )));
  connect(m_modelTodo, SIGNAL(rowsRemoved ( const QModelIndex , int, int )),
		this, SLOT(verifyStateTODO ( const QModelIndex , int, int )));
  connect( m_btnCloseTab, SIGNAL(clicked()), this, SLOT(closeCurrentTab()) );
	connect( twTODO, SIGNAL(currentChanged(int)), this, SLOT(enableCloseTabButton(int)) );
  connect( m_actionAbout, SIGNAL(triggered()), this, SLOT(aboutQTGZManager()));
  connect ( m_systemTrayIcon , SIGNAL( activated( QSystemTrayIcon::ActivationReason ) ),
		this, SLOT( execSystemTrayActivated ( QSystemTrayIcon::ActivationReason ) ) );
  connect( bFilterPackage, SIGNAL(pressed()), this, SLOT(clearFilterPackage()));
  connect( bFilterInstalledPackage, SIGNAL(pressed()), this, SLOT(clearFilterInstalledPackage()));
}

MainWindowImpl::~MainWindowImpl(){
  delete m_argList;
  delete m_modelDir;
  delete m_modelPackage;
  delete m_modelInstalledPackages;
  delete m_install;
  delete m_remove;
  delete m_downgrade;
  delete m_upgrade;
  delete m_reinstall;
  delete m_modelTodo;
}

void MainWindowImpl::loadSettings(){
  QString qtWindowStyle;
  m_verticalSplit = splitterVertical->saveState();
  m_horizontalSplit = splitterHorizontal->saveState();

  int fourPanelOrganizing = SettingsManager::instance()->getFourPanelOrganizing();

  if (fourPanelOrganizing % 2 != 0){
    actionHideRightView->setChecked(true);
    fourPanelOrganizing--;
  }

  switch(fourPanelOrganizing){
    case ectn_NORMAL : normalView(); actionNormalView->setChecked(true); break;
    case ectn_HIDE_BOTTON : minimizeLowerView(); actionMinimizeLowerView->setChecked(true); break;
    case ectn_MAXIMIZED : maximizeLowerView(); actionMaximizeLowerView->setChecked(true); break;
  }

  toolBar->setHidden(!SettingsManager::instance()->getShowToolBar());
  m_PackageListOrderedCol = SettingsManager::instance()->getPackageListOrderedCol();
  m_InstalledPackageListOrderedCol = SettingsManager::instance()->getInstalledPackageListOrderedCol();
  m_PackageListSortOrder = (Qt::SortOrder) SettingsManager::instance()->getPackageListSortOrder();
  m_InstalledPackageListSortOrder = (Qt::SortOrder) SettingsManager::instance()->getInstalledPackageListSortOrder();
  m_DefaultDirectory = SettingsManager::instance()->getDefaultDirectory();
  m_actionIconifyOnStart->setChecked(SettingsManager::instance()->getStartIconified());

  tvPackage->header()->setSortIndicator( m_PackageListOrderedCol, m_PackageListSortOrder );
  tvPackage->sortByColumn( m_PackageListOrderedCol, m_PackageListSortOrder );
  tvInstalledPackages->header()->setSortIndicator( m_InstalledPackageListOrderedCol, m_InstalledPackageListSortOrder );
  tvInstalledPackages->sortByColumn( m_InstalledPackageListOrderedCol, m_InstalledPackageListSortOrder );

  initializeDirTreeView();
  showPackagesInDirectory();
}

void MainWindowImpl::saveSettings(){
  int fourPanelOrganizing = 0;

  if (actionHideRightView->isChecked()) fourPanelOrganizing += ectn_HIDE_RIGHT;
  if (actionNormalView->isChecked()) fourPanelOrganizing += ectn_NORMAL;
  if (actionMinimizeLowerView->isChecked()) fourPanelOrganizing += ectn_HIDE_BOTTON;
  if (actionMaximizeLowerView->isChecked()) fourPanelOrganizing += ectn_MAXIMIZED;

  SettingsManager::instance()->setShowToolBar(!toolBar->isHidden());
  SettingsManager::instance()->setFourPanelOrganizing(fourPanelOrganizing);
  SettingsManager::instance()->setPackageListOrderedCol(m_PackageListOrderedCol);
  SettingsManager::instance()->setInstalledPackageListOrderedCol(m_InstalledPackageListOrderedCol);
  SettingsManager::instance()->setPackageListSortOrder(m_PackageListSortOrder);
  SettingsManager::instance()->setInstalledPackageListSortOrder(m_InstalledPackageListSortOrder);
  SettingsManager::instance()->setDefaultDirectory(m_DefaultDirectory);
  SettingsManager::instance()->setStartIconified(m_actionIconifyOnStart->isChecked());
}

void MainWindowImpl::show(){
  static bool firstExecution=true;
  static bool settingsRetrivied=false;

  if (firstExecution){
    firstExecution = false;

    if (m_argList->getSwitch("-starticonified") || (SettingsManager::instance()->getStartIconified())){
      loadSettings();
      hide();
    }
    else
      showMaximized();

    if (!isHidden()){
      loadSettings();
      settingsRetrivied = true;
    }
  }
  else{
    QMainWindow::show();
    if (!settingsRetrivied){
      loadSettings();
      settingsRetrivied = true;
    }
  }
}

void MainWindowImpl::changeDir(){
  dockDirectories->setWindowTitle(m_modelDir->filePath(tvDir->currentIndex()));
}

bool MainWindowImpl::eventFilter(QObject *obj, QEvent *event){
  if (obj->objectName() == "tabBar" && event->type() == QEvent::MouseButtonPress){
		QMouseEvent *mouseEvent = static_cast<QMouseEvent *>(event);
		for (int i=0; i<m_tabBar->count(); i++){
			if ( m_tabBar-> tabRect(i).contains( mouseEvent->pos() ) ){
				m_clickedTab = i;
				break;
			}
		}
		if  ((mouseEvent != 0) && ( mouseEvent->button() == Qt::RightButton ) && 
				(m_clickedTab != 0) && (m_clickedTab != 1)) {
			QMenu *menu = new QMenu(this);
			connect(menu->addAction(QIcon(":/resources/images/close.png"), tr("Close tab")), SIGNAL(triggered()), this, SLOT(closeClickedTab()) );
			if (m_tabBar->count() > 3) 
				connect(menu->addAction(QIcon(":/resources/images/close.png"), tr("Close all tabs")), SIGNAL(triggered()), this, SLOT(closeAllTabs()) );
			menu->exec(mouseEvent->globalPos());
			m_clickedTab = -1;
		}
	}
  else if (obj->objectName() == "tvPkgFileList" && event->type() == QEvent::KeyPress){
    QTreeView *tv = static_cast<QTreeView *>(obj);
    QKeyEvent *keyEvent = static_cast<QKeyEvent *>(event);

    if (keyEvent->key() == Qt::Key_Return) openFileOrDirectory(tv->currentIndex());
  }

  return false;
}

void MainWindowImpl::fileSystemWatcher_directoryChanged ( const QString& ){
	showPackagesInDirectory();
}

void MainWindowImpl::fileSystemWatcher_packageDirectoryChanged( const QString& ){
  refreshInstalledPackageTreeView();
  showPackagesInDirectory();
}

void MainWindowImpl::headerViewPackageList_click( int col ){
	m_PackageListOrderedCol = col;
	m_PackageListSortOrder = tvPackage->header()->sortIndicatorOrder();
}

void MainWindowImpl::headerViewInstalledPackageList_click( int col ){
	m_InstalledPackageListOrderedCol = col;
	m_InstalledPackageListSortOrder = tvInstalledPackages->header()->sortIndicatorOrder();
}

void MainWindowImpl::enableCloseTabButton( int ){
	if (twTODO->count() <= 2){
    m_btnCloseTab->setEnabled( false );
		twTODO->removeAction( actionCloseTab );
    m_lblStatus->setText("");
	}
	else if (twTODO->count() > 2){
    if (twTODO->currentIndex() > 1) m_btnCloseTab->setEnabled( true );
    else m_btnCloseTab->setEnabled( false );

		if (twTODO->count() >= 3){
			if ((twTODO->count() > 3) && (twTODO->currentWidget()->findChild<QAction*>("actionCloseAllTabs") == 0))
				twTODO->currentWidget()->addAction( actionCloseAllTabs );	

			//we verify if there is a tvPkgFileList active
			QTreeView *t = twTODO->currentWidget()->findChild<QTreeView*>("tvPkgFileList");
			if ( t != 0){ 
				t->setFocus();
				showFullPathOfObject( t->currentIndex() );
			}
			else{ 
        m_lblStatus->setText("");
			}
		}
    else twTODO->currentWidget()->removeAction( actionCloseAllTabs );
	}
}

void MainWindowImpl::verifyStateTODO( const QModelIndex , int, int ){
  if (( m_modelTodo->rowCount(m_modelTodo->item(0, 0)->index()) != 0 ) || ( m_modelTodo->rowCount(m_modelTodo->item(1, 0)->index()) != 0 ) ||
      ( m_modelTodo->rowCount(m_modelTodo->item(2, 0)->index()) != 0 ) || ( m_modelTodo->rowCount(m_modelTodo->item(3, 0)->index()) != 0 ) ||
      ( m_modelTodo->rowCount(m_modelTodo->item(4, 0)->index()) != 0 ) )
		actionExecuteActions->setEnabled( true );
	else actionExecuteActions->setEnabled( false );
}

void MainWindowImpl::insertInstallPackageAction(){		
  if (m_unixCommand != 0) return;
  foreach(SelectedPackage sp, getSelectedPackage()){
    if ((m_modelTodo->findItems(sp.getCompleteFileName(), Qt::MatchRecursive).size()==0)
			&& (Package::isValid(sp.getCompleteFileName())))
      m_install->appendRow(new QStandardItem(sp.getCompleteFileName()));  
  }

	if (!tvTODO->isExpanded(tvTODO->currentIndex())) tvTODO->expandAll();
	if (twTODO->currentIndex()!=0) twTODO->setCurrentIndex(0);
  conditionalGotoNormalView();
}

void MainWindowImpl::insertDowngradePackageAction(){		
  if (m_unixCommand != 0) return;
  foreach(SelectedPackage sp, getSelectedPackage()){
    if ((m_modelTodo->findItems(sp.getCompleteFileName(), Qt::MatchRecursive).size()==0)
			&& (Package::isValid(sp.getCompleteFileName())))
      m_downgrade->appendRow(new QStandardItem(sp.getCompleteFileName()));
	}

	if (!tvTODO->isExpanded(tvTODO->currentIndex())) tvTODO->expandAll();
  if (twTODO->currentIndex()!=0) twTODO->setCurrentIndex(0);
  conditionalGotoNormalView();
}

void MainWindowImpl::insertRemovePackageAction(){		
  if (m_unixCommand != 0) return;
  if (tvPackage->hasFocus()){
		foreach(SelectedPackage sp, getSelectedPackage()){  
      if (((m_modelTodo->findItems(sp.getCompleteFileName(), Qt::MatchRecursive).size()==0))
				&& (Package::isValid(sp.getCompleteFileName())))
        m_remove->appendRow(new QStandardItem(sp.getCompleteFileName()));
		}
	}
	else if (tvInstalledPackages->hasFocus()){
		foreach(SelectedPackage sp, getSelectedInstalledPackage()){      
      if ((m_modelTodo->findItems(sp.getCompleteFileName(), Qt::MatchRecursive).size()==0))
        m_remove->appendRow(new QStandardItem(sp.getCompleteFileName()));
		} 
	}

	if (!tvTODO->isExpanded(tvTODO->currentIndex())) tvTODO->expandAll();
	if (twTODO->currentIndex()!=0) twTODO->setCurrentIndex(0);
  conditionalGotoNormalView();
}

void MainWindowImpl::insertUpgradePackageAction(){		
  if (m_unixCommand != 0) return;
  foreach(SelectedPackage sp, getSelectedPackage()){
    if ((m_modelTodo->findItems(sp.getCompleteFileName(), Qt::MatchRecursive).size()==0)
				&& (Package::isValid(sp.getCompleteFileName())))
      m_upgrade->appendRow(new QStandardItem(sp.getCompleteFileName()));
	}

	if (!tvTODO->isExpanded(tvTODO->currentIndex())) tvTODO->expandAll();
	if (twTODO->currentIndex()!=0) twTODO->setCurrentIndex(0);
  conditionalGotoNormalView();
}

void MainWindowImpl::insertReinstallPackageAction(){		
  if (m_unixCommand != 0) return;
  foreach(SelectedPackage sp, getSelectedPackage()){
    if ((m_modelTodo->findItems(sp.getCompleteFileName(), Qt::MatchRecursive).size()==0)
				&& (Package::isValid(sp.getCompleteFileName())))
      m_reinstall->appendRow(new QStandardItem(sp.getCompleteFileName()));
	}

	if (!tvTODO->isExpanded(tvTODO->currentIndex())) tvTODO->expandAll();
	if (twTODO->currentIndex()!=0) twTODO->setCurrentIndex(0);
  conditionalGotoNormalView();
}

void MainWindowImpl::execContextMenuDirectories(QPoint point){	
	showPackagesInDirectory();
	QModelIndex mi = tvDir->currentIndex();
	QMenu menu(this);            

  if (!isFindDisabled()) menu.addAction(m_actionFindPackage);
  menu.addAction(actionOpenDirectory);
  menu.addAction(m_actionOpenTerminal);

  if (dockDirectories->windowTitle() != m_DefaultDirectory)
    menu.addAction(m_actionChangeDefaultDirectory);

  menu.addAction(actionCreate_Directory);
	menu.addAction(actionRemove_Directory);        		        		       		        		

	QPoint pt2 = tvDir->mapToGlobal(point);
	pt2.setY(pt2.y() + tvDir->header()->height());
	menu.exec(pt2);        	        	
}

void MainWindowImpl::diffToEachOther(){
  tvPackage->repaint(tvPackage->rect());
	QCoreApplication::processEvents();	
  CPUIntensiveComputing ri;
  SelectedPackage pkg1 = getSelectedPackage()[0];
	SelectedPackage pkg2 = getSelectedPackage()[1];

	for (int c=2; c<twTODO->count(); c++)
    if (twTODO->tabText(c) == ( pkg1.getFileName() + " DIFF")){
			twTODO->setCurrentIndex ( c );			
      conditionalGotoNormalViewSimple();
      return;
		} 

  QString out = UNIXCommand::executeDiffToEachOther(pkg1.getCompleteFileName(), pkg2.getCompleteFileName());
  if (out == 0){
    QMessageBox::critical(MainWindowImpl::returnMainWindow(), QObject::tr("Diff"),
                          QObject::tr("This package seems corrupted!"));
    return;
  }
  else if (out == ctn_PACKAGES_WITH_SAME_CONTENT){
    QMessageBox::information(MainWindowImpl::returnMainWindow(), QObject::tr("Diff"),
                          QObject::tr("The packages have the same file list!"));
    return;
  }

  conditionalGotoNormalViewSimple();
  createTabPkgDiff(pkg1.getFileName(), pkg2.getFileName(), out);
}

void MainWindowImpl::diffToInstalled(){
  tvPackage->repaint(tvPackage->rect());
	QCoreApplication::processEvents();	
  CPUIntensiveComputing ri;
  SelectedPackage pkg = getSelectedPackage()[0];

	for (int c=2; c<twTODO->count(); c++)
    if (twTODO->tabText(c) == ( pkg.getFileName() + " DIFF(inst)")){
			twTODO->setCurrentIndex ( c );			
      conditionalGotoNormalViewSimple();
      return;
		} 

  Result res = Package::getStatus(pkg.getFileName());  

  QString out = UNIXCommand::executeDiffToInstalled(pkg.getCompleteFileName(), res.getInstalledPackage());
  if (out == 0){
    QMessageBox::critical(MainWindowImpl::returnMainWindow(), QObject::tr("Diff"),
                          QObject::tr("This package seems corrupted!"));
    return;
  }

  conditionalGotoNormalViewSimple();
  createTabPkgDiff(pkg.getFileName(), res.getInstalledPackage(), out);
}

void MainWindowImpl::execContextMenuPackages(QPoint point){
  if ((m_modelPackage->rowCount()==0) || (tvPackage->selectionModel()->selectedRows().count()==0)) return;

	QMenu *menu = new QMenu(this);            
	QIcon icon;  

	//Which actions do we have to put? It depends on package status
	bool allSameType=true;
	bool allTGZ=false;	int tgzCount=0;
	bool allRPM=false; int rpmCount=0;
  qint64 lastType=-9999;

	QList<SelectedPackage> lsp = getSelectedPackage();	

	foreach(SelectedPackage ami, lsp){  
    if ((lastType != -9999) && (lastType != ami.getIcon().cacheKey())) allSameType = false;
    if (ami.getIcon().pixmap(QSize(22,22)).toImage() == ctn_ICON_RPM.pixmap(QSize(22,22)).toImage())
			rpmCount++;
		else tgzCount++;
    lastType = ami.getIcon().cacheKey();
	}

	if ( tgzCount == lsp.count() ) {
		allTGZ = true;
		allRPM = false;
	}
	else if ( rpmCount == lsp.count() ) {
		allTGZ = false;
		allRPM = true;
	}

	if (allTGZ){
		//First Info, so when the user mistakenly clicks in it, QTGZ does less processing.
		menu->addAction(actionPackageContent);

		if (tgzCount == 1){
			menu->addAction(actionPackageInfo); 	 
      if (!isFindDisabled()) menu->addAction(m_actionFindFileInPackage);
		}
	}
  else if ( (allRPM) && (UNIXCommand::hasTheExecutable(ctn_RPM2TGZBIN)) ){
    menu->addAction(m_actionTransformRPMinTGZ);
	}

	if ( lsp.count() == 1 ){   
    //Here, we include the logic for package diff to installed
		icon = lsp[0].getIcon();
    if ( (icon.pixmap(QSize(22,22)).toImage() == ctn_ICON_SUPERIOR.pixmap(QSize(22,22)).toImage()) ||
        (icon.pixmap(QSize(22,22)).toImage() == ctn_ICON_INFERIOR.pixmap(QSize(22,22)).toImage()) ||
        (icon.pixmap(QSize(22,22)).toImage() == ctn_ICON_OTHER_VERSION.pixmap(QSize(22,22)).toImage()) ) {
      menu->addAction(actionDiffToInstalled);
		}
	}
	else if (( lsp.count() == 2 ) && (allTGZ) ) {  
    //Here, we include the logic for package diff to each other
		if ( lsp[0].getFileName() != lsp[1].getFileName()){
			QString s1 = Package::getBaseName(lsp[0].getFileName());
			QString s2 = Package::getBaseName(lsp[1].getFileName());
      if (s1 == s2) menu->addAction(m_actionDiffToEachOther);
		}
	}

	if ((allSameType) && (allTGZ) ) {
		icon = lsp[0].getIcon();
    if (icon.pixmap(QSize(22,22)).toImage() == ctn_ICON_INFERIOR.pixmap(QSize(22,22)).toImage()) {
			menu->addAction(actionDelete_file);
			menu->addAction(actionDowngrade_package);        		        		
		}
    else if (icon.pixmap(QSize(22,22)).toImage() == ctn_ICON_SUPERIOR.pixmap(QSize(22,22)).toImage()){
			menu->addAction(actionDelete_file);
			menu->addAction(actionUpgrade_package);        		        		       		        		
		}
    else if (icon.pixmap(QSize(22,22)).toImage() == ctn_ICON_OTHER_VERSION.pixmap(QSize(22,22)).toImage()){
			menu->addAction(actionDelete_file);
			menu->addAction(actionUpgrade_package);        		        		       		        		
		}
    else if (icon.pixmap(QSize(22,22)).toImage() == ctn_ICON_INSTALLED.pixmap(QSize(22,22)).toImage()){
			menu->addAction(actionDelete_file);
			menu->addAction(actionReinstall_package);
			menu->addAction(actionRemove_package);        		        		       		        		       		
		}
    else if (icon.pixmap(QSize(22,22)).toImage() == ctn_ICON_NOT_INSTALLED.pixmap(QSize(22,22)).toImage()){
			menu->addAction(actionDelete_file);
			menu->addAction(actionInstall_package);        		        		       		        		       		
		}        	
    else if (icon.pixmap(QSize(22,22)).toImage() == ctn_ICON_INTERNAL_ERROR.pixmap(QSize(22,22)).toImage()) {
			menu->addAction(actionDelete_file);        		   		
		}
    else if (icon.pixmap(QSize(22,22)).toImage() == ctn_ICON_FROZEN.pixmap(QSize(22,22)).toImage()) {
			menu->addAction(actionDelete_file);        		   		
		}
	}
	else{
		menu->addAction(actionDelete_file);
	}	

	QPoint pt2 = tvPackage->mapToGlobal(point);
	pt2.setY(pt2.y() + tvPackage->header()->height());
	menu->exec(pt2);        	
}

void MainWindowImpl::execContextMenuInstalledPackages(QPoint point){
  if ((m_modelInstalledPackages->rowCount()==0) || (tvInstalledPackages->selectionModel()->selectedIndexes().count()==0)) return;

	QModelIndex mi = tvInstalledPackages->currentIndex();
	QMenu *menu = new QMenu(this);               	

	//Which actions do we have to put? It depends on package status
	menu->addAction(actionPackageContent);

	if ( tvInstalledPackages->selectionModel()->selectedIndexes().count() == 2){
		menu->addAction(actionInstalledPackageInfo);
    if (!isFindDisabled()) menu->addAction(m_actionFindFileInPackage);
  }

	bool allSameType = true;
  qint64 lastType=-9999;

	foreach(QModelIndex item, tvInstalledPackages->selectionModel()->selectedIndexes()){
    if ( item.column() == ctn_PACKAGE_ICON ){
      QModelIndex mi = m_proxyModelInstalledPackages->mapToSource(item);
      if ((lastType != -9999) && (lastType != m_modelInstalledPackages->item( mi.row(), mi.column() )->icon().cacheKey())){
				allSameType = false; 
				break;
			}
      lastType = m_modelInstalledPackages->item( mi.row(), mi.column() )->icon().cacheKey();
		}
	}

	if ( allSameType == true ){
    if ( lastType == ctn_ICON_FROZEN.cacheKey() ) menu->addAction(m_actionUnfreezePkg);
		else {
      menu->addAction(m_actionFreezePkg);
      menu->addAction(actionRemove_package);
		}
	}	        		        		       		        		       		

	QPoint pt2 = tvInstalledPackages->mapToGlobal(point);
	pt2.setY(pt2.y() + tvInstalledPackages->header()->height());
	menu->exec(pt2);        	
}

void MainWindowImpl::conditionalGotoNormalView(){
  if (!actionMaximizeLowerView->isChecked() && !actionNormalView->isChecked()){
    actionNormalView->setChecked(true);
    normalView();
  }
}

void MainWindowImpl::conditionalGotoNormalViewSimple(){
  if (!actionNormalView->isChecked()){
    actionNormalView->setChecked(true);
    normalView();
  }
}

void MainWindowImpl::transformTGZinLZM(){
  conditionalGotoNormalView();

  QStringList out;
  int tgzCount=0;
	QList<SelectedPackage> lsp = getSelectedPackage();	

	foreach(SelectedPackage ami, lsp){  
    if (ami.getIcon().pixmap(QSize(22,22)).toImage() != ctn_ICON_RPM.pixmap(QSize(22,22)).toImage()) tgzCount++;
	}

	if ( tgzCount == lsp.count() ){
		foreach(SelectedPackage sp, lsp){  
      out << sp.getCompleteFileName();
		}

    m_unixCommand = new UNIXCommand(this);

    QObject::connect(m_unixCommand, SIGNAL( started() ), this, SLOT( TGZ2LZMprocessStarted()));
    QObject::connect(m_unixCommand, SIGNAL( readyReadStandardOutput() ), this, SLOT( TGZ2LZMprocessReadOutput() ));
    QObject::connect(m_unixCommand, SIGNAL( finished ( int, QProcess::ExitStatus )),
                     this, SLOT( TGZ2LZMprocessFinished(int, QProcess::ExitStatus) ));
    QObject::connect(m_unixCommand, SIGNAL( readyReadStandardError() ), this, SLOT( TGZ2LZMprocessRaisedError() ));

    if (UNIXCommand::hasTheExecutable(ctn_MAKELZMBIN))
      m_unixCommand->transformTGZinLZM(out, ectn_MAKELZM);
    else if (UNIXCommand::hasTheExecutable(ctn_TGZ2LZMBIN))
      m_unixCommand->transformTGZinLZM(out, ectn_TGZ2LZM);
	}
}

void MainWindowImpl::transformRPMinTGZ(){
  QStringList out;
  QList<SelectedPackage> lsp = getSelectedPackage();

	foreach(SelectedPackage sp, lsp){
    out << sp.getCompleteFileName();
	}

  conditionalGotoNormalView();
  m_unixCommand = new UNIXCommand(this);

  QObject::connect(m_unixCommand, SIGNAL( started() ), this, SLOT( RPM2TGZprocessStarted() ));
  QObject::connect(m_unixCommand, SIGNAL( readyReadStandardOutput() ), this, SLOT( RPM2TGZprocessReadOutput() ));
  QObject::connect(m_unixCommand, SIGNAL( finished ( int, QProcess::ExitStatus )),
      this, SLOT( RPM2TGZprocessFinished(int, QProcess::ExitStatus) ));
  QObject::connect(m_unixCommand, SIGNAL( readyReadStandardError() ), this, SLOT( RPM2TGZprocessRaisedError() ));

  m_unixCommand->transformRPMinTGZ(out);
}

void MainWindowImpl::execContextMenutvTODO(QPoint point){
	if (tvTODO->selectionModel()->selectedRows().count()==0) return;
	QModelIndex mi = tvTODO->currentIndex();
	QMenu menu(this);               	

	//Which actions do we have to put? It depends on package status
  if ( (m_modelTodo->itemFromIndex(mi)->parent() == 0) && (m_modelTodo->rowCount(mi) > 0))
		menu.addAction(actionDelete_All_ActionFiles);
  else if ((m_modelTodo->itemFromIndex(mi)->parent() != 0))
		menu.addAction(actionDelete_ActionFile);

	QPoint pt2 = tvTODO->mapToGlobal(point);
	pt2.setY(pt2.y() + tvTODO->header()->height());
	menu.exec(pt2);        	
}

void MainWindowImpl::execSystemTrayActivated ( QSystemTrayIcon::ActivationReason ar ){
	if ( ar == QSystemTrayIcon::Trigger )  
		if ( this->isHidden() ){
      this->restoreGeometry( m_savedGeometry );
			if (this->isMinimized()) this->setWindowState(Qt::WindowNoState);
			this->show();
		}
		else {
      m_savedGeometry = this->saveGeometry();
			this->hide();
		}
}

void MainWindowImpl::keyReleaseEvent(QKeyEvent* ke){
	static int i=0;
	static int k=-9999;
	static int k_count=0;
  static QString cur_dir=m_modelDir->filePath(tvDir->currentIndex());

	if ((tvPackage->hasFocus()) && (ke->key() >= Qt::Key_A) && (ke->key() <= Qt::Key_Z)){
    QList<QStandardItem*> fi = m_modelPackage->findItems( ke->text(), Qt::MatchStartsWith, ctn_PACKAGE_NAME );
		if (fi.count() > 0){
      if ( (cur_dir != m_modelDir->filePath(tvDir->currentIndex()) ) || (ke->key() != k) || (fi.count() != k_count) ) i=0;

			foreach (QStandardItem* si, fi){
				QModelIndex mi = si->index();
        mi = m_proxyModelPackage->mapFromSource(mi);
        if (!m_proxyModelPackage->hasIndex(mi.row(), mi.column())) fi.removeAll(si);
			}

      if (fi.count()==0) return;

			QList<SelectedPackage> sp = getSelectedPackage();
			if (sp.count() == 1) {
				int a=0;
				while (a<=fi.count()-1) {
					QStandardItem* si = fi[a];
					if (si->text() == sp[sp.count()-1].getFileName()) break;
					a++;
					i=a;
				}
				if((a+1)<=fi.count()-1) { a++; i=a; }
				else { a=i=0; }	
			}

			tvPackage->selectionModel()->clear();
			QModelIndex mi = fi[i]->index();
      mi = m_proxyModelPackage->mapFromSource(mi);
			tvPackage->scrollTo(mi);		
      QModelIndex maux = m_proxyModelPackage->index( mi.row(), ctn_PACKAGE_ICON );
			tvPackage->selectionModel()->setCurrentIndex(mi, QItemSelectionModel::Select);
			tvPackage->selectionModel()->setCurrentIndex(maux, QItemSelectionModel::Select);
			tvPackage->setCurrentIndex(mi);
      selectInstalledPackage();

			if ((i <= fi.count()-1)) i++; 
			if (i == fi.count()) i = 0;
		}

		k = ke->key();
		k_count = fi.count();
    cur_dir=m_modelDir->filePath(tvDir->currentIndex());
	}

	else if ((tvInstalledPackages->hasFocus()) && (ke->key() >= Qt::Key_A) && (ke->key() <= Qt::Key_Z)) {
    QList<QStandardItem*> fi = m_modelInstalledPackages->findItems( ke->text(), Qt::MatchStartsWith, ctn_PACKAGE_NAME );
		if (fi.count() > 0){
			if ( (ke->key() != k) || (fi.count() != k_count) ) i=0;

			foreach (QStandardItem* si, fi){
				QModelIndex mi = si->index();
        mi = m_proxyModelInstalledPackages->mapFromSource(mi);
        if (!m_proxyModelInstalledPackages->hasIndex(mi.row(), mi.column())) fi.removeAll(si);
			}

      if (fi.count()==0) return;

			QList<SelectedPackage> sp = getSelectedInstalledPackage();
			if (sp.count() == 1) {
				int a=0;
				while (a<=fi.count()-1) {
					QStandardItem* si = fi[a];
					if (si->text() == sp[sp.count()-1].getFileName()) break;
					a++; i=a;
				}

				if ((a+1)<=fi.count()-1) { a++; i=a; }
				else a=i=0; 
			}

			tvInstalledPackages->selectionModel()->clear();
			QModelIndex mi = fi[i]->index();
      mi = m_proxyModelInstalledPackages->mapFromSource(mi);
			tvInstalledPackages->scrollTo(mi, QAbstractItemView::PositionAtCenter);		
      QModelIndex maux = m_proxyModelInstalledPackages->index(mi.row(), ctn_PACKAGE_ICON);
			tvInstalledPackages->selectionModel()->setCurrentIndex(mi, QItemSelectionModel::Select);
			tvInstalledPackages->selectionModel()->setCurrentIndex(maux, QItemSelectionModel::Select);

			if ((i <= fi.count()-1)) i++; 
			if (i == fi.count()) i = 0;
		}

		k = ke->key();
		k_count = fi.count();
		cur_dir = "";
	}

  else if ((ke->key() != Qt::Key_Shift) && (ke->key() != Qt::Key_Control) && (tvPackage->hasFocus() &&
           (ke->key() == Qt::Key_Down|Qt::Key_Up|Qt::Key_PageDown|Qt::Key_PageUp))) {
    tvPackage->scrollTo(tvPackage->currentIndex());
    selectInstalledPackage();
	}
	else if ((ke->key() == Qt::Key_Down|Qt::Key_Up|Qt::Key_PageDown|Qt::Key_PageUp)) {
		QTreeView *t = twTODO->currentWidget()->findChild<QTreeView*>("tvPkgFileList");
		if ( (t != 0)  && (t->hasFocus())) {
			t->scrollTo( t->currentIndex() );
			showFullPathOfObject( t->currentIndex() );
		}
	}
	else ke->ignore();
}

void MainWindowImpl::keyPressEvent(QKeyEvent* ke){
  if ( ((ke->key() < Qt::Key_A) || (ke->key() > Qt::Key_Z)) && ((ke->key() != Qt::Key_Shift) &&
       (ke->key() != Qt::Key_Control) && (tvPackage->hasFocus()))) {
		tvPackage->scrollTo(tvPackage->currentIndex());
		selectInstalledPackage();
	}
	else {
		QTreeView *t = twTODO->currentWidget()->findChild<QTreeView*>("tvPkgFileList");
		if ( (t != 0)  && (t->hasFocus())) {
			t->scrollTo( t->currentIndex() );
			showFullPathOfObject( t->currentIndex() );
		}
	}

  if (( tvPackage->hasFocus()) && ((ke->key() == Qt::Key_Minus) && (ke->modifiers() == Qt::ControlModifier)) &&
      (m_proxyModelPackage->rowCount() > 0) ){
    if (m_PackagesInDirFontSize > 8){
      m_PackagesInDirFontSize--;
      tvPackage->setFont(QFont("Verdana", m_PackagesInDirFontSize));
		}
	}

  else if (( tvPackage->hasFocus()) && ((ke->key() == Qt::Key_Equal) && (ke->modifiers() == Qt::ControlModifier)) &&
           (m_proxyModelPackage->rowCount() > 0) ){
    m_PackagesInDirFontSize++;
    tvPackage->setFont(QFont("Verdana", m_PackagesInDirFontSize));
	}

  if (( tvInstalledPackages->hasFocus()) && ((ke->key() == Qt::Key_Minus) && (ke->modifiers() == Qt::ControlModifier)) &&
      (m_proxyModelInstalledPackages->rowCount() > 0) ){
    if (m_InstalledPackagesFontSize > 8){
      m_InstalledPackagesFontSize--;
      tvInstalledPackages->setFont(QFont("Verdana", m_InstalledPackagesFontSize));
		}
	}

  else if (( tvInstalledPackages->hasFocus()) && ((ke->key() == Qt::Key_Equal) && (ke->modifiers() == Qt::ControlModifier))
        && (m_proxyModelInstalledPackages->rowCount() > 0) ){
    m_InstalledPackagesFontSize++;
    tvInstalledPackages->setFont(QFont("Verdana", m_InstalledPackagesFontSize));
	}

  else if ( (tvPackage->hasFocus()) && ((ke->key() == Qt::Key_L) && (ke->modifiers() == Qt::ControlModifier)) &&
            ( UNIXCommand::hasTheExecutable(ctn_TGZ2LZMBIN) || UNIXCommand::hasTheExecutable(ctn_MAKELZMBIN) )) {

    foreach(SelectedPackage ami, getSelectedPackage())
      if (ami.getIcon().pixmap(QSize(22,22)).toImage() == ctn_ICON_RPM.pixmap(QSize(22,22)).toImage()) return;

    QModelIndexList lmi = tvPackage->selectionModel()->selectedIndexes();
    if (lmi.size() > 0) transformTGZinLZM();
	}	

  else if (ke->key() == Qt::Key_F6) openDirectory();

	else if (ke->key() == Qt::Key_F4) openTerminal();

	else if ((ke->key() == Qt::Key_F8) && (tvPackage->hasFocus())){
		showPackageInfo();
	}

	else if ((ke->key() == Qt::Key_F8) && (tvInstalledPackages->hasFocus())){
		showInstalledPackageInfo();
	}	

	else if ((ke->key() == Qt::Key_Delete) && (tvPackage->hasFocus())){ 
		QModelIndexList lmi = tvPackage->selectionModel()->selectedIndexes();
		if (lmi.size() > 0) deleteFile();
	}  

	else if ((ke->key() == Qt::Key_Delete) && (tvTODO->hasFocus())){
		QModelIndex mi = tvTODO->currentIndex();

		int res=0;
    if ( (m_modelTodo->itemFromIndex(mi)->parent() == 0) && (m_modelTodo->rowCount(mi) > 0)){
			res = QMessageBox::question ( this, tr ( "Confirmation" ),
					tr ( "Are you sure you want to delete all these files?" ),
					QMessageBox::Yes | QMessageBox::No,
					QMessageBox::No );

      if (res == QMessageBox::Yes) m_modelTodo->removeRows (0, m_modelTodo->rowCount(mi), mi);
		}
    else if ((m_modelTodo->itemFromIndex(mi)->parent() != 0)){
			res = QMessageBox::question(this, tr("Confirmation"), 
					tr("Are you sure you want to delete this file?"), 
					QMessageBox::Yes | QMessageBox::No,
					QMessageBox::No);

      if (res == QMessageBox::Yes) m_modelTodo->removeRow(mi.row(), mi.parent());
		}
	}

	else if ((ke->key() == Qt::Key_W) && (ke->modifiers() == Qt::ControlModifier)){
    if (!actionMinimizeLowerView->isChecked()) closeCurrentTab();
	}

  else if (ke->key() == Qt::Key_Return && (tvPackage->hasFocus() || tvInstalledPackages->hasFocus())){
    showPackageContent();
  }  

  /*else if (ke->key() == Qt::Key_F && ke->modifiers() == Qt::ControlModifier){
    QTreeView *t = twTODO->currentWidget()->findChild<QTreeView*>("tvPkgFileList");
    if ( (t != 0) && (t->hasFocus())) findFileInPkgFileList();
    else findFile();
  }*/

	else ke->ignore();
}

void MainWindowImpl::_deallocateTabContent(int tabIndex){
  QTreeView *t = twTODO->widget(tabIndex)->findChild<QTreeView*>("tvPkgFileList");
  if ( t != 0){
    QStandardItemModel *sim = (QStandardItemModel*) t->model();
    sim->clear();
    delete sim;
    delete t;
  }
}

void MainWindowImpl::closeClickedTab(){
  if ((m_clickedTab != -1) && (m_clickedTab != 0) && (m_clickedTab != 1)){
    _deallocateTabContent(m_clickedTab);
    twTODO->removeTab(m_clickedTab);
  }
	if (twTODO->count() == 2){
		twTODO->setCurrentIndex(0);
		enableCloseTabButton(-1);	
	}
}

void MainWindowImpl::closeCurrentTab() {
	if (twTODO->currentWidget()->isActiveWindow() && (twTODO->currentIndex()!=0 && twTODO->currentIndex()!=1)){ 
    _deallocateTabContent(twTODO->currentIndex());
    twTODO->removeTab(twTODO->currentIndex());
		enableCloseTabButton(-1);	

		if (twTODO->count() == 2)	twTODO->setCurrentIndex(0);
	}	
	if (twTODO->count() == 2){
		enableCloseTabButton(-1);	
	}
}

void MainWindowImpl::closeAllTabs() {
	while ( twTODO->count() > 2 ){  
    _deallocateTabContent(twTODO->count()-1);
    twTODO->removeTab( twTODO->count()-1 );
		QCoreApplication::processEvents();	
	}

	twTODO->setCurrentIndex(0);
}

void MainWindowImpl::hideRightView(){
  if ((actionHideRightView->isChecked()) && (splitterVertical->sizes()[2] == 0)) return;

  static QByteArray stateV = splitterVertical->saveState();

	QList<int> l, rl;
  rl = splitterVertical->sizes();

  if ( rl[2] == 0 ) splitterVertical->restoreState( stateV );
	else
    splitterVertical->setSizes( l << dockDirectories->width() << dockPackages->width() << 0 );
}

void MainWindowImpl::normalView(){
  splitterVertical->restoreState(m_verticalSplit);
  splitterHorizontal->restoreState(m_horizontalSplit);
	if (actionHideRightView->isChecked()) hideRightView();

  QTreeView *t = twTODO->currentWidget()->findChild<QTreeView*>("tvPkgFileList");
  if (t){
    t->setFocus();
    if (t->currentIndex().isValid()) t->setCurrentIndex(t->currentIndex());
    else t->setCurrentIndex(t->model()->index(0,0));
    if (m_lblStatus->text()=="") showFullPathOfObject(t->currentIndex());
  }
  else if ((t = twTODO->currentWidget()->findChild<QTreeView*>("tvTODO"))) t->setFocus();
  else tvPackage->setFocus();
}

void MainWindowImpl::minimizeLowerView(){
  static QByteArray stateH = splitterHorizontal->saveState();
  m_lblStatus->setText("");
	tvPackage->setFocus();
  if (actionHideRightView->isChecked()) hideRightView();

	QList<int> l, rl;
  rl = splitterHorizontal->sizes();
  if ( rl[1] == 0 ) splitterHorizontal->restoreState( stateH );
	else
    splitterHorizontal->setSizes( l << twTODO->height() << 0 );
  if (actionHideRightView->isChecked()) hideRightView();
}

void MainWindowImpl::maximizeLowerView(){
  static QByteArray stateH = splitterHorizontal->saveState();

	QList<int> l, rl;
  rl = splitterHorizontal->sizes();
  if ( rl[0] == 0 ) splitterHorizontal->restoreState( stateH );
	else
    splitterHorizontal->setSizes( l << 0 << twTODO->maximumHeight());

  QTreeView *t = twTODO->currentWidget()->findChild<QTreeView*>("tvPkgFileList");
  if (t){
    t->setFocus();
    if (t->currentIndex().isValid()) t->setCurrentIndex(t->currentIndex());
    else t->setCurrentIndex(t->model()->index(0,0));
    if (m_lblStatus->text()=="") showFullPathOfObject(t->currentIndex());
  }
  else if ((t = twTODO->currentWidget()->findChild<QTreeView*>("tvTODO"))) t->setFocus();
  else twTODO->setFocus();
}

void MainWindowImpl::initializeActions(){  
  m_actionFreezePkg = new QAction( this );
  m_actionUnfreezePkg = new QAction( this );
  m_actionFreezePkg->setIcon(QIcon((":/resources/images/tgz_frozen.png")));
  m_actionFreezePkg->setText(tr("Freeze package"));
  m_actionUnfreezePkg->setIcon(QIcon((":/resources/images/tgz4.png")));
  m_actionUnfreezePkg->setText(tr("Unfreeze package"));
  m_actionTransformRPMinTGZ = new QAction( this );
  m_actionTransformRPMinTGZ->setIcon(QIcon((":/resources/images/tgz4.png")));
  m_actionTransformRPMinTGZ->setText(tr("Transform in TGZ"));
  m_actionChangeDefaultDirectory = new QAction( this );
  m_actionChangeDefaultDirectory->setIcon(QIcon((":/resources/images/favorites.png")));
  m_actionChangeDefaultDirectory->setText(tr("Set as default directory"));
  m_actionFindFileInPkgFileList = new QAction( this );
  m_actionFindFileInPkgFileList->setText(tr("Find a file"));
  m_actionFindFileInPkgFileList->setIcon(QIcon((":/resources/images/find.png")));
  m_actionFindFileInPackage = new QAction( this );
  m_actionFindFileInPackage->setText(tr("Find a file"));
  m_actionFindFileInPackage->setIcon(QIcon((":/resources/images/find.png")));
  actionFindFile->setIcon(QIcon((":/resources/images/find.png")));
  actionFindFile->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_F ));

  m_actionFindPackage = new QAction(this);
  m_actionFindPackage->setIcon(QIcon(":/resources/images/find.png"));
  m_actionFindPackage->setText(tr("Find a package"));
  m_actionFindPackage->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_K ));

  m_actionIconifyOnStart = new QAction(this);
  m_actionIconifyOnStart->setText(tr("Start with hidden window"));
  m_actionIconifyOnStart->setCheckable(true);
  m_actionIconifyOnStart->setChecked(false);

  m_actionShowToolBar = new QAction(this);
  m_actionShowToolBar->setText(tr("Show toolbar"));
  m_actionShowToolBar->setCheckable(true);
  m_actionShowToolBar->setChecked(SettingsManager::instance()->getShowToolBar());

  actionExit->setIcon(QIcon((":/resources/images/exit.png")));
  actionInstall_package->setIcon(QIcon((":/resources/images/1rightarrow.png")));
	actionReinstall_package->setIcon(QIcon(":/resources/images/reload2.png"));
	actionUpgrade_package->setIcon(QIcon((":/resources/images/2rightarrow.png")));
	actionDowngrade_package->setIcon(QIcon((":/resources/images/2leftarrow.png")));
	actionRemove_package->setIcon(QIcon((":/resources/images/close.png")));
	actionDelete_file->setIcon(QIcon((":/resources/images/tool_delete.png")));
	actionDelete_All_ActionFiles->setIcon(QIcon((":/resources/images/tool_delete.png")));
	actionDelete_ActionFile->setIcon(QIcon((":/resources/images/tool_delete.png")));
	actionCreate_Directory->setIcon(QIcon((":/resources/images/newFolder.png")));
	actionRemove_Directory->setIcon(QIcon((":/resources/images/trashcan.png")));
	actionAbout_QTGZManager->setIcon(QIcon((":/resources/images/QTGZManager.png")));
	actionAbout_QTGZManager->setShortcut( QKeySequence(Qt::Key_F1) );
  actionAbout_QTGZManager->setToolTip(tr("About..."));
	actionSnapshotInstalledPackages->setIcon(QIcon((":/resources/images/camera.png")));
	actionPackageContent->setIcon(QIcon(":/resources/images/content.png"));
	actionPackageInfo->setIcon(QIcon(":/resources/images/info.png"));
  actionInstalledPackageInfo->setIcon(QIcon(":/resources/images/info.png"));
	actionExecuteActions->setIcon(QIcon(":/resources/images/executeActions.png"));
	actionExecuteActions->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_E ));
  actionExecuteActions->setEnabled( false );
  actionSnapshotInstalledPackages->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_S ));
  actionExpand_All_Items->setIcon(QIcon(":/resources/images/expand.png"));
	actionCollapse_All_Items->setIcon(QIcon(":/resources/images/collapse.png"));
  m_actionExpandItem = new QAction(QIcon(":/resources/images/expand.png"), tr("Expand this item"), this);
  m_actionCollapseItem = new QAction(QIcon(":/resources/images/collapse.png"), tr("Collapse this item"), this);
  m_actionDiffToEachOther = new QAction(QIcon(":/resources/images/diff.png"), tr("Diff to each other"), this);
	actionOpenFile = new QAction(QIcon(":/resources/images/binary.png"), tr("Open file"), this);
  m_actionEditFile = new QAction(QIcon(":/resources/images/editfile.png"), tr("Edit file"), this);
	actionOpenDirectory = new QAction(QIcon(":/resources/images/folder.png"), tr("Open directory"), this);
  actionOpenDirectory->setShortcut(QKeySequence(Qt::Key_F6));
  m_actionOpenTerminal = new QAction(QIcon(":/resources/images/terminal.png"), tr("Open in terminal"), this);
  m_actionOpenTerminal->setShortcut(QKeySequence(Qt::Key_F4));
  actionHideRightView->setIcon(QIcon(":/resources/images/horizontalView.png"));
  actionHideRightView->setShortcut(QKeySequence(Qt::Key_F9));
  actionNormalView->setIcon(QIcon(":/resources/images/normalView.png"));
  actionNormalView->setShortcut(QKeySequence(Qt::Key_F10));
  actionMinimizeLowerView->setIcon(QIcon(":/resources/images/minimizedLowerView.png"));
  actionMinimizeLowerView->setShortcut(QKeySequence(Qt::Key_F11));
  actionMaximizeLowerView->setIcon(QIcon(":/resources/images/verticalView.png"));
  actionMaximizeLowerView->setShortcut(QKeySequence(Qt::Key_F12));
	actionCloseTab->setIcon(QIcon(":/resources/images/close.png"));	
	actionCloseAllTabs->setIcon(QIcon(":/resources/images/close.png"));
	actionCloseAllTabs->setObjectName("actionCloseAllTabs");
  actionDiffToInstalled->setIcon(QIcon(":/resources/images/diff.png"));
	actionMaximizeLowerView->setToolTip(actionMaximizeLowerView->text());
	actionHideRightView->setToolTip(actionHideRightView->text());
	actionExit->setToolTip(actionExit->text());
  actionExit->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_X ));
  actionExecuteActions->setToolTip(actionExecuteActions->text());
	actionMaximizeLowerView->setCheckable( true );
	actionMaximizeLowerView->setChecked( false );
	actionHideRightView->setCheckable( true );
	actionHideRightView->setChecked( false );
	actionMinimizeLowerView->setCheckable( true );
	actionMinimizeLowerView->setChecked( false );
	actionNormalView->setCheckable( true );
	actionNormalView->setChecked( true );

  connect(m_actionTransformRPMinTGZ, SIGNAL(triggered()), this, SLOT(transformRPMinTGZ()));
  connect(m_actionFreezePkg, SIGNAL(triggered()), this, SLOT(freezePackage()));
  connect(m_actionUnfreezePkg, SIGNAL(triggered()), this, SLOT(unfreezePackage()));
  connect(m_actionChangeDefaultDirectory, SIGNAL(triggered()), this, SLOT(changeDefaultDirectory()));
  connect(actionFindFile, SIGNAL(triggered()), this, SLOT(metaFindFile()));
  connect(m_actionFindPackage, SIGNAL(triggered()), this, SLOT(findPackage()));
  connect(actionOpenFile, SIGNAL(triggered()), this, SLOT(openFile()));
  connect(m_actionEditFile, SIGNAL(triggered()), this, SLOT(editFile()));
	connect(actionOpenDirectory, SIGNAL(triggered()), this, SLOT(openDirectory()));
  connect(m_actionOpenTerminal, SIGNAL(triggered()), this, SLOT(openTerminal()));
	connect(actionRefreshPackageLists, SIGNAL(triggered()), this, SLOT(refreshTreeViews()));
	connect(actionMaximizeLowerView, SIGNAL(triggered()), this, SLOT(maximizeLowerView()));
	connect(actionMinimizeLowerView, SIGNAL(triggered()), this, SLOT(minimizeLowerView()));
	connect(actionHideRightView, SIGNAL(triggered()), this, SLOT(hideRightView()));
	connect(actionNormalView, SIGNAL(triggered()), this, SLOT(normalView()));
	connect(actionExpand_All_Items, SIGNAL(triggered()), this, SLOT(expandAllContentItems()));
  connect(m_actionExpandItem,  SIGNAL(triggered()), this, SLOT(expandThisContentItems()));
  connect(m_actionCollapseItem,  SIGNAL(triggered()), this, SLOT(collapseThisContentItems()));
	connect(actionCollapse_All_Items, SIGNAL(triggered()), this, SLOT(collapseAllContentItems()));
	connect(actionCloseAllTabs, SIGNAL(triggered()), this, SLOT(closeAllTabs()));
	connect(actionCloseTab, SIGNAL(triggered()), this, SLOT(closeCurrentTab()));
	connect(actionExecuteActions, SIGNAL(triggered()), this, SLOT(executePackageActions()));
	connect(actionPackageInfo, SIGNAL(triggered()), this, SLOT(showPackageInfo()));
	connect(actionInstalledPackageInfo, SIGNAL(triggered()), this, SLOT(showInstalledPackageInfo()));
	connect(actionPackageContent, SIGNAL(triggered()), this, SLOT(showPackageContent()));
	connect(actionUpgrade_package, SIGNAL(triggered()), this, SLOT(insertUpgradePackageAction()));
	connect(actionReinstall_package, SIGNAL(triggered()), this, SLOT(insertReinstallPackageAction()));
	connect(actionInstall_package, SIGNAL(triggered()), this, SLOT(insertInstallPackageAction()));
	connect(actionDowngrade_package, SIGNAL(triggered()), this, SLOT(insertDowngradePackageAction()));
	connect(actionRemove_package, SIGNAL(triggered()), this, SLOT(insertRemovePackageAction()));		
	connect(actionDelete_file, SIGNAL(triggered()), this, SLOT(deleteFile()));
  connect(actionDiffToInstalled, SIGNAL(triggered()), this, SLOT(diffToInstalled()));
  connect(m_actionDiffToEachOther, SIGNAL(triggered()), this, SLOT(diffToEachOther()));
	connect(actionDelete_ActionFile, SIGNAL(triggered()), this, SLOT(deleteActionFile()));
	connect(actionDelete_All_ActionFiles, SIGNAL(triggered()), this, SLOT(deleteAllActionFiles()));
	connect(actionCreate_Directory, SIGNAL(triggered()), this, SLOT(createDirectory()));
	connect(actionRemove_Directory, SIGNAL(triggered()), this, SLOT(removeDirectory()));	    	   
	connect(actionExit, SIGNAL(triggered()), this, SLOT(exitApplication()));
	connect(actionAbout_QTGZManager, SIGNAL(triggered()), this, SLOT(aboutQTGZManager()));
	connect(actionSnapshotInstalledPackages, SIGNAL(triggered()), this, SLOT(takeSnapshotOfInstalledPackages()));
  connect(m_actionFindFileInPkgFileList, SIGNAL(triggered()), this, SLOT(findFileInPkgFileListExt()));
  connect(m_actionFindFileInPackage, SIGNAL(triggered()), this, SLOT(findFileInPackage()));  
  connect(m_actionShowToolBar, SIGNAL(triggered()), this, SLOT(showToolBar()));
}

void MainWindowImpl::takeSnapshotOfInstalledPackages(){
  conditionalGotoNormalView();
  twTODO->setCurrentIndex(1);
	QString msg;
	textEdit->clear();
  msg = tr("<h5>Please, wait while the snapshot of installed packages is being generated...</h5><br>");
	textEdit->insertHtml(msg);

  QString fileName = Package::dumpInstalledPackageList();
  msg = tr("<h5>Finished generating the snapshot of installed packages!</h5><br>"
      "<br>The name of the snapshot is: \"%1\"<br><br>").arg(fileName) +
      m_textForMoreInformation;

  textEdit->insertHtml(msg);
  textEdit->ensureCursorVisible();
}

void MainWindowImpl::showToolBar(){
  toolBar->setHidden(!toolBar->isHidden());
}

void MainWindowImpl::correctActionToolBarStatus(){
  m_actionShowToolBar->setChecked(toolBar->toggleViewAction()->isChecked());
}

void MainWindowImpl::refreshTreeViews(){
	showPackagesInDirectory();
}

void MainWindowImpl::initializeMenuBar(){
  menubar->setContextMenuPolicy(Qt::NoContextMenu);
  menuFile->addAction(actionExecuteActions);
  menuFile->addAction(actionFindFile);
  menuFile->addAction(m_actionFindPackage);
  menuFile->addAction(actionSnapshotInstalledPackages);
  menuFile->addSeparator();
  menuFile->addAction(actionExit);

  menuView->addAction(actionHideRightView);
  menuView->addAction(actionNormalView);
  menuView->addAction(actionMinimizeLowerView);
  menuView->addAction(actionMaximizeLowerView);
  menuView->addSeparator();
  menuView->addAction(m_actionShowToolBar);
  menuView->addAction(m_actionIconifyOnStart);

  menubar->setVisible( true );
  menubar->setStyleSheet(" QMenuBar { font: 12px; } ");
}

void MainWindowImpl::initializeToolBar(){
	QActionGroup *ag = new QActionGroup(this);

  toolBar->setWindowTitle(tr("Show toolbar"));
  connect(toolBar->toggleViewAction(), SIGNAL(triggered()), this, SLOT(correctActionToolBarStatus()));
  toolBar->setStyleSheet(" QToolBar { border: 5px; } ");
  toolBar->setToolButtonStyle ( Qt::ToolButtonIconOnly );
	toolBar->addAction(actionHideRightView);
	toolBar->addAction(ag->addAction(actionNormalView));
	toolBar->addAction(ag->addAction(actionMinimizeLowerView));
	toolBar->addAction(ag->addAction(actionMaximizeLowerView));
	toolBar->addSeparator();
	toolBar->addAction(actionFindFile);
	toolBar->addAction(actionExecuteActions);
	toolBar->addAction(actionSnapshotInstalledPackages);
	toolBar->addSeparator();
	toolBar->addAction(actionAbout_QTGZManager);
	toolBar->addAction(actionExit);
}

void MainWindowImpl::refreshPackageTreeView(){
  delete m_modelPackage;
  delete m_proxyModelPackage;
  initializePackageTreeView();
}

void MainWindowImpl::refreshInstalledPackageTreeView(){
  delete m_modelInstalledPackages;
  delete m_proxyModelInstalledPackages;
  initializeInstalledPackagesTreeView();
	reapplyInstalledPackagesFilter();
}

void MainWindowImpl::freezePackage(){
  int bkSize = m_frozenPkgList->count();
	foreach(QModelIndex item, tvInstalledPackages->selectionModel()->selectedIndexes()){
    if ( item.column() == ctn_PACKAGE_ICON ){
      QStandardItem *si = new QStandardItem( ctn_ICON_FROZEN, "Frozen" );
      QModelIndex mi = m_proxyModelInstalledPackages->mapToSource( item );
      m_modelInstalledPackages->setItem(mi.row(), mi.column(), si);
		}
    else if ( item.column() == ctn_PACKAGE_NAME ){
      *m_frozenPkgList << Package::getBaseName( m_proxyModelInstalledPackages->data(item).toString());
		}
	}

	refreshInstalledPackageTreeView();

  //Now we must rebuild the list of frozen packages in order to save it to the QSettings.
  m_frozenPkgList->save();
  if ( (m_modelPackage->rowCount()>0) && (bkSize != m_frozenPkgList->count()) ) showPackagesInDirectory();
}

void MainWindowImpl::unfreezePackage(){
  int bkSize = m_frozenPkgList->count();
	foreach(QModelIndex item, tvInstalledPackages->selectionModel()->selectedIndexes()){
    if ( item.column() == ctn_PACKAGE_ICON ){
      QStandardItem *si = new QStandardItem( ctn_ICON_UNFROZEN, "UnFrozen" );
      QModelIndex mi = m_proxyModelInstalledPackages->mapToSource( item );
      m_modelInstalledPackages->setItem(mi.row(), mi.column(), si);
		}		
    else if ( item.column() == ctn_PACKAGE_NAME ){
      m_frozenPkgList->removeAll( Package::getBaseName(m_proxyModelInstalledPackages->data(item).toString() ));
		}
	}

	refreshInstalledPackageTreeView();

  //Now we must rebuild the list of frozen packages in order to save it to the QSettings.
  m_frozenPkgList->save();
  if ( (m_modelPackage->rowCount()>0) && (bkSize != m_frozenPkgList->count()) ) showPackagesInDirectory();
}

void MainWindowImpl::aboutQTGZManager(){	
  if (this->isHidden()) show();
  conditionalGotoNormalView();

  QString aux("About");
	QString translated_about = QApplication::translate ( "MainWindow", aux.toUtf8(), 0, QApplication::UnicodeUTF8 );
	for (int c=2; c<twTODO->count(); c++)
		if (twTODO->tabText(c) == translated_about){
			twTODO->setCurrentIndex ( c );			
			return;
		} 

	QWidget *tabAbout = new QWidget();
	QGridLayout *gridLayoutX = new QGridLayout ( tabAbout );
	gridLayoutX->setSpacing ( 0 );
	gridLayoutX->setMargin ( 0 );

	QTextBrowser *text = new QTextBrowser(tabAbout);
	text->setReadOnly(true);
	text->setFrameShape(QFrame::NoFrame);
	text->setFrameShadow(QFrame::Plain);
  text->setOpenExternalLinks(true);
	gridLayoutX->addWidget ( text, 0, 0, 1, 1 );

  QString url = "qrc:/resources/help/help_" + QLocale::system().name() + ".html";
  text->setSource(QUrl(url));

  if (text->document()->isEmpty()){
    url = "qrc:/resources/help/help_en_US.html";
    text->setSource(QUrl(url));
  }

	text->show();

	int tindex = twTODO->addTab( tabAbout, QApplication::translate ( "MainWindow", aux.toUtf8(), 0, QApplication::UnicodeUTF8 ) );
	twTODO->setTabText(twTODO->indexOf(tabAbout), QApplication::translate("MainWindow", aux.toUtf8(), 0, QApplication::UnicodeUTF8));

	twTODO->setCurrentIndex ( tindex );
}

void MainWindowImpl::deleteActionFile(){
	QModelIndex mi = tvTODO->currentIndex();

	int res = QMessageBox::question ( this, tr ( "Confirmation" ),
			tr ( "Are you sure you want to delete this file?" ),
			QMessageBox::Yes | QMessageBox::No,
			QMessageBox::No );

	if ( res == QMessageBox::Yes ){
    m_modelTodo->removeRow ( mi.row(), mi.parent() );
	}
}

void MainWindowImpl::deleteAllActionFiles(){
	QModelIndex mi = tvTODO->currentIndex();

	int res = QMessageBox::question ( this, tr ( "Confirmation" ),
			tr ( "Are you sure you want to delete all these files?" ),
			QMessageBox::Yes | QMessageBox::No,
			QMessageBox::No );

	if ( res == QMessageBox::Yes ){
    m_modelTodo->removeRows (0, m_modelTodo->rowCount(mi), mi);
	}
}

void MainWindowImpl::openFileOrDirectory(const QModelIndex& mi){
  const QStandardItemModel *sim = qobject_cast<const QStandardItemModel*>(mi.model());
  QStandardItem *si = sim->itemFromIndex(mi);
  QDir d;
  QFile f;
  if ((si->icon().pixmap(QSize(22,22)).toImage() == ctn_ICON_BINARY.pixmap(QSize(22,22)).toImage()) && ( f.exists( m_lblStatus->text())) ) {
    openFile();
  }
}

void MainWindowImpl::metaFindFile(){
  if (isFindDisabled()) return;
  QTreeView *t = twTODO->currentWidget()->findChild<QTreeView*>("tvPkgFileList");
  if ( (t != 0) && (t->hasFocus())) findFileInPkgFileList();
  else findFile();
}

void MainWindowImpl::findFile(){
  if (isFindDisabled()) return;
  if (m_findDialog != 0) delete (m_findDialog);

  toggleToolBarStatus();
  m_findDialog = new FindDialogImpl( this );
  m_findDialog->setSearchPlace(ectn_INSIDE_INSTALLED_PACKAGES);
  //connect(m_findDialog, SIGNAL(accepted()), this, SLOT(toggleToolBarStatus()));
  m_findDialog->show();
}

void MainWindowImpl::findFileInPkgFileList(){
  if (isFindDisabled()) return;
  QTreeView *t = twTODO->currentWidget()->findChild<QTreeView*>("tvPkgFileList");
  if (t == 0) return;

  toggleToolBarStatus();
  QStandardItemModel *sim = (QStandardItemModel*) t->model();

  if (m_findDialog != 0) delete (m_findDialog);
  m_findDialog = new FindDialogImpl( this );
  QString headerLabel = sim->horizontalHeaderItem(0)->text();
  m_findDialog->setTargetPackage(headerLabel.section('"', 1, 1));
  m_findDialog->setSearchPlace(ectn_INSIDE_QSTDITEMMODEL); 
  m_findDialog->setQStandardItemModel(qobject_cast<QStandardItemModel*>(t->model()));
  //connect(m_findDialog, SIGNAL(accepted()), this, SLOT(toggleToolBarStatus()));
  m_findDialog->show();
}

void MainWindowImpl::findFileInPkgFileListExt(){
  if (isFindDisabled()) return;
  QTreeView *t = twTODO->currentWidget()->findChild<QTreeView*>("tvPkgFileList");
  if (t == 0) return;

  toggleToolBarStatus();
  QStandardItemModel *sim = (QStandardItemModel*) t->model();

  if (m_findDialog != 0) delete (m_findDialog);
  m_findDialog = new FindDialogImpl( this );
  QString headerLabel = sim->horizontalHeaderItem(0)->text();
  m_findDialog->setTargetPackage(headerLabel.section('"', 1, 1));
  m_findDialog->setTargetDir(showFullPathOfObject(t->currentIndex()));
  m_findDialog->setSearchPlace(ectn_INSIDE_QSTDITEMMODEL);
  m_findDialog->setQStandardItemModel(qobject_cast<QStandardItemModel*>(t->model()),
                                      sim->itemFromIndex(t->currentIndex()));
  //connect(m_findDialog, SIGNAL(accepted()), this, SLOT(toggleToolBarStatus()));
  m_findDialog->show();
}

bool MainWindowImpl::isFindDisabled(){
  return !actionFindFile->isEnabled();
}

void MainWindowImpl::toggleToolBarStatus(){
  m_actionFindPackage->setEnabled(!m_actionFindPackage->isEnabled());
  actionFindFile->setEnabled(!actionFindFile->isEnabled());
  //actionExit->setEnabled(!actionExit->isEnabled());
}

void MainWindowImpl::findFileInPackage(){
  if (showPackageContent()) findFileInPkgFileList();
}

void MainWindowImpl::findPackage(){
  if (isFindDisabled()) return;

  toggleToolBarStatus();
  if (m_findDialog != 0) delete (m_findDialog);
  m_findDialog = new FindDialogImpl( this );
  m_findDialog->setTargetDir(dockDirectories->windowTitle());
  m_findDialog->setSearchPlace(ectn_INSIDE_DIRECTORY);
  //connect(m_findDialog, SIGNAL(accepted()), this, SLOT(toggleToolBarStatus()));
  m_findDialog->show();
}

void MainWindowImpl::openFile(){
  UNIXCommand::openFile(m_lblStatus->text());
}

void MainWindowImpl::editFile(){
  UNIXCommand::editFile(m_lblStatus->text());
}

void MainWindowImpl::openDirectory(){
  QString targetDir("");
  if ((tvPackage->hasFocus()) || (tvDir->hasFocus())) targetDir = m_modelDir->filePath(tvDir->currentIndex());
  else if (m_lblStatus->text() != ""){
    targetDir = m_lblStatus->text();
  }
  if (targetDir != ""){
    UNIXCommand::openDirectory(targetDir);
  }
}

void MainWindowImpl::openTerminal(){
	QString targetDir;
	QTreeView *t = twTODO->currentWidget()->findChild<QTreeView*>("tvPkgFileList");
  if ((t) && (t->hasFocus()) && (m_lblStatus->text() != "")){
    QFileInfo fi(m_lblStatus->text());
    if (fi.isDir()) targetDir = m_lblStatus->text();
		else targetDir = fi.path(); 
	}
	else if	(dockDirectories->windowTitle() != "") targetDir = dockDirectories->windowTitle();

  UNIXCommand::openTerminal(targetDir);
}

void MainWindowImpl::deleteFile(){
  QDir d(m_modelDir->filePath(tvDir->currentIndex()));
	d.setFilter(QDir::Files);		
	QList<SelectedPackage> lmi = getSelectedPackage();		
	int res;

	if (lmi.size() > 1)
		res = QMessageBox::question(this, tr("Confirmation"), 
				tr("Are you sure you want to delete these files?"), 
				QMessageBox::Yes | QMessageBox::No,
				QMessageBox::No);
	else
		res = QMessageBox::question(this, tr("Confirmation"), 
				tr("Are you sure you want to delete this file?"), 
				QMessageBox::Yes | QMessageBox::No,
				QMessageBox::No);

	if (res == QMessageBox::Yes){
		foreach (SelectedPackage mi, lmi){
      QString fileToDelete = mi.getCompleteFileName();
			QFile::remove(fileToDelete);			
		}			
	}
}

void MainWindowImpl::exitApplication(){
	m_reallyWannaClose = true;
  close();
}

bool MainWindowImpl::close(){
	bool res = QWidget::close();
	if ( res == true ){ 
    qApp->exit();
		return( true );
	}
	else return( false );
}

void MainWindowImpl::closeEvent(QCloseEvent *event){
  m_savedGeometry = this->saveGeometry();
  saveSettings();

  if ((m_reallyWannaClose) && ((m_reinstall->rowCount() != 0) || (m_install->rowCount() != 0) ||
      (m_remove->rowCount() != 0) || (m_downgrade->rowCount() != 0) || (m_upgrade->rowCount() != 0))){
		int res = QMessageBox::question(this, tr("Confirmation"), 
				tr("There are actions waiting for execution!\n"
						"Are you sure you want to exit?"), 
						QMessageBox::Yes | QMessageBox::No,
						QMessageBox::No);

		if (res == QMessageBox::Yes) event->accept(); 
		else event->ignore();
	}
	m_reallyWannaClose = false;
}

void MainWindowImpl::initializeDirTreeView(){
  m_modelDir = new QDnDDirModel(this);
  m_modelDir->setFilter(QDir::AllDirs|QDir::CaseSensitive|QDir::NoDotAndDotDot);
	QModelIndex index;
	
  QString initialDir = m_argList->getSwitchArg("-initialdir", "");
  if ((initialDir != "") and (QFile::exists(initialDir))) index = m_modelDir->index(initialDir);
  else if (initialDir == "") initialDir = SettingsManager::instance()->getDefaultDirectory();

  index = m_modelDir->setRootPath(initialDir);
  m_modelDir->setIconProvider(new QDnDDirModelIconProvider(initialDir));

 	QString style ("QDockWidget::title { "
     "border: 1px solid gray;"
     "border-top-right-radius: 2px;"
     "border-bottom-right-radius: 2px;"
     "text-align: left;"
     "background-color: lightgray;"
     "padding-left: 3px;}");
	dockDirectories->setStyleSheet(style);

  tvDir->setModel(m_modelDir);
	tvDir->setColumnHidden(1, true);
	tvDir->setColumnHidden(2, true);
	tvDir->setColumnHidden(3, true);
  tvDir->setFont(QFont("Verdana", m_DirectoryFontSize));
	tvDir->header()->hide();
	tvDir->setStyleSheet( treeViewStyleSheet());
	tvDir->setAcceptDrops(true);
	tvDir->setDropIndicatorShown(true);
	tvDir->setCurrentIndex(index);
	tvDir->expand(index);
	update();
	tvDir->scrollTo(tvDir->currentIndex());
	changeDir();
}

QString MainWindowImpl::filterPackageFoundStyleSheet(){
	return QString("QLineEdit{ color: black; "
		 "background-color: rgb(255, 255, 200);"
     "border-color: rgb(206, 204, 197);}"
	);	
}

QString MainWindowImpl::filterPackageNotFoundStyleSheet(){
	return QString("QLineEdit{ color: white; "
		 "background-color: rgb(207, 135, 142);"
		 "border-color: rgb(206, 204, 197);}"
	);	
}

QString MainWindowImpl::treeViewStyleSheet(){
	return QString("QTreeView::branch:has-siblings:!adjoins-item {"
      "   border-image: url(:/resources/styles/vline.png) 0;}"
			"QTreeView::branch:has-siblings:adjoins-item {"
      "    border-image: url(:/resources/styles/branch-more.png) 0;}"
			"QTreeView::branch:!has-children:!has-siblings:adjoins-item {"
      "   border-image: url(:/resources/styles/branch-end.png) 0;}"
			"QTreeView::branch:has-children:!has-siblings:closed,"
			"QTreeView::branch:closed:has-children:has-siblings {"
			"       border-image: none;"
      "        image: url(:/resources/styles/branch-closed_BW.png);}"
			"QTreeView::branch:open:has-children:!has-siblings,"
			"QTreeView::branch:open:has-children:has-siblings  {"
			"       border-image: none;"
			"        image: url(:/resources/styles/branch-open_BW.png);}");
}

void MainWindowImpl::initializePackageTreeView(){
 	QString style ("QDockWidget::title { "
		 "text-align: right;" 
		 "background: transparent;" 
		 "padding-right: 5px;}");
	dockPackages->setStyleSheet(style);

  m_proxyModelPackage = new QSortFilterProxyModel(this);
  m_modelPackage = new QDnDStandardItemModel(this);
  m_proxyModelPackage->setSourceModel(m_modelPackage);
  m_proxyModelPackage->setFilterKeyColumn(1);

	tvPackage->setItemDelegate(new tvPackagesItemDelegate(tvPackage));
  tvPackage->setFont(QFont("Verdana", m_PackagesInDirFontSize));
	tvPackage->setEditTriggers(QAbstractItemView::NoEditTriggers);
	tvPackage->setVerticalScrollMode(QAbstractItemView::ScrollPerItem);		
	tvPackage->setAllColumnsShowFocus( true );
  tvPackage->setModel(m_proxyModelPackage);
	tvPackage->setSortingEnabled( true );
	tvPackage->sortByColumn( 1, Qt::AscendingOrder);
	tvPackage->setIndentation( 0 );
	tvPackage->header()->setSortIndicatorShown(true);
	tvPackage->header()->setClickable(true);
	tvPackage->header()->setMovable(false);
	tvPackage->header()->setDefaultAlignment( Qt::AlignCenter );
	tvPackage->header()->setResizeMode( QHeaderView::Fixed );

	//Prepare it for drag operations
	tvPackage->setSelectionMode(QAbstractItemView::ExtendedSelection);
	tvPackage->setDragEnabled(true);
  connect(leFilterPackage, SIGNAL(textChanged (const QString&)), this, SLOT(reapplyPackageFilter()));
}

void MainWindowImpl::initializeInstalledPackagesTreeView(){
 	QString style ("QDockWidget::title { "
		 "text-align: right;" 
     "background: transparent;" 
     "padding-right: 5px;}");
	dockInstalledPackages->setStyleSheet(style);

  m_proxyModelInstalledPackages = new QSortFilterProxyModel(this);
  m_modelInstalledPackages = new QStandardItemModel(this);
  m_proxyModelInstalledPackages->setSourceModel(m_modelInstalledPackages);
  m_proxyModelInstalledPackages->setFilterKeyColumn(1);

  tvInstalledPackages->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	tvInstalledPackages->setItemDelegate(new tvPackagesItemDelegate(tvInstalledPackages));

  QDir d(ctn_PACKAGES_DIR);
	d.setFilter(QDir::Files);		
	QStringList list = d.entryList();

  QStandardItem *parentItem = m_modelInstalledPackages->invisibleRootItem();
	QList<QStandardItem*> lIcons, lNames;

  if ( m_frozenPkgList->isEmpty() ){
		foreach( QString s, list ){
      lIcons << new QStandardItem( ctn_ICON_UNFROZEN, "_UnFrozen" );
			lNames << new QStandardItem( s );
		}
	}
	else{
		foreach( QString s, list ){
      if ( m_frozenPkgList->indexOf( QRegExp(Package::getBaseName(s)), 0 ) == -1 )
        lIcons << new QStandardItem( ctn_ICON_UNFROZEN, "_UnFrozen" );
			else
        lIcons << new QStandardItem( ctn_ICON_FROZEN, "_Frozen" );

			lNames << new QStandardItem( s ); 
		}
	}

	parentItem->insertColumn(0, lIcons );
	parentItem->insertColumn(1, lNames );

  QStringList sl;
  m_modelInstalledPackages->setHorizontalHeaderLabels(sl << "" << tr("Name"));

  dockInstalledPackages->setWindowTitle(tr("%1 Installed Packages").arg(QString::number(list.size())));
  tvInstalledPackages->setFont(QFont("Verdana", m_InstalledPackagesFontSize));
  tvInstalledPackages->setEditTriggers(QAbstractItemView::NoEditTriggers);
	tvInstalledPackages->setSelectionMode(QAbstractItemView::ExtendedSelection);
  tvInstalledPackages->setModel(m_proxyModelInstalledPackages);
	tvInstalledPackages->setSortingEnabled( true );
	tvInstalledPackages->sortByColumn( 1, Qt::AscendingOrder);
	tvInstalledPackages->setIndentation( 0 );
	tvInstalledPackages->setAllColumnsShowFocus( true );
	tvInstalledPackages->header()->setSortIndicatorShown(true);
	tvInstalledPackages->header()->setClickable(true);
	tvInstalledPackages->header()->setMovable(false);
	tvInstalledPackages->setColumnWidth(0, 24);	
	tvInstalledPackages->setColumnWidth(1, 50); 
	tvInstalledPackages->header()->setSortIndicator( m_InstalledPackageListOrderedCol, m_InstalledPackageListSortOrder );	
	tvInstalledPackages->sortByColumn( m_InstalledPackageListOrderedCol, m_InstalledPackageListSortOrder );
	tvInstalledPackages->header()->setDefaultAlignment( Qt::AlignCenter );
	tvInstalledPackages->header()->setResizeMode( QHeaderView::Fixed );
  connect(leFilterInstalledPackages, SIGNAL(textChanged (const QString&)), this, SLOT(reapplyInstalledPackagesFilter()));
}

void MainWindowImpl::initializeTodoTreeView(){
  m_modelTodo = new QDnDStandardItemModel(this);

  QFont f("Verdana", m_TodoFontSize);
	tvTODO->setFont(f);		
  tvTODO->setModel(m_modelTodo);
	tvTODO->setEditTriggers(QAbstractItemView::NoEditTriggers);
	tvTODO->setDropIndicatorShown(true);
	tvTODO->setAcceptDrops(true);
	tvTODO->header()->setSortIndicatorShown(false);
	tvTODO->header()->setClickable(false);
	tvTODO->header()->setMovable(false);
	tvTODO->header()->setDefaultAlignment( Qt::AlignCenter );
	tvTODO->setStyleSheet( treeViewStyleSheet()	);

	textEdit->setFont(QFont("Verdana", 11));
  m_modelTodo->setSortRole(0);
  m_modelTodo->setColumnCount(0);

  m_downgrade = new QStandardItem(ctn_ICON_INFERIOR, ctn_TODO_DOWNGRADE);
  m_install = new QStandardItem(ctn_ICON_NOT_INSTALLED, ctn_TODO_INSTALL);
  m_remove = new QStandardItem(ctn_ICON_INTERNAL_ERROR, ctn_TODO_REMOVE);
  m_upgrade = new QStandardItem(ctn_ICON_SUPERIOR, ctn_TODO_UPGRADE);
  m_reinstall = new QStandardItem(ctn_ICON_INSTALLED, ctn_TODO_REINSTALL);

  QStandardItem* parentItem = m_modelTodo->invisibleRootItem();
  parentItem->appendRow(m_downgrade);
  parentItem->appendRow(m_install);
  parentItem->appendRow(m_reinstall);
  parentItem->appendRow(m_remove);
  parentItem->appendRow(m_upgrade);

  m_modelTodo->setHorizontalHeaderLabels(QStringList() << tr("Package Actions ( Ctrl+E to execute )"));
	twTODO->setCurrentIndex(0);
}

void MainWindowImpl::createTabPkgDiff(const QString pkg, const QString installedPkg, const QString diff ){
	QWidget *tabPkgFileList = new QWidget();
	QGridLayout *gridLayoutX = new QGridLayout ( tabPkgFileList );
	gridLayoutX->setSpacing ( 0 );
	gridLayoutX->setMargin ( 0 );

	QTextBrowser *text = new QTextBrowser(tabPkgFileList);
	text->setReadOnly(true);
	text->setFrameShape(QFrame::NoFrame);
	text->setFrameShadow(QFrame::Plain);
	gridLayoutX->addWidget ( text, 0, 0, 1, 1 );

	QString html;
  html += tr("<table border=\"1\" bgcolor=\"#FBFFFC\"><tr><th colspan=\"2\"><h3>Diff to %1</h3></th></tr>")
          .arg(installedPkg);
  html += "<tr bgcolor=\"#FBFFFC\"><td><font color=\"#5C311F\"><h3>" + pkg +
          "</h3></td><td><font color=\"#5C311F\"><h3>" + installedPkg + "</h3></td></tr>";

	//We must first parse this diff content to produce human-readable text
	QStringList ldiff = diff.split("\n");
	foreach (QString s, ldiff){
		if ((s[0] != '<') && (s[0] != '>')) continue;
		if (s[0] == '<'){
      html += "<tr><td><h5><font color=\"darkgreen\">" + s.mid(2) + "</h5></td><td></td></tr>";
		}		
		else if (s[0] == '>') {
      html += "<tr><td></td><td><h5><font color=\"#940305\">" + s.mid(2) + "</h5></td></tr>";
		}
	}

  html += "</table>";
	text->insertHtml(html); 
  QString aux;
  if (installedPkg.endsWith(ctn_TGZ_PACKAGE_EXTENSION) || installedPkg.endsWith(ctn_TXZ_PACKAGE_EXTENSION)) aux = pkg + " DIFF";
  else aux = pkg + " DIFF(inst)";
	int tindex = twTODO->addTab( tabPkgFileList, QApplication::translate ( "MainWindow", aux.toUtf8(), 0, QApplication::UnicodeUTF8 ) );
	twTODO->setTabText(twTODO->indexOf(tabPkgFileList), QApplication::translate("MainWindow", aux.toUtf8(), 0, QApplication::UnicodeUTF8));
	twTODO->setCurrentIndex ( tindex ); 
	text->verticalScrollBar()->setValue(0);
}

void MainWindowImpl::createTabPkgFileList(const QString pkgName, const QStringList fileList){
  conditionalGotoNormalView();
  QWidget *tabPkgFileList = new QWidget(this);
	QGridLayout *gridLayoutX = new QGridLayout ( tabPkgFileList );
	gridLayoutX->setSpacing ( 0 );
	gridLayoutX->setMargin ( 0 );
	QStandardItemModel *modelPkgFileList = new QStandardItemModel(this);
	QStandardItemModel *fakeModelPkgFileList = new QStandardItemModel(this);
	QTreeView *tvPkgFileList = new QTreeView(tabPkgFileList);
  QFont f("Verdana", m_PkgListFontSize);

  tvPkgFileList->setFont(f);  
	tvPkgFileList->setEditTriggers(QAbstractItemView::NoEditTriggers);
	tvPkgFileList->setDropIndicatorShown(false);
	tvPkgFileList->setAcceptDrops(false);  
	tvPkgFileList->header()->setSortIndicatorShown(false);
	tvPkgFileList->header()->setClickable(false);
	tvPkgFileList->header()->setMovable(false);
	tvPkgFileList->setFrameShape(QFrame::NoFrame);
	tvPkgFileList->setFrameShadow(QFrame::Plain);
	tvPkgFileList->setObjectName("tvPkgFileList");
  tvPkgFileList->setStyleSheet( treeViewStyleSheet() );

	modelPkgFileList->setSortRole(0);
	modelPkgFileList->setColumnCount(0);

	gridLayoutX->addWidget ( tvPkgFileList, 0, 0, 1, 1 );
	QStandardItem* fakeRoot = fakeModelPkgFileList->invisibleRootItem();
	QStandardItem* root = modelPkgFileList->invisibleRootItem();
	QStandardItem *bkpDir, *item, *bkpItem=root, *parent;
	bool first=true;
	bkpDir = root;

  foreach ( QString file, fileList ){
		QFileInfo fi ( file );
    if ( file.endsWith ( '/' ) ){
      if ( first == true ){
        item = new QStandardItem ( ctn_ICON_FOLDER, file );
        item->setAccessibleDescription("directory " + item->text());
				fakeRoot->appendRow ( item );
			}
      else{
        if ( file.indexOf ( bkpDir->text() ) != -1 ){
          item = new QStandardItem ( ctn_ICON_FOLDER, file );
          item->setAccessibleDescription("directory " + item->text());
					bkpDir->appendRow ( item );
				}
        else{
					parent = bkpItem->parent();
          do{
						if ( parent == 0 || file.indexOf ( parent->text() ) != -1 ) break;
						parent = parent->parent();
					}
					while ( parent != fakeRoot );

          item = new QStandardItem ( ctn_ICON_FOLDER, file );
          item->setAccessibleDescription("directory " + item->text());
					if ( parent != 0 ) parent->appendRow ( item );
					else fakeRoot->appendRow ( item );
				}
			}
			bkpDir = item;
		}
    else{
      item = new QStandardItem ( ctn_ICON_BINARY, fi.fileName() );
      item->setAccessibleDescription("file " + item->text());
      parent = bkpDir;
      do{
				if ( parent == 0 || file.indexOf ( parent->text() ) != -1 ) break;
				parent = parent->parent();
			}
			while ( parent != fakeRoot );
			parent->appendRow ( item );
		}

		bkpItem = item;
		first = false;
	}

	root = fakeRoot;
	fakeModelPkgFileList->sort(0);
	modelPkgFileList = fakeModelPkgFileList;
	tvPkgFileList->setModel(modelPkgFileList);
	tvPkgFileList->header()->setDefaultAlignment( Qt::AlignCenter );
  modelPkgFileList->setHorizontalHeaderLabels( QStringList() <<
    tr("Content of \"%1\" ( Ctrl+W to close this tab )").arg(pkgName));

  QList<QStandardItem*> lit = modelPkgFileList->findItems( "/", Qt::MatchStartsWith | Qt::MatchRecursive );

	foreach( QStandardItem* it, lit ){
		QFileInfo fi( it->text() );
		if ( fi.isFile() == false ){
			QString s( it->text() );
			s.remove(s.size()-1, 1);
			s = s.right(s.size() - s.lastIndexOf('/') -1);
			it->setText( s );
		}
	}

	QString aux("&"+pkgName);
	int tindex = twTODO->addTab( tabPkgFileList, QApplication::translate ( "MainWindow", aux.toUtf8(), 0, QApplication::UnicodeUTF8 ) );
	twTODO->setTabText(twTODO->indexOf(tabPkgFileList), QApplication::translate("MainWindow", aux.toUtf8(), 0, QApplication::UnicodeUTF8));
	twTODO->setCurrentIndex ( tindex );

	tvPkgFileList->setContextMenuPolicy(Qt::CustomContextMenu);
	connect(tvPkgFileList, SIGNAL(customContextMenuRequested(QPoint)), 
			this, SLOT(execContextMenuPkgFileList(QPoint)));
	connect(tvPkgFileList, SIGNAL(clicked (const QModelIndex&)), 
			this, SLOT(showFullPathOfObject(const QModelIndex&))); 
  connect(tvPkgFileList, SIGNAL(doubleClicked (const QModelIndex&)),
			this, SLOT(openFileOrDirectory(const QModelIndex&))); 
  connect(tvPkgFileList, SIGNAL(pressed (const QModelIndex&)),
      tvPkgFileList, SIGNAL(clicked (const QModelIndex&)));
  connect(tvPkgFileList, SIGNAL(activated(const QModelIndex)), tvPkgFileList,
      SIGNAL(clicked(const QModelIndex)));
  tvPkgFileList->installEventFilter(this);
}

QString MainWindowImpl::showFullPathOfObject( const QModelIndex & index ){
  if (!index.isValid()) return "";

	const QStandardItemModel *sim = qobject_cast<const QStandardItemModel*>( index.model() );	

	QStringList sl;	
	QModelIndex nindex;
	QString str;
	sl << sim->itemFromIndex( index )->text();

	nindex = index;
	while (1){
		nindex = sim->parent( nindex ); 
		if ( nindex != sim->invisibleRootItem()->index() ) sl << sim->itemFromIndex( nindex )->text();
		else break;
	}
	str = QDir::separator() + str;

	for ( int i=sl.count()-1; i>=0; i-- ){
		if ( i < sl.count()-1 ) str += QDir::separator();
		str += sl[i];
	}

  m_lblStatus->setText( str );
  return str;
}

void MainWindowImpl::execContextMenuPkgFileList(QPoint point){
	QTreeView *t = twTODO->currentWidget()->findChild<QTreeView*>("tvPkgFileList");
	if (t == 0) return;

  QModelIndex mi = t->currentIndex();
	QMenu menu(this);            
	QStandardItemModel *sim = qobject_cast<QStandardItemModel*>(t->model());
	QStandardItem *si = sim->itemFromIndex(mi);
	if (si == 0) return;
  if (si->hasChildren() && (!t->isExpanded(mi))) menu.addAction( m_actionExpandItem );
  if (si->hasChildren() && (t->isExpanded(mi))) menu.addAction( m_actionCollapseItem );

	if (menu.actions().count() > 0) menu.addSeparator(); 
	menu.addAction( actionCollapse_All_Items );
	menu.addAction( actionExpand_All_Items );
	menu.addSeparator();

	QDir d;
  QFile f(m_lblStatus->text()); QFileInfo fi(f);
  if ( si->icon().pixmap(QSize(22,22)).toImage() == ctn_ICON_FOLDER.pixmap(QSize(22,22)).toImage() ){
    if ( d.exists( m_lblStatus->text() )){
      menu.addAction( actionOpenDirectory );
      menu.addAction( m_actionOpenTerminal );
    }
    if (sim->hasChildren(mi) && (!isFindDisabled())) menu.addAction( m_actionFindFileInPkgFileList );
	}  
  else if ( f.exists( m_lblStatus->text() ) && (fi.size() > 0) ) {
		menu.addAction ( actionOpenFile );
	}
  if (f.exists( m_lblStatus->text()) && UNIXCommand::isTextFile(m_lblStatus->text()))
    menu.addAction( m_actionEditFile );

	menu.addSeparator();
	menu.addAction( actionCloseTab );
	if (twTODO->count() > 3) 
		menu.addAction( actionCloseAllTabs );

	QPoint pt2 = t->mapToGlobal(point);
	pt2.setY(pt2.y() + t->header()->height());
	menu.exec(pt2);        	        	
}

void MainWindowImpl::expandAllContentItems(){
	QTreeView *tv =  twTODO->currentWidget()->findChild<QTreeView *>("tvPkgFileList") ;		
	if ( tv != 0 ){
		tv->repaint(tv->rect());
 		QCoreApplication::processEvents();
		tv->expandAll();
	}
}

void MainWindowImpl::_expandItem(QTreeView* tv, QStandardItemModel* sim, QModelIndex* mi){
  for (int i=0; i<sim->rowCount(*mi); i++){
    if (sim->hasChildren(*mi)){
      tv->expand(*mi);
      QModelIndex mi2 = mi->child(i, 0);
      _expandItem(tv, sim, &mi2);
		}
  }
}

void MainWindowImpl::_collapseItem(QTreeView* tv, QStandardItemModel* sim, QModelIndex mi){
	for (int i=0; i<sim->rowCount(mi); i++){
		if (sim->hasChildren(mi)){			
	 		QCoreApplication::processEvents();
			tv->collapse(mi);
			QModelIndex mi2 = mi.child(i, 0);
			_collapseItem(tv, sim, mi2);
		}
	}
}

void MainWindowImpl::expandThisContentItems(){
	QTreeView *tv =  twTODO->currentWidget()->findChild<QTreeView *>("tvPkgFileList") ;
	if ( tv != 0 ){
		tv->repaint(tv->rect());
    QCoreApplication::processEvents();
		QStandardItemModel *sim = qobject_cast<QStandardItemModel*>(tv->model());
		QModelIndex mi = tv->currentIndex();
    if (sim->hasChildren(mi))	_expandItem(tv, sim, &mi);
	} 
}

void MainWindowImpl::collapseThisContentItems(){
	QTreeView *tv =  twTODO->currentWidget()->findChild<QTreeView *>("tvPkgFileList") ;
	if ( tv != 0 ){
		tv->repaint(tv->rect());
 		QCoreApplication::processEvents();
		QStandardItemModel *sim = qobject_cast<QStandardItemModel*>(tv->model());
		QModelIndex mi = tv->currentIndex();
		if (sim->hasChildren(mi))	_collapseItem(tv, sim, mi);
	} 
}

void MainWindowImpl::collapseAllContentItems(){
	QTreeView *tv =  twTODO->currentWidget()->findChild<QTreeView *>("tvPkgFileList") ;
	if ( tv != 0 ) tv->collapseAll();
}

void  MainWindowImpl::removeAbsoluteDir( QStandardItemModel *im, QModelIndex index ){
	if (index.isValid() == false) return;
	else if ( im->itemFromIndex(index)->hasChildren() ) {
		QStandardItem* si = im->itemFromIndex(index)->child(0, 0);
		removeAbsoluteDir(im, im->index(0, 0, si->index()));		
	}
	else{
		QStandardItem *item = im->itemFromIndex(index);
		QFileInfo fi( item->text() );
		if (fi.isDir()) item->setText( fi.path() );
	}
}

void MainWindowImpl::createDirectory(){
	QModelIndex index = tvDir->currentIndex();
	if (!index.isValid())
		return;

	QString dirName = QInputDialog::getText(this,
			tr("Create directory"),
			tr("Directory name"));

	if ((dirName != 0) && (dirName.length() > 0)) {
    if (!m_modelDir->mkdir(index, dirName).isValid())
			QMessageBox::information(this, tr("Create directory"),
					tr("Failed to create the directory"));
	}		
}

void MainWindowImpl::removeDirectory(){
	QModelIndex index = tvDir->currentIndex();

	if (!index.isValid()) return;

	bool ok;
  if (m_modelDir->fileInfo(index).isDir()) {
		int res;
		res = QMessageBox::question(this, tr("Confirmation"), 
				tr("Are you sure you want to remove this directory?"), 
				QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
		if (res == QMessageBox::Yes){
      ok = m_modelDir->rmdir(index);
      if (!ok) QMessageBox::information(this, tr("Remove"),
                 tr("Failed to remove") + " " + (m_modelDir->fileName(index)));
		}
	}	  
}

void MainWindowImpl::changeDefaultDirectory(){
  if (dockDirectories->windowTitle() != ""){
    m_DefaultDirectory = dockDirectories->windowTitle();
    SettingsManager::instance()->setDefaultDirectory(m_DefaultDirectory);
    initializeDirTreeView();
  }
}

void MainWindowImpl::showPackagesInDirectory(){
	changeDir();	

	QPointer<CPUIntensiveComputing> gRi;	
  if (!m_fsw.directories().isEmpty()) m_fsw.removePath( m_fsw.directories()[0] );
  if (tvDir->currentIndex().isValid()) m_fsw.addPath( m_modelDir->filePath(tvDir->currentIndex() ));
  m_modelPackage->clear();
  QDir d(m_modelDir->filePath(tvDir->currentIndex()));

	QStringList fl;
  QStringList list = d.entryList(fl << "*" + ctn_TXZ_PACKAGE_EXTENSION << "*" +
    ctn_TGZ_PACKAGE_EXTENSION << "*" + ctn_RPM_PACKAGE_EXTENSION,
		QDir::Files, QDir::Name); 
	if ( list.count() > 0 )	gRi = new CPUIntensiveComputing();		

  QList<QStandardItem*> items, icons;

	foreach (QString str, list) {
    Result res = Package::getStatus(str);
		QStandardItem *s;

		switch (res.getClassification()) { 
    case ectn_RPM:
			s = new QStandardItem(
          ctn_ICON_RPM, "_RPM");
			icons << s;
			s = new QStandardItem( str );	
			items << s; break;

    case ectn_FROZEN :
			s = new QStandardItem(
          ctn_ICON_FROZEN, "_Frozen");
			icons << s;
			s = new QStandardItem( str );	
			items << s; break;

    case ectn_INTERNAL_ERROR :
			s = new QStandardItem(
          ctn_ICON_INTERNAL_ERROR, "_Error");
			icons << s;
			s = new QStandardItem( str );	
			items << s; break;

    case ectn_INFERIOR_VERSION :
			s = new QStandardItem(
          ctn_ICON_INFERIOR, "_Inferior");
			icons << s;
			s = new QStandardItem( str );    			
			items << s; break;

    case ectn_SUPERIOR_VERSION:
			s = new QStandardItem(
          ctn_ICON_SUPERIOR, "_Superior");
			icons << s;
			s = new QStandardItem( str );    						
			items << s; break;

    case ectn_OTHER_VERSION:
			s = new QStandardItem(
          ctn_ICON_OTHER_VERSION, "_OtherVersion");
			icons << s;
			s = new QStandardItem( str );    						
			items << s; break;

    case ectn_INSTALLED :
			s = new QStandardItem(
          ctn_ICON_INSTALLED, "_Installed");
			icons << s;
			s = new QStandardItem( str );    						
			items << s; break;

    case ectn_NOT_INSTALLED :
			s = new QStandardItem(
          ctn_ICON_NOT_INSTALLED, "_Not installed");
			icons << s;
			s = new QStandardItem( str );    						
			items << s; break; 

		default: 
			s = new QStandardItem(
          ctn_ICON_INTERNAL_ERROR, "_Error");
			icons << s;
			s = new QStandardItem( str );    						
			items << s;	    	
		}    	
	}
	
  m_modelPackage->appendColumn(icons);
  m_modelPackage->appendColumn(items);
  m_modelPackage->sort(1);
  m_modelPackage->setHorizontalHeaderLabels(QStringList() << "" << tr("Name"));

	QString msg;
	if (items.size()>=1){
		if (items.size()>1) dockPackages->setWindowTitle(tr("%1 Packages in Directory").arg(
				QString::number(items.size())));
		else dockPackages->setWindowTitle(tr("%1 Package in Directory").arg(QString::number(items.size())));  	
	}
  else dockPackages->setWindowTitle(ctn_LABEL_TREEVIEW_PACKAGES);

	tvPackage->setColumnWidth(0, 24);	
	tvPackage->header()->setSortIndicator( m_PackageListOrderedCol, m_PackageListSortOrder );	
	tvPackage->sortByColumn( m_PackageListOrderedCol, m_PackageListSortOrder );
	tvPackage->header()->setMovable(false);
	tvPackage->header()->setResizeMode( QHeaderView::Fixed );
  if (leFilterPackage->text() != "") reapplyPackageFilter();

  if (m_modelPackage->rowCount() > 0) {
    QModelIndex mi = m_proxyModelPackage->index(0, 0);
		tvPackage->setCurrentIndex(mi);
		tvPackage->scrollTo(mi);
		selectInstalledPackage();    
		tvPackage->setFocus();
	}		
	
  if (gRi) delete gRi;
}

void MainWindowImpl::selectInstalledPackage(){
  if (m_modelPackage->rowCount() == 0) return;

  QDir d(m_modelDir->filePath(tvDir->currentIndex()));
	d.setFilter(QDir::Files);	

	QModelIndex mi;

	QList<SelectedPackage> lt = getSelectedPackage();
	if (lt.count() == 0){
    QModelIndex mi = m_proxyModelInstalledPackages->index(0, 0);
		tvInstalledPackages->selectionModel()->clear();
		tvInstalledPackages->scrollTo(mi, QAbstractItemView::PositionAtCenter);
	} 

	else if (lt.count() == 1) tvInstalledPackages->selectionModel()->clear(); 
	
	foreach( SelectedPackage sp, lt ){
    Result res = Package::getStatus(sp.getFileName());
		if (res.getInstalledPackage().size() > 0){
			QList<QStandardItem*> l = 
        m_modelInstalledPackages->findItems( res.getInstalledPackage(), Qt::MatchStartsWith, ctn_PACKAGE_NAME );

			if (l.count() > 0){
				QStandardItem* aux = l[0];

				mi = 	aux->index();
        mi = m_proxyModelInstalledPackages->mapFromSource(mi);
        if (!m_proxyModelInstalledPackages->hasIndex(mi.row(), mi.column())) return;
			}
      else mi = m_proxyModelInstalledPackages->index(0, 0);

			tvInstalledPackages->scrollTo(mi, QAbstractItemView::PositionAtCenter);		
      QModelIndex maux = m_proxyModelInstalledPackages->index(mi.row(), 0);
			tvInstalledPackages->selectionModel()->setCurrentIndex(mi, QItemSelectionModel::Select);
			tvInstalledPackages->selectionModel()->setCurrentIndex(maux, QItemSelectionModel::Select);

		}
		else if (res.getInstalledPackage().size() == 0){
      QModelIndex mi = m_proxyModelInstalledPackages->index(0, 0);
			tvInstalledPackages->selectionModel()->clear();
			tvInstalledPackages->scrollTo(mi, QAbstractItemView::PositionAtCenter);
		} 
	}
}

void MainWindowImpl::positionInInstalledPkgList( const QString& pkg ){
  if (actionHideRightView->isChecked()) actionHideRightView->trigger();
  if (!actionMinimizeLowerView->isChecked() && !actionNormalView->isChecked()){
    actionNormalView->setChecked(true);
    normalView();
  }

  tvInstalledPackages->setFocus();
	tvInstalledPackages->selectionModel()->clear();	

  QList<QStandardItem*> l = m_modelInstalledPackages->findItems( pkg, Qt::MatchStartsWith, ctn_PACKAGE_NAME );
	QStandardItem* aux;
	QModelIndex mi;

	if (l.count() > 0){
		aux = l[0];
		mi = 	aux->index();
    mi = m_proxyModelInstalledPackages->mapFromSource(mi);
    if (!m_proxyModelInstalledPackages->hasIndex(mi.row(), mi.column())) return;
	}
  else return;

	tvInstalledPackages->scrollTo(mi, QAbstractItemView::PositionAtCenter);		
  QModelIndex maux = m_proxyModelInstalledPackages->index(mi.row(), 0);
	tvInstalledPackages->selectionModel()->setCurrentIndex(mi, QItemSelectionModel::Select);
	tvInstalledPackages->selectionModel()->setCurrentIndex(maux, QItemSelectionModel::Select);
}

void MainWindowImpl::positionInPkgList( const QString &dir, const QString &pkg ){
  tvPackage->setFocus();
  tvPackage->selectionModel()->clear();

  //Let's change the directory first
  if (dir != dockDirectories->windowTitle()){
    QModelIndex index = m_modelDir->setRootPath(dir);
    tvDir->setCurrentIndex(index);
    tvDir->scrollTo(index);
    showPackagesInDirectory();
    tvPackage->setCurrentIndex(QModelIndex());
  }

  QList<QStandardItem*> l = m_modelPackage->findItems( pkg, Qt::MatchStartsWith, ctn_PACKAGE_NAME );
  QStandardItem* aux;
  QModelIndex mi;

  if (l.count() > 0){
    aux = l[0];
    mi = 	aux->index();
    mi = m_proxyModelPackage->mapFromSource(mi);
    if (!m_proxyModelPackage->hasIndex(mi.row(), mi.column())) return;
  }
  else return;

  tvPackage->scrollTo(mi, QAbstractItemView::PositionAtCenter);
  QModelIndex maux = m_proxyModelPackage->index(mi.row(), 0);
  tvPackage->selectionModel()->setCurrentIndex(mi, QItemSelectionModel::Select);
  tvPackage->selectionModel()->setCurrentIndex(maux, QItemSelectionModel::Select);

  if (!actionMinimizeLowerView->isChecked() && !actionNormalView->isChecked()){
    actionNormalView->setChecked(true);
    normalView();
  }
}

void MainWindowImpl::positionInPkgFileList( const QString &fileName, const QString &directory ){
  QTreeView *t = twTODO->currentWidget()->findChild<QTreeView*>("tvPkgFileList");
  if ( t != 0){
    conditionalGotoNormalView();

    QStandardItemModel *sim = qobject_cast<QStandardItemModel*>(t->model());
    QList<QStandardItem*> fileList = sim->findItems(fileName, Qt::MatchContains|Qt::MatchRecursive, 0);

    foreach(QStandardItem *it, fileList){
      if(it->icon().pixmap(QSize(22,22)).toImage() == ctn_ICON_FOLDER.pixmap(QSize(22,22)).toImage()) continue;

      QString fullPath = showFullPathOfObject(it->parent()->index());
      if (it->text() == fileName && fullPath == directory){
        t->setCurrentIndex(it->index());
        t->scrollTo(t->currentIndex());
        showFullPathOfObject(it->index());
        break;
      }
    }
  }
}

QList<SelectedPackage> MainWindowImpl::getSelectedPackage(){
	QList<SelectedPackage> lsp;
  QDir d(m_modelDir->filePath(tvDir->currentIndex()));
	d.setFilter(QDir::Files);	
	QList<QIcon> licons;
	int c=0;

  foreach(QModelIndex item, tvPackage->selectionModel()->selectedRows(ctn_PACKAGE_ICON)){
    QModelIndex mi = m_proxyModelPackage->mapToSource(item);
    QStandardItem *si = m_modelPackage->item( mi.row(), ctn_PACKAGE_ICON );
		if (si != 0) 
			licons << si->icon();
	}

  foreach(QModelIndex item, tvPackage->selectionModel()->selectedRows(ctn_PACKAGE_NAME)){
    QModelIndex mi = m_proxyModelPackage->mapToSource(item);
    QStandardItem *si = m_modelPackage->item( mi.row(), ctn_PACKAGE_NAME );
		if (si != 0){ 
			lsp << SelectedPackage(d.absolutePath(), si->text(), licons[c]);
			c++;
		}
	}

	return lsp;		
}

QList<SelectedPackage> MainWindowImpl::getSelectedInstalledPackage(){
	QList<SelectedPackage> lsp;
	foreach(QModelIndex item, tvInstalledPackages->selectionModel()->selectedIndexes()){
    if ( item.column() == ctn_PACKAGE_NAME ){
      lsp << SelectedPackage(ctn_PACKAGES_DIR, m_proxyModelInstalledPackages->data(item, 0).toString());
		}
	}

	return lsp;	
}

void MainWindowImpl::showPackageInfo(){
	tvPackage->repaint(tvPackage->rect());
	QCoreApplication::processEvents();	

	QList<SelectedPackage> lsp = getSelectedPackage();
	if (lsp.count() > 1) return;
	foreach(SelectedPackage sp, lsp){       	
    if ( sp.getIcon().pixmap(QSize(22,22)).toImage() == ctn_ICON_RPM.pixmap(QSize(22,22)).toImage() ) continue;

		CPUIntensiveComputing *ri = new CPUIntensiveComputing();
		QString sb = Package::getInformation(sp.getCompleteFileName(), false);
		delete ri;
		if ((sb != 0) && (sb.size() > 0)) QMessageBox::information(this, tr("Package %1 info").arg(sp.getFileName()),  sb);
		else QMessageBox::critical(this, tr("Package %1 info").arg(sp.getFileName()), tr("Sorry, no info provided!"));    
	}
}

void MainWindowImpl::showInstalledPackageInfo(){
	tvInstalledPackages->repaint(tvInstalledPackages->rect());
	QCoreApplication::processEvents();	
	QList<SelectedPackage> lsp = getSelectedInstalledPackage();

	foreach(SelectedPackage sp, lsp){
		CPUIntensiveComputing *ri = new CPUIntensiveComputing();
		QString sb = Package::getInformation(sp.getCompleteFileName(), true);
		delete ri;
		if ((sb != 0) && (sb.size() > 0)){
			QMessageBox::information(this, tr("Package %1 info").arg(sp.getFileName()), "<html>" + sb + "</html>");
		}
		else QMessageBox::critical(this, tr("Package %1 info").arg(sp.getFileName()), tr("Sorry, no info provided!"));    
	}
}

//Returns true if the package (installed or not) has a content
bool MainWindowImpl::execPackageContent(){
	CPUIntensiveComputing ri;
	QStringList res = m_fw->result();	
	QString package = res[0];
	res.removeAt(0);
	if ( !res.isEmpty() ) 
		createTabPkgFileList(package, res);  
	else{
    ri.restoreDefaultCursor();
    while (QApplication::overrideCursor() and QApplication::overrideCursor()->shape()==Qt::WaitCursor)
      ri.restoreDefaultCursor();

		QMessageBox::critical(this, tr("Package %1").arg(package), tr("This package seems corrupted!"));	
	}
  return !res.isEmpty();
}

QStringList _showPackageContent(const QString& fileName, const QString& completeFileName, bool b){
	QStringList sl;
	sl << fileName << Package::getContents(completeFileName, b);
	return sl;
}

//Shows the Package (installed or not) list of files and returns true if the package has a content
bool MainWindowImpl::showPackageContent(){
  tvPackage->repaint(tvPackage->rect());
	QCoreApplication::processEvents();	
  CPUIntensiveComputing ri;
  QFuture<QStringList> f;
	bool alreadyExists=false;
  bool res=false;

	if (tvInstalledPackages->hasFocus()){
		foreach( SelectedPackage sp, getSelectedInstalledPackage() ){  			
			alreadyExists = false;

			//First we check if there's not a tab opened with this pkgName contents
			for (int c=2; c<twTODO->count(); c++)
				if (twTODO->tabText(c) == ("&" + sp.getFileName())){
					twTODO->setCurrentIndex ( c );			
					alreadyExists = true;
					break;
				} 
			if (alreadyExists) continue;

			QString pkg(sp.getCompleteFileName());			
			f = run(_showPackageContent, sp.getFileName(), pkg, true);
			m_fw->setFuture(f);
			m_fw->waitForFinished();
			QCoreApplication::processEvents();	
      res=execPackageContent();
		}
	}
	else{
		foreach( SelectedPackage sp, getSelectedPackage() ){  			
			alreadyExists = false;

			//First we check if there's not a tab opened with this pkgName contents
			for (int c=2; c<twTODO->count(); c++)
				if (twTODO->tabText(c) == ("&" + sp.getFileName())){
					twTODO->setCurrentIndex ( c );			
					alreadyExists = true;
					break;
				} 
			if (alreadyExists) continue;

			QString pkg(sp.getCompleteFileName());
			f = run(_showPackageContent, sp.getFileName(), pkg, false);
			m_fw->setFuture(f);
			m_fw->waitForFinished();
			QCoreApplication::processEvents();
      res=execPackageContent();
		}			
	}
  if (alreadyExists) res = true;
  if (res==true) conditionalGotoNormalView();
  return res;
}

void MainWindowImpl::executePackageActions(){ 
  QStringList out;
  for (int r=0; r<m_remove->rowCount(); r++){
    QFileInfo fi(m_remove->child(r)->text());
		out << "cd \"" << fi.path() << "\" \n";
    out << ctn_PACKAGE_REMOVE << " " << fi.fileName() << "\n";
	}
  for (int r=0; r<m_downgrade->rowCount(); r++){
    QFileInfo fi(m_downgrade->child(r)->text());
		out << "cd \"" << fi.path() << "\" \n";
    out << ctn_PACKAGE_UPGRADE << " " << fi.fileName() << "\n";
	}
  for (int r=0; r<m_upgrade->rowCount(); r++){
    QFileInfo fi(m_upgrade->child(r)->text());
		out << "cd \"" << fi.path() << "\" \n";
    out << ctn_PACKAGE_UPGRADE << " " << fi.fileName() << "\n";
	}
  for (int r=0; r<m_install->rowCount(); r++){
    QFileInfo fi(m_install->child(r)->text());
		out << "cd \"" << fi.path() << "\" \n";
    out << ctn_PACKAGE_INSTALL << " " << fi.fileName() << "\n";
	}
  for (int r=0; r<m_reinstall->rowCount(); r++){
    QFileInfo fi(m_reinstall->child(r)->text());
		out << "cd \"" << fi.path() << "\" \n";
    out << ctn_PACKAGE_REINSTALL << " " << fi.fileName() << "\n";
	}

	textEdit->clear();
	textEdit->setHtml("");

  conditionalGotoNormalView();
  m_unixCommand = new UNIXCommand(this);

  QObject::connect(m_unixCommand, SIGNAL( started() ), this, SLOT( ACTIONSprocessStarted()));
  QObject::connect(m_unixCommand, SIGNAL( readyReadStandardOutput() ), this, SLOT( ACTIONSprocessReadOutput() ));
  QObject::connect(m_unixCommand, SIGNAL( finished ( int, QProcess::ExitStatus )),
                   this, SLOT( ACTIONSprocessFinished(int, QProcess::ExitStatus) ));
  QObject::connect(m_unixCommand, SIGNAL( readyReadStandardError() ), this, SLOT( ACTIONSprocessRaisedError() ));

  m_unixCommand->executePackageActions(out);
}

void MainWindowImpl::ACTIONSprocessRaisedError(){
	twTODO->setCurrentIndex(1);
	QString msg;
  msg = tr("<br><h5>It seems an error has occurred...</h5><br>") + m_textForMoreInformation;
  textEdit->insertHtml("<pre>" + m_unixCommand->readAllStandardError() + "</pre>");
	textEdit->insertHtml(msg);
	textEdit->ensureCursorVisible();
}

void MainWindowImpl::ACTIONSprocessStarted(){
	twTODO->setCurrentIndex(1);
	QString msg;
	textEdit->clear();
	actionExecuteActions->setEnabled( false );
  msg = tr("<h5>Please, wait while the actions are being executed...</h5><br>");
	textEdit->insertHtml(msg);
}

void MainWindowImpl::ACTIONSprocessFinished(int, QProcess::ExitStatus){
	twTODO->setCurrentIndex(1);
	QString msg;
  m_lblStatus->setText("");

	if (textEdit->toHtml().indexOf("<pre style") != -1){
    m_install->removeRows(0, m_install->rowCount());
    m_remove->removeRows(0, m_remove->rowCount());
    m_downgrade->removeRows(0, m_downgrade->rowCount());
    m_upgrade->removeRows(0, m_upgrade->rowCount());
    m_reinstall->removeRows(0, m_reinstall->rowCount());
    delete m_modelInstalledPackages;
    delete m_proxyModelInstalledPackages;
    initializeInstalledPackagesTreeView();

    msg = tr("<br><h5>Finished executing package actions!</h5><br>") + m_textForMoreInformation;
	}
	else{
    msg = tr("<br><h5>Package actions's execution were cancelled!</h5><br>") + m_textForMoreInformation;
		actionExecuteActions->setEnabled( true );
	}
	
	reapplyInstalledPackagesFilter();	
	textEdit->insertHtml(msg);
	textEdit->ensureCursorVisible();
  delete m_unixCommand;
  m_unixCommand = 0;
  QFile::remove(ctn_TEMP_ACTIONS_FILE);
}

void MainWindowImpl::ACTIONSprocessReadOutput(){
	twTODO->setCurrentIndex(1);
  textEdit->insertHtml("<pre>" + m_unixCommand->readAllStandardOutput() + "</pre>");
	textEdit->ensureCursorVisible();
}

void MainWindowImpl::TGZ2LZMprocessReadOutput(){
	twTODO->setCurrentIndex(1);
  textEdit->insertHtml("<pre>" + m_unixCommand->readAllStandardOutput() + "</pre>");
	textEdit->ensureCursorVisible();
}

void MainWindowImpl::RPM2TGZprocessReadOutput(){
	twTODO->setCurrentIndex(1);
  textEdit->insertHtml("<pre>" + m_unixCommand->readAllStandardOutput() + "</pre>");
	textEdit->ensureCursorVisible();
}

void MainWindowImpl::TGZ2LZMprocessStarted(){
	twTODO->setCurrentIndex(1);
	QString msg;
	textEdit->clear();
	actionExecuteActions->setEnabled( false );
  msg = tr("<h5>Please, wait while the TGZ to LZM convertion is being executed...</h5><br>");
	textEdit->insertHtml(msg);
}

void MainWindowImpl::RPM2TGZprocessStarted(){
	twTODO->setCurrentIndex(1);
	QString msg;
	textEdit->clear();
	actionExecuteActions->setEnabled( false );
  msg = tr("<h5>Please, wait while the RPM to TGZ convertion is being executed...</h5><br>");
	textEdit->insertHtml(msg);
}

void MainWindowImpl::TGZ2LZMprocessFinished(int, QProcess::ExitStatus){
	twTODO->setCurrentIndex(1);
	QString msg;		
  m_lblStatus->setText("");

	if (textEdit->toHtml().indexOf("<pre style") != -1){
    msg = tr("<br><h5>Finished executing TGZ to LZM convertion!</h5><br>") + m_textForMoreInformation;
	}
	else{
    msg = tr("<br><h5>TGZ to LZM convertion was cancelled!</h5><br>") + m_textForMoreInformation;
	}

	textEdit->insertHtml(msg);
	textEdit->ensureCursorVisible();
  delete m_unixCommand;
  m_unixCommand = 0;
  QFile::remove(ctn_TEMP_ACTIONS_FILE);
}

void MainWindowImpl::RPM2TGZprocessFinished(int, QProcess::ExitStatus){
	twTODO->setCurrentIndex(1);
	QString msg;		
  m_lblStatus->setText("");
  msg = tr("<br><h5>Finished executing RPM to TGZ convertion!</h5><br>") + m_textForMoreInformation;
	textEdit->insertHtml(msg);
	textEdit->ensureCursorVisible();
  delete m_unixCommand;
  m_unixCommand = 0;
  QFile::remove(ctn_TEMP_ACTIONS_FILE);
}

void MainWindowImpl::TGZ2LZMprocessRaisedError(){
	twTODO->setCurrentIndex(1);
  textEdit->insertHtml("<pre>" + m_unixCommand->readAllStandardError() + "</pre><br><br>");
	textEdit->ensureCursorVisible();
}

void MainWindowImpl::RPM2TGZprocessRaisedError(){
	twTODO->setCurrentIndex(1);
	QString msg;
  msg = tr("<br><h5>It seems an error has occurred...</h5><br>") + m_textForMoreInformation;
  textEdit->insertHtml("<pre>" + m_unixCommand->readAllStandardError() + "</pre>");
	textEdit->insertHtml(msg);
	textEdit->ensureCursorVisible();
}

void MainWindowImpl::reapplyPackageFilter(){
	QRegExp regExp(leFilterPackage->text(), Qt::CaseInsensitive, QRegExp::FixedString);
  m_proxyModelPackage->setFilterRegExp(regExp);
  int numPkgs = m_proxyModelPackage->rowCount();

	if (leFilterPackage->text() != ""){
		if (numPkgs > 0) leFilterPackage->setStyleSheet(filterPackageFoundStyleSheet());
		else leFilterPackage->setStyleSheet(filterPackageNotFoundStyleSheet());
	}
	else{
		leFilterPackage->setStyleSheet("");
    showPackagesInDirectory();
	}

	if (numPkgs > 1)
		dockPackages->setWindowTitle(tr("%1 Packages in Directory").arg(QString::number(numPkgs)));
	else if (numPkgs == 1)
		dockPackages->setWindowTitle(tr("1 Package in Directory"));
	else
		dockPackages->setWindowTitle(tr("0 Packages in Directory"));

	tvPackage->selectionModel()->clear();
  QModelIndex mi = m_proxyModelPackage->index(0, 0);
	tvPackage->setCurrentIndex(mi);
	tvPackage->scrollTo(mi);
	selectInstalledPackage();
}

void MainWindowImpl::reapplyInstalledPackagesFilter(){
	QRegExp regExp(leFilterInstalledPackages->text(), Qt::CaseInsensitive, QRegExp::FixedString);
  m_proxyModelInstalledPackages->setFilterRegExp(regExp);
  int numInstPkgs = m_proxyModelInstalledPackages->rowCount();

	if (leFilterInstalledPackages->text() != ""){
		if (numInstPkgs > 0) leFilterInstalledPackages->setStyleSheet(filterPackageFoundStyleSheet());
		else leFilterInstalledPackages->setStyleSheet(filterPackageNotFoundStyleSheet());
	}
	else{
		leFilterInstalledPackages->setStyleSheet("");
	}

	if (numInstPkgs > 1)
		dockInstalledPackages->setWindowTitle(tr("%1 Installed Packages").arg(QString::number(numInstPkgs)));
	else if (numInstPkgs == 1)
		dockInstalledPackages->setWindowTitle(tr("1 Installed Package"));
	else
		dockInstalledPackages->setWindowTitle(tr("No Installed Package found!"));

	tvInstalledPackages->selectionModel()->clear();
  QModelIndex mi = m_proxyModelInstalledPackages->index(0, 1);
	tvInstalledPackages->scrollTo(mi, QAbstractItemView::PositionAtCenter);		
  QModelIndex maux = m_proxyModelInstalledPackages->index(0, 0);
	tvInstalledPackages->selectionModel()->setCurrentIndex(maux, QItemSelectionModel::Select);
	tvInstalledPackages->selectionModel()->setCurrentIndex(mi, QItemSelectionModel::Select);	
}

void MainWindowImpl::clearFilterPackage(){
  leFilterPackage->clear();
  leFilterPackage->setFocus();
}

void MainWindowImpl::clearFilterInstalledPackage(){
  leFilterInstalledPackages->clear();
  leFilterInstalledPackages->setFocus();
}
