/*
 * This file is part of QTGZManager, an open-source GUI for Slackware pkgtools.
 * Copyright (C) 2006  Alexandre Albuquerque Arnt
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Contact e-mail : Alexandre Albuquerque Arnt <aaarnt@gmail.com>
 * Program URL   : http://jtgzmanager.sf.net
 *
 */

#include "unixcommand.h"
#include "package.h"
#include <QString>
#include <QStringList>
#include <QProcess>
#include <QFile>
#include <QFileInfo>
#include <QByteArray>
#include <QMessageBox>
#include <QTextStream>

bool UNIXCommand::isAnInstanceRunning(){
  QStringList slParam;
  QProcess proc;
  slParam << "-C";
  slParam << "qtgzmanager";
  proc.start("ps", slParam);
  proc.waitForStarted();
  proc.waitForFinished();

  //First, we verify if there is already a running instance of QTGZManager...
  QString out = proc.readAll();
  if  (out.count("qtgzmanager")>1) 	return true;
  else return false;
}

bool UNIXCommand::hasTheExecutable( const QString& exeName ){
  QStringList slParam;
  QProcess* proc = new QProcess();
  slParam << exeName;
  proc->start("whereis", slParam);
  proc->waitForFinished();
  QString out = proc->readAllStandardOutput();

  delete proc;
  if (out.count(exeName) == 1) return false;
  else return true;
}

bool UNIXCommand::isTextFile( const QString& fileName ){
  QProcess *p = new QProcess();
  QStringList s(fileName);
  p->start( "file", s );
  p->waitForFinished();
  QByteArray output = p->readAllStandardOutput();
  delete p;

  int from = output.indexOf(":", 0)+1;
  return ((output.indexOf( "ASCII", from ) != -1) || (output.indexOf( "text", from ) != -1));
}

void UNIXCommand::editFile( const QString& fileName ){
  QProcess *process = new QProcess(qApp->activeWindow());
  QStringList s;

  if (QDir::homePath() != ctn_ROOT_HOME){
    s << "-d";
    s << "-t";
    s << "--noignorebutton";
    s << "kwrite " + fileName;
    process->startDetached("kdesu", s);
  }
  else{
    s << "kwrite " + fileName;
    process->startDetached("/bin/sh", s);
  }
}

void UNIXCommand::openFile( const QString& fileName ){
  QFile f(fileName);
  QFileInfo fi(f);
  if (fi.size() == 0) return;

  if (!isTextFile(fileName)){
    int res = QMessageBox::question(qApp->activeWindow(), QObject::tr("Confirmation"),
        QObject::tr("This file does not appear to be a simple text.\n"
            "Are you sure you want to run it?"),
            QMessageBox::Yes | QMessageBox::No,
            QMessageBox::No);

    if ( res == QMessageBox::No ) return;
  }

  QProcess *p = new QProcess(qApp->activeWindow());
  QStringList s;
  s << "exec";
  s << "file:" + fileName;
  p->startDetached( "kfmclient" , s );
}

void UNIXCommand::openDirectory( const QString& dirName ){
  QProcess *p = new QProcess(qApp->activeWindow());
  QStringList s;
  s << "newTab";

  QFileInfo f(dirName);
  if (f.exists()){
    s << dirName;
    p->startDetached( "kfmclient" , s );
  }
}

void UNIXCommand::openTerminal( const QString& dirName ){
  QProcess *p = new QProcess(qApp->activeWindow());
  QStringList s("--workdir");

  QFileInfo f(dirName);
  if (f.exists()){
    s << dirName;
    p->startDetached( "konsole" , s );
  }
}

QString UNIXCommand::executeDiffToEachOther( QString pkg1, QString pkg2 ){
  QStringList sl = Package::getContents(pkg1);
  QCoreApplication::processEvents();
  QStringList sl2 = Package::getContents(pkg2);
  QCoreApplication::processEvents();

  if ( sl.isEmpty() && sl2.isEmpty() ) return 0;

  sl.sort();
  sl2.sort();
  QFile fPkg("/tmp/tempPkg");
  QFile fIPkg("/tmp/tempIPkg");

  fPkg.open(QIODevice::ReadWrite | QIODevice::Text);
  fIPkg.open(QIODevice::ReadWrite | QIODevice::Text);
  QTextStream tsp(&fPkg);
  QTextStream tsip(&fIPkg);

  foreach(QString s, sl) tsp << s << endl;
  foreach(QString s, sl2) tsip << s << endl;

  QStringList slParam;
  QProcess proc;

  slParam << "--suppress-common-lines";
  slParam << fPkg.fileName();
  slParam << fIPkg.fileName();

  proc.start("diff", slParam);
  proc.waitForStarted();
  proc.waitForFinished();

  fPkg.close();
  fIPkg.close();
  fPkg.remove();
  fIPkg.remove();

  QString result = proc.readAllStandardOutput();

  if (result != 0)
    return result;
  else return ctn_PACKAGES_WITH_SAME_CONTENT;
}

QString UNIXCommand::executeDiffToInstalled( QString pkg, QString installedPackage ){
  QStringList sl = Package::getContents(ctn_PACKAGES_DIR + QDir::separator() + installedPackage, true);
  QCoreApplication::processEvents();
  QStringList sl2 = Package::getContents(pkg);
  QCoreApplication::processEvents();

  if ( sl2.isEmpty() ) return 0;

  sl.sort();
  sl2.sort();

  QFile fPkg("/tmp/tempPkg");
  QFile fIPkg("/tmp/tempIPkg");

  fPkg.open(QIODevice::ReadWrite | QIODevice::Text);
  fIPkg.open(QIODevice::ReadWrite | QIODevice::Text);
  QTextStream tsp(&fPkg);
  QTextStream tsip(&fIPkg);

  foreach(QString s, sl2) tsp << s << endl;
  foreach(QString s, sl) tsip << s << endl;

  //Here, we execute the diff
  QStringList slParam;
  QProcess proc;

  slParam << "--suppress-common-lines";
  slParam << fPkg.fileName();
  slParam << fIPkg.fileName();
  proc.start("diff", slParam);
  proc.waitForStarted();
  proc.waitForFinished();

  fPkg.close();
  fIPkg.close();
  fPkg.remove();
  fIPkg.remove();

  return proc.readAllStandardOutput();
}

void UNIXCommand::executePackageActions( const QStringList& commandList ){
  QFile ftemp(ctn_TEMP_ACTIONS_FILE);
  ftemp.open(QIODevice::ReadWrite|QIODevice::Text);
  ftemp.setPermissions(QFile::ReadOwner|QFile::WriteOwner|QFile::ExeOwner|
      QFile::ReadUser|QFile::WriteUser|QFile::ExeUser);
  QTextStream out(&ftemp);
  QStringList s;
  bool rootUser = (QDir::homePath() == ctn_ROOT_HOME);

  if (!rootUser){
    s << "-d";
    s << "-t";
    s << "--noignorebutton";
  }
  else s << "-c";

  foreach(QString line, commandList) out << line;
  out.flush();
  ftemp.close();
  s << ctn_TEMP_ACTIONS_FILE;

  if (rootUser) m_process->start("/bin/sh", s);
  else m_process->start("kdesu", s);
}

void UNIXCommand::transformTGZinLZM( const QStringList& commandList, LZMCommand commandUsed ){
  QFile ftemp(ctn_TEMP_ACTIONS_FILE);
  ftemp.open(QIODevice::ReadWrite|QIODevice::Text);
  ftemp.setPermissions(QFile::ReadOwner|QFile::WriteOwner|QFile::ExeOwner|
      QFile::ReadUser|QFile::WriteUser|QFile::ExeUser);
  QTextStream out(&ftemp);
  QStringList args;
  bool rootUser = (QDir::homePath() == ctn_ROOT_HOME);

  if (!rootUser){
    args << "-d";
    args << "-t";
    args << "--noignorebutton";
  }
  else args << "-c";

  QFileInfo fi(commandList[0]);
  m_process->setWorkingDirectory(fi.absolutePath());

  if (commandUsed == ectn_TGZ2LZM) {
    foreach(QString line, commandList){
      QFileInfo fi(line);
      QString newFile = fi.fileName();
      newFile = newFile.replace(".tgz", ".lzm");
      out << "tgz2lzm " << line << " " << newFile << "\n";
    }
  }
  else if (commandUsed == ectn_MAKELZM) {
    foreach(QString line, commandList)
      out << "make-lzm " << line << "\n";
  }

  out.flush();
  ftemp.close();
  args << ctn_TEMP_ACTIONS_FILE;

  if (!rootUser) m_process->start("kdesu", args);
  else m_process->start("/bin/sh", args);
}

void UNIXCommand::transformRPMinTGZ(const QStringList& commandList){
  QFile ftemp(ctn_TEMP_ACTIONS_FILE);
  ftemp.open(QIODevice::ReadWrite|QIODevice::Text);
  ftemp.setPermissions(QFile::ReadOwner|QFile::WriteOwner|QFile::ExeOwner|
      QFile::ReadUser|QFile::WriteUser|QFile::ExeUser);
  QTextStream out(&ftemp);
  QStringList args("-c");

  QFileInfo fi(commandList[0]);
  m_process->setWorkingDirectory(fi.absolutePath());

  foreach(QString line, commandList)
    out << "rpm2tgz " << line << "\n";

  out.flush();
  ftemp.close();
  args << ctn_TEMP_ACTIONS_FILE;

  m_process->start("/bin/sh", args) ;
}

void UNIXCommand::processReadyReadStandardOutput(){
  m_readAllStandardOutput = m_process->readAllStandardOutput();
}

void UNIXCommand::processReadyReadStandardError(){
  m_readAllStandardError = m_process->readAllStandardError();
  m_errorString = m_process->errorString();
}

QString UNIXCommand::readAllStandardOutput(){
  return m_readAllStandardOutput;
}

QString UNIXCommand::readAllStandardError(){
  return m_readAllStandardError;
}

QString UNIXCommand::errorString(){
  return m_errorString;
}

UNIXCommand::UNIXCommand(QObject *parent): QObject(){
  m_process = new QProcess(parent);

  QObject::connect(m_process, SIGNAL( started() ), this,
                   SIGNAL( started() ));
  QObject::connect(this, SIGNAL( started() ), this,
                   SLOT( processReadyReadStandardOutput() ));

  QObject::connect(m_process, SIGNAL( readyReadStandardOutput() ), this,
                   SIGNAL( readyReadStandardOutput() ));
  QObject::connect(this, SIGNAL( readyReadStandardOutput() ), this,
                   SLOT( processReadyReadStandardOutput() ));

  QObject::connect(m_process, SIGNAL( finished ( int, QProcess::ExitStatus )), this,
                   SIGNAL( finished ( int, QProcess::ExitStatus )) );
  QObject::connect(this, SIGNAL( finished ( int, QProcess::ExitStatus )), this,
                   SLOT( processReadyReadStandardOutput() ));

  QObject::connect(m_process, SIGNAL( readyReadStandardError() ), this,
                   SIGNAL( readyReadStandardError() ));
  QObject::connect(this, SIGNAL( readyReadStandardError() ), this,
                   SLOT( processReadyReadStandardError() ));
}
