/*
 * lirccd v0.8 - Middle layer daemon between lircd and its clients
 * Copyright (C) 2003  Fredrik Tolf (fredrik@dolda2000.cjb.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <stdarg.h>

#include "command.h"
#include "commands.h"

struct stackent *stack = NULL;
struct command *commands = NULL;
struct instruction **labels = NULL;
struct symbol *symbols = NULL;
struct function *functions = NULL;
struct callstackent callstack[STACK_DEPTH];
int curcall;
int curscope = 0;
static int curlabel = 0;

void printcmd(struct instruction *ins)
{
    int i, labl;
    struct command *cur;
    
    for(; ins != NULL; ins = ins->next)
    {
	labl = -1;
	for(i = 0; i < curlabel; i++)
	{
	    if(labels[i] == ins)
	    {
		labl = i;
		break;
	    }
	}
	if(labl < 0)
	    printf("\t");
	else
	    printf("%i:\t", labl);
	if(ins->func == NULL)
	{
	    if(ins->pushval == NULL)
	    {
		printf("NOOP\n");
	    } else {
		switch(ins->pushval->type)
		{
		case VAL_NULL:
		    printf("PUSHNULL\n");
		    break;
		case VAL_INT:
		    printf("PUSHINT %i\n", ins->pushval->val.num);
		    break;
		case VAL_FLOAT:
		    printf("PUSHFL %f\n", ins->pushval->val.flnum);
		    break;
		case VAL_STRING:
		    printf("PUSHSTR %s\n", ins->pushval->val.string.buf);
		    break;
		case VAL_ARRAYSTART:
		    printf("[\n");
		    break;
		default:
		    printf("BUG!\n");
		    break;
		}
	    }
	} else {
	    for(cur = commands; cur != NULL; cur = cur->next)
	    {
		if(cur->func == ins->func)
		{
		    printf("%s\n", cur->name);
		    break;
		}
	    }
	    if(cur == NULL)
		printf("Function %p not found\n", ins->func);
	}
    }
}

struct value *pop(void)
{
    struct stackent *ent;
    struct value *val;
    
    if(stack == NULL)
	return(NULL);
    ent = stack;
    stack = ent->next;
    val = ent->val;
    free(ent);
    return(val);
}

struct value* push(struct value *val)
{
    struct stackent *ent;
    
    val->ref++;
    ent = (struct stackent *)malloc(sizeof(struct stackent));
    ent->val = val;
    ent->next = stack;
    stack = ent;
    return(val);
}

int label(struct instruction *ins)
{
    labels = (struct instruction **)realloc((void *)labels, sizeof(struct instruction *) * (curlabel + 1));
    labels[curlabel] = ins;
    return(curlabel++);
}

void beginscope(void)
{
    curscope++;
}

void endscope(void)
{
    struct symbol *cur, *next;
    
    curscope--;
    for(cur = symbols; cur != NULL; cur = next)
    {
	next = cur->next;
	if(cur->scope > curscope)
	    freesym(cur);
    }
}

void freefunc(struct function *func)
{
    freecode(func->code);
    free(func->name);
    free(func);
}

struct instruction *getfunc(unsigned char *name)
{
    struct function *cur;
    
    for(cur = functions; cur != NULL; cur = cur->next)
    {
	if(!strcmp(name, cur->name))
	    return(cur->code);
    }
    return(NULL);
}

struct function *newfunc(unsigned char *name, struct instruction *code)
{
    struct function *new;
    
    new = (struct function *)malloc(sizeof(struct function));
    new->next = functions;
    new->name = strcpy((unsigned char *)malloc(strlen(name) + 1), name);
    new->code = code;
    functions = new;
    return(new);
}

void freesym(struct symbol *sym)
{
    struct symbol *cur;
    
    if(sym == symbols)
    {
	symbols = sym->next;
    } else {
	for(cur = symbols; cur->next != sym; cur = cur->next);
	cur->next = sym->next;
    }
    relval(sym->val);
    free(sym->name);
    free(sym);
}

struct value *getsym(unsigned char *name)
{
    struct symbol *cur;
    
    for(cur = symbols; cur != NULL; cur = cur->next)
    {
	if(!strcmp(name, cur->name))
	{
	    cur->val->ref++;
	    return(cur->val);
	}
    }
    return(NULL);
}

struct symbol *newsym(unsigned char *name, struct value *val)
{
    struct symbol *new;
    
    new = (struct symbol *)malloc(sizeof(struct symbol));
    new->next = symbols;
    new->name = strcpy((unsigned char *)malloc(strlen(name) + 1), name);
    new->val = val;
    new->scope = curscope;
    val->ref++;
    symbols = new;
    return(new);
}

struct value *newval(void)
{
    struct value *new;
    
    new = (struct value *)malloc(sizeof(struct value));
    new->ref = 1;
    new->type = VAL_NULL;
    return(new);
}

void makenull(struct value *val)
{
    int i;
    
    switch(val->type)
    {
    case VAL_STRING:
	free(val->val.string.buf);
	break;
    case VAL_ARRAY:
	for(i = 0; i < val->val.array.len; i++)
	    relval(val->val.array.values[i]);
	free(val->val.array.values);
	break;
    }
    val->type = VAL_NULL;
}

void relval(struct value *val)
{
    if(val == NULL)
	return;
    if(!(--val->ref))
    {
	makenull(val);
	free(val);
    }
}

struct value *valmakeint(struct value *val, int num)
{
    makenull(val);
    val->type = VAL_INT;
    val->val.num = num;
    return(val);
}

struct value *valmakefloat(struct value *val, double num)
{
    makenull(val);
    val->type = VAL_FLOAT;
    val->val.flnum = num;
    return(val);
}

struct value *valprintf(struct value *val, unsigned char *format, ...)
{
    va_list args;
    int ret;
    
    makenull(val);
    val->type = VAL_STRING;
    val->val.string.buf = (unsigned char *)malloc(val->val.string.buflen = 10);
    va_start(args, format);
    if((ret = vsnprintf(val->val.string.buf, val->val.string.buflen, format, args)) >= val->val.string.buflen)
    {
	val->val.string.buf = (unsigned char *)realloc((void *)val->val.string.buf, val->val.string.buflen = (ret + 1));
	vsnprintf(val->val.string.buf, val->val.string.buflen, format, args);
    }
    va_end(args);
    val->val.string.len = ret;
    return(val);
}

struct value *valmakestr(struct value *val, unsigned char *str)
{
    makenull(val);
    val->type = VAL_STRING;
    val->val.string.buf = strcpy((unsigned char *)malloc(val->val.string.buflen = (strlen(str) + 1)), str);
    val->val.string.len = strlen(val->val.string.buf);
    return(val);
}

struct value *valmakestr2(struct value *val, unsigned char *str, int len)
{
    makenull(val);
    val->type = VAL_STRING;
    val->val.string.buf = memcpy((unsigned char *)malloc(val->val.string.buflen = (len + 1)), str, len);
    val->val.string.buf[len] = 0;
    val->val.string.len = len;
    return(val);
}

struct value *valcatstr(struct value *val, unsigned char *str, int len)
{
    int olen;
    
    val->type = VAL_STRING;
    olen = val->val.string.len;
    val->val.string.buf = (unsigned char *)realloc((void *)val->val.string.buf, val->val.string.buflen = ((val->val.string.len += len) + 1));
    memcpy(val->val.string.buf + olen, str, len);
    val->val.string.buf[val->val.string.len] = 0;
    return(val);
}

struct value *valmakenullarray(struct value *val)
{
    makenull(val);
    val->type = VAL_ARRAY;
    val->val.array.len = 0;
    val->val.array.values = (struct value **)malloc(sizeof(struct value *));
    return(val);
}

struct value *valcatarray(struct value *arr, struct value *val)
{
    arr->val.array.values = (struct value **)realloc((void *)arr->val.array.values, sizeof(struct value *) * (arr->val.array.len + 1));
    arr->val.array.values[arr->val.array.len] = val;
    arr->val.array.len++;
    val->ref++;
    return(arr);
}

int valbool(struct value *val)
{
    switch(val->type)
    {
    case VAL_NULL:
	return(0);
    case VAL_INT:
	return(!!val->val.num);
    case VAL_FLOAT:
	return(!!val->val.flnum);
    case VAL_STRING:
	return(!!val->val.string.len);
    case VAL_ARRAY:
	return(!!val->val.array.len);
    }
    return(0);
}

struct value *storeval(struct value *dest, struct value *src)
{
    int i;
    
    makenull(dest);
    switch(dest->type = src->type)
    {
    case VAL_INT:
	dest->val.num = src->val.num;
	break;
    case VAL_FLOAT:
	dest->val.flnum = src->val.flnum;
	break;
    case VAL_STRING:
	dest->val.string.buf = memcpy(malloc(dest->val.string.buflen = src->val.string.buflen), src->val.string.buf, (dest->val.string.len = src->val.string.len) + 1);
	break;
    case VAL_ARRAY:
	dest->val.array.values = (struct value **)malloc((dest->val.array.len = src->val.array.len) * sizeof(struct value *));
	for(i = 0; i < dest->val.array.len; i++)
	{
	    dest->val.array.values[i] = src->val.array.values[i];
	    dest->val.array.values[i]->ref++;
	}
	break;
    }
    return(dest);
}

struct value *dupval(struct value *val)
{
    struct value *new, *cur;
    
    new = newval();
    storeval(new, val);
    return(new);
}

void freecode(struct instruction *code)
{
    struct instruction *next;
    
    while(code != NULL)
    {
	next = code->next;
	if(code->pushval != NULL)
	    relval(code->pushval);
	free(code);
	code = next;
    }
}

struct instruction *insconcat(struct instruction *ins, ...)
{
    va_list insl;
    struct instruction *nins, *lins;
    
    va_start(insl, ins);
    lins = ins;
    while((nins = va_arg(insl, struct instruction *)) != NULL)
    {
	for(; lins->next != NULL; lins = lins->next);
	lins->next = nins;
	lins = nins;
    }
    va_end(insl);
    return(ins);
}

struct instruction *inspush(struct value *val)
{
    struct instruction *new;
    
    new = (struct instruction *)malloc(sizeof(struct instruction));
    new->next = NULL;
    new->func = NULL;
    new->pushval = dupval(val);
    return(new);
}

struct instruction *insmake(unsigned char *cmd)
{
    struct instruction *new;
    
    new = (struct instruction *)malloc(sizeof(struct instruction));
    new->next = NULL;
    new->func = findcommand(cmd);
    new->pushval = NULL;
    return(new);
}

void (*findcommand(unsigned char *name))(void)
{
    struct command *cur;
    
    for(cur = commands; cur != NULL; cur = cur->next)
    {
	if(!strcmp(name, cur->name))
	    return(cur->func);
    }
    return(NULL);
}

void addcommand(unsigned char *name, void (*func)(void))
{
    struct command *new;
    
    new = (struct command *)malloc(sizeof(struct command));
    new->next = commands;
    new->name = strcpy((unsigned char *)malloc(strlen(name) + 1), name);
    new->func = func;
    commands = new;
}

void checkstack(void)
{
    if(stack != NULL)
    {
	fprintf(stderr, "warning: non-empty stack\n");
	while(stack != NULL)
	    relval(pop());
    }
}

struct value *runfunc(unsigned char *name, int release, ...)
{
    struct instruction *code;
    struct value *v;
    va_list args;
    
    if((code = getfunc(name)) == NULL)
	return(NULL);
    va_start(args, release);
    for(v = va_arg(args, struct value *); v != NULL; v = va_arg(args, struct value *))
    {
	push(v);
	if(release)
	    relval(v);
    }
    va_end(args);
    run(code);
    v = pop();
    checkstack();
    return(v);
}

void run(struct instruction *code)
{
    struct value *v;
    
    beginscope();
    callstack[curcall = 0].ins = code;
    while(curcall >= 0)
    {
	while(callstack[curcall].ins != NULL)
	{
	    callstack[curcall].ins = (code = callstack[curcall].ins)->next;
	    if(code->func == NULL)
	    {
		if(code->pushval != NULL)
		{
		    v = dupval(code->pushval);
		    push(v);
		    relval(v);
		}
	    } else {
		code->func();
	    }
	}
	if(--curcall >= 0)
	{
	    while(curscope > callstack[curcall].scope)
		endscope();
	}
    }
    endscope();
    if(curscope > 0)
    {
	fprintf(stderr, "warning: curscope does not equal zero\n");
	while(curscope > 0)
	    endscope();
    }
}

void initcommand(void)
{
    initcommands();
}
