/*
 * lirccd v0.8 - Middle layer daemon between lircd and its clients
 * Copyright (C) 2003  Fredrik Tolf (fredrik@dolda2000.cjb.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>

#include "command.h"

#define LOG(s) (fprintf(stderr, "%s: %s\n", __FUNCTION__, (s)))

#define POP_NUM(var, num) \
( { int __i; \
    for(__i = 0; __i < (num); __i++) \
    { \
        if(((var)[((num) - 1) - __i] = pop()) == NULL) \
        { \
            LOG("not enough values"); \
            return; \
        } \
    } \
  } \
)

static void cmd_drop(void)
{
    if(pop() == NULL)
	LOG("not enough values");
}

static void cmd_swap(void)
{
    struct value *v[2];
    
    POP_NUM(v, 2);
    push(v[1]);
    push(v[0]);
    relval(v[0]);
    relval(v[1]);
}

static void cmd_dup(void)
{
    struct value *v, *v2;
    
    if((v = pop()) == NULL)
    {
	LOG("not enough values");
	return;
    }
    v2 = dupval(v);
    push(v);
    push(v2);
    relval(v);
    relval(v2);
}

static void cmd_clone(void)
{
    struct value *v;
    
    if((v = pop()) == NULL)
    {
	LOG("not enough values");
	return;
    }
    push(v);
    push(v);
    relval(v);
}

static void cmd_load(void)
{
    struct value *v, *v2;
    
    if((v = pop()) == NULL)
    {
	LOG("not enough values");
	return;
    }
    if(v->type != VAL_STRING)
    {
	LOG("cannot load from non-string value");
	return;
    }
    if((v2 = getsym(v->val.string.buf)) == NULL)
	newsym(v->val.string.buf, v2 = newval());
    push(v2);
    relval(v2);
    relval(v);
}

static void cmd_store(void)
{
    struct value *v[2];
    
    POP_NUM(v, 2);
    storeval(v[0], v[1]);
    push(v[0]);
    relval(v[0]);
    relval(v[1]);
}

static void cmd_makevar(void)
{
    struct value *v, *v2;
    
    if((v = pop()) == NULL)
    {
	LOG("not enough values");
	return;
    }
    if(v->type != VAL_STRING)
    {
	LOG("cannot make non-string variable name");
	return;
    }
    newsym(v->val.string.buf, v2 = newval());
    push(v2);
    relval(v2);
    relval(v);
}

static void cmd_jump(void)
{
    struct value *v;
    
    if((v = pop()) == NULL)
    {
	LOG("not enough values");
	return;
    }
    if(v->type != VAL_INT)
    {
	LOG("cannot jump to non-int label");
	return;
    }
    callstack[curcall].ins = labels[v->val.num];
    relval(v);
}

static void cmd_cjump(void)
{
    struct value *v, *cond;
    
    if((cond = pop()) == NULL)
    {
	LOG("not enough values");
	return;
    }
    if((v = pop()) == NULL)
    {
	LOG("not enough values");
	relval(cond);
	return;
    }
    if(v->type != VAL_INT)
    {
	LOG("cannot jump to non-int label");
	relval(v);
	relval(cond);
	return;
    }
    if(valbool(cond))
	callstack[curcall].ins = labels[v->val.num];
    relval(v);
    relval(cond);
}

static void cmd_call(void)
{
    struct value *v;
    struct instruction *code;
    
    if((v = pop()) == NULL)
    {
	LOG("not enough values");
	return;
    }
    if(v->type != VAL_STRING)
    {
	relval(v);
	LOG("cannot jump to non-string function");
	return;
    }
    if((code = getfunc(v->val.string.buf)) == NULL)
    {
	relval(v);
	LOG("call to non-existant function");
	relval(push(newval()));
	return;
    }
    relval(v);
    callstack[curcall].scope = curscope;
    callstack[++curcall].ins = code;
}

static void cmd_return(void)
{
    if(--curcall >= 0)
    {
	while(curscope > callstack[curcall].scope)
	    endscope();
    }
}

static void cmd_array(void)
{
    struct value *v, *v2;
    
    valmakenullarray(v2 = newval());
    while(1)
    {
	if((v = pop()) == NULL)
	{
	    LOG("no array start value");
	    return;
	}
	if(v->type == VAL_ARRAYSTART)
	    break;
	valcatarray(v2, v);
	relval(v);
    }
    push(v2);
    relval(v2);
}

static void cmd_index(void)
{
    struct value *v[2];
    struct value *new;
    unsigned char buf[2];
    
    POP_NUM(v, 2);
    if(v[1]->type != VAL_INT)
    {
	LOG("cannot look up non-int index");
	return;
    }
    new = newval();
    if(v[0]->type == VAL_ARRAY)
    {
	if((v[1]->val.num < 0) || (v[1]->val.num > (v[0]->val.array.len - 1)))
	{
	    LOG("warning: index out of bounds");
	} else {
	    relval(new);
	    new = v[0]->val.array.values[v[1]->val.num];
	    new->ref++;
	}
    }
    if(v[0]->type == VAL_STRING)
    {
	if((v[1]->val.num < 0) || (v[1]->val.num > (v[0]->val.string.len - 1)))
	{
	    LOG("warning: index out of bounds");
	} else {
	    buf[0] = v[0]->val.string.buf[v[1]->val.num];
	    buf[1] = 0;
	    valmakestr(new, buf);
	}
    }
    push(new);
    relval(new);
    relval(v[0]);
    relval(v[1]);
}

static void cmd_makeint(void)
{
    struct value *v, *v2;
    
    if((v = pop()) == NULL)
    {
	LOG("not enough values");
    }
    v2 = newval();
    v2->type = VAL_INT;
    switch(v->type)
    {
    case VAL_INT:
	v2->val.num = v->val.num;
	break;
    case VAL_FLOAT:
	v2->val.num = (int)v->val.flnum;
	break;
    case VAL_STRING:
	v2->val.num = atoi(v->val.string.buf);
	break;
    case VAL_ARRAY:
	v2->val.num = v->val.array.len;
	break;
    }
    push(v2);
    relval(v);
    relval(v2);
}

static void cmd_makefloat(void)
{
    struct value *v, *v2;
    
    if((v = pop()) == NULL)
    {
	LOG("not enough values");
    }
    v2 = newval();
    v2->type = VAL_FLOAT;
    switch(v->type)
    {
    case VAL_INT:
	v2->val.flnum = (double)v->val.num;
	break;
    case VAL_FLOAT:
	v2->val.flnum = v->val.flnum;
	break;
    case VAL_STRING:
	v2->val.flnum = atof(v->val.string.buf);
	break;
    case VAL_ARRAY:
	v2->val.flnum = (double)v->val.array.len;
	break;
    }
    push(v2);
    relval(v);
    relval(v2);
}

static void cmd_makestring(void)
{
    struct value *v, *v2;
    int ret;
    
    if((v = pop()) == NULL)
    {
	LOG("not enough values");
    }
    v2 = newval();
    switch(v->type)
    {
    case VAL_NULL:
	valmakestr(v2, "(null)");
	break;
    case VAL_INT:
	valprintf(v2, "%i", v->val.num);
	break;
    case VAL_FLOAT:
	valprintf(v2, "%d", v->val.flnum);
	break;
    case VAL_STRING:
	v2 = dupval(v);
	break;
    case VAL_ARRAY:
	break;
    }
    push(v2);
    relval(v);
    relval(v2);
}

static void cmd_makearray(void)
{
    struct value *v, *v2;
    int ret;
    
    if((v = pop()) == NULL)
    {
	LOG("not enough values");
    }
    v2 = newval();
    v2->type = VAL_ARRAY;
    v2->val.array.values = (struct value **)malloc(sizeof(struct value *));
    v2->val.array.values[0] = v;
    v2->val.array.len = 1;
    v->ref++;
    push(v2);
    relval(v2);
    relval(v);
}

static void cmd_inc(void)
{
    struct value *v;
    
    if((v = pop()) == NULL)
    {
	LOG("not enough values");
    }
    switch(v->type)
    {
    case VAL_INT:
	v->val.num++;
	break;
    case VAL_FLOAT:
	v->val.flnum++;
	break;
    }
    push(v);
    relval(v);
}

static void cmd_dec(void)
{
    struct value *v;
    
    if((v = pop()) == NULL)
    {
	LOG("not enough values");
    }
    switch(v->type)
    {
    case VAL_INT:
	v->val.num--;
	break;
    case VAL_FLOAT:
	v->val.flnum--;
	break;
    }
    push(v);
    relval(v);
}

static void cmd_add(void)
{
    int i;
    struct value *v[2];
    struct value *new;
    int olen;
    
    POP_NUM(v, 2);
    if((v[0]->type == VAL_NULL) || (v[1]->type == VAL_NULL))
    {
	LOG("null value");
	return;
    }
    new = newval();
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_INT))
    {
	new->type = VAL_INT;
	new->val.num = v[0]->val.num + v[1]->val.num;
    }
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_INT))
    {
	new->type = VAL_FLOAT;
	new->val.flnum = v[0]->val.flnum + (double)v[1]->val.num;
    }
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_FLOAT))
    {
	new->type = VAL_FLOAT;
	new->val.flnum = (double)v[0]->val.num + v[1]->val.flnum;
    }
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_FLOAT))
    {
	new->type = VAL_FLOAT;
	new->val.flnum = v[0]->val.flnum + v[1]->val.flnum;
    }
    if((v[0]->type == VAL_STRING) && (v[1]->type == VAL_STRING))
    {
	new->type = VAL_STRING;
	new->val.string.buf = (unsigned char *)malloc(new->val.string.buflen = ((new->val.string.len = (v[0]->val.string.len + v[1]->val.string.len)) + 1));
	memcpy(new->val.string.buf, v[0]->val.string.buf, v[0]->val.string.len);
	memcpy(new->val.string.buf + v[0]->val.string.len, v[1]->val.string.buf, v[1]->val.string.len);
	new->val.string.buf[new->val.string.len] = 0;
    }
    if(v[0]->type == VAL_ARRAY)
    {
	if(v[1]->type == VAL_ARRAY)
	{
	    storeval(new, v[0]);
	    olen = new->val.array.len;
	    new->val.array.values = (struct value **)realloc((void *)new->val.array.values, sizeof(struct value *) * (new->val.array.len += v[1]->val.array.len));
	    for(i = 0; i < v[1]->val.array.len; i++)
	    {
		new->val.array.values[i + olen] = v[1]->val.array.values[i];
		v[1]->val.array.values[i]->ref++;
	    }
	} else {
	    storeval(new, v[0]);
	    valcatarray(new, v[1]);
	}
    }
    if(new->type == VAL_NULL)
	LOG("type mismatch");
    relval(v[0]);
    relval(v[1]);
    push(new);
    relval(new);
}

static void cmd_sub(void)
{
    struct value *v[2];
    struct value *new;
    int i;
    
    POP_NUM(v, 2);
    if((v[0]->type == VAL_NULL) || (v[1]->type == VAL_NULL))
    {
       LOG("null value");
       return;
    }
    new = newval();
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_INT))
    {
	new->type = VAL_INT;
	new->val.num = v[0]->val.num - v[1]->val.num;
    }
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_INT))
    {
	new->type = VAL_FLOAT;
	new->val.flnum = v[0]->val.flnum - (double)v[1]->val.num;
    }
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_FLOAT))
    {
	new->type = VAL_FLOAT;
	new->val.flnum = (double)v[0]->val.num - v[1]->val.flnum;
    }
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_FLOAT))
    {
	new->type = VAL_FLOAT;
	new->val.flnum = v[0]->val.flnum - v[1]->val.flnum;
    }
    if(v[0]->type == VAL_ARRAY)
    {
	if(v[1]->type == VAL_INT)
	{
	    storeval(new, v[0]);
	    if((v[1]->val.num < 0) || (v[1]->val.num >= v[0]->val.array.len))
	    {
		LOG("warning: subtraction index out of bounds");
	    } else {
		new->val.array.len--;
		relval(new->val.array.values[v[1]->val.num]);
		for(i = v[1]->val.num; i < new->val.array.len; i++)
		    new->val.array.values[i] = new->val.array.values[i + 1];
		new->val.array.values = (struct value **)realloc((void *)new->val.array.values, sizeof(struct value *) * (new->val.array.len));
	    }
	}
    }
    if(new->type == VAL_NULL)
	LOG("type mismatch");
    relval(v[0]);
    relval(v[1]);
    push(new);
    relval(new);
}

static void cmd_mul(void)
{
    struct value *v[2];
    struct value *new;
    
    POP_NUM(v, 2);
    if((v[0]->type == VAL_NULL) || (v[1]->type == VAL_NULL))
    {
       LOG("null value");
       return;
    }
    new = newval();
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_INT))
    {
	new->type = VAL_INT;
	new->val.num = v[0]->val.num * v[1]->val.num;
    }
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_INT))
    {
	new->type = VAL_FLOAT;
	new->val.flnum = v[0]->val.flnum * (double)v[1]->val.num;
    }
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_FLOAT))
    {
	new->type = VAL_FLOAT;
	new->val.flnum = (double)v[0]->val.num * v[1]->val.flnum;
    }
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_FLOAT))
    {
	new->type = VAL_FLOAT;
	new->val.flnum = v[0]->val.flnum * v[1]->val.flnum;
    }
    if(new->type == VAL_NULL)
	LOG("type mismatch");
    relval(v[0]);
    relval(v[1]);
    push(new);
    relval(new);
}

static void cmd_div(void)
{
    struct value *v[2];
    struct value *new;
    
    POP_NUM(v, 2);
    if((v[0]->type == VAL_NULL) || (v[1]->type == VAL_NULL))
    {
       LOG("null value");
       return;
    }
    new = newval();
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_INT))
    {
	new->type = VAL_INT;
	new->val.num = v[0]->val.num / v[1]->val.num;
    }
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_INT))
    {
	new->type = VAL_FLOAT;
	new->val.flnum = v[0]->val.flnum / (double)v[1]->val.num;
    }
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_FLOAT))
    {
	new->type = VAL_FLOAT;
	new->val.flnum = (double)v[0]->val.num / v[1]->val.flnum;
    }
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_FLOAT))
    {
	new->type = VAL_FLOAT;
	new->val.flnum = v[0]->val.flnum / v[1]->val.flnum;
    }
    if(new->type == VAL_NULL)
	LOG("type mismatch");
    relval(v[0]);
    relval(v[1]);
    push(new);
    relval(new);
}

static void cmd_not(void)
{
    struct value *v, *v2;
    
    if((v = pop()) == NULL)
    {
	LOG("not enough values");
	return;
    }
    v2 = valmakeint(newval(), !valbool(v));
    push(v2);
    relval(v);
    relval(v2);
}

static void cmd_eq(void)
{
    struct value *v[2], *new;
    
    POP_NUM(v, 2);
    new = newval();
    new->type = VAL_INT;
    new->val.num = 0;
    if((v[0]->type == VAL_NULL) && (v[1]->type == VAL_NULL))
	new->val.num = 0;
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_INT))
	new->val.num = (v[0]->val.num == v[1]->val.num);
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_FLOAT))
	new->val.num = (v[0]->val.flnum == v[1]->val.flnum);
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_FLOAT))
	new->val.num = ((double)v[0]->val.num == v[1]->val.flnum);
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_INT))
	new->val.num = (v[0]->val.flnum == (double)v[1]->val.num);
    if((v[0]->type == VAL_STRING) && (v[1]->type == VAL_STRING))
    {
	if(v[0]->val.string.len == v[1]->val.string.len)
	    new->val.num = !memcmp(v[0]->val.string.buf, v[1]->val.string.buf, v[0]->val.string.len);
    }
    push(new);
    relval(new);
    relval(v[0]);
    relval(v[1]);
}

static void cmd_ne(void)
{
    struct value *v[2], *new;
    
    POP_NUM(v, 2);
    new = newval();
    new->type = VAL_INT;
    new->val.num = 0;
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_INT))
	new->val.num = (v[0]->val.num != v[1]->val.num);
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_FLOAT))
	new->val.num = (v[0]->val.flnum != v[1]->val.flnum);
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_FLOAT))
	new->val.num = ((double)v[0]->val.num != v[1]->val.flnum);
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_INT))
	new->val.num = (v[0]->val.flnum != (double)v[1]->val.num);
    if((v[0]->type == VAL_STRING) && (v[1]->type == VAL_STRING))
    {
	if(v[0]->val.string.len == v[1]->val.string.len)
	    new->val.num = !!memcmp(v[0]->val.string.buf, v[1]->val.string.buf, v[0]->val.string.len);
	else
	    new->val.num = 1;
    }
    push(new);
    relval(new);
    relval(v[0]);
    relval(v[1]);
}

static void cmd_gt(void)
{
    struct value *v[2], *new;
    
    POP_NUM(v, 2);
    new = newval();
    new->type = VAL_INT;
    new->val.num = 0;
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_INT))
	new->val.num = (v[0]->val.num > v[1]->val.num);
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_FLOAT))
	new->val.num = (v[0]->val.flnum > v[1]->val.flnum);
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_FLOAT))
	new->val.num = ((double)v[0]->val.num > v[1]->val.flnum);
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_INT))
	new->val.num = (v[0]->val.flnum > (double)v[1]->val.num);
    if((v[0]->type == VAL_STRING) && (v[1]->type == VAL_STRING))
    {
	if(v[0]->val.string.len < v[1]->val.string.len)
	    new->val.num = 0;
	if(v[0]->val.string.len == v[1]->val.string.len)
	    new->val.num = (memcmp(v[0]->val.string.buf, v[1]->val.string.buf, v[0]->val.string.len) > 0);
	if(v[0]->val.string.len > v[1]->val.string.len)
	    new->val.num = 1;
    }
    push(new);
    relval(new);
    relval(v[0]);
    relval(v[1]);
}

static void cmd_lt(void)
{
    struct value *v[2], *new;
    
    POP_NUM(v, 2);
    new = newval();
    new->type = VAL_INT;
    new->val.num = 0;
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_INT))
	new->val.num = (v[0]->val.num < v[1]->val.num);
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_FLOAT))
	new->val.num = (v[0]->val.flnum < v[1]->val.flnum);
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_FLOAT))
	new->val.num = ((double)v[0]->val.num < v[1]->val.flnum);
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_INT))
	new->val.num = (v[0]->val.flnum < (double)v[1]->val.num);
    if((v[0]->type == VAL_STRING) && (v[1]->type == VAL_STRING))
    {
	if(v[0]->val.string.len < v[1]->val.string.len)
	    new->val.num = 1;
	if(v[0]->val.string.len == v[1]->val.string.len)
	    new->val.num = (memcmp(v[0]->val.string.buf, v[1]->val.string.buf, v[0]->val.string.len) < 0);
	if(v[0]->val.string.len > v[1]->val.string.len)
	    new->val.num = 0;
    }
    push(new);
    relval(new);
    relval(v[0]);
    relval(v[1]);
}

static void cmd_ge(void)
{
    struct value *v[2], *new;
    
    POP_NUM(v, 2);
    new = newval();
    new->type = VAL_INT;
    new->val.num = 0;
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_INT))
	new->val.num = (v[0]->val.num >= v[1]->val.num);
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_FLOAT))
	new->val.num = (v[0]->val.flnum >= v[1]->val.flnum);
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_FLOAT))
	new->val.num = ((double)v[0]->val.num >= v[1]->val.flnum);
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_INT))
	new->val.num = (v[0]->val.flnum >= (double)v[1]->val.num);
    if((v[0]->type == VAL_STRING) && (v[1]->type == VAL_STRING))
    {
	if(v[0]->val.string.len < v[1]->val.string.len)
	    new->val.num = 0;
	if(v[0]->val.string.len == v[1]->val.string.len)
	    new->val.num = (memcmp(v[0]->val.string.buf, v[1]->val.string.buf, v[0]->val.string.len) >= 0);
	if(v[0]->val.string.len > v[1]->val.string.len)
	    new->val.num = 1;
    }
    push(new);
    relval(new);
    relval(v[0]);
    relval(v[1]);
}

static void cmd_le(void)
{
    struct value *v[2], *new;
    
    POP_NUM(v, 2);
    new = newval();
    new->type = VAL_INT;
    new->val.num = 0;
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_INT))
	new->val.num = (v[0]->val.num <= v[1]->val.num);
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_FLOAT))
	new->val.num = (v[0]->val.flnum <= v[1]->val.flnum);
    if((v[0]->type == VAL_INT) && (v[1]->type == VAL_FLOAT))
	new->val.num = ((double)v[0]->val.num <= v[1]->val.flnum);
    if((v[0]->type == VAL_FLOAT) && (v[1]->type == VAL_INT))
	new->val.num = (v[0]->val.flnum <= (double)v[1]->val.num);
    if((v[0]->type == VAL_STRING) && (v[1]->type == VAL_STRING))
    {
	if(v[0]->val.string.len < v[1]->val.string.len)
	    new->val.num = 1;
	if(v[0]->val.string.len == v[1]->val.string.len)
	    new->val.num = (memcmp(v[0]->val.string.buf, v[1]->val.string.buf, v[0]->val.string.len) <= 0);
	if(v[0]->val.string.len > v[1]->val.string.len)
	    new->val.num = 0;
    }
    push(new);
    relval(new);
    relval(v[0]);
    relval(v[1]);
}

void initcommands(void)
{
    addcommand("DROP", cmd_drop);
    addcommand("SWAP", cmd_swap);
    addcommand("DUP", cmd_dup);
    addcommand("CLONE", cmd_clone);
    addcommand("LOAD", cmd_load);
    addcommand("STORE", cmd_store);
    addcommand("MAKEVAR", cmd_makevar);
    addcommand("BEGINSCOPE", beginscope);
    addcommand("ENDSCOPE", endscope);
    addcommand("JUMP", cmd_jump);
    addcommand("CJUMP", cmd_cjump);
    addcommand("CALL", cmd_call);
    addcommand("RETURN", cmd_return);
    addcommand("ARRAY", cmd_array);
    addcommand("INDEX", cmd_index);
    addcommand("MAKEINT", cmd_makeint);
    addcommand("MAKEFLOAT", cmd_makefloat);
    addcommand("MAKESTRING", cmd_makestring);
    addcommand("MAKEARRAY", cmd_makearray);
    addcommand("EQ", cmd_eq);
    addcommand("NE", cmd_ne);
    addcommand("GT", cmd_gt);
    addcommand("LT", cmd_lt);
    addcommand("GE", cmd_ge);
    addcommand("LE", cmd_le);
    addcommand("INC", cmd_inc);
    addcommand("DEC", cmd_dec);
    addcommand("ADD", cmd_add);
    addcommand("SUB", cmd_sub);
    addcommand("MUL", cmd_mul);
    addcommand("DIV", cmd_div);
    addcommand("NOT", cmd_not);
}
