/*
 * lirccd v0.8 - Middle layer daemon between lircd and its clients
 * Copyright (C) 2003  Fredrik Tolf (fredrik@dolda2000.cjb.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdlib.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <sys/poll.h>
#include <errno.h>

static int lircc_fd = -1;
static unsigned char databuf[1024], retbuf[1024];
static int datalen = 0;

int lircc_getstr_r(unsigned char *buf, int buflen, int block)
{
    int ret;
    unsigned char *p;
    struct pollfd pf;
    
    if(lircc_fd < 0)
	return(-2);
    while(1)
    {
	if((datalen > 0) && ((p = memchr(databuf, '\n', datalen)) != NULL))
	{
	    if(p - databuf >= buflen)
	    {
		ret = -1;
	    } else {
		memcpy(buf, databuf, p - databuf);
		buf[p - databuf] = 0;
		ret = p - databuf;
	    }
	    p++;
	    memmove(databuf, p, datalen - (p - databuf));
	    datalen -= p - databuf;
	    return(ret);
	}
	if(datalen == 1024)
	    datalen = 0;
	if(!block)
	{
	    pf.fd = lircc_fd;
	    pf.events = POLLIN;
	    pf.revents = 0;
	    if(poll(&pf, 1, 0) == 0)
		return(0);
	    if(pf.revents & (POLLERR | POLLHUP | POLLNVAL))
	    {
		close(lircc_fd);
		lircc_fd = -1;
		return(-2);
	    }
	}
	ret = read(lircc_fd, databuf + datalen, 1024 - datalen);
	if(((ret < 0) && (errno != EINTR)) || (ret == 0))
	{
	    close(lircc_fd);
	    lircc_fd = -1;
	    return(-2);
	}
	if(ret > 0)
	    datalen += ret;
    }
}

unsigned char *lircc_getstr(int block)
{
    if(lircc_getstr_r(retbuf, 1024, block) <= 0)
	return(NULL);
    else
	return(retbuf);
}

int lircc_init(unsigned char *appname, unsigned char *sockname)
{
    unsigned char *homedir;
    struct sockaddr_un name;
    int ret;
    
    if((sockname == NULL) && ((homedir = getenv("HOME")) == NULL))
	return(-1);
    name.sun_family = AF_UNIX;
    if(sockname == NULL)
	sprintf(name.sun_path, "%s/.lircc", homedir);
    else
	strcpy(name.sun_path, sockname);
    if((lircc_fd = socket(PF_UNIX, SOCK_STREAM, 0)) < 0)
	return(-1);
    if(connect(lircc_fd, (struct sockaddr *)&name, sizeof(name)) < 0)
    {
	close(lircc_fd);
	return(-1);
    }
    ret = sprintf(databuf, "%s\n", appname);
    if(send(lircc_fd, databuf, ret, MSG_NOSIGNAL) < ret)
    {
	close(lircc_fd);
	return(-1);
    }
    return(lircc_fd);
}

void lircc_cleanup(void)
{
    if(lircc_fd >= 0)
	close(lircc_fd);
}
