/*
 * lirccd v0.8 - Middle layer daemon between lircd and its clients
 * Copyright (C) 2003  Fredrik Tolf (fredrik@dolda2000.cjb.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <regex.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <signal.h>
#include <sys/wait.h>
#include <errno.h>
#include <getopt.h>

#include "lirccd.h"
#include "command.h"
#include "usercommands.h"

#define LIRC_DEVICE "/dev/lircd"

struct client *clients = NULL;
struct binding *bindings = NULL;
struct child *children = NULL;
struct alarm *alarms = NULL;
int running;
int endbinding;
static unsigned char sockname[128];
static unsigned char buf[1024];

void sighandler(int signum)
{
    switch(signum)
    {
    case SIGCHLD:
	/* Just making sure that there is a signal handler, so that
	   the select call gets interrupted */
	break;
    case SIGHUP:
    case SIGINT:
    case SIGTERM:
	exit(0);
    }
}

void freechild(struct child *child)
{
    struct child *cur;
    
    if(child == children)
    {
	children = child->next;
    } else {
	for(cur = children; cur->next != child; cur = cur->next);
	cur->next = child->next;
    }
    if(child->notify != NULL)
	free(child->notify);
    free(child->desc);
    free(child);
}

struct child *getchild(int id)
{
    struct child *cur;
    
    for(cur = children; cur != NULL; cur = cur->next)
    {
	if(cur->id == id)
	    return(cur);
    }
    return(NULL);
}

struct child *regchild(unsigned char *desc, int pid)
{
    static int curid = 0;
    struct child *new;
    
    new = (struct child *)malloc(sizeof(struct child));
    new->id = curid++;
    new->desc = strcpy(malloc(strlen(desc) + 1), desc);
    new->pid = pid;
    new->running = 1;
    new->stopped = 0;
    new->next = children;
    new->notify = NULL;
    children = new;
    return(new);
}

void checkchildren(void)
{
    struct child *cur;
    pid_t pid;
    
    if((pid = waitpid(-1, NULL, WNOHANG)) > 0)
    {
	for(cur = children; cur != NULL; cur = cur->next)
	{
	    if(cur->pid == pid)
	    {
		cur->stopped = 1;
		break;
	    }
	}
    }
    for(cur = children; cur != NULL; cur = cur->next)
    {
	if(cur->running && cur->stopped)
	{
	    cur->running = 0;
	    cur->stopped = 0;
	    if(cur->notify != NULL)
		relval(runfunc(cur->notify, 1, valmakeint(newval(), cur->id), NULL));
	}
    }
}

void freealarm(struct alarm *alarm)
{
    struct alarm *cur;
    
    if(alarm == alarms)
    {
	alarms = alarm->next;
    } else {
	for(cur = alarms; cur->next != alarm; cur = cur->next);
	cur->next = alarm->next;
    }
    free(alarm->func);
    free(alarm);
}

struct alarm *regalarm(time_t time, unsigned char *func)
{
    static int curid = 0;
    struct alarm *new;
    
    new = (struct alarm *)malloc(sizeof(struct alarm));
    new->id = curid++;
    new->time = time;
    new->func = strcpy(malloc(strlen(func) + 1), func);
    new->next = alarms;
    alarms = new;
    return(new);
}

void checkalarms(void)
{
    struct alarm *cur, *next;
    time_t curtime;
    
    curtime = time(NULL);
    for(cur = alarms; cur != NULL; cur = next)
    {
	next = cur->next;
	if(curtime >= cur->time)
	{
	    relval(runfunc(cur->func, 1, valmakeint(newval(), cur->id), NULL));
	    freealarm(cur);
	}
    }
}

struct client *newclient(int fd)
{
    struct client *new;
    
    new = (struct client *)malloc(sizeof(struct client));
    new->fd = fd;
    new->state = 0;
    new->appname = NULL;
    new->next = clients;
    clients = new;
    return(new);
}

void freeclient(struct client *client)
{
    struct client *cur;
    
    if(client == clients)
    {
	clients = client->next;
    } else {
	for(cur = clients; cur->next != client; cur = cur->next);
	cur->next = client->next;
    }
    close(client->fd);
    if(client->appname != NULL)
	free(client->appname);
    free(client);
}


void addbinding(struct binding **dest, struct binding *binding)
{
    struct binding *cur;
    
    if(*dest == NULL)
    {
	*dest = binding;
	return;
    }
    for(cur = *dest; cur->next != NULL; cur = cur->next);
    cur->next = binding;
}

struct binding *newbinding(unsigned char *remote, unsigned char *button, struct instruction *code)
{
    struct binding *new;
    int ret;
    
    new = (struct binding *)malloc(sizeof(struct binding));
    new->next = NULL;
    new->child = NULL;
    new->code = code;
    new->flags = 0;
    if(ret = regcomp(&new->remote, remote, REG_EXTENDED | REG_NOSUB))
    {
	regerror(ret, &new->remote, buf, 1024);
	fprintf(stderr, "%s\n", buf);
	exit(1);
    }
    if(ret = regcomp(&new->button, button, REG_EXTENDED | REG_NOSUB))
    {
	regerror(ret, &new->remote, buf, 1024);
	fprintf(stderr, "%s\n", buf);
	exit(1);
    }
    return(new);
}

void parsecodes()
{
    int i;
    int fd, serverfd, newfd;
    struct sockaddr_un name;
    FILE *stream;
    int repeat, len;
    int done, ret;
    unsigned char *p;
    unsigned char *remote, *button;
    struct binding *cur, **bindstack;
    struct value *v;
    int stacklevel;
    fd_set rfds;
    struct timeval tv;
    struct client *client, *nclient;
    
    if((fd = socket(PF_UNIX, SOCK_STREAM, 0)) < 0)
    {
	perror("socket");
	exit(1);
    }
    name.sun_family = AF_UNIX;
    strcpy(name.sun_path, LIRC_DEVICE);
    if(connect(fd, (struct sockaddr *)&name, sizeof(name)) < 0)
    {
	perror("could not connect to " LIRC_DEVICE);
	exit(1);
    }
    running = 1;
    if((stream = fdopen(fd, "r")) == NULL)
    {
	perror("fdopen");
	exit(1);
    }
    if((serverfd = socket(PF_UNIX, SOCK_STREAM, 0)) < 0)
    {
	perror("socket");
	exit(1);
    }
    name.sun_family = AF_UNIX;
    strcpy(name.sun_path, sockname);
    unlink(sockname);
    if(bind(serverfd, (struct sockaddr *)&name, sizeof(name)) < 0)
    {
	perror("could not bind server socket");
	exit(1);
    }
    if(listen(serverfd, 16) < 0)
    {
	perror("could not listen on server socket");
	exit(1);
    }
    remote = malloc(1024);
    button = malloc(1024);
    bindstack = (struct binding **)malloc(sizeof(struct binding *) * 128);
    stacklevel = 0;
    bindstack[0] = bindings;
    while(running)
    {
	FD_ZERO(&rfds);
	FD_SET(fd, &rfds);
	FD_SET(serverfd, &rfds);
	for(client = clients; client != NULL; client = client->next)
	{
	    FD_SET(client->fd, &rfds);
	}
	tv.tv_sec = 1;
	tv.tv_usec = 0;
	if(select(FD_SETSIZE, &rfds, NULL, NULL, &tv) < 0)
	{
	    if(errno == EINTR)
	    {
		FD_ZERO(&rfds);
	    } else {
		perror("select");
		break;
	    }
	}
	checkalarms();
	checkchildren();
	for(client = clients; client != NULL; client = nclient)
	{
	    nclient = client->next;
	    if(FD_ISSET(client->fd, &rfds))
	    {
		ret = read(client->fd, buf, 1024);
		if(ret <= 0)
		{
		    freeclient(client);
		    continue;
		} else if(client->state == 0) {
		    if((p = memchr(buf, '\n', ret)) != NULL)
		    {
			ret = p - buf;
			client->state = 1;
		    }
		    if(client->appname == NULL)
			len = 0;
		    else
			len = strlen(client->appname);
		    client->appname = (unsigned char *)realloc((void *)client->appname, len + ret + 1);
		    memcpy(client->appname + len, buf, ret);
		    client->appname[len + ret] = 0;
		}
	    }
	}
	if(FD_ISSET(serverfd, &rfds))
	{
	    if((newfd = accept(serverfd, NULL, 0)) >= 0)
	    {
		newclient(newfd);
	    }
	}
	if(FD_ISSET(fd, &rfds))
	{
	    if(fgets(buf, 1024, stream) == NULL)
		break;
	    sscanf(buf, "%*s %x %s %s", &repeat, button, remote);
	    done = 0;
	    v = valmakestr(newval(), button);
	    relval(storeval(getsym("button"), v));
	    relval(v);
	    v = valmakestr(newval(), remote);
	    relval(storeval(getsym("remote"), v));
	    relval(v);
	    v = newval();
	    v->type = VAL_INT;
	    v->val.num = repeat;
	    relval(storeval(getsym("repeat"), v));
	    relval(v);
	    for(i = stacklevel; i >= 0; i--)
	    {
		for(cur = bindstack[i]; cur != NULL; cur = cur->next)
		{
		    if((i == stacklevel) || (cur->flags & BIND_RECURSIVE))
		    {
			if(!regexec(&cur->remote, remote, 0, NULL, 0) && !regexec(&cur->button, button, 0, NULL, 0))
			{
			    endbinding = 0;
			    if(cur->code != NULL)
			    {
				run(cur->code);
				checkstack();
			    }
			    if(endbinding)
			    {
				stacklevel--;
				done = 1;
				break;
			    }
			    if(cur->child != NULL)
			    {
				bindstack[++stacklevel] = cur->child;
				done = 1;
				break;
			    }
			    if(cur->flags & BIND_STOP)
			    {
				done = 1;
				break;
			    }
			}
		    }
		}
		if(done)
		    break;
	    }
	}
    }
    free(remote);
    free(button);
    fclose(stream);
}

int main(int argc, unsigned char **argv)
{
    int o;
    int status;
    unsigned char *homedir;
    int nofork;
    FILE *config;
    struct value *v;
    struct instruction *init;
    
    if((homedir = getenv("HOME")) == NULL)
    {
	fprintf(stderr, "HOME variable not set\n");
	exit(1);
    }
    if(getuid() && !geteuid())
    {
	fprintf(stderr, "lirccd should never be run as suid root\n");
	exit(1);
    }
    sprintf(sockname, "%s/.lircc", homedir);
    sprintf(buf, "cpp %s/.lirccrc", homedir);
    nofork = 0;
    while((o = getopt(argc, (char **)argv, "hC:s:f")) >= 0)
    {
	switch(o)
	{
	case 'C':
	    sprintf(buf, "cpp %s", optarg);
	    break;
	case 's':
	    strcpy(sockname, optarg);
	    break;
	case 'f':
	    nofork = 1;
	    break;
	case 'h':
	case '?':
	case ':':
	default:
	    fprintf(stderr, "usage: lirccd [-h] [-C configfile] [-s sockfile]\n");
	    exit(1);
	}
    }
    initcommand();
    regusercommands();
    if((config = popen(buf, "r")) == NULL)
    {
	perror(argv[1]);
	return(1);
    }
    yyrestart(config);
    yyparse();
    if((status = pclose(config)) == -1)
    {
	perror("pclose");
	return(1);
    }
    if(!WIFEXITED(status) || WEXITSTATUS(status))
    {
	fprintf(stderr, "cpp terminated abnormally (%i)\n", WEXITSTATUS(status));
	return(1);
    }
    v = newval();
    newsym("button", v);
    relval(v);
    v = newval();
    newsym("remote", v);
    relval(v);
    v = newval();
    newsym("repeat", v);
    relval(v);
    signal(SIGCHLD, sighandler);
    signal(SIGHUP, sighandler);
    signal(SIGINT, sighandler);
    signal(SIGTERM, sighandler);
    if((init = getfunc("init")) != NULL)
    {
	run(init);
	pop();
	checkstack();
    }
    srand(time(NULL));
    if(!nofork)
    {
	if(fork())
	    exit(0);
    }
    parsecodes();
    return(0);
}
