/*
 * lirccd v0.8 - Middle layer daemon between lircd and its clients
 * Copyright (C) 2003  Fredrik Tolf (fredrik@dolda2000.cjb.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <unistd.h>
#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <glob.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <sys/poll.h>
#include <sys/socket.h>
#include <ctype.h>

#include "command.h"
#include "lirccd.h"

static int popvalues(int num, ...)
{
    int i;
    va_list args;
    struct value **v;
    int type;
    
    va_start(args, num);
    for(i = 0; i < num; i++)
    {
	type = va_arg(args, int);
	v = va_arg(args, struct value **);
	if((*v = pop()) == NULL)
	{
	    fprintf(stderr, "warning: popvalues empty stack\n");
	    return(1);
	}
	if((type != -1) && ((*v)->type != type))
	{
	    fprintf(stderr, "warning: popvalues type mismatch\n");
	    return(1);
	}
    }
    va_end(args);
    return(0);
}

static void echoval(struct value *val, int level)
{
    int i;
    
    for(i = 0; i < level; i++)
	putchar('\t');
    switch(val->type)
    {
    case VAL_NULL:
	printf("(null)\n");
	break;
    case VAL_INT:
	printf("%i\n", val->val.num);
	break;
    case VAL_FLOAT:
	printf("%f\n", val->val.flnum);
	break;
    case VAL_STRING:
	printf("\"%s\"\n", val->val.string.buf);
	break;
    case VAL_ARRAY:
	printf("(array)\n");
	for(i = 0; i < val->val.array.len; i++)
	    echoval(val->val.array.values[i], level + 1);
	break;
    }
}

static void cmd_echo(void)
{
    struct value *v;
    
    if((v = pop()) == NULL)
	return;
    echoval(v, 0);
    relval(v);
    relval(push(newval()));
}

static void cmd_endbinding(void)
{
    endbinding = 1;
    relval(push(newval()));
}

static void cmd_exec(void)
{
    struct value *v;
    int ret;
    
    if(popvalues(1, VAL_STRING, &v))
	return;
    ret = system(v->val.string.buf);
    relval(v);
    v = valmakeint(newval(), WEXITSTATUS(ret));
    relval(push(v));
}

static void cmd_execarray(void)
{
    int i;
    struct value *cmd, *args, *pid;
    unsigned char **cargs;
    int ret;
    struct child *new;
    
    if(popvalues(2, VAL_ARRAY, &args, VAL_STRING, &cmd))
	return;
    if(!(ret = fork()))
    {
	cargs = (unsigned char **)malloc(sizeof(unsigned char *) * args->val.array.len + 1);
	for(i = 0; i < args->val.array.len; i++)
	    cargs[i] = args->val.array.values[i]->val.string.buf;
	cargs[i] = NULL;
	execvp(cmd->val.string.buf, (char * const *)cargs);
	exit(127);
    }
    if(ret > 0)
    {
	new = regchild(cmd->val.string.buf, ret);
	pid = valmakeint(newval(), new->id);
    } else {
	pid = newval();
    }
    push(pid);
    relval(pid);
    relval(cmd);
    relval(args);
}

static void cmd_running(void)
{
    struct value *id;
    struct child *child;
    int running;
    
    if(popvalues(1, VAL_INT, &id))
	return;
    running = 0;
    if((child = getchild(id->val.num)) != NULL)
	running = child->running;
    relval(push(valmakeint(newval(), running)));
}

static void cmd_killchild(void)
{
    struct value *id;
    struct child *child;
    
    if(popvalues(1, VAL_INT, &id))
	return;
    if((child = getchild(id->val.num)) != NULL)
    {
	if(child->running)
	    kill(child->pid, SIGTERM);
    }
    relval(push(newval()));
    relval(id);
}

static void cmd_killchild2(void)
{
    struct value *id, *sig;
    struct child *child;
    
    if(popvalues(2, VAL_INT, &sig, VAL_INT, &id))
	return;
    if((child = getchild(id->val.num)) != NULL)
    {
	if(child->running)
	    kill(child->pid, sig->val.num);
    }
    relval(push(newval()));
    relval(id);
    relval(sig);
}

static void cmd_discardchild(void)
{
    struct value *id;
    struct child *child;
    
    if(popvalues(1, VAL_INT, &id))
	return;
    if((child = getchild(id->val.num)) != NULL)
	freechild(child);
    relval(push(newval()));
}

static void cmd_childnotify(void)
{
    struct value *id, *func;
    struct child *child;
    
    if(popvalues(2, VAL_STRING, &func, VAL_INT, &id))
	return;
    if((child = getchild(id->val.num)) != NULL)
    {
	if(child->notify != NULL)
	    free(child->notify);
	child->notify = strcpy(malloc(strlen(func->val.string.buf) + 1), func->val.string.buf);
    }
    relval(push(newval()));
    relval(func);
    relval(id);
}

static void cmd_expand(void)
{
    int i;
    struct value *str, *del, *arr, *new;
    int start, d;
    
    if(popvalues(2, VAL_STRING, &del, VAL_STRING, &str))
	return;
    d = *(del->val.string.buf);
    relval(del);
    arr = valmakenullarray(newval());
    start = 0;
    for(i = 0; i < str->val.string.len; i++)
    {
	if(str->val.string.buf[i] == d)
	{
	    if(i > start)
	    {
		new = valmakestr2(newval(), str->val.string.buf + start, i - start);
		valcatarray(arr, new);
		relval(new);
	    }
	    start = i + 1;
	}
    }
    if(i > start)
    {
	new = valmakestr2(newval(), str->val.string.buf + start, i - start);
	valcatarray(arr, new);
	relval(new);
    }
    push(arr);
    relval(arr);
    relval(str);
}

static void cmd_file(void)
{
    struct value *v;
    int fd, ret;
    unsigned char *buf;
    
    if(popvalues(1, VAL_STRING, &v))
	return;
    fd = open(v->val.string.buf, O_RDONLY);
    relval(v);
    if(fd < 0)
    {
	relval(push(newval()));
	return;
    }
    v = valmakestr(newval(), "");
    buf = malloc(65536);
    while((ret = read(fd, buf, 65536)) > 0)
	valcatstr(v, buf, ret);
    free(buf);
    close(fd);
    push(v);
    relval(v);
}

static struct value *readdump(int fd)
{
    int i;
    struct value *v;
    
    v = newval();
    read(fd, &v->type, sizeof(v->type));
    switch(v->type)
    {
    case VAL_INT:
	read(fd, &v->val.num, sizeof(v->val.num));
	break;
    case VAL_FLOAT:
	read(fd, &v->val.flnum, sizeof(v->val.flnum));
	break;
    case VAL_STRING:
	read(fd, &v->val.string.len, sizeof(v->val.string.len));
	read(fd, v->val.string.buf = (unsigned char *)malloc(v->val.string.len + 1), v->val.string.len);
	v->val.string.buf[v->val.string.len] = 0;
	break;
    case VAL_ARRAY:
	read(fd, &v->val.array.len, sizeof(v->val.array.len));
	v->val.array.values = (struct value **)malloc(v->val.string.len * sizeof(struct value *));
	for(i = 0; i < v->val.array.len; i++)
	    v->val.array.values[i] = readdump(fd);
	break;
    }
    return(v);
}

static void cmd_rdump(void)
{
    struct value *f, *v;
    int fd;
    
    if(popvalues(1, VAL_STRING, &f))
	return;
    if((fd = open(f->val.string.buf, O_RDONLY)) < 0)
    {
	relval(push(newval()));
	return;
    }
    v = readdump(fd);
    close(fd);
    relval(push(v));
    relval(f);
}

static void dumpval(struct value *v, int fd)
{
    int i;
    
    write(fd, &v->type, sizeof(v->type));
    switch(v->type)
    {
    case VAL_INT:
	write(fd, &v->val.num, sizeof(v->val.num));
	break;
    case VAL_FLOAT:
	write(fd, &v->val.flnum, sizeof(v->val.flnum));
	break;
    case VAL_STRING:
	write(fd, &v->val.string.len, sizeof(v->val.string.len));
	write(fd, v->val.string.buf, v->val.string.len);
	break;
    case VAL_ARRAY:
	write(fd, &v->val.array.len, sizeof(v->val.array.len));
	for(i = 0; i < v->val.array.len; i++)
	    dumpval(v->val.array.values[i], fd);
	break;
    }
}

static void cmd_wdump(void)
{
    struct value *f, *v;
    int fd;
    
    if(popvalues(2, -1, &v, VAL_STRING, &f))
	return;
    if((fd = open(f->val.string.buf, O_WRONLY | O_CREAT, 0666)) < 0)
    {
	relval(push(valmakeint(newval(), 1)));
	return;
    }
    dumpval(v, fd);
    close(fd);
    relval(push(valmakeint(newval(), 0)));
    relval(v);
    relval(f);
}

static void cmd_pipe(void)
{
    struct value *v;
    FILE *stream;
    int ret;
    unsigned char *buf;
    
    if(popvalues(1, VAL_STRING, &v))
	return;
    stream = popen(v->val.string.buf, "r");
    relval(v);
    if(stream == NULL)
    {
	relval(push(newval()));
	return;
    }
    v = valmakestr(newval(), "");
    buf = malloc(65536);
    while((ret = fread(buf, 1, 65536, stream)) > 0)
	valcatstr(v, buf, ret);
    free(buf);
    pclose(stream);
    push(v);
    relval(v);
}

static void cmd_glob(void)
{
    int i, c;
    struct value *v, *v2;
    glob_t globbuf;
    
    if(popvalues(2, VAL_INT, &v2, VAL_STRING, &v))
	return;
    if(glob(v->val.string.buf, GLOB_MARK, NULL, &globbuf))
    {
	relval(v);
	relval(v2);
	relval(push(valmakenullarray(newval())));
	return;
    }
    relval(v);
    v = valmakenullarray(newval());
    c = valbool(v2);
    for(i = 0; i < globbuf.gl_pathc; i++)
    {
	if((globbuf.gl_pathv[i][strlen(globbuf.gl_pathv[i]) - 1] != '/') ^ c)
	    valcatarray(v, valmakestr(newval(), globbuf.gl_pathv[i]));
    }
    push(v);
    relval(v);
    relval(v2);
}

static void cmd_getenv(void)
{
    struct value *v;
    unsigned char *ret;
    
    if(popvalues(1, VAL_STRING, &v))
	return;
    ret = getenv(v->val.string.buf);
    relval(v);
    if(ret == NULL)
    {
	relval(push(newval()));
	return;
    }
    v = valmakestr(newval(), ret);
    push(v);
    relval(v);
}

static void cmd_time(void)
{
    relval(push(valmakeint(newval(), time(NULL))));
}

static void cmd_mktime(void)
{
    struct value *s, *m, *h, *d, *mo, *y;
    struct tm *tm;
    time_t ret;
    
    if(popvalues(6, VAL_INT, &s, VAL_INT, &m, VAL_INT, &h, VAL_INT, &d, VAL_INT, &mo, VAL_INT, &y))
	return;
    ret = time(NULL);
    tm = localtime(&ret);
    if(y->val.num >= 0)
	tm->tm_year = y->val.num;
    if(mo->val.num >= 0)
	tm->tm_mon = mo->val.num;
    if(d->val.num >= 0)
	tm->tm_mday = d->val.num;
    if(h->val.num >= 0)
	tm->tm_hour = h->val.num;
    if(m->val.num >= 0)
	tm->tm_min = m->val.num;
    if(s->val.num >= 0)
	tm->tm_sec = s->val.num;
    relval(y);
    relval(mo);
    relval(d);
    relval(h);
    relval(m);
    relval(s);
    ret = mktime(tm);
    relval(push(valmakeint(newval(), ret)));
}

static void cmd_alarm(void)
{
    struct value *time, *func;
    int ret;
    
    if(popvalues(2, VAL_STRING, &func, VAL_INT, &time))
	return;
    ret = regalarm(time->val.num, func->val.string.buf)->id;
    relval(time);
    relval(func);
    relval(push(valmakeint(newval(), ret)));
}

static void cmd_calarm(void)
{
    struct value *id, *ret;
    struct alarm *cur;
    
    if(popvalues(1, VAL_INT, &id))
	return;
    for(cur = alarms; (cur != NULL) && (cur->id != id->val.num); cur++);
    if(cur != NULL)
    {
	freealarm(cur);
	ret = valmakeint(newval(), 0);
    } else {
	ret = valmakeint(newval(), 1);
    }
    relval(id);
    relval(push(ret));
}

static void cmd_strchr(void)
{
    struct value *s, *c;
    void *t;
    int ret;
    
    if(popvalues(2, VAL_STRING, &c, VAL_STRING, &s))
	return;
    t = memchr(s->val.string.buf, *c->val.string.buf, s->val.string.len);
    if(t == NULL)
	ret = -1;
    else
	ret = (int)t - (int)s->val.string.buf;
    relval(s);
    relval(c);
    relval(push(valmakeint(newval(), ret)));
}

static void cmd_strrchr(void)
{
    struct value *s, *c;
    void *t, *t1, *t2;
    int len, ret;
    
    if(popvalues(2, VAL_STRING, &c, VAL_STRING, &s))
	return;
    ret = 0;
    t = NULL;
    t1 = s->val.string.buf;
    len = s->val.string.len;
    while(1)
    {
	t2 = memchr(t1, *c->val.string.buf, len);
	if(t2 == NULL)
	    break;
	len -= (int)t2 - (int)t1;
	t1 = t2;
	t = t1;
	t1++;
    }
    if(t == NULL)
	ret = -1;
    else
	ret = (int)t - (int)s->val.string.buf;
    relval(s);
    relval(c);
    relval(push(valmakeint(newval(), ret)));
}

static void cmd_strbeg(void)
{
    struct value *s, *p, *v;
    int len;
    
    if(popvalues(2, VAL_INT, &p, VAL_STRING, &s))
	return;
    if((len = p->val.num) >= s->val.string.len)
	len = 0;
    v = valmakestr2(newval(), s->val.string.buf, len);
    push(v);
    relval(s);
    relval(p);
    relval(v);
}

static void cmd_strmid(void)
{
    struct value *s, *p1, *p2, *v;
    int len;
    
    if(popvalues(2, VAL_INT, &p2, VAL_INT, &p1, VAL_STRING, &s))
	return;
    if(p1->val.num >= s->val.string.len)
    {
	v = valmakestr(newval(), "");
    } else {
	if((len = p2->val.num) > s->val.string.len - p1->val.num)
	    len = s->val.string.len - p1->val.num;
	v = valmakestr2(newval(), s->val.string.buf + p1->val.num, len);
    }
    push(v);
    relval(s);
    relval(p1);
    relval(p2);
    relval(v);
}

static void cmd_strend(void)
{
    struct value *s, *p, *v;
    
    if(popvalues(2, VAL_INT, &p, VAL_STRING, &s))
	return;
    if(p->val.num >= s->val.string.len)
	v = valmakestr(newval(), "");
    else
	v = valmakestr2(newval(), s->val.string.buf + p->val.num, s->val.string.len - p->val.num);
    push(v);
    relval(s);
    relval(p);
    relval(v);
}

static void cmd_sendstring(void)
{
    struct value *a, *s;
    struct client *cur;
    struct pollfd pf;
    
    if(popvalues(2, VAL_STRING, &s, VAL_STRING, &a))
	return;
    for(cur = clients; cur != NULL; cur = cur->next)
    {
	if((cur->state == 1) && !strcmp(a->val.string.buf, cur->appname))
	{
	    pf.fd = cur->fd;
	    pf.events = POLLOUT;
	    poll(&pf, 1, 0);
	    if(pf.revents & POLLOUT)
		send(cur->fd, s->val.string.buf, s->val.string.len, MSG_DONTWAIT | MSG_NOSIGNAL);
	}
    }
    relval(a);
    relval(s);
    relval(push(newval()));
}

static void cmd_rand(void)
{
    struct value *m;
    
    if(popvalues(1, VAL_INT, &m))
	return;
    relval(push(valmakeint(newval(), rand() % m->val.num)));
    relval(m);
}

static void cmd_strlower(void)
{
    struct value *s, *n;
    unsigned char *p;
    
    if(popvalues(1, VAL_STRING, &s))
	return;
    n = storeval(newval(), s);
    for(p = n->val.string.buf; p < n->val.string.buf + n->val.string.len; p++)
	*p = tolower(*p);
    relval(s);
    relval(push(n));
}

void regusercommands(void)
{
    addcommand("echo", cmd_echo);
    addcommand("endbinding", cmd_endbinding);
    addcommand("exec", cmd_exec);
    addcommand("execarray", cmd_execarray);
    addcommand("running", cmd_running);
    addcommand("killchild", cmd_killchild);
    addcommand("killchild2", cmd_killchild2);
    addcommand("discardchild", cmd_discardchild);
    addcommand("childnotify", cmd_childnotify);
    addcommand("expand", cmd_expand);
    addcommand("file", cmd_file);
    addcommand("rdump", cmd_rdump);
    addcommand("wdump", cmd_wdump);
    addcommand("pipe", cmd_pipe);
    addcommand("glob", cmd_glob);
    addcommand("getenv", cmd_getenv);
    addcommand("time", cmd_time);
    addcommand("mktime", cmd_mktime);
    addcommand("alarm", cmd_alarm);
    addcommand("calarm", cmd_calarm);
    addcommand("strchr", cmd_strchr);
    addcommand("strrchr", cmd_strrchr);
    addcommand("strbeg", cmd_strbeg);
    addcommand("strmid", cmd_strmid);
    addcommand("strend", cmd_strend);
    addcommand("sendstring", cmd_sendstring);
    addcommand("rand", cmd_rand);
    addcommand("strlower", cmd_strlower);
}
