/********************************************************************************
*                                                                               *
*                        F O X  D B U S  S U P P O R T                          *
*                                                                               *
*********************************************************************************
* Copyright (C) 2007-2009 by Sander Jansen. All Rights Reserved.                *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************
* $Id$                                                                          *
********************************************************************************/
#ifndef FXDBUS_H
#define FXDBUS_H

#ifndef DBUS_BUS_H
#include <dbus/dbus.h>
#endif

/// Global Functions
namespace FX {

/**
* FXDBusConnection is a thin wrapper of DBusConnection with the purpose of integrating it
* into the FOX event loop (1.6), so that activity on DBus will be properly handled.
* DBusConnections may only be managed by one FXDBusConnection. The APIs are strictly enforcing this
* by keeping a global (threadsafe) map of FXDBusConnection/DBusConnection references.
*
* DBus will automatically use FOX's thread functionality.
*
* In the future I hope to support the new FXReactor framework in FOX.
*
*/
class FXDBusConnection : public FXObject {
FXDECLARE(FXDBusConnection)
private:
  DBusConnection * dc;
private:
  FXDBusConnection(const FXDBusConnection &);
  FXDBusConnection &operator=(const FXDBusConnection&);
public:
  enum {
    ID_HANDLE = 1,
    ID_TIMEOUT,
    ID_LAST,
    };
public:
  long onHandleRead(FXObject*,FXSelector,void*);
  long onHandleWrite(FXObject*,FXSelector,void*);
  long onHandleExcept(FXObject*,FXSelector,void*);
  long onTimeOut(FXObject*,FXSelector,void*);
protected:

  /**
  * Dispatch when handle hnd is signaled with mode.
  */
  //virtual FXbool doHandle(FXInputHandle hnd,FXint mode);

  /**
  * Dispatch when timeout expires.
  */
  //virtual FXbool doTimeout(FXTime t);


public:
  /**
  *  Construct non active Dbus Connection Hook
  */
  FXDBusConnection();

  /**
  *  Manage an already existing DBusConnection.
  *  If not owned, FXDBusConnection will increase the ref count of the connection.
  *  return false if already managed by other FXDBusConnection
  */
  FXbool open(DBusConnection*,FXbool owned=true);

  /**
  *  Open Service
  *  return false if already managed by other FXDBusConnection
  */
  FXbool open(const FXString & name,FXbool dedicated=false);

  /**
  *  Open Standard Bus
  *  return false if already managed by other FXDBusConnection
  */
  FXbool open(DBusBusType bustype=DBUS_BUS_SESSION,FXbool dedicated=false);

  /**
  * Return FXDBusConnection for given DBusConnection. Return NULL if not found.
  */
  static FXDBusConnection * find(DBusConnection * dc);

  /**
  * Init the event loop for all connections
  */
  static void initEventLoop();

 /**
  * Return DBUS version
  */
	static FXString dbusversion();

  /**
  * Setup Callback Hooks
  */
  virtual void setup_event_loop();

  /**
  * Return DBusConnection
  */
  DBusConnection * connection() const { return dc; }

  /**
  * Returns whether we're connected or not
  */
  FXbool connected() const;

  /**
  * Returns whether we're  authenticated
  */
  FXbool authenticated() const;


  /**
  * Flush
  */
  void flush();

  /**
  * Send with Reply
  */
  FXbool sendWithReply(DBusMessage * msg,FXint timeout,FXObject*,FXSelector);


  /**
  * Destructor. Existing DBusConnection will be unreffed.
  */
  virtual ~FXDBusConnection();
  };
  
  }

#endif



