/***************************************************************************
 *   Copyright (C) 2007 by Reinhard Tchorz                                 *
 *   mail@rt-sw.de                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <KApplication>
#include <KAboutData>
#include <KCmdLineArgs>
#include <KActionCollection>
#include <KMessageBox>
#include <KFileDialog>
#include <KConfigGroup>
#include <KGlobal>
#include <QDir>
#include <QTextStream>
#include <QGridLayout>
#include <QGroupBox>
#include <QHeaderView>
#include <QCloseEvent>
#include <QInputDialog>
#include <QTimer>
#include <unistd.h>
#include "qdirwatch.h"


using namespace std;



cWatchDir::cWatchDir(QString path, int mask, bool enabled)
{
    watchedFile = NULL;
    if ( (fd = inotify_init()) > 0 ) {
        watchedFile = strdup(path.toAscii().constData());
        bEnabled = enabled;
        user_mask = mask;
        bRecursive = mask & IN_RECURSIVE;
        mask &= ~IN_RECURSIVE;
        if (user_mask & IN_MOVED_FROM)
            user_mask |= IN_UNMOUNT;
        wmask = mask | IN_CREATE | IN_DELETE_SELF;
    }
}


cWatchDir::~cWatchDir()
{
    close(fd);
    free(watchedFile);
}


// Delete all watches. Call from the end of the thread
void cWatchDir::deleteWatches()
{
watchEntry *i;

    foreach(i,dirlist) {
        inotify_rm_watch(fd,i->cookie);
        free(i);
    }
    dirlist.clear();
}


void cWatchDir::errorBox(int error_nb)
{
    error(watchedFile,error_nb);    // send a message to main thread
    deleteWatches();
    bEnabled = false;
}


void cWatchDir::save(KConfigGroup &settings, int k)
{
char key[32];

    sprintf(key,"F%i",k);
    settings.writeEntry(key,QString(watchedFile));
    sprintf(key,"U%i",k);
    settings.writeEntry(key,user_mask);
    sprintf(key,"E%i",k);
    settings.writeEntry(key,bEnabled);
}


// Add file or subdirectory to watch
int cWatchDir::addWatch(const char *name)
{
int c;
watchEntry *entry;

    if ( (c=inotify_add_watch(fd,name,wmask)) >= 0 ) {
        entry = (watchEntry*)malloc(sizeof(entry)+strlen(name)+1);
        entry->cookie = c;
        strcpy(entry->file,name);
        dirlist.push_back(entry);
        return 0;
    }
    return errno;
}


int cWatchDir::addfile(const char *file)
{
struct stat st;
char absfile[512];
const char *pf;

    pf = file;
    if (file[0]!='/') {              // relative path
        strcpy(absfile,pf = get_current_dir_name());
        strcat(strcat(absfile,"/"),file);
        free((void*)pf);
        pf = absfile;
    }
    if (!lstat(pf,&st)) {
        if (S_ISDIR(st.st_mode)) {
            if (bRecursive)         // watch subdirectories?
                return addDir(pf);
            else
                return addWatch(pf);
        }
        else
            if (S_ISREG(st.st_mode))
                return addWatch(pf);
    }
    else
        return errno;
    return 0;
}


int cWatchDir::addDir(const char *name)
{
int r;
DIR *dp;
struct dirent *ep;
struct stat st;
char dir[512];

    if (bStop)  return 0;
    if ( (r=addWatch(name)) != 0 )
        return r;
    if ( (dp=opendir(name)) != NULL ) {
        while( !bStop && (ep = readdir(dp))!=NULL ) {
            sprintf(dir,"%s/%s",name,ep->d_name);
            lstat(dir,&st);
            if (S_ISDIR(st.st_mode)) {
                if (strcmp(ep->d_name,".")!=0  &&  strcmp(ep->d_name,"..")!=0)
                    addDir(dir);
            }
        }
        closedir(dp);
    }
    return 0;
}


// The user changed the wanted events
void cWatchDir::setMask(int mask)
{
bool run = isRunning();
    stop();
    user_mask = mask;
    bRecursive = mask & IN_RECURSIVE;
    mask &= ~IN_RECURSIVE;
    if (user_mask & IN_MOVED_FROM)
        user_mask |= IN_UNMOUNT;
    wmask = mask | IN_CREATE | IN_DELETE_SELF;
    if(run) {       // restart the thread
        wait();
        start();
    }
}


// The user clicks on the left side in the file
// allStoped: monitoring is not enabled
// en: this the single watch is enabled or not
void cWatchDir::setEnabled(bool en, bool allStoped)
{
    if (en != bEnabled) {
        bEnabled = en;
        if (en && !allStoped)
            start();
        else
            stop();
    }
}


// The thread function
void cWatchDir::run()
{
int nr, result;
struct pollfd fds;
struct stat st;
QList<watchEntry*>::iterator i;
inotify_event *pev;
char *fp;

    bStop = false;
    if (fd > 0 && bEnabled) {
        if ( (nr=addfile(watchedFile)) != 0 ) { // if an error occurs, show error and exit
            errorBox(nr);
            return;
        }
        fds.fd = fd;
        fds.events = POLLIN;
        while (!bStop && poll(&fds,1,100)>0) {   // clear cache
            nr = read(fd,buffer,BUFFERSIZE);
        }
        while (!bStop) {
            if ( (result=poll(&fds,1,POLL_TIMEOUT))>0 ) {
                if ( (nr=read(fd,buffer,BUFFERSIZE)) >0 ) {
                    pev = (inotify_event*)buffer;
                    while ( !bStop && nr > 0 )  {
                        for(i=dirlist.begin(); i!=dirlist.end(); ++i)
                            if ((*i)->cookie == pev->wd) {
                                if (user_mask & pev->mask) {
                                    if (pev->len) {
                                        fp = (char*)malloc(strlen((*i)->file)+strlen(pev->name)+2);
                                        sprintf(fp,"%s/%s",(*i)->file,pev->name);
                                        eventWatch(fp,pev->mask);
                                    }
                                    else
                                        eventWatch(strdup((*i)->file),pev->mask);
                                }
                                if (pev->mask & IN_DELETE_SELF) {
                                    dirlist.erase(i);
                                    free(*i);
                                }
                                if (pev->mask & IN_CREATE) {
                                    fp = (char*)malloc(strlen((*i)->file)+strlen(pev->name)+2);
                                    sprintf(fp,"%s/%s",(*i)->file,pev->name);
                                    if (!stat(fp,&st) && S_ISDIR(st.st_mode))
                                        addDir(fp);
                                    free(fp);
                                }
                                break;
                            }
                        nr -= sizeof(inotify_event) + pev->len;
                        pev = (inotify_event*) ( (unsigned char*)pev + sizeof(inotify_event) + pev->len );
                    }
                }
            }
            else if (result<0) {    // error
                errorBox(errno);
                return;
            }
        }
        deleteWatches();
    }
}

// ---------------------------------------------------------------------------------------------


myTreeWidget::myTreeWidget(QWidget *parent) : QTreeWidget(parent)
{
    filter = NULL;
    setItemsExpandable(false);
    setSelectionMode(QAbstractItemView::NoSelection);
    setColumnCount(3);
    setHeaderLabels(QStringList()<<i18n("Time")<<i18n("Event")<<i18n("Filename"));
    icons[redLED] = QIcon(":/redled.png");
    icons[yellowLED] = QIcon(":/yellowled.png");
    icons[greenLED] = QIcon(":/greenled.png");
}

// Add a new entry in the filelist
// Called from cWatchDir
void myTreeWidget::addEvent(const char *file,int mask)
{
static const char *eventName[] = {
    I18N_NOOP("deleted"), I18N_NOOP("write"), I18N_NOOP("overflow"), I18N_NOOP("read"),
    I18N_NOOP("closed"), I18N_NOOP("opened"), I18N_NOOP("attribute"), I18N_NOOP("create"),
    I18N_NOOP("move from"), I18N_NOOP("move to"), I18N_NOOP("renamed"), I18N_NOOP("umounted") };
static int eventMask[] = {
    IN_DELETE|IN_DELETE_SELF, IN_MODIFY, IN_Q_OVERFLOW, IN_ACCESS, IN_CLOSE, IN_OPEN,
    IN_ATTRIB, IN_CREATE, IN_MOVED_FROM, IN_MOVED_TO, IN_MOVE_SELF, IN_UNMOUNT };
int i_icon, i_event;
QTime time;
QTreeWidgetItem *item;

    time = QTime::currentTime();
    if (topLevelItemCount() >= MAX_ITEMS)
        delete topLevelItem(0);
    for(i_event=0; i_event<int(sizeof(eventMask)/sizeof(eventMask[0])); i_event++)
        if(mask & eventMask[i_event]) {
            break;
        }
    if (i_event < int(sizeof(eventMask)/sizeof(eventMask[0]))) {
        i_icon = redLED;
        if (i_event >= 3)
            i_icon = greenLED;
        if (i_event >= 6)
            i_icon = yellowLED;
        item = new QTreeWidgetItem(this);
        item->setText(0,time.toString(Qt::TextDate));
        item->setIcon(0,icons[i_icon]);
        item->setText(1,i18n(eventName[i_event]));
        item->setText(2,QString::fromLocal8Bit(file));
        if (filter && !QDir::match(*filter,file))
            item->setHidden(true);
    }
    free((void*)file);
}


// Save filelist entries as a simple tabbed file.
void myTreeWidget::saveFiles()
{
int i,c;
QString filename;
QTreeWidgetItem *item;

    filename = KFileDialog::getSaveFileName();
    if (!filename.isEmpty()) {
        QFile file(filename);
        if (file.open(QIODevice::WriteOnly | QIODevice::Text)) {
            QTextStream str(&file);
            c = topLevelItemCount();
            for(i=0; i<c; i++)  {
                if ( (item = topLevelItem(i)) != NULL && !item->isHidden())
                    str << item->data(0,Qt::DisplayRole).toString() << "\t" \
                        << item->data(1,Qt::DisplayRole).toString() << "\t" \
                        << item->data(2,Qt::DisplayRole).toString() << "\n";
            }
        }
        else
            KMessageBox::error(this,i18n("Couldn't create %1").arg(filename));
    }
}


void myTreeWidget::filterList(QString &fileFilter)
{
int i,c;
QTreeWidgetItem *item;

    filter = &fileFilter;
    c = topLevelItemCount();
    for(i=0; i<c; i++) {
        item = topLevelItem(i);
        if ( !QDir::match(fileFilter,item->data(2,Qt::DisplayRole).toString()) )
            item->setHidden(true);
    }
}


// Disable file filtering
void myTreeWidget::showAll()
{
int i,c;

    filter = NULL;
    for(c=topLevelItemCount(),i=0; i<c; i++)
        topLevelItem(i)->setHidden(false);
}


// Returns number of visible items
int myTreeWidget::getItemsCount()
{
int c, i;

    if (filter == NULL)
        c = topLevelItemCount();
    else {
        c = 0;
        for(i=topLevelItemCount()-1; i>=0; i--)
            if(!topLevelItem(i)->isHidden())
                c++;
    }
    return c;
}

// ---------------------------------------------------------------------------------------------

MainWindow::MainWindow(QWidget *parent) : KXmlGuiWindow(parent)
{
KAction *action;    // for temp use

    bStop = true;
    bFilter = false;
    undo_WatchedFile = NULL;
    filelist = new myTreeWidget(this);

    iStart = new KIcon(":/start.png");
    iStop = new KIcon(":/stop.png");

    // Add actions
    aFilter = new KAction(KIcon("edit-find"),i18n("Filter"),this);
    aFilter->setShortcut(Qt::CTRL + Qt::Key_F);
    connect(aFilter, SIGNAL(triggered(bool)), this, SLOT(EnterFileFilter()));
    actionCollection()->addAction("find", aFilter);
    action = new KAction(KIcon("edit-clear"),i18n("Clear"),this);
    action->setShortcut(Qt::CTRL+Qt::Key_C);
    connect(action, SIGNAL(triggered()),this,SLOT(clearList()));
    actionCollection()->addAction("clear", action);

    aSort = new KAction(KIcon("view-sort-ascending"),i18n("Sorted"),this);
    aSort->setShortcut(Qt::CTRL+Qt::Key_A);
    aSort->setCheckable(true);
    connect(aSort,SIGNAL(triggered()),this,SLOT(sorted()));
    actionCollection()->addAction("sort", aSort);

    aAlternate = new KAction(i18n("Alternate Colors"),this);
    aAlternate->setCheckable(true);
    connect(aAlternate,SIGNAL(triggered()),this,SLOT(altColors()));
    actionCollection()->addAction("altcolors",aAlternate);

    action = new KAction(KIcon("document-new"),i18n("Add file..."),this);
    action->setShortcut(Qt::CTRL+Qt::Key_P);
    connect( action, SIGNAL(triggered()), this, SLOT(AddFile()) );
    actionCollection()->addAction("addfile",action);

    action = new KAction(KIcon("inode-directory"),i18n("Add directory..."),this);
    action->setShortcut(Qt::CTRL+Qt::Key_D);
    connect( action, SIGNAL(triggered()), this, SLOT(AddDir()) );
    actionCollection()->addAction("adddir",action);

    aUndo = new KAction(KIcon("edit-undo"),i18n("Undo deleted watch"),this);
    aUndo->setShortcut(Qt::CTRL+Qt::Key_Z);
    aUndo->setEnabled(false);
    connect( aUndo, SIGNAL(triggered()), this, SLOT(Undo()) );
    actionCollection()->addAction("undo",aUndo);

    aDel = new KAction(KIcon("edit-delete"),i18n("Delete"),this);
    aDel->setShortcut(Qt::Key_Delete);
    aDel->setEnabled(false);
    connect( aDel, SIGNAL(triggered()), this, SLOT(DelDirFile()) );
    actionCollection()->addAction("delwatch",aDel);

    aStartStop = new KAction(*iStart,i18n("Start"),this);
    aStartStop->setShortcut(Qt::Key_F2);
    aStartStop->setEnabled(false);
    connect( aStartStop, SIGNAL(triggered()), this, SLOT(Start()) );
    actionCollection()->addAction("start",aStartStop);

    KStandardAction::quit(this, SLOT(close()), actionCollection());
    KStandardAction::save(filelist, SLOT(saveFiles()),actionCollection());

//  Read max_user_watches from /proc
    {
    int fd, n;
    char digit[32];
        if ( (fd=open("/proc/sys/fs/inotify/max_user_watches",O_RDONLY)) != 0 ) {
            n=read(fd,digit,sizeof digit);
            ::close(fd);
            max_user_watches = atoi(digit);
        }
    }
    {
    static const char *ckTitle[] = {
        I18N_NOOP("write"), I18N_NOOP("create"), I18N_NOOP("delete"), I18N_NOOP("move"),
        I18N_NOOP("attribute"), I18N_NOOP("open and close"), I18N_NOOP("read"),
        I18N_NOOP("watch subdirectories") };
    QGridLayout *gridLayout1, *panelLayout;
    QGroupBox *groupBox;
    QWidget *panelWidget;
    int events;
        splitter = new QSplitter(Qt::Vertical);
        panelWidget = new QWidget();
        panelLayout = new QGridLayout(panelWidget);
        panelLayout->setSpacing(0);
        panelLayout->setMargin(0);
//      Groupbox Files and Directories
        groupBox = new QGroupBox(i18n("Files and Directories"));
        gridLayout1 = new QGridLayout(groupBox);
        lsDir = new KListWidget(groupBox);
        gridLayout1->addWidget(lsDir, 0, 0, 1, 1);
        connect( lsDir, SIGNAL(itemSelectionChanged ()), this, SLOT(DirSelChanged()) );
        connect( lsDir, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(DirlistClicked(QListWidgetItem*)) );
        panelLayout->addWidget(groupBox, 1, 1);
//      Groupbox Events
        gridLayout1 = new QGridLayout(groupBox=new QGroupBox(i18n("Events")));
        for(events=0; events<eCount; events++) {
            ckEvents[events] = new QCheckBox(i18n(ckTitle[events]),groupBox);
            gridLayout1->addWidget(ckEvents[events], events%4, events/4);
            connect( ckEvents[events], SIGNAL(clicked()), this, SLOT(EventBoxClicked()) );
        }
        panelLayout->addWidget(groupBox, 1, 0);
        splitter->addWidget(filelist);
        splitter->addWidget(panelWidget);
        setCentralWidget(splitter);
        updateEventPanel(IN_MODIFY|IN_CREATE|IN_DELETE|IN_DELETE_SELF|
                          IN_MOVED_FROM|IN_MOVED_TO|IN_MOVE_SELF|IN_RECURSIVE);
        setupGUI();
        status = statusBar();
        for(events=0; events<eStatusLast; events++)
            status->addWidget(statusWidget[events]=new QLabel(),0);
    }
//  Read config data
    {
    int m, i;
    bool en;
    char num[32];
    QString file;
    QHeaderView *header;
    KConfigGroup settings(KGlobal::config(),"settings");

        for(i=0; ; i++) {
            sprintf(num,"F%d",i);
            file = settings.readEntry(num,QString());
            if (!file.isEmpty()) {
                sprintf(num,"U%i",i);
                m = settings.readEntry(num,0);
                sprintf(num,"E%i",i);
                en = settings.readEntry(num,true);
                addNewWatch(file,m,en);
            }
            else
                break;
            lsDir->setCurrentItem(lsDir->item(0));
        };
        header = filelist->header();
        for(i=0; i<2; i++) {
            sprintf(num,"H%d",i);
            header->resizeSection(i,settings.readEntry(num,100));
        }
        {
        QList<int> def;
            def << 4096 << 16;
            splitter->setSizes(settings.readEntry("fh",def));
        }
        fileFilter = settings.readEntry("Filter",QString());
        filelist->setSortingEnabled(settings.readEntry("srt",false));
        filelist->setAlternatingRowColors(settings.readEntry("alt",false));
        aSort->setChecked(filelist->isSortingEnabled());
        aAlternate->setChecked(filelist->alternatingRowColors());
    }
    timer = new QTimer(this);   // used for statusbar update
    connect(timer, SIGNAL(timeout()), this, SLOT(updateStatusBar()));
    timer->start(2000);
    updateStatusBar(eMode);
}


// Enable or disable a watch
void MainWindow::DirlistClicked(QListWidgetItem *item)
{
cWatchDir *watch;

    watch = (cWatchDir*)(item->data(Qt::UserRole).toULongLong());
    watch->setEnabled(item->checkState() == Qt::Checked,bStop);
}


// Show selected events in the eventpanel
void MainWindow::updateEventPanel(int c)
{
int i;

    for(i=eCount-1; i>=0; i--)  {
        ckEvents[i]->setCheckState((c & event_mask[i])? Qt::Checked : Qt::Unchecked);
    }
}


void MainWindow::EventBoxClicked()
{
QListWidgetItem *item;
cWatchDir *watch;

    if ((item=lsDir->currentItem()) != NULL) {
        watch = (cWatchDir*)(item->data(Qt::UserRole).toULongLong());
        watch->setMask(buildMask());
    }
}


void MainWindow::AddDir()
{
    AddFile(true);
}


void MainWindow::AddFile()
{
    AddFile(false);
}


void MainWindow::AddFile(bool bDir)
{
QString file;

    if (bDir)
        file = KFileDialog::getExistingDirectory(QDir::homePath());
    else
        file = KFileDialog::getOpenFileName(QDir::homePath());
    if (!file.isEmpty())
        addNewWatch(file,buildMask(),true);
}


// If selection of the watches list changed, show the corresponding in
// the event panel
void MainWindow::DirSelChanged()
{
QListWidgetItem *item;
cWatchDir *watch;

    aDel->setEnabled(true);
    if ((item=lsDir->currentItem()) != NULL) {
        watch = (cWatchDir*)(item->data(Qt::UserRole).toULongLong());
        updateEventPanel(watch->getMask());
    }
}


// Removes a watched directory or file
void MainWindow::DelDirFile()
{
QList<cWatchDir*>::iterator i;
QListWidgetItem *item;
cWatchDir *watch;

    if ( (item = lsDir->currentItem()) != NULL ) {
        watch = (cWatchDir*)(item->data(Qt::UserRole).toULongLong());
        watch->stop();
        delete item;
        aDel->setEnabled(lsDir->count());
        for(i=watches.begin(); i!=watches.end(); ++i)
            if ( (*i) == watch) {
                watches.erase(i);
                break;
            }
        delete undo_WatchedFile;
        undo_WatchedFile = strdup(watch->getWatchedFile());
        undo_mask = watch->getMask();
        aUndo->setEnabled(true);
        watch->wait();
        delete watch;
    }
}



void MainWindow::Undo()
{
    addNewWatch(undo_WatchedFile,undo_mask,true);
    delete undo_WatchedFile;
    undo_WatchedFile = NULL;
    aUndo->setEnabled(false);
}


void MainWindow::closeEvent(QCloseEvent *event)
{
    saveSettings();
    event->accept();
}


void MainWindow::saveSettings()
{
int c,i;
cWatchDir *wd;
char num[32];
QHeaderView *header;
KConfigGroup settings(KGlobal::config(),"settings");

    hide();
    settings.deleteGroup();
// stop all watch threads
    i = 0;
    foreach(wd,watches) {
        wd->stop();
        wd->save(settings,i);
        i++;
    }
    header = filelist->header();
    for(c=header->count()-1,i=0 ; i<c; i++)  {
        sprintf(num,"H%d",i);
        settings.writeEntry(num,header->sectionSize(i));
    }
    settings.writeEntry("srt",filelist->isSortingEnabled());
    settings.writeEntry("alt",filelist->alternatingRowColors());
    settings.writeEntry("Filter",fileFilter);
    settings.writeEntry("fh",splitter->sizes());
    foreach(wd,watches) {
        wd->wait();
        delete wd;
    }
}



// Build a mask for inotify from the selected events
int MainWindow::buildMask()
{
int i, user_mask = 0;

    for(i=0; i<eCount; i++)
        if (ckEvents[i]->checkState())
            user_mask |= event_mask[i];
    return user_mask;
}


void MainWindow::addNewWatch(const QString& dir, int flags, bool enabled)
{
cWatchDir *watch;
int i;
struct stat st;
QListWidgetItem *item;

    if (!stat(dir.toAscii().constData(),&st)) {
// Check if the path already registered
        for(i=0; i<lsDir->count() ;i++) {
            if ( (item=lsDir->item(i)) != NULL ) {
                if (item->text() == dir) {
                    lsDir->setCurrentItem(item);
                    return;
                }
            }
        }
        watch = new cWatchDir(dir,flags,enabled);
        connect( watch, SIGNAL(eventWatch(const char*,int)), filelist, SLOT(addEvent(const char*,int)) );
        connect( watch, SIGNAL(error(const char*,int)), this, SLOT(errorBox(const char*,int)) );
        watches.push_back(watch);
        item = new QListWidgetItem(QIcon((S_ISDIR(st.st_mode))? ":/folder" : ":/file"),dir,lsDir);
        item->setData(Qt::UserRole,QVariant((unsigned long long)watch));
        item->setCheckState((enabled)? Qt::Checked : Qt::Unchecked);
        lsDir->setCurrentItem(item);
        if (!bStop)
            watch->start();
        aStartStop->setEnabled(true);
    }
}



// Start or stop watching events
void MainWindow::Start()
{
cWatchDir* i;

    bStop = !bStop;
    if (bStop) {
        aStartStop->setText(i18n("&Start"));
        aStartStop->setIcon(*iStart);
    }
    else {
        aStartStop->setText(i18n("&Stop"));
        aStartStop->setIcon(*iStop);
    }
    updateStatusBar(eMode);
    if (bStop)
        foreach(i, watches)
            i->stop();
    else
        foreach(i, watches)
            i->start();
}


void MainWindow::sorted()
{
bool b;

    b = !filelist->isSortingEnabled();
    if (!b)
        filelist->sortItems(0,Qt::AscendingOrder);
    filelist->setSortingEnabled(b);
}


void MainWindow::clearList()
{
    filelist->clear();
    updateStatusBar(eItems);
}


void MainWindow::altColors()
{
bool b;

    b = !filelist->alternatingRowColors();
    filelist->setAlternatingRowColors(b);
}


void MainWindow::EnterFileFilter()
{
bool ok;

    if (bFilter) {
        filelist->showAll();
        aFilter->setText(i18n("&Filter..."));
        bFilter = false;
    }
    else {
        QString tmp = QInputDialog::getText(this,i18n("Filter"),i18n("Same rules as in the shell are valid"),
                        QLineEdit::Normal,fileFilter,&ok);
        if (ok && !tmp.isEmpty()) {
            fileFilter = tmp;
            aFilter->setText(i18n("&Filter off"));
            filelist->filterList(fileFilter);
            bFilter = true;
        }
    }
}



void MainWindow::updateStatusBar(int field)
{
QString label;

    if (status->isVisible()) {
        switch(field) {
            case eMode:
                if(bStop)
                    label = i18n("Watching: stopped");
                else
                    label = i18n("Watching: running");
                break;
            case eItems:
                label = i18n("Events: %1").arg(filelist->getItemsCount());
                break;
            case eInotify:
                {
                int c=0, w=0;
                QListWidgetItem *item;
                cWatchDir* i;
                    foreach(i,watches) {
                        w += i->getWatchesCount();
                    }
                    if ( (item=lsDir->currentItem()) != NULL )
                        c = ((cWatchDir*)(item->data(Qt::UserRole).toULongLong()))->getWatchesCount();
                    label = i18n("%1 watches (total %2, max %3)").arg(c).arg(w).arg(max_user_watches);
                }
                break;
        }
        label = label + "     ";
        statusWidget[field]->setText(label);
    }
}


// called by the timer
void MainWindow::updateStatusBar()
{
    updateStatusBar(eItems);
    updateStatusBar(eInotify);
}


// Errors from a watch thread
void MainWindow::errorBox(const char* path, int code)
{
QList<QListWidgetItem*> fitems;

    KMessageBox::error(this,QString(path)+"\n\n"+strerror(code),i18n("Error"));
    fitems = lsDir->findItems(path,Qt::MatchExactly);
    if (!fitems.empty())
        fitems.at(0)->setCheckState(Qt::Unchecked);
}


// ---------------------------------------------------------------------------------------------

KAboutData aboutData( "kdirwatch", 0,
      ki18n("KDirWatch"), "1.0",
      ki18n("Watched changes of files and directories"),
      KAboutData::License_GPL,
      ki18n("Copyright (c) 2009 R.Tchorz") );


int main(int argc, char *argv[])
{
MainWindow *window;
    aboutData.addAuthor(ki18n("Reinhard Tchorz"), ki18n("Current maintainer"), "tchorz@t-online.de");
    aboutData.setTranslator(ki18nc("NAME OF TRANSLATORS", "Your names"), ki18nc("EMAIL OF TRANSLATORS", "Your emails"));
    KCmdLineArgs::init(argc,argv,&aboutData);
    KApplication app;
    window = new MainWindow();
    window->show();
    app.exec();
    return 0;
}
